#!/bin/bash

# ANSI color codes
COLOR_WARN='\033[0;33m'   # Yellow for warnings
COLOR_ERROR='\033[0;31m'  # Red for errors
COLOR_RESET='\033[0m'     # Reset to default color

declare -A HTTP_ERROR_RESPONSE_STATUS
HTTP_ERROR_RESPONSE_STATUS['400']="Bad Request (400)"
HTTP_ERROR_RESPONSE_STATUS['404']="Not Found (404)"
HTTP_ERROR_RESPONSE_STATUS['408']="Request Timeout (408)"
HTTP_ERROR_RESPONSE_STATUS['421']="Misdirected Request (421)"
HTTP_ERROR_RESPONSE_STATUS['429']="Too Many Requests (429)"
HTTP_ERROR_RESPONSE_STATUS['500']="Internal Server Error (500)"
HTTP_ERROR_RESPONSE_STATUS['502']="Bad Gateway (502)"
HTTP_ERROR_RESPONSE_STATUS['503']="Service Unavailable (503)"
HTTP_ERROR_RESPONSE_STATUS['504']="Gateway Timeout (504)"

## This dictionary will contain the upstream port values that can be configured for a component.
# This will be used to validate the proxymap.txt file while installing.
declare -A VALID_UPSTREAM_PORTS

VALID_UPSTREAM_PORTS['finesse']='8445,443'
VALID_UPSTREAM_PORTS['ids']='8553'
VALID_UPSTREAM_PORTS['cuic']='8444'
VALID_UPSTREAM_PORTS['livedata']='443'
VALID_UPSTREAM_PORTS['cuic_1261']='8444,8447'
VALID_UPSTREAM_PORTS['livedata_1261']='12005,12008'
VALID_UPSTREAM_PORTS['cloudconnect']='8445'
VALID_UPSTREAM_PORTS['idp-adfs3']='443'
VALID_UPSTREAM_PORTS['chat']='5280'

PRINT_FORMAT="%10s|%15s|%40s|%12s|%14s|%20s\n"

# Harded system have the following values
SOMAX=511
KERNEL_MAX_MSG=65536


# Mapping created for test port to original ports
# used in hot reload
declare -A TEST_PORT_TO_ORIGINAL_PORT_MAP

# port counter to get test port numbers for hot_reload
TEST_PORT_COUNTER=19900

function log_info () {
    echo -e "[INFO] $@"
}

function log_warn () {
    echo -e "${COLOR_WARN}[WARN] $@${COLOR_RESET}"
}

function log_error () {
    echo -e "${COLOR_ERROR}[ERROR] $@${COLOR_RESET}" >&2
}

exit_code=1

function exitall() {
    error_code=$1
    if ! [[ "$exit_code" =~ ^[0-9]+$ ]]; then
        # Invalid exit - defalt 1
        exit_code=1
    fi
    kill -USR1 $$
    exit $exit_code
}

trap 'exit $exit_code' USR1

# Store the test port values with the actual port values and store it in a temp file.
# laucher_utilities.sh starts child bash (run.sh) and which internally reads the actual port values configured in component env for the components.
# There is no way we can communicate the parrent shell about the mapping port info, hence redirecting the mapping info into a temp file which is passed while starting the child shell.
# Same file will be used by parrent shell after the child shell exit.
function store_test_port_map_to_file() {
	local tofile
	tofile=$1
	# Iterate over the content of the dictionary
	for key in "${!TEST_PORT_TO_ORIGINAL_PORT_MAP[@]}"; do
		echo "$key=${TEST_PORT_TO_ORIGINAL_PORT_MAP[$key]}">>"${tofile}"
	done
}

# Function to check given value is valid port number
function is_valid_port_value() {
	local port
	port="$1"
	if [[ "$port" =~ ^[0-9]+$ && "$port" -ge 1023 && "$port" -le 65535 ]]; then
		return 0 # Valid port value
	else
		log_error "Invalid port number $port. Exiting."
		exitall 104
	fi
}

# Function to get the free port from the available ports
# If the counter value is already used, get the next available one.
function get_next_free_port() {
	((TEST_PORT_COUNTER++))
	local count
	count=$(netstat -anp | grep -c ":$TEST_PORT_COUNTER ")
	while [ "$count" -ne 0 ]
	do
  		((TEST_PORT_COUNTER++))
	done
	log_info "New test port value $TEST_PORT_COUNTER, checking the port value is valid."
	is_valid_port_value $TEST_PORT_COUNTER
}

# Function to check proxy map file is present or not
function is_proxymap_present() {
	if [[ ! -f  "${PROXYMAP_FILE}" ]]; then
		log_info "proxymap.txt file doesn't present."
		return 1
	fi
	return 0
}

# This function exports value for the given key.
# It is used to configure the test port for the components to start a test container.
function export_key_value() {
	local old_val
	local new_val
	set -o allexport
	old_val=$(eval "echo \$${1}")
	export "${1}=${2}"
	new_val=$(eval "echo \$${1}")
	log_info "Exporting new value for ${1}. Existing value ${old_val}, new value ${new_val}"
	set +o allexport
}

function is_podman_installed() {
    if [ -x "$(command -v podman)" ]; then
        return 0
    fi
    log_error "Cannot find 'podman' in PATH."
    return 1
}



function is_iptables_running() {
    if systemctl is-active --quiet iptables; then
        return 0
    fi
    return 1
}

# Function print dot till given time
# Arg: total time to wait
# Prints dot for every 5 seconds till tital time is complete.
function wait_and_print_dots() {
    local time_to_wait
	time_to_wait=$1
    local time_elapsed
	time_elapsed=0
    local interval
	interval=5
    while [ "$time_elapsed" -le "$time_to_wait" ]; do
    	echo -n "."
    	sleep $interval
    	time_elapsed=$((time_elapsed + $interval))
    done
}

# Get new temp file
# Optional parameter to skip setting the trap on exit
function get_new_temp_file() {
	local filename
	local temp_file=$1
	if [ -z "$1" ]; then
		temp_file=/tmp/openresty.installer.XXXXXX
	fi
	filename=$(mktemp -q ${temp_file}) || { log_error "Failed to create temporary file. Exiting."; exitall 351; }
	if [ $? -ne 0 ]; then
    	log_error "Temp file creation failed. Exiting."
		exitall 101
	fi
	echo "${filename}"
}


function get_component() {
	echo "$1" | sed "s/livedata/ld/g" | sed "s/finesse/fin/g" | sed "s/idp-adfs3/idp/g" |sed 's/_.*//' | sed 's/_//' | tr '[:lower:]' '[:upper:]'
}

function is_nslookup_present() {
    if command -v nslookup &>/dev/null; then
        return 0
    else
        return 1
    fi
}

# Function to check nslookup is present.
# If not alert user that some checks may not work without the command
function check_nslookup_or_get_user_confirmation() {
    if is_nslookup_present; then
        return 0
    else
       get_user_confirmation_or_exit_with_error "${1}" "${2}" "${3}"
    fi
}


# Validate two of the kernel flags set as part of hardening, if they are not updated kernel hardening was not performed.
function check_kernel_hardend_or_get_user_confirmation() {
    local somax=$(sysctl net.core.somaxconn | awk '{print $3}')
    local kernel_msg_max=$(sysctl kernel.msgmax | awk '{print $3}')
    if [[ $somax -ge $SOMAX  &&  $kernel_msg_max -ge $KERNEL_MAX_MSG ]]; then
        return 0
    else
        get_user_confirmation_or_exit_with_error "${1}" "${2}" "${3}"
    fi
}

# Validate disk io speed is more than the recommended, else give warning.
function check_diskspeed_or_get_user_confirmation() {
    local output
    local speed
    local unit
    local scale_factor
    local threshold_speed
    scale_factor=${4}
    threshold_speed_mb=$((2 * scale_factor))
    # get the speed of the disk i/o
    output=$(dd if=/dev/zero of=/root/junk bs=512 count=1000 oflag=dsync 2>&1)
    # extract speed value eg 3.4 MB/s 424 kB/s 1.0 GB/s
    speed=$(echo "$output" | grep -o '[0-9.]* [A-Za-z]B/s' | awk '{print $1}')
    unit=$(echo "$output" | grep -o '[0-9.]* [A-Za-z]B/s' | awk '{print $2}')

    # convert based on unit
    if [ "$unit" == "kB/s" ]; then
        speed=$(echo "$speed / 1000" |bc -l)
    elif [ "$unit" == "GB/s" ]; then
        speed=$(echo "$speed * 1000" | bc -l )
    fi


    # we have now in MB/s value
    log_info "Current disk speed ${speed}MB/s and recommened speed is ${threshold_speed_mb}MB/s"
    
    result=$(echo "$speed >= $threshold_speed_mb" | bc -l)
    if [ "$result" -eq 1 ]; then
        # all good
        return 0
    else
        get_user_confirmation_or_exit_with_error "${1}" "${2}" "${3}"
    fi
}

# Generic function to get user confirmation and exit if N is presssed.
function get_user_confirmation_or_exit_with_error() {
    if get_user_confirmation "${1}" "${2}"; then
        return 0
    else
        log_error "${3}"
        exitall 291
    fi
}

# Function to purge old log files.
# Incase the log file is so huge eg: more than 5-10GB, it will not be rotated by log rotator properly.
# Hence, before starting we can purge the old logs.
function purge_huge_logfiles() {
    if [ ! -e "${1}" ]; then
        # File not exists
        return 0
    fi
    # Take backup of last 1lakh lines before purge
    LAST_N_LINES=100000
    # 2GB Threshold
    SIZE_THRESHOLD=2048
    SIZE_IN_MB=$(( $(stat -c%s ${1}) / 1024 / 1024 ))
    log_info "${1} file size is ${SIZE_IN_MB}MB"
    if [[ (! -z $SIZE_IN_MB) && ($SIZE_IN_MB -gt $SIZE_THRESHOLD) ]]
    then
        log_info "Size of the file is greater than ${SIZE_THRESHOLD}MB, truncating file to have last $LAST_N_LINES lines only."
        TMP_FILE=$(mktemp /tmp/logfile.XXXX) || { log_error "Failed to create temporary file."; exitall 351; }
        echo "Temp file name is $TMP_FILE"
        tail -$LAST_N_LINES ${1} > $TMP_FILE
        cat $TMP_FILE > ${1}
        # :? is needed for all rm -rf commands as a preventive measure to not to delete the entire filesystem if the value is empty.
        rm -rf ${TMP_FILE:?}
    fi
}

# Generic function to get user confirmation for the given prompt
# arg1: Prompt message
# arg2: true if auto confirmation is enabled
# returns y or n for the prompt
function get_user_confirmation() {
    local response=""
    local prompt="${1:-Are you sure? [y/N]}"
    if [[ "$2" == "true" ]]; then
        return 0
    fi
    # Display the multiline prompt message using the $'...' syntax for newlines
    echo -e -n "$COLOR_WARN"    # Display the promt in color text
    read -r -p $"${prompt}" userInput
    echo -e -n "$COLOR_RESET"
    # lower case
    userInput="${userInput,,}"
    # Check if the user entered "yes" or "y"
    if [[ "${userInput}" == "yes" || "${userInput}" == "y" ]]; then
        return 0
    else
        return 1
    fi
}

# Function to delete the temporary file
function cleanup() {
    rm -f "${1}"
}

# function to print troubleshoot headers
# arg1: header format with separator "|"
function print_header() {
	printf "${PRINT_FORMAT}" "Test Type" "Component" "Host" "Status" "Latency (sec)" "Description"
}

# Function builds the error description using the http response code
# arg1: http response code
function build_description() {
	local message=""
    # For external CC, there is no publically available API to check health, using task api, which will return 401 response, that is fine to see the status of CC.
	if [[ "${1}" == "200" || "${1}" == "STATE_IN_SERVICE" || "${1}" == "IN_SERVICE" || "${1}" == "401" ]]; then
		message="OK"
	elif [ "${1}" == "OUT_OF_SERVICE" ]; then
		message="${1}"
	else
		if [[ -z "${1}" || -z "${HTTP_ERROR_RESPONSE_STATUS[$1]}" ]]; then
			message=${HTTP_ERROR_RESPONSE_STATUS['408']}
		elif [[ "${1}" != "200" ]]; then
			message=${HTTP_ERROR_RESPONSE_STATUS["${1}"]}
		fi
	fi
	echo $message
}

# function to get the http code from curl response 
# the curl response is a combination of http_code & total_time separated by ','
# arg1: curl response as string
function get_http_status() {
	echo "$( echo "${1}" | awk -F, '{print $1}' )"
}

# function to get the total time from curl response 
# the curl response is a combination of http_code & total_time separated by ','
# arg1: curl response as string
function get_response_time() {
	echo "$( echo "${1}" | awk -F, '{print $2}' )"
}

# function to print the parameters corresponding to their headers
# arg1: test type
# arg2: http response status
# arg3: component name
# arg4: component hostname
# arg5: total response time
# arg6: status description
function print_response() {
	local description=${6}
	if [ -z "${description}" ]; then
		description=$(build_description ${2})
	fi
    # We are checking the reachability check, hence 401 is reachable.
	if [[ "${2}" == "200" || "${2}" == "STATE_IN_SERVICE" || "${2}" == "IN_SERVICE" || "${2}" == "401" ]]; then
        printf "${PRINT_FORMAT}" "${1}" "${3}" "${4}" "ACTIVE" "${5}" "${description}"
	else
    	printf "${PRINT_FORMAT}" "${1}" "${3}" "${4}" "NOT ACTIVE" "${5}" "${description}"
    fi
}


function get_prop_val {
	# ignore commented lines. xargs trims the whitespaces.
	grep "${1}" "${2}" | grep -o '^[^#]*' | cut -d '=' -f2 | xargs
}


#	The function get_prop_val should be replaced with this implementation,
#	and the usages should be updated. This method doesn't require the caller
#	to check if the property is inside a comment, and only returns the value
#	if the full property name matches. i.e., if there are two properties,
#	prop_a and prop_ab, this method returns the value of each property properly
#	while get_prop_val returns the values of both the properties when
#	called with get_prop_val "prop_a".
function get_prop_val_clean {
	grep "^[\s]*${1}=" "${2}" | cut -d '=' -f2 | xargs
}


# Returns the value without any spaces
# If the value is 1, 0.2, 3.4 -> return 1,0.2,3.4
function get_prop_val_space_stripped {
    local out
    out=$(get_prop_val "${1}" "${2}")
    echo $out | tr -d ' '
}

function check_dir_exist_or_exit() {
	if [ ! -d "${1}" ]; then
		log_error "Directory ${1} does not exist. Exiting"
		exitall 101
	fi
}

function check_file_exist_or_exit() {
	if [ ! -f "${1}" ]; then
		log_error "File ${1} does not exist. Exiting"
		exitall 102
	fi
}

#Function to check if an env file is readable and writable
function is_file_readable_and_writable() {
    local file="$1"
    check_file_exist_or_exit "$file"
    if [ ! -r "$file" ]; then
        log_error "File ${file} is not readable. Exiting"
    fi
    if [ ! -w "$file" ]; then
        log_error "File ${file} is not writable. Exiting"
    fi
}

create_lb_configurations() {
	local ip_list
	local real_ip_header
	local file
	ip_list="${1}"
	real_ip_header="${2}"
	file="${3}"
	echo "" > "${file}"

	IFS='| ' read -r -a ips <<< "$ip_list"
	for ip in "${ips[@]}"
	do
		if [[ "$ip" =~ ^[0-9]+\.[0-9]+\.[0-9]+\.[0-9]+$ ]]; then
			echo "set_real_ip_from ${ip};" >> "${file}"
		else
			logInfo "[ERROR] NGX_LOAD_BALANCER_IPS should contain only IP addresses. Exiting."
			exitall 108
		fi
	done
	echo "real_ip_header ${real_ip_header};" >> "${file}"
	echo "real_ip_recursive on;" >> "${file}"
}


function parse_ips_and_create_list() {
	local ip_list
	local out_file
	local value
	ip_list="${1}"
	out_file="${2}"
	value="${3}"
	# Split trsuted hosts property value on | delimiter and add to file
    IFS='| ' read -r -a ips <<< "$ip_list"
    for ip in "${ips[@]}"
    do
		echo "${ip} \"${value}\";" >> "${out_file}"
	done
}

function is_bc_installed() {
    if command -v bc &> /dev/null; then
        return 0
    else
        log_error "bc command is not installed. Exiting."
        exitall 118
    fi
}

function is_route_installed() {
    if command -v route &> /dev/null; then
        return 0
    else
        log_error "route command is not installed. Exiting."
        exitall 118
    fi
}

function is_ip_installed() {
    if command -v ip &> /dev/null; then
        return 0
    else
        log_error "ip command is not installed. Exiting."
        exitall 118
    fi
}


# Function that checks enternal nic configured has multiple IPs in it
function nic_has_single_ip() {
    local nic
	local ips
	local count
	nic="$1"
    local ips
	ips=$(ip addr show dev "$nic" | grep 'inet ' | awk '{print $2}')
    count=$(echo "$ips" | wc -l)
    if [ "$count" -eq 1 ]; then
       return 0
    fi
	return 1
}

function is_ip_part_of_the_nic() {
	local nic
	local ip
	local count
	nic="$1"
	ip="$2"
	count=$(ip addr show "$1" | grep 'inet ' |grep "$2" | wc -l)
	if [ "$count" -eq 1 ]; then
       return 0
    fi
	return 1
}

# Function to get the IP address of the given nic
function get_nic_ip() {
    local nic
	local ip_address
	nic=$1
	ip_address=$(ip addr show dev "$nic" | grep 'inet ' | awk '{print $2}' | cut -d'/' -f1)
    if [ -z "${ip_address}" ]; then
		log_error "No IP address configured on the nic ${nic}. Exiting"
        exitall 259
    fi
    echo "${ip_address}"
}

function get_external_nic() {
    source /etc/reverse-proxy/network.env
    echo "$EXTERNAL_INTERFACE"
}

function get_external_ip() {
		local external_nic=$(get_external_nic)
    local external_ip="${PROXY_BINDING_IP}"
    if [ -z "${PROXY_BINDING_IP}" ]; then
      if nic_has_single_ip "${external_nic}"; then
        external_ip=$(get_nic_ip "${external_nic}")
      else
        log_error "External nic configured has multiple IPs. Please provide a valid IP to bind from this list as the value for PROXY_BINDING_IP in installer.env configuration file."
          exitall 260
      fi
    else
      if ! is_ip_part_of_the_nic "${external_nic}" "${external_ip}"; then
        log_error "PROXY_BINDING_IP is not configured on the external nic configured. Exiting."
        exitall 260
      fi
    fi
    echo "${external_ip}"
}

function get_count_of_internal_ips() {
    local external_nic=$(get_external_nic)
    local count
    count=$(ip -o -4 addr show | awk '{print $2, $4}' | grep -v "$external_nic" | grep -v "lo" | grep -v "podman" | wc -l)
    echo "${count}"
}

function get_private_ip() {
  local external_nic=$(get_external_nic)
	local private_ip="${PROXY_BINDING_INTERNAL_IP}"
	if [ -z "${PROXY_BINDING_INTERNAL_IP}" ]; then
	  local internal_ip_count=$(get_count_of_internal_ips)
	  if [ "$internal_ip_count" -gt 1 ]; then
      log_error "Multiple IPs are available in the system. Please provide the exact IP to bind to in PROXY_BINDING_INTERNAL_IP property, in installer.env configuration file."
      exitall 260
    else
      private_ip=$(ip -o -4 addr show | awk '{print $2, $4}' | grep -v "${external_nic}" | grep -v "lo" | grep -v "podman" | awk '{print $2}' | cut -d'/' -f1)
    fi
  fi
	echo "${private_ip}"
}

# Function to get the IP address of the given hostname using the given DNS server list
function get_upstream_ip() {
	local hostname
	local dns_entries
	local output
	hostname=$1
	dns_entries=$2

	IFS='| ' read -r -a items <<< "$dns_entries"

	# Iterate over each item
	for dns_server in "${items[@]}"; do
		output=$(nslookup "${hostname}" "${dns_server}" | grep Address | tail -n1 | awk '{print $2}')
		if [[ ! -z ${output} ]]; then
			echo "${output}"
			return 0
		fi
	done
  return 1
}

function log_to_file() {
    local log_file
    local log_message
    log_message=$1
    log_file=$2
    echo `date "+%m/%d/%Y %H:%M:%S :"` "${log_message}" >> "${log_file}"
}

function is_selinux_enabled() {
    local selinux_status=$(getenforce)
 	if [ "$selinux_status" != "Disabled" ]; then
   		return 0
    fi
    return 1
}

function is_container_selinux_installed() {
    local package="container-selinux"
  	if rpm -q "$package" &>/dev/null; then
    	return 0
    fi
    return 1
}
