#!/bin/bash

####################################################################################################
# This file contains all the other validations that has to be done after completeing the upgrade   #
# Initial validations can be added in primary_validation_utils.sh file.						       #
####################################################################################################

NOFILE_LIMIT_2K=102400
CPU_LIMIT_2K=2
MEM_LIMIT_2K_IN_GB=4
MEM_SWAP_LIMIT_2K_IN_GB=8
MEM_RES_2K_IN_GB=2

function is_same_container_image {
	log_info "Validating is the podman images are same? $1"
	local existing_image_name
	local image_name_from_env
	# Extract from the image name "ImageName": "localhost/reverse-proxy-openresty-container:15.0.1-SNAPSHOT",
	# Previously installer.env had only reverse-proxy-openresty-container:15.0.1-SNAPSHOT without "localhost"/reverse-proxy-openresty-container:15.0.1-SNAPSHOT
	# Now, installer.env itself has image name prepended with localhost.
	# Podman inspect will have the image name starts with "localhost/" if it is loaded with localhost.
	existing_image_name=$(podman inspect "$1" | grep "ImageName" | awk -F': ' '/ImageName/ {print $2}' |  tr -d '"' |  tr -d ',')
	image_image_name_from_env="${2}"

	if [ "${existing_image_name}" == "${image_image_name_from_env}" ]; then
		log_info "Same image name given, proceeding further."
		return 0
	fi

	log_error "To use hot reload, the running container image should be exactly the same as the image being started up. Exiting."
	exitall 204
}

# Check container is running and image name is same to reload
# Else exit with error
function is_valid_container_to_reload() {
	local real_container_name
	local container_image_name
	real_container_name=$1
	container_image_name=$2
	if ! is_container_running "$real_container_name" ; then
		log_error "Container $real_container_name is not running, hence cannot be reloaded. Exiting."
		exitall 209
	fi
	if ! is_same_container_image "$real_container_name" "$container_image_name" ; then
		log_error "The podman image used to reload $real_container_name different, hence cannot be reloaded. Exiting"
		exitall 210
	fi
}


# Function to check the given container is running or not.
# Print the log file accordkingly
function check_test_container_not_running() {
	local test_container_name
	test_container_name=$1
	if is_container_running "$test_container_name"; then
		log_error "Test container with name [$test_container_name] is already running, stop it and try reload. Exiting."
		exitall 209
	fi
}


function is_container_running() {
	local container_id
	container_id="$1"
	log_info "Checking the container name ${container_id} is running.."
	if [ "$( podman container inspect -f '{{.State.Status}}' "$container_id" )" = "running" ]; then
		log_info "Container ${container_id} is running."
		return 0
	fi
	log_info "Container ${container_id} is NOT running."
	return 1
}

# Function to suggest and get user confirmation for the installation
function suggest_podman_resource_params() {
	check_no_file_limit
	local max_agents
	max_agents=$(nearest_2k $MAX_NUMBER_OF_AGENTS)
	# For now check these validations only for 2k, 4k and 6k setup
	if [[ $max_agents -gt 6000 ]]; then
		log_warn "Profile with more than 6000 agents are untested."
		return 0
	fi
	check_host_resources
	check_container_configurations
}


## Weightage
# IdS		0.11
# IdP		0.01
# Finesse	0.60
# CUIC		0.15
# LD		0.11
# CC		0.01
# Chat		0.01
# If a container configured with 2K agents + one set of all these will give total weight age of 1.

IDS_WEIGHT=0.11
IDP_WEIGHT=0.01
FINESSE_WEIGHT=0.60
CUIC_WEIGHT=0.15
LD_WEIGHT=0.11
CC_WEIGHT=0.01
CHAT_WEIGHT=0.01
AGENTS_COUNT_WEIGHT=0.0005
# When number of components or agents are increased, add this 15% of scalling buffer as well to find the hardward requirement.
SCALING_BUFFER=0.15

declare -A COMPONENTS_COUNT
COMPONENTS_COUNT['FIN']=0
COMPONENTS_COUNT['CUIC']=0
COMPONENTS_COUNT['LD']=0
COMPONENTS_COUNT['IDS']=0
COMPONENTS_COUNT['IDP']=0
COMPONENTS_COUNT['CLOUDCONNECT']=0
COMPONENTS_COUNT['CHAT']=0

# Function to calculate the factor based on weights of each components and number of such components being installed.
# For default 2k setup and want to install for 2000 agents and one component per each type, we require 2 CPU and 4GB RAM - the output factor for such use case is 1.
# Incase if the 4000 agents and 2 Ids + 2 finesse + 2 LD is installed then factor will be 3.08, rounding off value will be 3, then our 6CPU and 12GB RAM is recommended.
# The Weightage of the components are based on the current understanding of the component loads in our dev and SA setup. It can be finetuned later.
function get_scale_factor() {
    local ids
    local idp
    local finesse
    local cuic
    local ld
    local cc
    local chat
	local max_agents

	ids=${COMPONENTS_COUNT['IDS']}
	idp=${COMPONENTS_COUNT['IDP']}
	finesse=${COMPONENTS_COUNT['FIN']}
	cuic=${COMPONENTS_COUNT['CUIC']}
	ld=${COMPONENTS_COUNT['LD']}
	cc=${COMPONENTS_COUNT['CLOUDCONNECT']}
	chat=${COMPONENTS_COUNT['CHAT']}
	max_agents=$(nearest_2k $MAX_NUMBER_OF_AGENTS)
	
	# Calculate the factor
    local factor=$(echo "$IDS_WEIGHT * $ids + $IDP_WEIGHT * $idp + $FINESSE_WEIGHT * $finesse + $CUIC_WEIGHT * $cuic + $LD_WEIGHT * $ld + $CC_WEIGHT * $cc + $CHAT_WEIGHT * $chat" | bc)
	factor=$(echo "$factor * $AGENTS_COUNT_WEIGHT * $max_agents" | bc)
	# Check if the factor is greater than 1, and add 15% if true
    if (( $(echo "$factor > 1" | bc -l) )); then
        factor=$(echo "$factor + ($factor * ${SCALING_BUFFER})" | bc)
    fi
	factor=$(echo "if ($factor > 1) $factor else 1" | bc)
	factor=$(printf "%.0f" "$factor")
    echo ${factor}
}


## Weightage for max_conns
# IdS		0.02
# IdP		0.02
# Finesse	1.85
# CUIC		TBA
# LD		TBA
# CC		TBA
# If a container configured with 2K agents + one set of all these will give total weight age of 1.
declare -A MAX_CONNS_WEIGHT
MAX_CONNS_WEIGHT['IDS']=0.03
MAX_CONNS_WEIGHT['IDP']=0.1
MAX_CONNS_WEIGHT['FIN']=1.85
MAX_CONNS_WEIGHT['CUIC']=0		#TODO : update after cuic and LD validations
MAX_CONNS_WEIGHT['LD']=0
MAX_CONNS_WEIGHT['CLOUDCONNECT']=0
MAX_CONNS_WEIGHT['CHAT']=0

# Derive the number of max connection for the given component based on the weight assiged above
function get_max_conns_for_component() {
	local component_name
	component_name=${1}
	local max_conns
	local env_dir
	env_dir=${2}

	max_agents=$(nearest_2k $MAX_NUMBER_OF_AGENTS)

	local max_conn_factor=${MAX_CONNS_WEIGHT["${component_name}"]}
	
	local ids_factors=$(get_ids_idp_max_conns_factor "${component_name}" "${env_dir}")

	max_conns=$(echo "$max_conn_factor * $max_agents * $ids_factors" | bc)
	
	if (( $(echo "${max_conns} > ${NGX_NUM_WKR_CONN}" | bc -l) )); then
		max_conns=$(echo "if (${max_conns} > ${NGX_NUM_WKR_CONN}) 0 else $max_conns" | bc)
	fi
	max_conns=$(printf "%.0f" "${max_conns}")
	echo ${max_conns}
}

# To get the number non IDS and IDP component the ids has to support
function get_ids_idp_max_conns_factor() {
	local component=${1}
	local env_dir=${2}
	if [[ "$component" != "IDS" || "$component" != "IDP" || "$component" == "CHAT" ]]; then
		echo 1
		return 0
	fi
	local total_components=0
	for ENV in "${env_dir}"/*.env; do
		if [ "${ENV}" == "${ENV_DIR}/core.env" ] || [ "${ENV}" == "${ENV_DIR}/dirs.env" ]; then
			continue
		else
			local template_type=$(get_prop_val '^TEMPLATE_TYPE' ${ENV})
			if [[ "$template_type" == "ids" || "$template_type" == "idp-adfs3" || "$key" == "chat" ]]; then
				continue
			fi
			total_components=$((total_components + 1))
		fi
	done
	total_components=$(echo "if (${total_components} < 1) 1 else $total_components" | bc)
	echo $total_components
}


# function get the nearest 2000.
function nearest_2k() {
    local number
	number=$1
	
    if [ -z "$number" ]; then
		echo 2000
		return 0
	fi
	if [ "$(echo $number | tr -d '[:digit:]' | xargs)" == "" ]; then
		echo 2000
		return 0
	fi

    local remainder=$(( number % 2000 ))

    if [ "$remainder" -ge 1000 ]; then
        echo $(( number + 2000 - remainder ))
    else
        echo $(( number - remainder ))
    fi
}

# For 2k setup, expects 2vCVP and 4GB RAM
# for 4k setup it should be double, factor is 2
function check_host_resources() {
	local factor
	factor=$(get_scale_factor)
	local cpu_limit
	cpu_limit=$((CPU_LIMIT_2K * factor))
	# round off to nearest int
	cpu_limit=$(printf "%.0f" "$cpu_limit")
	local mem_limit
	mem_limit=$((MEM_LIMIT_2K_IN_GB * factor))
	# round off to nearest int
	mem_limit=$(printf "%.0f" "$mem_limit")
	local available_cpus
	available_cpus=$(nproc)
	# get free memory in GB
	local available_mem
	available_mem=$(free -h | awk '/^Mem:/ {sub(/[Gg][i]*/, "", $2); print $2}')
	available_mem=$(printf "%.0f" "$available_mem")

	if [[ $available_cpus -lt 2 || $available_mem  -lt 4 || $available_cpus -lt $cpu_limit || $available_mem -lt $mem_limit ]]; then
		log_error "Not enough resources configured for the container. Available CPUs $available_cpus, Available Memeory ${available_mem}GB. Recommended to have minimum ${cpu_limit}vCPU and ${mem_limit}GB of available memeory. Exiting."
		exitall 214
	fi
}

# Validate the container resource configurations and check the values are within the limit for the number of agents configured + number of components.
function check_container_configurations() {
	local factor
	local mem_limit
	local cpu_limit
	local swap_limit
	local mem_res_limit
	local nofile_limit
	
	local recommended_cpu
	local recommended_mem
	local recommended_res
	local recommended_swap_limit
	local recommended_nofile

	factor=$(get_scale_factor)
	mem_limit=$(echo $MEM_LIMIT | tr -d '[:alpha:]')
	cpu_limit=${CPU_LIMIT%.*}
	swap_limit=$(echo $MEM_SWAP_LIMIT | tr -d '[:alpha:]')
	mem_res_limit=$(echo $MEM_RES | tr -d '[:alpha:]')
	nofile_limit=$(get_current_nofile_limit)
	recommended_cpu=$((CPU_LIMIT_2K * factor))
	recommended_mem=$((MEM_LIMIT_2K_IN_GB * factor))
	recommended_res=$((MEM_RES_2K_IN_GB * factor))
	recommended_swap_limit=$((MEM_SWAP_LIMIT_2K_IN_GB * factor))
	recommended_nofile=$((NOFILE_LIMIT_2K * factor))

	if [[ $cpu_limit -gt $recommended_cpu || $mem_limit -gt $recommended_mem || $mem_res_limit -gt $recommended_res || $swap_limit -gt $recommended_swap_limit || $nofile_limit -gt $recommended_nofile ]]; then
		check_configs_and_get_user_confirmation "${recommended_cpu}" "${recommended_mem}" "${recommended_swap_limit}" "${recommended_res}" "${recommended_nofile}" "more"
	elif [[ $cpu_limit -lt $recommended_cpu || $mem_limit -lt $recommended_mem || $mem_res_limit -lt $recommended_res || $swap_limit -lt $recommended_swap_limit || $nofile_limit -gt $recommended_nofile ]]; then
		check_configs_and_get_user_confirmation "${recommended_cpu}" "${recommended_mem}" "${recommended_swap_limit}" "${recommended_res}" "${recommended_nofile}" "less"
	fi
}
