#!/bin/bash

####################################################################################################
# This file contains all the initial validations that has to be done before starting the installer.#
# Other validation utils can be added in other_validation_utils.sh file.						   #
####################################################################################################

CURRENT_SCRIPTPATH=$(dirname "${0}")

## This dictionary will contain the upstream port values that can be configured for a component.
# This will be used to validate the proxymap.txt file while installing.
declare -A VALID_UPSTREAM_PORTS

VALID_UPSTREAM_PORTS['finesse']='8445,443'
VALID_UPSTREAM_PORTS['ids']='8553'
VALID_UPSTREAM_PORTS['cuic']='8444'
VALID_UPSTREAM_PORTS['livedata']='443'
VALID_UPSTREAM_PORTS['cuic_1261']='8444,8447'
VALID_UPSTREAM_PORTS['livedata_1261']='12005,12008'
VALID_UPSTREAM_PORTS['cloudconnect']='8445'
VALID_UPSTREAM_PORTS['idp-adfs3']='443'
VALID_UPSTREAM_PORTS['chat']='5280'


# Skips installer checks for the below variables.
# Some variables which are not default present in master_env can be enabled in sample_env
# Eg: NGX_SSL_DHPARAM
NGX_SKIP_INSTALLER_CHECK_VARIABLES=("NGX_SSL_DHPARAM" "CONFIG_VERSION" "PROXY_BINDING_IP" "PROXY_BINDING_INTERNAL_IP" "NGX_CONFIGURED_COMPONENTS_LIST" "NGX_FIN_USERS_URL")

CONFIGS_DIR=${CURRENT_SCRIPTPATH}/reverse-proxy-openresty-configs/configs
REVERSE_PROXY_OPENRESTY_CONFIGS_DIR=${CURRENT_SCRIPTPATH}/reverse-proxy-openresty-configs/
TEMPLATE_DIR=${CURRENT_SCRIPTPATH}/reverse-proxy-openresty-configs/configs/templates
MASTER_ENV_DIR=${CURRENT_SCRIPTPATH}/reverse-proxy-openresty-configs/configs/master_envs
MAIN_INSTALLER=${CURRENT_SCRIPTPATH}/reverse-proxy-openresty-configs/configs/installer.env


function check_proxymap_has_no_duplicate_entries() {
	local proxymapfile
	proxymapfile=${1}
	if ! is_proxymap_present ; then
		return 0
	fi
	# Check if there are any duplicate entry on the proxy map
	local count
	count=$(grep -o '^[^#]*' "${proxymapfile}"  | cut -d '=' -f1 | sed 's/:443$//' |  uniq -c | sort -r | head -n1 | xargs | cut -d ' ' -f1 | xargs)
	if [ -z $count ]; then
		return 0
	fi

	if [[ $count -gt 1 ]]; then
		local duplicate_upstream
		duplicate_upstream=$(grep -o '^[^#]*' "${proxymapfile}"  | cut -d '=' -f1 | sed 's/:443$//' |  uniq -c | sort -r | head -n1 | xargs | cut -d ' ' -f2 | xargs)
		log_error "There are more than one entry populated for this upstream entry ${duplicate_upstream}. Exiting."
		exitall 201
	fi
}


function check_ld_failover_hosts_reachable() {
	local component_type
	local ENV
	local container_dns_resolver
	component_type="$1"
	ENV="$2"
	container_dns_resolver="$3"
	if [[ "${component_type}" != "livedata"* ]]; then
		return 0
	fi

	local ld_backed
	ld_backed=$(get_prop_val "^NGX_LD_BACKEND_FAILOVER" "${ENV}")
	ld_backed=$(echo "$ld_backed" | cut -d ':' -f 1)
	if [[ ! -z $ld_backed ]]; then
		check_upstream_reachable "${ld_backed}" "${container_dns_resolver}"
	else
		log_error "NGX_LD_BACKEND_FAILOVER is not configured. Exiting"
		exitall 133
	fi
	ld_backed=$(get_prop_val "^NGX_LD_BACKEND_PROXY_FAILOVER" "${ENV}")
	ld_backed=$(echo "$ld_backed" | cut -d ':' -f 1)
	if [[ ! -z $ld_backed ]]; then
		check_upstream_reachable "${ld_backed}" "${container_dns_resolver}"
	else
		log_error "NGX_LD_BACKEND_PROXY_FAILOVER is not configured. Exiting"
		exitall 133
	fi
	log_info "Validated Livedata backend failover host reachablity."
}

function check_file_is_readable_or_exit() {
	if [ ! -r "${1}" ]; then
		log_error "File ${1} is not readable for user \"$USER\". Exiting."
		exitall 101
	fi
}

function checkmd5sum(){
    local md5sum_value
	local script_path
	script_path="${3}"
	md5sum_value=$(find "${1}"/"${2}" -type f | sort | xargs cat | md5sum | cut -d " " -f1)
    if [[ -z $(grep "${md5sum_value}" "${script_path}"/md5sum.txt | grep "${2}") ]]; then
        log_warn "Master ${2} have been altered. Note: Some of the pre-install checks that are based on the ${2} configurations will be skipped."
    fi
}


function check_for_number_of_ids_instance() {
	local cmd_out
	cmd_out=$(grep -r "TEMPLATE_TYPE=ids" "${1}" | cut -d '=' -f2 | wc -l)
	if [ "$cmd_out" -gt 1 ]; then
		log_error "Number of IdS instance should not be more than 1. Exiting."
		exitall 102
	fi

}

function check_more_than_one_env_version() {
	local cmd_out
	cmd_out=$(grep -r "TEMPLATE_TYPE=${2}" "${1}" | cut -d '=' -f2 | uniq | wc -w)
	if [ "$cmd_out" -gt 1 ]; then
		log_error "Multiple versions of env files detected for ${2}, retain one type and retry. Exiting."
		exitall 102
	fi
}


function check_upstream_reachable() {
	local hostname
	local dns_entries
	local output
	hostname=$1
	dns_entries=$2

	IFS='| ' read -r -a items <<< "$dns_entries"

	# Iterate over each item
	for dns_server in "${items[@]}"; do
		output=$(nslookup "${hostname}" "${dns_server}" | grep "Name:" | cut -d ':' -f2 | xargs)
		if [[ ! -z ${output} ]]; then
			log_info "Valid server configured and it is reachable. Hostname: ${hostname}, DNS server: ${dns_server}"
			return 0
		fi	
	done
	log_error "Hostname configured is not reachable using DNS server entries. Hostname: ${hostname}, DNS server: ${dns_entries}. Exiting."
	exitall 213
}


#  Function to compare the existing proxymap to check if that matching with the env file.
#  If the proxymap config is not matching with the .env configuration exit.
function check_proxymap_content() {
	local proxy_host
	local proxy_port
	local upstream_hostname
	local template_type
	local proxymapfile
	proxy_host=${1}
	proxy_port=${2}
	upstream_hostname=${3}
	template_type=${4}
	proxymapfile="${5}"
	# Get the LHS of the greped line, if empty try with 
	local lhs
	lhs=$(grep -o '^[^#]*' "${proxymapfile}" | grep -i "=\s*${proxy_host}:${proxy_port}\s*$" | cut -d '=' -f1 | xargs)
	if [[ -z "$lhs" ]]; then
		# Assuming the proxy map is configured for port 443 without explict mention
		lhs=$(grep -o '^[^#]*' "${proxymapfile}" | grep -i "=\s*${proxy_host}\s*$" | cut -d '=' -f1 | xargs)
		if [[ -z "$lhs" ]]; then
			log_error "No valid proxy configuration present in proxymap for ${proxy_host}:${proxy_port}."
			exitall 101
		fi
	fi
	local configured_port="443"
	if [[ "$lhs" == *:* ]]; then
		# LHS contains :, split the value and find the port, else default 443
		configured_port=$(echo "${lhs}" | cut -d ':' -f 2)
	fi
	lhs=$(echo "${lhs}" | cut -d ':' -f 1)
	# ${lhs,,} is for case in-sensitive match
	if [ "${lhs,,}" != "${upstream_hostname,,}" ]; then
		log_error "Upstream configured on proxymap is wrong for ${proxy_host}:${proxy_port}. Expected ${upstream_hostname} but configured in proxymap.txt is ${lhs}, exitall ing."
		exitall  101
	fi
	# Validate the upstream port value
	validate_proxymap_upstream_port "${template_type}" "${configured_port}"
}

# Function to validate the configured upstream port
function validate_proxymap_upstream_port() {
    local template_type
    local configured_port
	template_type="$1"
	configured_port="$2"
	# If no configured port, then assume it is 443
	configured_port=${configured_port:="443"}
	# This will have the the list of ports possible for the upstream, one of them should match
    possible_ports=$(echo "${VALID_UPSTREAM_PORTS[$template_type]}" | tr ',' ' ')
	local port
    for port in ${possible_ports};
    do
        if [ "$port" == "$configured_port" ]; then
			return 0
        fi
    done
	# If the configured upstream port is not valid, exit
    if [[ "$match_found" -ne 1 ]]; then
        log_error "Upstream port configured on proxymap.txt is wrong for template ${template_type}. Possible values are [${possible_ports}] but configured is ${configured_port}, Exiting."
        exitall 101
    fi
}


# This function checks the speed of the nics has minimum gigabitspeed configured.
# Because the performance of the proxy is verymuch depends on the speed of the network configuration.
function verify_nic_speed() {
	# Get all interface names
	interfaces=$(ip link | awk -F: '$0 !~ "lo|vir|wl|podman[0-9]|docker[0-9]|^[^0-9]"{print $2}')
	# Get speed value and check it is > 1000Mbps
	for  nic in $interfaces; do
		speed=$(ethtool $nic | awk '/Speed:/ {print $2}' | tr -d '[:alpha:]' | tr -d '/')
		if ! [[ "$speed" =~ ^[0-9]+$ ]] || [[ -z "$speed" ]]; then
			continue;
		fi
		if [[ -z $speed || $speed -ge 1000 ]]; then
			log_info "$nic has a connection speed of $speed Mbps."
		else
			log_error "$nic does not have a connection speed greater than or equal to 1000 Mbps. Exiting."
			exitall 204
		fi
	done
}

function verify_selinux() {
	local core_env
	core_env=${1}
	local is_selinux_ignored
	is_selinux_ignored=$(get_prop_val "NGX_IGNORE_SELINUX" "${core_env}")
	if [ "${is_selinux_ignored}" == "false" ]; then
		# SeLinux on that box should be enabled and container-selinux should be installed
		if ! is_selinux_enabled; then
			log_error "SELinux is not enabled on the host. Exiting."
			exitall 292
		fi

		if ! is_container_selinux_installed; then
			log_error "The package container-selinux is NOT installed. Exiting."
    		exitall 292
		fi
		bind_mounts=":Z"
		security_opts_disabled=""
		log_info "SeLinux is enforced."
	else
		log_warn "SeLinux enforcement is disabled."
	fi
}

function check_valid_user_agent_validation() {
	local core_env
	core_env=${1}
	local is_valid_user_agent_validation
	is_valid_user_agent_validation=$(get_prop_val "NGX_USE_REGEX_TO_VALIDATE_USER_AGENT" "${core_env}")
	if [ "${is_valid_user_agent_validation}" != "false" ] &&  [ "${is_valid_user_agent_validation}" != "true" ]; then
		log_error "Invalid NGX_USE_REGEX_TO_VALIDATE_USER_AGENT value configured in core.env"
		exitall 236
	fi
}

function check_for_unreplaced_env_vars() {
	
	local unreplaced_env_var
	unreplaced_env_var=$(grep -r -v '^\s*#' "${1}"| grep -F '{NGX_' )
	# Extract unreplaced variable
	unreplaced_env_var=$(echo "${unreplaced_env_var}" | cut -d "{" -f2 | cut -d "}" -f1)

	if [[ -n "${unreplaced_env_var}" ]]; then
		log_error "${unreplaced_env_var} variable is missing in component ${2} & ENV file ${ENV}. Exiting."
		exitall 105
	fi

}
function check_for_duplicate_env_vars() {
	local file
	#Gets the duplicate variables in a ENV file
	local duplicate_variable
	file=${1}
	duplicate_variable=$(sort "$file"| cut -d'=' -f1 | tr -d '\t'  | tr -d ' ' | uniq -d | grep -v '^\s*#' | grep -v '^ *1 ')
	# If duplicatevariable its not empty exit
	if [[ -n "$duplicate_variable" ]];then
		log_error "Following variables were found to be duplicate in file $file. Exiting."
		echo "$duplicate_variable"
		exitall 106
	fi

}

function get_upgrade_script_and_validate_upgrade_path() {
	local file_type
	local DEPLOYMENT_ENV_DIR
	file_type="$1"
	DEPLOYMENT_ENV_DIR="$2"
	template_type=$(get_template_type "${file_type}" "${DEPLOYMENT_ENV_DIR}")
	config_upgrade_file=$(get_upgrade_file "${template_type}")
	log_info "Config verifying file $ENV by $config_upgrade_file"
	check_upgrade_script_has_valid_path "$config_upgrade_file"
}

function check_upgrade_script_has_valid_path() {
	local config_upgrade_file
	config_upgrade_file="${1}"

	local version_path=$(get_version_path "${config_upgrade_file}")
	log_info "Config upgrade path found from env file ${version_path}"
	if ! is_sorted "${version_path}"; then
		log_error "VERSION_PATH value in ${config_upgrade_file} is not sorted. It must be in ascending order."
		exitall 241
	fi
}

function find_properties_without_values() {
	local file="$1"

	if [ -f "$file" ]; then
		# filter all properties that don't have values PROPERTY= 
		properties_without_values=$(grep -E '^[A-Za-z_][A-Za-z0-9_]*=' ${file} | grep -v -E '=[^=]')
		if [ -n "$properties_without_values" ]; then
			log_error "There are properties defined without anyvalue for it in file ${file}. Exiting."
			exitall 243
		fi
	fi
}

# Function to return the template type
function get_template_type() {
	local env_file
	local template_type
	local DEPLOYMENT_ENV_DIR
	env_file="$1"
	DEPLOYMENT_ENV_DIR="$2"
	if [ "${env_file}" == "${DEPLOYMENT_ENV_DIR}/core.env" ]; then
		template_type="core"
	elif [ "${env_file}" == "${DEPLOYMENT_ENV_DIR}/dirs.env" ]; then
		template_type="dirs"
	else
		template_type=$(get_prop_val '^TEMPLATE_TYPE' "${env_file}")
	fi
	echo ${template_type}
}

function check_masterfolders_integrity (){
    checkmd5sum "${CONFIGS_DIR}" "templates" "$REVERSE_PROXY_OPENRESTY_CONFIGS_DIR"
    checkmd5sum "${CONFIGS_DIR}" "master_envs" "$REVERSE_PROXY_OPENRESTY_CONFIGS_DIR"
}


function do_system_validations() {
	local AUTO_PROMT_CONFIMATION
	AUTO_PROMT_CONFIMATION=${1}
	check_nslookup_or_get_user_confirmation "[WARN] 'nslookup' command is not available, some of the installation checks might not work properly, would you like to continue the installation? [Y|N]" "${AUTO_PROMT_CONFIMATION}" "nslookup command not available. Exiting."
	check_kernel_hardend_or_get_user_confirmation "[WARN] OS specific configuration hardening is not done, but it is recommended. Would you like to continue without hardening? [Y|N]" "${AUTO_PROMT_CONFIMATION}" "OS Hardening recommended. Exiting."
	check_diskspeed_or_get_user_confirmation "[WARN] Reverse proxy is sensitive to disk blockages if any (eg if SAN is used). Throughput is lesser than expected this will result in slow preformance of reverse proxy, would you like to continue the installation? [Y|N]" "${AUTO_PROMT_CONFIMATION}" "Slow disk i/o speed. Exiting." "$(get_scale_factor)"
}


function validate_cert_property_for() {
	local filePath="${1}"
	local emptyFileError="${2}"
	local invalidFileError="${3}"
	if [ -z "${filePath}" ]; then
		echo "${emptyFileError}"
		exitall 110
	else
		local cert_path_relative
		local cert_path_in_host
		cert_path_relative="${filePath//$NGX_SSL_DIR/}"
		cert_path_in_host="${HOST_SSL_VOL}${cert_path_relative}"
		if [ ! -r "$cert_path_in_host" ] || [ ! -s "$cert_path_in_host" ]; then
			echo "${invalidFileError}"
			log_error "Missing or empty file ${cert_path_in_host}"
			exitall 110
		fi
	fi
}

function check_log_level_value() {
	local value="$1"
  	case "$value" in
    	debug|info|warn|error|crit|alert|emerg)
			return 0
      	;;
    *)
      	log_error "Invalid log level \"${value}\" specified for variable NGX_ERR_LOG_LEVEL. Allowed values are debug|info|warn|error|crit|alert|emerg."
		exitall 110
      	;;
  	esac
}

function check_ssl_dir_and_files() {
	if [ ! -f "${HOST_SSL_VOL}/${SSL_CERT_NAME}" ] || [ ! -f "${HOST_SSL_VOL}/${SSL_KEY_NAME}" ]; then
		if [ ! "${CREATE_SELF_SIGNED_SSL_CERT}" == "true" ]; then
			local error_details=""
			if [ ! -d "$HOST_SSL_VOL" ]; then
				error_details="Create the directory and place the Certificate and Key file."
			fi
			log_error "SSL Key and Certificate are not present on the specificed location ${HOST_SSL_VOL}. ${error_details}"
			exitall 1
		else
			log_info "SSL Key and Certificates are not present on the specified location, proceeding as CREATE_SELF_SIGNED_SSL_CERT is set to true."
		fi
	fi
	# Change the private key permissions. Only root can modify other users can do readonly.
	chmod 644 "${HOST_SSL_VOL}"/"${SSL_KEY_NAME}"
	log_info "SSL dir and files are present on ${HOST_SSL_VOL}"

	local liveLogShippingStatus="${NGX_LIVE_LOG_SHIPPING_ENABLED}"
	local liveLogShippingTlsMode="${NGX_LIVE_LOG_SERVER_CRT_AUTH}"
	if [ "${liveLogShippingStatus,,}" == "true" ] && [ "${liveLogShippingTlsMode}" == "1" ]; then
		log_info "Live log shipping enabled and secured with TLS. Validating the certificates and keys"
		local liveLogShippingServerCert="${NGX_LIVE_LOG_SHIPPING_SERVER_CERT}"
		local liveLogShippingClientCert="${NGX_LIVE_LOG_SHIPPING_CLIENT_CERT}"
		local liveLogShippingClientCertKey="${NGX_LIVE_LOG_SHIPPING_CLIENT_KEY}"
		validate_cert_property_for "${liveLogShippingServerCert}" \
		"[ERROR] Secured live log shipping is enabled, but the logging server certificate path is empty. Configure the variable NGX_LIVE_LOG_SHIPPING_SERVER_CERT. Exiting." \
		"[ERROR] Secured live log shipping is enabled, but the certificate for the logging server is missing, not readable, or invalid. Exiting."

		validate_cert_property_for "${liveLogShippingClientCert}" \
		"[ERROR] Secured live log shipping is enabled, but the reverse proxy client certificate path is empty. Configure the variable NGX_LIVE_LOG_SHIPPING_CLIENT_CERT. Exiting." \
		"[ERROR] Secured live log shipping is enabled, but the certificate for the reverse proxy client is missing, not readable, or invalid. Exiting."

		validate_cert_property_for "${liveLogShippingClientCertKey}" \
		"[ERROR] Secured live log shipping is enabled, but the reverse proxy client certificate key is empty. Configure the variable NGX_LIVE_LOG_SHIPPING_CLIENT_KEY. Exiting." \
		"[ERROR] Secured live log shipping is enabled, but the certificate key for the reverse proxy client is missing, not readable, or invalid. Exiting."
		log_info "Validated the live log shipping certificates and keys"
	fi
}


function compare_customerenvs_masterenvs() {

    local customerenv
	local customer_comp_type
	customerenv="${1}"
	customer_comp_type=$(get_prop_val '^TEMPLATE_TYPE' "${customerenv}" | tr -d '"')
	if [ -z "${customer_comp_type}" ]; then
		log_error "TEMPLATE_TYPE variable missing in file ${customerenv}. Exiting."
		exitall 108
	fi

	local masterenv="${MASTER_ENV_DIR}/${customer_comp_type}.env"
	# Installer env is not in same as master_env dir
	if [ "${customer_comp_type}" == "installer" ]; then
		masterenv="${MAIN_INSTALLER}"
	fi
	
	if [[ ! -f  "$masterenv" ]]; then
		log_error "Master env file doesn't exists under $masterenv, Exiting."
		exitall 108;
	fi
	
	#Get the keys from customer envs and master envs for comparision
    
	cut -d= -f1 "$customerenv" | sort |uniq | grep -o '^[^#]*' > customerenv.tmp
	cut -d= -f1 "$masterenv" | sort |uniq | grep -o '^[^#]*' > masterenv.tmp


	compare_variables "customerenv.tmp" "masterenv.tmp" "Below unused variable(s) found in $customerenv, Exiting."

	compare_variables "masterenv.tmp" "customerenv.tmp" "Below extra variable(s) found in $customerenv, Exiting."

	# remove tmp file created above for comparision of customer envs and master envs
	rm -rf ./*.tmp

	local master_comp_type
	 master_comp_type=$(get_prop_val '^TEMPLATE_TYPE' "${masterenv}")

	if [[ "$customer_comp_type" != "$master_comp_type" ]]; then
		log_error "Unknown TEMPLATE_TYPE ${customer_comp_type} found in file ${customerenv}, Exiting."
		exitall 108
	fi


}

function compare_variables() {
    local first_file
    local second_file
    local error_message
	first_file="${1}"
	second_file="${2}"
	error_message="${3}"

    local diff_variables
	diff_variables=$(diff -B "$first_file" "$second_file" | grep ">" | sed 's/> //g'  | sed '/^$/d')


    for item in "${NGX_SKIP_INSTALLER_CHECK_VARIABLES[@]}"
    do
       	local tmp_variable
		tmp_variable=$(printf "%s" "$diff_variables" | sed "s/$item//g" | sed '/^$/d')
        diff_variables="${tmp_variable}"
    done

    if [[ -n "$diff_variables" ]]; then
        log_error "$error_message"
        log_info "$diff_variables"
        exitall 108
    fi
}


function validate_envs_with_main() {
	local DEPLOYMENT_ENV_DIR
	DEPLOYMENT_ENV_DIR="${1}"
	for ENV in "${DEPLOYMENT_ENV_DIR}"/*.env; do
		if [ "${ENV}" == "${DEPLOYMENT_ENV_DIR}/core.env" ] || [ "${ENV}" == "${DEPLOYMENT_ENV_DIR}/dirs.env" ]; then
			continue
		fi
		compare_customerenvs_masterenvs "${ENV}"
	done
}

# Old configuration used to have *.autobot.cvp in search domain used to work with CentOS
# Existing if the domain names contains *
function check_no_asterisk_in_dns_entries() {
    local input_string
	input_string="$1"

    if [[ "$input_string" == *\** ]]; then
		log_error "CONTAINER_DNS_SEARCH_DOMAIN should not contain *. Exiting."
		exitall 198
    fi
}

function check_cron_service_or_exit() {
	local service_name
	service_name="crond"

	if systemctl is-active --quiet "$service_name"; then
    	log_info "Cron service '$service_name' is running."
	else
		log_warn "Warning: Cron service '$service_name' is NOT running! Attempting to start..."
		if systemctl start "$service_name"; then
			log_info "Cron service '$service_name' started successfully."
		else
			log_error "Error: Failed to start the cron service '$service_name'." 
			exit 223
		fi
	fi
}

function check_or_create_nginxuser() {
	local installer_env=${1}
	local core_env=${2}
	local ignore_selinx=$(get_prop_val '^NGX_IGNORE_SELINUX' "${core_env}")
	if [ "${ignore_selinx}" == "true" ]; then
		return 0
	fi

    local username=$(get_prop_val '^NGX_USER_USERID' "${installer_env}")
    local groupname=$(get_prop_val '^NGX_USER_USERGROUP' "${installer_env}")
    local required_uid=$(get_prop_val '^NGX_USER_UID' "${installer_env}")


    #Check if the group exists, if not, create it
    if ! grep -q "^$groupname:" /etc/group >/dev/null; then
        log_info "Group '$groupname' does not exist. Creating it... UID=${required_uid} groupid=${groupname}"
        groupadd -g "${required_uid}" "${groupname}"
        if [ $? -ne 0 ]; then
            log_error "Error: Failed to create group '$groupname'."
            exitall 110
        fi
    fi

    #Check if the user exists
    if id "$username" >/dev/null 2>&1; then
        #GetUID
        actual_uid=$(id -u "$username")

        #Compare the UID
        if [ "$actual_uid" -eq "$required_uid" ]; then
            log_info "User '$username' exists and has UID $required_uid."
        else
            log_error "Error: User '$username' exists but has UID $actual_uid instead of $required_uid."
            exitall 110
        fi
    else
        #Create the user with the specified UID and group
        log_info "User '$username' does not exist. Creating it with UID $required_uid..."
        useradd -u "$required_uid" -g "$groupname" "$username"
        if [ $? -eq 0 ]; then
            log_info "User '$username' created successfully with UID $required_uid and group '$groupname'."
        else
            log_error "Error: Failed to create user '$username' with UID ${required_uid} and groupname ${groupname}."
            exitall 100
        fi
    fi
}

function do_initial_validations_before_start_or_hot_reload() {
	local DEPLOYMENT_ENV_DIR
	local INSTALLER_ENV
	local AUTO_PROMT_CONFIMATION
	INSTALLER_ENV="${1}"
	DEPLOYMENT_ENV_DIR="${2}"
	AUTO_PROMT_CONFIMATION="${3}"

	do_system_validations "${AUTO_PROMT_CONFIMATION}"
	check_no_asterisk_in_dns_entries "${CONTAINER_DNS_SEARCH_DOMAIN}"
	verify_nic_speed
	verify_selinux "${DEPLOYMENT_ENV_DIR}"/core.env

	is_file_readable_and_writable "${INSTALLER_ENV}"
	check_dir_exist_or_exit "${DEPLOYMENT_ENV_DIR}"

	is_file_readable_and_writable "${DEPLOYMENT_ENV_DIR}"/core.env
	is_file_readable_and_writable "${DEPLOYMENT_ENV_DIR}"/dirs.env

	for file in "${DEPLOYMENT_ENV_DIR}"/*; do
        log_info "Checking file: ${file}"
        is_file_readable_and_writable "${file}"
    done

	check_dir_exist_or_exit "${TEMPLATE_DIR}"/core
	check_dir_exist_or_exit "${TEMPLATE_DIR}"/common-includes

	check_more_than_one_env_version "${DEPLOYMENT_ENV_DIR}" "cuic"
	check_more_than_one_env_version "${DEPLOYMENT_ENV_DIR}" "livedata"

	(
		source ${INSTALLER_ENV}
		PROXYMAP_FILE=${HOST_WORKING_DIR}/html/proxymap.txt
		check_proxymap_has_no_duplicate_entries "${PROXYMAP_FILE}"
		# Purge old huge log files - do only on start, not on hot_reload.
		purge_huge_logfiles ${HOST_LOGS_VOL}/access.log
		purge_huge_logfiles ${HOST_LOGS_VOL}/error.log
		check_ssl_dir_and_files
	)

	if [ "$(ls -A "${DEPLOYMENT_ENV_DIR}" | wc -l )" -lt 2 ]; then
    	log_error "No component env configs provided in ${DEPLOYMENT_ENV_DIR}."
    	exitall 105
	fi

	find_properties_without_values "${INSTALLER_ENV}"

	local installer_env_upgrade_file
	installer_env_upgrade_file=$(get_upgrade_file "installer")
	check_upgrade_script_has_valid_path "$installer_env_upgrade_file" "$DEPLOYMENT_ENV_DIR"
	
	for ENV in "${DEPLOYMENT_ENV_DIR}"/*.env; do
		find_properties_without_values "$ENV"
		get_upgrade_script_and_validate_upgrade_path "$ENV" "$DEPLOYMENT_ENV_DIR"
		if [ "${ENV}" == "${DEPLOYMENT_ENV_DIR}/core.env" ] || [ "${ENV}" == "${DEPLOYMENT_ENV_DIR}/dirs.env" ]; then
			continue
		else
			# Sanitize the input env files by removing additional spaces and tabs
			# do it only for the key values not the comments
			sed '/^[^#]/s/[ \t]//g' "${ENV}" > "${ENV}".tmp && mv "${ENV}".tmp "${ENV}"
			#Check the template type exists 
			local component_type
			component_type=$(get_prop_val '^TEMPLATE_TYPE' "${ENV}" | tr -d '"')
			check_dir_exist_or_exit "${TEMPLATE_DIR}/${component_type}"
			check_for_duplicate_env_vars "${ENV}"
			# We cannot compare the main envs to the provided envs before upgrading them.
			# Wait till upgrade envs complete
			# compare_customerenvs_masterenvs "${ENV}"
		fi
	done

	check_masterfolders_integrity

}

