--[[
   Resources List
   The purpose of Resources List is to hold the allowed Desktop URLs.
   Also has functionality to fetch backend api and parse it. This is invoked by Nginx timer at every fixed interval.
]]

local accept = "application/json"
local constants = require('constants')

local ResourcesList = {}
local tempResourceVariable 
local count = 0
local desktopurlcount = ngx.shared.desktopurlcount

--[[
   This function is used to create an instance of Resources List object
   Input:
      - serverName - proxy server name
      - url - which to be used for retrieving list of allowed resources using nginx timer scheduler
   
   Returns:
      Resources Lists Object specific to proxy server name
]]
function ResourcesList:new (serverName, url)
   self.__index = self
   return setmetatable({
      serverName = serverName,
      url = url,
      resources = {}
  }, self)
end

--[[
   This function is invoked by Nginx timer.
   The pupose of this function is to trigger/fetch HTTP GET request to backend finesse server defined in self.url
   Returns
      - response payload in string
]]
function ResourcesList:fetchResourcesList()
   ngx.update_time()
   local startTime = ngx.now()
   local http = require "resty.http"
   local httpc = http.new()

   local body = nil
   
   -- Fetch the response
   local res, err = httpc:request_uri(self.url, { method = "GET", headers = { ["Accept"] = accept, ["User-Agent"] = constants.USER_AGENT_HEADER  }, ssl_verify = false })
   if res and res.status == ngx.HTTP_OK then
      ngx.update_time()
      body = res.body 
      ngx.log(ngx.NOTICE, "API url: [", self.url, "]  Succeeded on worker process: [", ngx.worker.id(), "]")
   else
      ngx.log(ngx.ERR, "API url: [", self.url, "]  Failed with error: [", err, "] and response [", res and res.status ,"] on worker process: [", ngx.worker.id(), "]")
   end
   return body
end

--[[
   This function parses the fetched response from backend server and populate/initializes self.resources (contains all allowed resources )
]]
function ResourcesList:parseResources(resourcesString)
   ngx.update_time()
   local startTime = ngx.now()
   local _resources = {}
   local desktopurllist = ngx.shared.desktopurllist
   -- Flush all the entries, which are put into expired state 
   desktopurllist:flush_all()
   -- Flush all the expired entries
   desktopurllist:flush_expired()
	   
   -- since the rest api returns the paths in format of [path1,path2], we need to take substring from 2nd position from start and 2nd from end.(lua index starts with1)
   local resourcesFromAPI = resourcesString:sub(2, -2)   
	for urlPath in string.gmatch(resourcesFromAPI, '([^,]+)') do
		  tempResourceVariable = string.gsub(urlPath, "%s+", "")
		  tempResourceVariable =  tempResourceVariable:sub(2, -2)
   		  desktopurllist:set(tempResourceVariable:sub(1, -1),true)
   		 count = count + 1
	end
   desktopurlcount:set("size",count)
   ngx.update_time()
   local utils = require("utils")
   local constants = require("constants")
   -- update last update of the thirdpartyurl list
   utils.set_last_poll_time(constants.DESKTOP_POLLER_NAME)
   ngx.log(ngx.NOTICE, "For worker process (resources_list): [", ngx.worker.id(), "] total parse time for parsing resources is: ", (ngx.now() - startTime))
end

return ResourcesList 