
--[[
   Unauthenticated Resources Manager 
   The purpose of Unauthenticated Resources Manager is to maintain the 3rd party gadget Resources List Object for each proxy server name.
   NOTE: Unauthenticated Resources Manager is defined per worker process and data is not shared among them
   Functionality:
   - Get the Desktop Resources List, Desktop Url list for the each proxy server name and url from which allowed resources list needs to be determined.
   - If resources List Object is not present, then initialise resources list object, schedule nginx timer with handler at every fixed interval to refresh the urls list.
   - Manually invoke the handler to populated for this first time instead of waiting for scheduler to populate.
]]

local ResourcesList = require('resources_list')
local utils = require('utils')
local constants = require('constants')
local delay = constants.DESKTOP_POLL_INTERVAL
local fast_poller_interval = constants.FAST_POLLER_INTERVAL

--[[
   Handler function to populate the resources list object with allowed url paths.
   
   Input:
      - premature - argument takes a boolean value indicating whether it is a premature timer expiration or not
         Premature timer expiration happens when the Nginx worker process is trying to shut down
      - desktopResourcesInstance - instance of Desktop Resources List Object
]]
local handler = function (premature, desktopResourcesInstance)
   if premature then return end
   ngx.update_time()
   local startTime = ngx.now()

   if utils.has_value_populated(constants.DESKTOP_POLLER_NAME) and not utils.is_valid_poll(constants.DESKTOP_POLLER_NAME, delay) then
      return
   end


   -- ngx.log(ngx.DEBUG, desktopResourcesInstance.serverName, " ",  desktopResourcesInstance.url)
   local ResourcesListXML = desktopResourcesInstance:fetchResourcesList()
   if ResourcesListXML ~= nil then
      desktopResourcesInstance:parseResources(ResourcesListXML)
   end
   ngx.update_time()
   ngx.log(ngx.NOTICE, "Handler for worker process: unauthenticate_desktopresources [", ngx.worker.id() , "] and url :[" , desktopResourcesInstance.url, "] takes total processing time as: [", (ngx.now() - startTime), "]")
   collectgarbage()
end

--[[
   Function to schedule the Nginx timer at every fixed internal
   Input:
      - desktopResourcesInstance - instance of resources List Object
]]

local startNginxTimer = function (desktopResourcesInstance)
   ngx.log(ngx.NOTICE, "Starting Nginx timer to fetch desktop Resource list every ", fast_poller_interval, " seconds for proxy server: ",desktopResourcesInstance.serverName, " on worker process:", ngx.worker.id())
   local ok, err = ngx.timer.every(fast_poller_interval, handler, desktopResourcesInstance)
   if not ok then
      ngx.log(ngx.NOTICE, "Failure - while invoking Thread scheduler for fetching desktopResource list: ", err, " on worker process: [", ngx.worker.id(), " ] and url: [", desktopResourcesInstance.url, "]")
   else
      ngx.log(ngx.NOTICE, "Success - Thread scheduler for fetching desktop resource list on worker process: [", ngx.worker.id(), " ] and url: [", desktopResourcesInstance.url, "]")
   end
   ngx.timer.at(0, handler, desktopResourcesInstance)
end

local UnauthenticatedDesktopResourcesManager = {}

--[[
   Function to get the Desktop urls which dont require authentication for a proxy server name with backend upstream url
   If Desktop url List Object is not present, then initialise the resource list object, schedule nginx timer with handler at every fixed interval to refresh the  list from api.
   Input:
      - serverName - proxy server name
      - url - which to be used for retrieving list of allowed desktop urls  without authentication  using nginx timer scheduler
   
   Retuns:
      - desktop reosurces Object responsible for getting the data from api.
]]
function UnauthenticatedDesktopResourcesManager.getDesktopResources (serverName, url)

	local sharedinstance = ngx.shared.timerthreadsstore
 		local resty_lock = require "resty.lock"
        local value  =  sharedinstance:get("desktopurlthreadstarted")
        if  value then       -- cache hit
            if utils.is_poller_running(constants.DESKTOP_POLLER_NAME) then
               ngx.log(ngx.DEBUG, "Poller is running for ", constants.DESKTOP_POLLER_NAME, " worker id [", ngx.worker.id(), "].")
               return
            else
               ngx.log(ngx.ERR, "Poller stopped for ", constants.DESKTOP_POLLER_NAME, " worker id [", ngx.worker.id(), "]. Starting it.")
            end
        end    
		local lock = resty_lock:new("timerthreadsstore")
        local elapsed = lock:lock("desktopurlkey")
        if not elapsed then
              return
        end
	    -- lock successfully acquired!
	    -- someone might have already put the value into the cache
	    -- so we check it here again
	    -- Double check

	    local threadstarted, err = sharedinstance:get("desktopurlthreadstarted")
	    if threadstarted == "started" then
            if utils.is_poller_running(constants.DESKTOP_POLLER_NAME) then
               local ok, err = lock:unlock()
               if not ok then
                     ngx.log(ngx.ERR, "Failed to unlock in desktopresourcesmanager [", ngx.worker.id(), " ] ")
               end
               return
            else
               ngx.log(ngx.ERR, "Poller stopped for ", constants.DESKTOP_POLLER_NAME, ", worker id [", ngx.worker.id(), "]. Starting it.")
            end
	    end
	    
	   -- Start the Timer 
      local desktopResourcesInstance = ResourcesList:new(serverName, url)

      -- make the first call is done inline - to make sure the values are populated first before starting thread.
      local ResourcesListXML = desktopResourcesInstance:fetchResourcesList()
      if ResourcesListXML ~= nil then
         desktopResourcesInstance:parseResources(ResourcesListXML)
      end

      startNginxTimer(desktopResourcesInstance)
      
      -- Item never expires, so not setting expiry.
      local success, err =  sharedinstance:set("desktopurlthreadstarted", "started")
	    if not success then
            local success, err = lock:unlock()
            if not success then
            	ngx.log(ngx.ERR, "Failed to unlock  in desktopresourcesmanager[", ngx.worker.id(), " ] ")
                return 
            end
            return 
	    end

		-- unlock
        local success, err = lock:unlock()
        if not success then
        	ngx.log(ngx.ERR, "Failed to unlock  in desktopresourcesmanager [", ngx.worker.id(), " ] ")
            return
        end   
end

return UnauthenticatedDesktopResourcesManager 
 