--[[
   User List
   The purpose of User List is to hold the allowed users and also check if user existence is users list
   Also has functinality to fetch backend api and parse it. This is invoked by Nginx timer at every fixed interval
]]
local constants = require('constants')
local utils = require('utils')
local pattern = "<loginId>[^<]*</loginId>"
local patternForLoginName = "<loginName>[^<]*</loginName>"
local alternateHostPattern = "\"alternateHost\":\"([^:\"]*)"
local accept = "application/xml"
local acceptJSON = "application/json"
local shareduserlist = ngx.shared.userlist
-- creating the count also inside shared memory because of performance issues looping through the ngx dictionary again for getting count.
local userscount = ngx.shared.userscount
local fast_poller_interval = constants.FAST_POLLER_INTERVAL

local UsersList = {}

--[[
   This function is used to create an instance of Users List object
   Input:
      - serverName - proxy server name
      - url - which to be used for retrieving list of allowed users using nginx timer scheduler
   
   Returns:
      Users Lists Object specific to proxy server name
]]
function UsersList:new (serverName, serverPort, url, delay)
   self.__index = self
   return setmetatable({
      serverName = serverName,
      serverPort = serverPort,
      url = url,
      delay = delay,
      users = {}
  }, self)
end

--[[
   This function is invoked by Nginx timer.
   The pupose of this function is to trigger/fetch HTTP GET request to backend finesse server defined in self.url
   Returns
      - response payload in string
]]
function UsersList:fetchUsersList()
   ngx.update_time()
   local startTime = ngx.now()
   local http = require "resty.http"
   local httpc = http.new()

   local body = nil

   -- Fetch the response
   local res, err = httpc:request_uri(self.url, { method = "GET", headers = { ["Accept"] = accept, ["User-Agent"] = constants.USER_AGENT_HEADER  }, ssl_verify = false })
   if res and res.status == ngx.HTTP_OK then
      ngx.update_time()
      body = res.body 
   else
      ngx.log(ngx.ERR, "API [",self.url, "] Failed with error: [", err, "] and response [", res and res.status ,"] on worker process: [", ngx.worker.id(), "]")
   end
   return body
end
--[[
   This function is invoked by Nginx timer.
   The pupose of this function is to trigger/fetch HTTP GET request to backend finesse server for desktop config.
   Returns
      - response payload in string
]]
function UsersList:fetchAlternateHost()
   local http = require "resty.http"
   local httpc = http.new()

   local body = nil

   -- Fetch the response
   local desktopConfigURL = "https://" .. self.serverName .. ":" .. self.serverPort .. "/desktop/api/DesktopConfig"
   local res, err = httpc:request_uri(desktopConfigURL, { method = "GET", headers = { ["Accept"] = acceptJSON, ["User-Agent"] = constants.USER_AGENT_HEADER  }, ssl_verify = false })
   if res and res.status == ngx.HTTP_OK then
      body = res.body 
   else
      ngx.log(ngx.ERR, "API [",self.url, "] Failed with error: [", err, "] and response [", res and res.status ,"] on worker process: [", ngx.worker.id(), "]")
   end
   return body
end

--[[
   This function parses the fetched response from backend server based on pattern and populate/initializes self.users (contains all allowed users )
]]
function UsersList:parseUsers(UsersListXML)
   ngx.update_time()
   local startTime = ngx.now()
   local _users = {}
   local i, j, k, l, count = 0, 0, 0, 0, 0
   
   local shareduserlist = ngx.shared.userlist

   while true do
      i ,j = string.find(UsersListXML, pattern , i+1)
      if i == nil then break end
      local str = string.sub(UsersListXML, i + 9, j-10)

      k ,l = string.find(UsersListXML, patternForLoginName , k+1)
      if k == nil then break end
      local loginName = string.sub(UsersListXML, k + 11, l-12)
      local expireAfter = self.delay + fast_poller_interval
      -- Required only if single timer is created per worker for sharing.
      -- _users[str] = true
      -- auto expiry after 900 seconds
      ngx.log(ngx.NOTICE, "For worker process: [", ngx.worker.id(), "] added key on shareduserlist: user_id ", self.serverName .. str, ", username:", self.serverName .. loginName, " expire after (seconds) : ", expireAfter)
      shareduserlist:set(self.serverName .. str, true, expireAfter)
      shareduserlist:set(self.serverName .. loginName, true, expireAfter)
      count = count + 1
   end
   -- Flush all the expired entries
   shareduserlist:flush_expired()

   userscount:set("size",count)
   self.users = _users
   utils.set_last_poll_time(constants.USERS_POLLER_NAME)
   ngx.update_time()
   ngx.log(ngx.NOTICE, "For worker process: [", ngx.worker.id(), "] total parse time is: ", (ngx.now() - startTime))
end
--[[
   This function parses the fetched response from backend server based on pattern and populate/initializes self.alternateHost for each hostname.
]]
function UsersList:parseDesktopConfig(DesktopConfigJSON)
   local _users = {}
   local i, j, count = 0, 0, 0
   local sharedAlternateHosts = ngx.shared.alternatehosts
   local  _,_,alternateHost = string.find(DesktopConfigJSON, alternateHostPattern)
   sharedAlternateHosts:set(alternateHost,self.serverName,self.delay+60)
   -- Flush all the expired entries
   sharedAlternateHosts:flush_expired()
   utils.set_last_poll_time(constants.ALTERNATE_HOST_POLLER_NAME)
end

-- function to check if user exists in the users list
-- Required only if single timer is created per worker.
function UsersList:userExists (user)
   return self.users[user] == true
end

-- function to check if users table is empty
-- Required only if single timer is created per worker.
function UsersList:isEmpty ()
   return next(self.users) == nil
end

return UsersList
