--[[
   Users List Manager 
   The purpose of User List Manager is to maintain the Users List Object for each proxy server name.
   NOTE: User List Manager is defined per worker process and data is not shared among them
   Functionality:
   - Get the Users List for the each proxy server name and url from which allowed users needs to be determined.
   - If Users List Object is not present, then initialize users list object, schedule nginx timer with handler at every fixed interval to refresh the users list.
   - Manually invoke the handler to populated for this first time instead of waiting for scheduler to populate.
]]
local constants = require('constants')
local UsersList = require('users_list')
local utils = require('utils')

local delay = constants.USERS_POLL_INTERVAL -- in seconds
local fast_poller_interval = constants.FAST_POLLER_INTERVAL

--[[
   Handler function to populate the users list object with allowed users.
   
   Input:
      - premature - argument takes a boolean value indicating whether it is a premature timer expiration or not
         Premature timer expiration happens when the Nginx worker process is trying to shut down
      - userInstance - instance of User List Object
]]
local handler = function (premature, usersInstance)
   if premature then return end

   ngx.update_time()
   local startTime = ngx.now()
   ngx.log(ngx.DEBUG, "starting handler for ", constants.USERS_POLLER_NAME, " utils.has_value_populated(constants.USERS_POLLER_NAME) ", utils.has_value_populated(constants.USERS_POLLER_NAME), " utils.is_valid_poll(constants.USERS_POLLER_NAME, delay) ", utils.is_valid_poll(constants.USERS_POLLER_NAME, delay))
   if not utils.has_value_populated(constants.USERS_POLLER_NAME) or utils.is_valid_poll(constants.USERS_POLLER_NAME, delay) then
      local usersListXML = usersInstance:fetchUsersList()
      if usersListXML ~= nil then
         usersInstance:parseUsers(usersListXML)
      end
   end


   if not utils.has_value_populated(constants.ALTERNATE_HOST_POLLER_NAME) or utils.is_valid_poll(constants.ALTERNATE_HOST_POLLER_NAME, delay) then
      local alternateHostsJSON = usersInstance:fetchAlternateHost()
      if alternateHostsJSON ~= nil then
         usersInstance:parseDesktopConfig(alternateHostsJSON)
      end
   end

   if not utils.has_value_populated(constants.UPLOAD_LIMIT_POLLER_NAME) or utils.is_valid_poll(constants.UPLOAD_LIMIT_POLLER_NAME, delay) then
      -- Get the upload limit as well and update cache while getting the agent details
      utils.get_and_update_upload_limit_cache()
   end
   
   
   

   ngx.update_time()
   ngx.log(ngx.DEBUG, "Handler for worker process: [", ngx.worker.id() , "] and url :[" , usersInstance.url, "] takes total processing time as: [", (ngx.now() - startTime), "]")
   collectgarbage()
end

--[[
   Function to schedule the Nginx timer at every fixed internal
   Input:
      - userInstance - instance of User List Object
]]
local startNginxTimer = function (usersInstance)
   ngx.log(ngx.NOTICE, "Starting Nginx timer to fetch User list every ", fast_poller_interval, " seconds for proxy server: ", usersInstance.serverName, " on worker process:", ngx.worker.id())
   local ok, err = ngx.timer.every(fast_poller_interval, handler, usersInstance)
   if not ok then
      ngx.log(ngx.ERR, "Failure - while invoking Thread scheduler for fetching user list: ", err, " on worker process: [", ngx.worker.id(), " ] and url: [", usersInstance.url, "]")
   else
      ngx.log(ngx.NOTICE, "Success - Thread scheduler for fetching user list on worker process: [", ngx.worker.id(), " ] and url: [", usersInstance.url, "]")
   end
   ngx.timer.at(0, handler, usersInstance)
end

local UsersListManager = {}

--[[
   Function to get the users list object for a proxy server name with backend upstream url which to be used for retrieving list of allowed users
   If Users List Object is not present, then initialise users list object, schedule nginx timer with handler at every fixed interval to refresh the users list.
   Input:
      - serverName - proxy server name
      - url - which to be used for retrieving list of allowed users using nginx timer scheduler
   
   Retuns:
      - Users Lists Object specific to proxy server name
]]
function UsersListManager.getUserList(serverName, serverPort, url)

	   local sharedinstance = ngx.shared.timerthreadsstore
 		local resty_lock = require "resty.lock"
        local value  =  sharedinstance:get("getusersthreadstarted" .. serverName)
        if  value then       -- cache hit
            if utils.is_poller_running(constants.USERS_POLLER_NAME) then
               ngx.log(ngx.DEBUG, "poller for ", constants.USERS_POLLER_NAME, " is running.")
               return
            else
               ngx.log(ngx.ERR, "poller for ", constants.USERS_POLLER_NAME, " is not running, hence starting again.")
            end
        end    
		local lock = resty_lock:new("timerthreadsstore")
        local elapsed = lock:lock("userlistkey" .. serverName)
        if not elapsed then
              return
        end
	    -- lock successfully acquired!
	    -- someone might have already put the value into the cache
	    -- so we check it here again
	    -- Double check

	    local threadstarted, err = sharedinstance:get("getusersthreadstarted" .. serverName)
	    if threadstarted == "started" then
            if utils.is_poller_running(constants.USERS_POLLER_NAME) then
               local ok, err = lock:unlock()
               if not ok then
                     ngx.log(ngx.ERR, "Failed to unlock timer store in userslistmanager [", ngx.worker.id(), " ] ")
               end
               return
            else
               ngx.log(ngx.ERR, "poller for ", constants.USERS_POLLER_NAME, " is not running, hence starting again.")
            end
	    end
	    
		-- start the timer
      local usersInstance = UsersList:new(serverName, serverPort, url, delay)
      startNginxTimer(usersInstance)
      
      -- Item never expires, so not setting expiry.
      local success, err =  sharedinstance:set("getusersthreadstarted" .. serverName, "started")
	    if not success then
            local success, err = lock:unlock()
            if not success then
            	ngx.log(ngx.ERR, "Failed to unlock timer store in userslistmanager [", ngx.worker.id(), " ] ")
                return 
            end
            return 
	    end

		-- unlock
        local success, err = lock:unlock()
        if not success then
        	ngx.log(ngx.ERR, "Failed to unlock timer store in userslistmanager [", ngx.worker.id(), " ] ")
            return
        end  
end

return UsersListManager
