#!/bin/bash

#
# This script generates reverse proxy related configurations as per 
# provided environment data
#
# Copyright: Cisco Systems Copyright (c) 2021 by Cisco
#
CURRENT_SCRIPTPATH=$(dirname "${0}")

source "${CURRENT_SCRIPTPATH}"/setenv.sh
source "${CURRENT_SCRIPTPATH}"/../iptables_utilities.sh
source "${CURRENT_SCRIPTPATH}"/../common_utils.sh
source "${CURRENT_SCRIPTPATH}"/../primary_validation_utils.sh
source "${CURRENT_SCRIPTPATH}"/../other_validation_utils.sh

CONFIGS_DIR=${CURRENT_SCRIPTPATH}/configs
TEMPLATE_DIR=${CURRENT_SCRIPTPATH}/configs/templates
TARGET_DIR=${CURRENT_SCRIPTPATH}/configs_out
CONF_OUTPUT_DIR=${TARGET_DIR}/conf
MASTER_ENV_DIR=${CURRENT_SCRIPTPATH}/configs/master_envs
LUA_OUTPUT_DIR=${TARGET_DIR}/lua
LOG_SHIPPING_CONFIG_OUTPUT_DIR=${TARGET_DIR}/rsyslog.d
PROXYMAP_FILE=${HOST_WORKING_DIR}/html/proxymap.txt
HTTP2=""

# This dictionary keeps the values of reverse proxy host + port combination configured for a component
# {"hulk-reverseproxy:8445":"finesse"}
# When the same host port is being configured on the another compoent - look up on this map will give the existing component
# will help to exit the code when duplicate hostport is configured in multiple components.
declare -A HOSTPORT_TEMPLATE_MAP


function usage() {
	log_info "USAGE: run.sh <ENV-DIR> <OPTIONAL_OUT_FILE>"
	log_info "All params are mandatory"
	exitall 1
}


## Validates the properties realted to live log shipping.
function validate_live_log_shipping_configurations() {
	local liveLogShippingStatus="${NGX_LIVE_LOG_SHIPPING_ENABLED}"
	local liveLogShippingHost="${NGX_LIVE_LOG_SHIPPING_SERVER_HOST}"
	local liveLogShippingPort="${NGX_LIVE_LOG_SHIPPING_SERVER_PORT}"
	local liveLogShippingTlsMode="${NGX_LIVE_LOG_SERVER_CRT_AUTH}"
	local liveLogShippingServerCert="${NGX_LIVE_LOG_SHIPPING_SERVER_CERT}"
	local liveLogShippingClientCert="${NGX_LIVE_LOG_SHIPPING_CLIENT_CERT}"
	local liveLogShippingClientCertKey="${NGX_LIVE_LOG_SHIPPING_CLIENT_KEY}"
	if [ "${liveLogShippingStatus,,}" == "true" ]; then

		log_info "Live log shipping is enabled"

		if [ -z "${liveLogShippingHost}" ]; then
			log_error "Live log shipping server is not configured when live log shipping is enabled. Configure the variable NGX_LIVE_LOG_SHIPPING_SERVER_HOST. Exiting."
			exitall 110
		fi

		check_upstream_reachable "${liveLogShippingHost}" "${CONTAINER_DNS_RESOLVER}"

		if [ -z "${liveLogShippingPort}" ]; then
			log_error "Live log shipping server port is not configured when live log shipping is enabled. Configure the variable NGX_LIVE_LOG_SHIPPING_SERVER_PORT. Exiting."
			exitall 110
		fi

		if [ "${liveLogShippingTlsMode}" == "1" ]; then
			log_info "TLS enabled for live log shipping"
			if [ -z "${liveLogShippingServerCert}" ] || [ -z "${liveLogShippingClientCert}" ] || [ -z "${liveLogShippingClientCertKey}" ]; then
				log_error "Secure live log shipping is enabled, but server the certificate, the client certificate or the client key are not present. Exiting."
				exitall 110
			fi
		elif [ "${liveLogShippingTlsMode}" == "0" ]; then
			log_warn "TLS disabled for live log shipping"
		else
			log_error "Invalid value for the property NGX_LIVE_LOG_SERVER_CRT_AUTH. It should be 1 for secured live log shipping and 0 otherwise. Exiting"
			exitall 110
		fi
	else
		log_info "Live log shipping is disabled"
	fi
}

function create_dirs() {
	log_info "Creating dirs"
	#:? is needed for all rm -rf commands as a preventive measure to not to delete the entire filesystem if the value is empty.
	rm -rf "${TARGET_DIR:?}"/*
	mkdir -p "${CONF_OUTPUT_DIR}"/core
	mkdir -p "${CONF_OUTPUT_DIR}"/components
	mkdir -p "${CONF_OUTPUT_DIR}"/common-includes
}

function copy_core_conf() {
	log_info "Copying html and lua configs to ${TARGET_DIR}/"
	cp -r "${CURRENT_SCRIPTPATH}"/configs/html "${TARGET_DIR}"/
	cp -r "${CURRENT_SCRIPTPATH}"/configs/lua "${TARGET_DIR}"/
	cp -r "${CURRENT_SCRIPTPATH}"/configs/rsyslog.d "${TARGET_DIR}"/
	log_info "Copying core configs"
	cp -f "${TEMPLATE_DIR}"/nginx.conf "${CONF_OUTPUT_DIR}"/
	cp -f "${TEMPLATE_DIR}"/mime.types "${CONF_OUTPUT_DIR}"/
	cp -rf "${TEMPLATE_DIR}"/common-includes "${CONF_OUTPUT_DIR}"
	cp -rf "${TEMPLATE_DIR}"/core "${CONF_OUTPUT_DIR}"
}


function set_http2_support() {
	local core_env
	local http2_enabled
	core_env=${1}
	http2_enabled=$(get_prop_val '^NGX_HTTP2_SUPPORT' "${core_env}")
	if [ "${http2_enabled,,}" == "on" ]; then
		log_info "http2 support is enabled."
		HTTP2="http2"
	else
		log_warn "http2 support is disabled."
	fi
}

function process_core_env() {
	set -o allexport
	source "${ENV_DIR}"/dirs.env
	set +o allexport
	local hostname
	hostname=$(hostname)
	if [ "${CONTAINER_NETWORK_MODE}" != "host" ]; then
		log_info "Container network mode is ${CONTAINER_NETWORK_MODE}, hence using ${CONTAINER_NAME} as hostname of the container."
		hostname="${CONTAINER_NAME}"
	fi
	if [[ "${START_TEST_CONTAINER}" == "true" ]];  then
		local existing_static_files_port
		existing_static_files_port="${NGX_PRXY_STATIC_FILES_PORT}"
		local existing_status_port
		existing_status_port="${NGX_PRXY_STATUS_PORT}"
		get_next_free_port
		local test_static_file_port
		test_static_file_port="$TEST_PORT_COUNTER"
		get_next_free_port
		local test_status_port
		test_status_port="$TEST_PORT_COUNTER"
		export_key_value "NGX_PRXY_STATIC_FILES_PORT" "$test_static_file_port"
		export_key_value "NGX_PRXY_STATUS_PORT" "$test_status_port"
		echo "${NGX_PRXY_STATIC_FILES_PORT}=${existing_static_files_port}">>"${TEMP_FILE_TO_STORE_PORT_MAPS}"
		echo "${NGX_PRXY_STATUS_PORT}=${existing_status_port}">>"${TEMP_FILE_TO_STORE_PORT_MAPS}"
	fi
	log_info "Creating configs for core in dir ${CONF_OUTPUT_DIR}/core from env ${ENV_DIR}/core.env"
	CURRENT_SCRIPTPATH="${CURRENT_SCRIPTPATH}" ENV_DIR="${ENV_DIR}" CONF_OUTPUT_DIR="${CONF_OUTPUT_DIR}" NGX_CONTAINER_NAME="${hostname}" sh -c '"${CURRENT_SCRIPTPATH}"/generate_config.sh "${ENV_DIR}"/core.env "${CONF_OUTPUT_DIR}"/core'
	log_info "Creating configs for core in dir ${CONF_OUTPUT_DIR}/common-includes from env ${ENV_DIR}/core.env"
	CURRENT_SCRIPTPATH="${CURRENT_SCRIPTPATH}" ENV_DIR="${ENV_DIR}" CONF_OUTPUT_DIR="${CONF_OUTPUT_DIR}" sh -c '${CURRENT_SCRIPTPATH}/generate_config.sh ${ENV_DIR}/core.env ${CONF_OUTPUT_DIR}/common-includes'
	log_info "Creating configs for core in dir ${CONF_OUTPUT_DIR}/nginx.conf from env ${ENV_DIR}/core.env"
	local ignore_selinx=$(get_prop_val 'NGX_IGNORE_SELINUX' "${ENV_DIR}/core.env")
	local add_user_line_in_nginx_conf="user root root;"
	if [ "${ignore_selinx}" == "false" ]; then
		add_user_line_in_nginx_conf="user nonrootuser nonrootusergroup;"
		log_info "SeLinux enforced. using usergroup and user as ${add_user_line_in_nginx_conf}"
	fi
	CURRENT_SCRIPTPATH="${CURRENT_SCRIPTPATH}" ENV_DIR="${ENV_DIR}" CONF_OUTPUT_DIR="${CONF_OUTPUT_DIR}" NGX_USER="${add_user_line_in_nginx_conf}" NGX_DNS_RSLVR=$(echo "$CONTAINER_DNS_RESOLVER" | sed 's/|/ /g') sh -c '${CURRENT_SCRIPTPATH}/generate_config.sh ${ENV_DIR}/core.env ${CONF_OUTPUT_DIR}/nginx.conf'
	log_info "Creating log live shipping configs for core in dir ${LOG_SHIPPING_CONFIG_OUTPUT_DIR}/ from env ${ENV_DIR}/core.env"
	CURRENT_SCRIPTPATH="${CURRENT_SCRIPTPATH}" ENV_DIR="${ENV_DIR}" CONF_OUTPUT_DIR="${LOG_SHIPPING_CONFIG_OUTPUT_DIR}" sh -c '${CURRENT_SCRIPTPATH}/generate_config.sh ${ENV_DIR}/core.env ${CONF_OUTPUT_DIR}'
	
}

# Function to find same proxy host:port is configured multiple component files.
function process_env_proxy_host_port() {
	local ENV
	ENV=${1}
	local template_type
	local result_template_type
	local hostnamekey
	local portkey
	local proxy_hostname
	local proxy_port
	local proxyhostport
	template_type=$(get_prop_val '^TEMPLATE_TYPE' "${ENV}")
	result_template_type=$(get_component "$template_type")
	hostnamekey="NGX_PRXY_${result_template_type}_HOSTNAME"
	portkey="NGX_PRXY_${result_template_type}_PORT"
	proxy_hostname=$(get_prop_val "^${hostnamekey}" "${ENV}")
	proxy_port=$(get_prop_val "^${portkey}" "${ENV}")
	proxyhostport="$proxy_hostname:$proxy_port"
	log_info "${proxyhostport} is configured for component ${template_type}"
	local existing_val
	existing_val=${HOSTPORT_TEMPLATE_MAP[$proxyhostport]}
	log_info "Checking ${proxyhostport} is configured already? $existing_val"
	if [ -v ${HOSTPORT_TEMPLATE_MAP[$proxyhostport]} ]; then
		log_info "${proxyhostport} is not configured yet, assigning it for component ${template_type} in map."
		HOSTPORT_TEMPLATE_MAP[$proxyhostport]="${template_type}"
	else
		log_error "Error while processing ${template_type} env. Configured ${proxyhostport} is already configured in component '${existing_val}'."
		exitall 1
	fi
}

function process_component_env_files() {
	local NGX_SET_UPSTREAM_MAX_CONNS
	NGX_SET_UPSTREAM_MAX_CONNS=${1}
	# As core.env and dirs.env is already processed so we are checking to be greater than 2
	if [ "$(ls -A "${ENV_DIR}" | wc -l )" -gt 2 ]; then
		for ENV in "${ENV_DIR}"/*.env; do
			if [ "${ENV}" == "${ENV_DIR}/core.env" ] || [ "${ENV}" == "${ENV_DIR}/dirs.env" ]; then
				continue
			else
				local component_type
				component_type=$(get_prop_val '^TEMPLATE_TYPE' "${ENV}")
				process_component_env "${ENV}" "${NGX_SET_UPSTREAM_MAX_CONNS}"
				check_for_unreplaced_env_vars "${CONF_OUTPUT_DIR}" "${component_type}"
				process_env_proxy_host_port "${ENV}"
			fi
		done
	else
    	log_error "No component env configs provided in ${ENV_DIR}, Exiting."
	fi
	store_test_port_map_to_file "${TEMP_FILE_TO_STORE_PORT_MAPS}"
}

function process_component_env() {
	local ENV
	local component_type
	ENV="${1}"
	local NGX_SET_UPSTREAM_MAX_CONNS
	NGX_SET_UPSTREAM_MAX_CONNS="${2}"
	component_type=$(get_prop_val '^TEMPLATE_TYPE' "${ENV}")
	source "${ENV}"
 
	generate_component_configs "${ENV}" "${component_type}" "${NGX_SET_UPSTREAM_MAX_CONNS}"
	
}

function configure_test_ports_from_list(){
  local proxy_port_key_val_list=$1
  local template_type=$2
  for key_val in ${proxy_port_key_val_list[@]}
  do

    get_next_free_port
    local test_port_value
    test_port_value="$TEST_PORT_COUNTER"

    if [ "${template_type}" == "chat" ]; then
      proxy_port_key=$(echo $key_val | cut -d '=' -f1)
      proxy_port_val=$(echo $key_val | cut -d ':' -f2 | xargs)
      proxy_port_host_val=$(echo $key_val | cut -d '=' -f2 | cut -d ':' -f1 | xargs)
      export_key_value "${proxy_port_key}" "$proxy_port_host_val:${test_port_value}"
    else
      proxy_port_key=$(echo $key_val | cut -d '=' -f1)
      proxy_port_val=$(echo $key_val | cut -d '=' -f2 | xargs)
      export_key_value "${proxy_port_key}" "${test_port_value}"
    fi

    TEST_PORT_TO_ORIGINAL_PORT_MAP[${test_port_value}]="${proxy_port_val}"
  done
}


function generate_component_configs() {
	local ENV
	local component_type
	local template_type
	local result_template_type
	local hostnamekey
	local hostname
	local component_dir_name
	local valid_referrers
	local NGX_SET_UPSTREAM_MAX_CONNS
	ENV="${1}"
	component_type="${2}"
	NGX_SET_UPSTREAM_MAX_CONNS="${3}"
	template_type=$(get_prop_val '^TEMPLATE_TYPE' "${ENV}")
	result_template_type=$(get_component "$template_type")
	hostnamekey="NGX_${result_template_type}_HOSTNAME"
	hostname=$(get_prop_val "^${hostnamekey}" "${ENV}")
	component_dir_name=$(echo "${result_template_type}_${hostname}" | tr '[:upper:]' '[:lower:]')
	valid_referrers=$(echo "${NGX_VALID_REFERRERS}" | tr '[:upper:]' '[:lower:]')

	if is_nslookup_present; then
		check_upstream_reachable "${hostname}" "${CONTAINER_DNS_RESOLVER}"
	fi

	check_ld_failover_hosts_reachable "${component_type}" "${ENV}" "${CONTAINER_DNS_RESOLVER}"

	# If proxymap file is present, validate the config with it.
	# Remove the proxymap config from the HOST_VOL/html location, if this check to be skipped.
	if is_proxymap_present; then
		local proxy_hostname
		local proxy_port
		local proxy_hostname_val
		local proxy_port_val
		proxy_hostname="NGX_PRXY_${result_template_type}_HOSTNAME"
		proxy_port="NGX_PRXY_${result_template_type}_PORT"
		proxy_hostname_val=$(eval "echo \$${proxy_hostname}")
		proxy_port_val=$(eval "echo \$${proxy_port}")
		check_proxymap_content "${proxy_hostname_val}" "${proxy_port_val}" "${hostname}" "${template_type}" "${PROXYMAP_FILE}"
	fi

	log_info "component_dir_name for ${ENV} is ${component_dir_name}"
	
	log_info "Creating configs for ${component_type} in dir ${component_dir_name} from env ${ENV}"

	mkdir -p "${CONF_OUTPUT_DIR}"/components/"${component_dir_name}"


	cp -rf "${TEMPLATE_DIR}"/"${component_type}"/* "${CONF_OUTPUT_DIR}"/components/"${component_dir_name}"

	local max_conns="0"
	if [  "${NGX_SET_UPSTREAM_MAX_CONNS}" == "true" ]; then
		max_conns=$(get_max_conns_for_component "${result_template_type}" ${ENV_DIR})
	fi
	log_info "Setting max_conns for upstream component ${result_template_type} is ${max_conns} "

	if [[ "${START_TEST_CONTAINER}" == "true" ]];  then

	  local proxy_port_key_val_list=$(grep -r '^[^#]*PRXY_.*_PORT' "${ENV}" | grep -v "NGX_PRXY_CHAT_PORT")
    configure_test_ports_from_list "${proxy_port_key_val_list}" "${template_type}"

    local chat_ports=$(grep -r '^[^#]*CHAT_.*_PROXY' "${ENV}")
    configure_test_ports_from_list "${chat_ports}" "${template_type}"

		CURRENT_SCRIPTPATH="${CURRENT_SCRIPTPATH}" NGX_MAX_CONN="${max_conns}" ENV_DIR="${ENV_DIR}" CONF_OUTPUT_DIR="${CONF_OUTPUT_DIR}" ENV="${ENV}" NGX_VALID_REFERRERS="${valid_referrers}" component_dir_name="${component_dir_name}" NGX_COMP_DIR_NAME="${component_dir_name}" NGX_HTTP2="${HTTP2}" sh -c '${CURRENT_SCRIPTPATH}/generate_config.sh ${ENV} ${CONF_OUTPUT_DIR}/components/${component_dir_name}'
	else

		CURRENT_SCRIPTPATH="${CURRENT_SCRIPTPATH}" NGX_MAX_CONN="${max_conns}" ENV_DIR="${ENV_DIR}" CONF_OUTPUT_DIR="${CONF_OUTPUT_DIR}" ENV="${ENV}" NGX_VALID_REFERRERS="${valid_referrers}" component_dir_name="${component_dir_name}" NGX_COMP_DIR_NAME="${component_dir_name}" NGX_HTTP2="${HTTP2}" sh -c '${CURRENT_SCRIPTPATH}/generate_config.sh ${ENV} ${CONF_OUTPUT_DIR}/components/${component_dir_name}'
	fi
	if [ "${component_type}" == "cloudconnect" ]; then
		local cloudconnect_clients_ip
		cloudconnect_clients_ip=$(get_prop_val 'NGX_CLOUDCONNECT_CLIENT_IPS' "${ENV}")
		parse_ips_and_create_list "${cloudconnect_clients_ip}" "${CONF_OUTPUT_DIR}/components/${component_dir_name}/cloudconnect_clients_ip_list" "true"
		more ${CONF_OUTPUT_DIR}/components/${component_dir_name}/cloudconnect_clients_ip_list >> ${CONF_OUTPUT_DIR}/core/skip_user_agent_check
	fi

	# Handle Lua scripts env replacement: CSCwd57545
	CURRENT_SCRIPTPATH="${CURRENT_SCRIPTPATH}" ENV_DIR="${ENV_DIR}" CONF_OUTPUT_DIR="${LUA_OUTPUT_DIR}" ENV="${ENV}" NGX_COMP_DIR_NAME="${component_dir_name}" NGX_AUTHENTICATE_WEBSOCKET="${NGX_AUTHENTICATE_WEBSOCKET}" sh -c '${CURRENT_SCRIPTPATH}/generate_config.sh ${ENV} ${CONF_OUTPUT_DIR}'
	validate_mtls "${component_type}" "${ENV}"
	log_info "Updating Lua scripts from env parameters from env ${ENV} and placing them back ${LUA_OUTPUT_DIR}"

	
}


#
#	This function validates whether the mTLS
#	certificate is present in the respective directory
#	if mTLS is enabled for the component.
#
function validate_mtls() {
	local mtls
	local upstream_server_cert
	mtls=$(get_prop_val_clean 'NGX_PRXY_SSL_VERIFY' "${2}")
	upstream_server_cert=$(get_prop_val_clean 'NGX_PRXY_SSL_TRUST_CRT' "${2}")
	if [ "${mtls,,}" == "on" ]; then
		log_info "Mutual TLS validation is enabled for ${1}."

		if [ -z "${upstream_server_cert}" ]; then
			log_error "Mutual TLS validation is enabled for ${1}, but the upstream server certificate path in NGX_PRXY_SSL_TRUST_CRT is empty. Exiting."
			exitall 109
		fi
		local cert_path_relative
		local cert_path_in_host
		cert_path_relative="${upstream_server_cert#\$\{NGX_SSL_DIR\}}"
		cert_path_in_host="${HOST_SSL_VOL}${cert_path_relative}"
		if [ ! -r "$cert_path_in_host" ] || [ ! -s "$cert_path_in_host" ]; then
			log_error "Mutual TLS validation is enabled for ${1}, but the upstream server certificate ${cert_path_in_host} is not present, not readable or invalid. Exiting."
			exitall 109
		fi
	else
		log_info "Mutual TLS validation is disabled for ${1}."
	fi
}


function run_validations() {
  check_for_number_of_ids_instance "${ENV_DIR}"
  validate_live_log_shipping_configurations
  source ${ENV_DIR}/core.env

  if [ "${NGX_IPTABLES_HARDENING}" != "1" ]; then
    log_warn "Iptables hardening turned off for container ${CONTAINER_NAME}."
    return
  fi

  if [[ "${NGX_AUTHENTICATE_WEBSOCKET}" != "true" && "${NGX_AUTHENTICATE_WEBSOCKET}" != "false" ]]; then
      log_error "NGX_AUTHENTICATE_WEBSOCKET property value in core.env should be either true or false"
      exitall 1
  fi
  if [ ! -f /etc/reverse-proxy/network.env ]; then
    log_error "Iptables hardening turned on for container ${CONTAINER_NAME}. Run OS hardening script with -i parameter to set external interface."
    exitall 1
  fi

  source /etc/reverse-proxy/network.env
  log_info "EXTERNAL_INTERFACE: $EXTERNAL_INTERFACE"
  NGX_EXTERNAL_INTERFACE=$EXTERNAL_INTERFACE

  is_ip_installed
  is_route_installed
  check_interface_or_exit ${NGX_EXTERNAL_INTERFACE}
  if is_iptables_running; then
    log_info "Iptables service is running."
    warn_if_dns_ips_are_external_ips
    for ENV in ${ENV_DIR}/*.env; do
        if [ "${ENV}" == "${ENV_DIR}/core.env" ] || [ "${ENV}" == "${ENV_DIR}/dirs.env" ]; then
          continue
        else
          validate_iptable_ratelimits "${ENV}"
        fi
    done
  else
    log_error "Iptables service is not running, Iptables hardening cannot be done."
    exitall 1
  fi
}

function validate_iptable_ratelimits(){
  local ENV
  ENV=${1}
  local IPTABLES_RATELIMITS_KEYS=("IPTABLES_CONNECTION_LIMIT_ABOVE" "IPTABLES_LOG_LIMIT_BURST" "IPTABLES_LOG_LIMIT" "IPTABLES_HASH_LIMIT_UPTO" "IPTABLES_HASH_LIMIT_BURST")
  for key in "${IPTABLES_RATELIMITS_KEYS[@]}"; do
    local value=$(get_prop_val_clean "$key" "${ENV}")
      if [ "$key" == "IPTABLES_HASH_LIMIT_UPTO" ]; then
        [[ $value =~ ^[0-9]+/sec$ ]] || { log_error "${key} in ${ENV} is not in expected format. Input Provided=$value. Expected Format \"<Integer>/sec\" .Eg. \"5/sec\""; exitall 107; }
        continue
      fi
      if [ "$key" == "IPTABLES_LOG_LIMIT" ]; then
        [[ $value =~ ^[0-9]+/min$ ]] || { log_error "${key} in ${ENV} is not in expected format. Input Provided=$value. Expected Format \"<Integer>/min\" .Eg. \"1/min\""; exitall 107; }
        continue
      fi
      [[ $value =~ ^[0-9]+$ ]] || { log_error "${key} in ${ENV} is not in expected format. Input Provided=$value. Expected Format <Integer> .Eg. 10"; exitall 107; }
  done
}

function warn_if_dns_ips_are_external_ips() {
    local ips_of_internal_interfaces=$(get_ip_of_internal_interfaces ${NGX_EXTERNAL_INTERFACE})
    ips_of_internal_interfaces=$(echo "$ips_of_internal_interfaces" | awk -F',' '{for(i=1;i<=NF;i++){sub(/\.[0-9]+$/, "", $i)}; print}' | tr ' ' ',')
	local internal_ips_pattern=$(echo "$ips_of_internal_interfaces" | tr ',' '|' | sed 's/^/\\b/; s/$/\\b/')
    # Use grep to check if the IP is in the pattern
  IFS='| ' read -r -a items <<< "$CONTAINER_DNS_RESOLVER"
  for ip in "${items[@]}"; do
    if ! printf "%s" "${ip}" | grep -E -q "${internal_ips_pattern}"; then
      log_warn "CONTAINER_DNS_RESOLVER IP ${ip} configured is not resolvable through the internal interface."
    fi
  done

}



# This function will help to add extra line on the config based on the server_token off configurations.
# One extra line should be added when the server_token is set to off on the configuration to remove the server header itself.
# server_tokne off just removes the version of the server used.
clear_server_response_header() {
	log_info "NGX_SERVER_TOKENS is configured as ${NGX_SERVER_TOKENS}."
	if [ "${NGX_SERVER_TOKENS}" == "off" ]; then
		echo "more_clear_headers \"Server\";" >> "${CONF_OUTPUT_DIR}"/core/common.conf
	fi
}

function main() {
	ENV_DIR=${1}
	TEMP_FILE_TO_STORE_PORT_MAPS="${2}"
	run_validations
	log_info "######################## Building openresty configurations ########################"
	create_dirs
	copy_core_conf
	source "${ENV_DIR}"/core.env
	process_core_env
	set_http2_support "${ENV_DIR}"/core.env
	>${CONF_OUTPUT_DIR}/core/skip_user_agent_check
	process_component_env_files $(get_prop_val '^NGX_SET_UPSTREAM_MAX_CONNS' "${ENV_DIR}"/core.env)
	clear_server_response_header
	parse_ips_and_create_list "${NGX_LOCALHOST_IPS}" "${CONF_OUTPUT_DIR}/core/reverseproxy_localhost_ips" "true"
	more ${CONF_OUTPUT_DIR}/core/reverseproxy_localhost_ips >> ${CONF_OUTPUT_DIR}/core/skip_user_agent_check
	parse_ips_and_create_list "${NGX_RATELIMIT_DISABLE_IPS}" "${CONF_OUTPUT_DIR}/core/ratelimit_disable_ip_list" ""
	if [ -z "${NGX_LOAD_BALANCER_IPS}" ]; then
		log_info "No Load balancer IPs configured."
		if [ -z "${NGX_LOAD_BALANCER_REAL_IP_HEADER}" ]; then
			log_info "No real IP header configured."
		else
			log_error "NGX_LOAD_BALANCER_REAL_IP_HEADER should be configured only when NGX_LOAD_BALANCER_IPS is configured, Exiting."
			exitall 108
		fi
	else
		if [ -z "${NGX_LOAD_BALANCER_REAL_IP_HEADER}" ]; then
			log_error "NGX_LOAD_BALANCER_REAL_IP_HEADER is empty. It should contain header details when NGX_LOAD_BALANCER_IPS is configured, Exiting."
			exitall 108
		fi
		# Configure Load balancer IPs
		create_lb_configurations "${NGX_LOAD_BALANCER_IPS}" "${NGX_LOAD_BALANCER_REAL_IP_HEADER}" "${CONF_OUTPUT_DIR}/core/lb_conf"
		# Configure not to consider load balancer IPs for rate limiting
		parse_ips_and_create_list "${NGX_LOAD_BALANCER_IPS}" "${CONF_OUTPUT_DIR}/core/ratelimit_disable_ip_list" ""
	fi
	# Remove duplicate if any from skip_user_agent_check
	log_info "Removing duplicate entires from skip_user_agent_check file if any"
	tmp_file=$(get_new_temp_file)
	sort ${CONF_OUTPUT_DIR}/core/skip_user_agent_check | uniq > ${tmp_file} && mv ${tmp_file} ${CONF_OUTPUT_DIR}/core/skip_user_agent_check
	
	log_info "######################## Openresty configurations generated ########################"
	exit 0
}

([ "$#" -gt 2 ] || [ "$#" -lt 1 ]) && usage || main "$@"


