/**************************************************************************
*       Copyright (c) 2006, Cisco Systems, All Rights Reserved
***************************************************************************
*
*  File:    PromptEntry.h
*  Date:    08/2006
*
***************************************************************************
*
*   Prompt Entry class implementation for the Client API.
*
***************************************************************************/

#ifndef _PROMPTENTRY_
#define _PROMPTENTRY_


#include "PromptEntryBase.h"

/**
 * When Authentication requires a user to enter credentials or view a banner in
 * conjunction with their VPN activation, one or more PromptEntry objects are 
 * created.  Each PromptEntry typically contains a label and value.  The
 * value can be set with a default value that the user can then change.
 *
 * PromptEntry instances are collected into a list and delivered in a single
 * instance of the ConnectPromptInfo class.
 *
 * When the selections or values are complete (using setValue method) for all
 * the PromptEntry instances, simply call the API method
 * ClientIfc::UserSubmit to alert the API that it can
 * process the responses and proceed with VPN establishment.
 *
 * An example of accessing individual PromptEntry and their values can be
 * found in ClientImpl::setUserData
 *
 */


class VPN_VPNAPI PromptEntry : public PromptEntryBase
{
    public:

        /**
         * use this method to get the current value set in the prompt entry.
         */
        const tstring& getValue() const;

        /**
         * use this method to set the user selection.  If a default value is
         * present, it's value will be used unless this method in invoked.
         * Returns true if the value is successfully set.
         */
        bool setValue(const tstring& value);


        /**
         * The fixed name associated with this prompt entry.
         * This represents a non-translated fixed entity, whereas the
         * label is a translated entry.
         */
        const tstring &getPromptName() const;


        /**
         * Set/get the label associated with this prompt entry.
         * This value is translated if a translation is available.
         */
        const tstring &getPromptLabel() const;


        /**
         * Return the type of prompt entry.  See the enum PromptType for
         * the possible types.
         */
        PromptType getPromptType() const;

        /**
         * Get the enabled state of this prompt entry which indicates if
         * it can be edited.
         */
        bool isEnabled() const;


        /**
         * Get the visible state of this prompt entry which indicates if
         * it should be displayed.
         */
        bool isVisible() const;


        /**
         * If a prompt entry has a list of possible selection, (like Prompt_Combo 
         * and Prompt_Checkbox in ::PromptType enum in api.h), that list is
         * available via this method.  For example, a checkbox type prompt 
         * would return "true" and "false" as options.  The values returned could for
         * example, be displayed in a combo box selection.
         */
        const std::list<tstring> &getValueOptions() const;


        /**
         * Use this prompt entry for group values
         */
        bool isEntryGroup() const;

        /*
         * Returns whether this prompt entry is read only (IE
         * it does not require user input)
         */
        bool isReadOnly() const;


        static tstring Username;          /**< Identifies the PromptEntry instance
                                               requesting a username.
                                               See getPromptName() method and example
                                               in ClientImpl::setUserData() */
        static tstring Password;          /**< Identifies PromptEntry instance
                                               requesting a password.
                                               See getPromptName() method and example
                                               in ClientImpl::setUserData() */
        static tstring SecondaryUsername; /**< Identifies PromptEntry instance
                                               requesting secondary username. */
        static tstring SecondaryPassword; /**< Identifies PromptEntry instance
                                               requesting secondary password. */
        static tstring GroupList;         /**< Identifies PromptEntry instance
                                               with group list. */
        static tstring Banner;            /**< Identifies PromptEntry instance
                                               containing banner. */
        static tstring Pin;              /**< Identifies PromptEntry PIN */
        static tstring VerifyPin;        /**< Identifies PromptEntry Verify PIN */
        static tstring NetAccess;        /**< Identifies the PromptEntry displaying
                                              the network access state. */

        // The following methods are used to configure the PromptEntry
        // and do not need to be used by a client application.


        PromptEntry(tstring PromptName,
                    tstring PromptLabel,
                    PromptType promptType = Prompt_Input,
                    const tstring& DefaultValue = EmptyString,
                    ApiStringMap LabelValues = EmptyLabelValues);

        ~PromptEntry();


};


#endif // _PROMPTENTRY_
