/**************************************************************************
*   Copyright (c) 2008, 2022-2023 Cisco Systems, Inc.
*   All Rights Reserved. Cisco Highly Confidential.
***************************************************************************
*
*   File: api.h
*   Date: 01/2008
*
***************************************************************************
*   VPN GUI API module interface types
***************************************************************************/
#ifndef _APISTDHEADER_
#define _APISTDHEADER_

/**
 * @file
 * This file contains some basic compiler definitions as well as common enums.
 */

//Not compatible with MIDL
#if !defined(__midl)
#ifdef _WIN32
    #pragma warning(disable:4251 4786)

    #ifndef UNICODE
        #define UNICODE
    #endif // UNICODE

    #ifndef _UNICODE
        #define _UNICODE
    #endif // _UNICODE

    #ifndef tstring
/** std::wstring */
        #define tstring std::wstring    /**< my wstring description */
    #endif // tstring

#else // non-windows

    #ifndef tstring
        #define tstring std::string
    #endif // tstring

#endif // _WIN32

#ifdef _UNICODE
    #define tostream std::wostream
#else
    #define tostream std::ostream
#endif /* UNICODE */


//used when including implementation files directly in an EXE.
#ifdef _NOEXPORTDLL
    #define VPN_VPNAPI
#else
    #ifdef _WIN32
        #ifdef VPN_APIEXPORTS //api
            #define VPN_VPNAPI __declspec(dllexport)
        #else
            #define VPN_VPNAPI __declspec(dllimport)
        #endif
    #else    
        #ifdef VPN_APIEXPORTS
            #define VPN_VPNAPI  __attribute__((visibility("default")))
        #else
            #define VPN_VPNAPI
        #endif
    #endif //_WIN32
#endif //NOEXPORTDLL

#ifndef OUT
#define OUT
#endif

#ifdef __cplusplus //only include if C++ is being used, 
                   //C code also includes api.h for COM proxy of enumerators.
#include <string>
#include <map>
#include <list>

typedef std::map<tstring, tstring> ApiStringMap;
typedef std::map<tstring, std::list<tstring> > ApiStringListMap;

#endif //__cplusplus
#endif //#if !defined(__midl)


/***** PUT ONLY SHARED ENUMS EXPOSED TO USERS OF API FROM THIS POINT UNTIL END *****\
********* make sure to add the [v1_enum] inside a __midl define to new enums ********
\******************** This is also compiled with IDL compiler **********************/

#include "GlobalEnums.h"
/**
 * MessageType
 * presents a level of severity associated with messages that are
 * sent to the API.  The severity can be useful for deciding how a message is
 * to be shown.  A UI might decide based on type to show a message as
 * a modal dialog versus a message written to the status area for an existing UI.
 */
#if defined(__midl)
[v1_enum] /*serialize as 32 bits*/
#endif
enum MessageType
{
    MsgType_Error,      /**< Issue usually requiring user to acknowledge */
    MsgType_Alert,      /**< Warning message that needs to be shown to user. */
    MsgType_Warn,       /**< Less severe, not required to be shown to user */
    MsgType_Info,       /**< General message providing status, progress, etc. */
    MsgType_Status,     /**< Can be used to indicate unexpected tunnel status change. */
    MsgType_UrlError    /**< Error message that provides additional information by opening browser. */
};


/**
 * Identifies the type of token that was used successfully when SDI
 * Authentication is in use.
 */
#if defined(__midl)
[v1_enum] /*serialize as 32 bits*/
#endif
enum SDITokenType 
{ 
    SDITT_NONE, 
    SDITT_HARDWARE, 
    SDITT_SOFTWARE 
};

/**
 * Provides the current state of the VPN tunnel.
 */
#if defined(__midl)
[v1_enum] /*serialize as 32 bits*/
#endif
enum VPNState
{
    CONNECTED     = STATE_CONNECTED,        /**< VPN is active */
    DISCONNECTED  = STATE_DISCONNECTED,     /**< VPN is inactive */
    CONNECTING    = STATE_CONNECTING,       /**< VPN is being established */
    DISCONNECTING = STATE_DISCONNECTING,    /**< VPN is being terminated */
    RECONNECTING  = STATE_RECONNECTING,     /**< VPN is being re-connected.  This state 
                                                 can occur due to network or other
                                                 temporary problems.  The state
                                                 indicates that the VPN is temporarily
                                                 unavailable and indicates the
                                                 connection is being re-established. */
    PAUSING       = STATE_PAUSING,          /**< VPN is being paused. */
    PAUSED        = STATE_PAUSED,           /**< VPN is paused. */
    SSOPOLLING    = STATE_SSOPOLLING,       /**< API is doing auth-poll, VPN is disconnected. */
    UNKNOWN       = ~0
};

/**
 * Provides the current sub-state of the VPN tunnel.
 */
#if defined(__midl)
[v1_enum] /*serialize as 32 bits*/
#endif
enum VPNSubState
{
    VPNSS_NORMAL                            = VCSS_NORMAL,
    VPNSS_INDEFINITE_DELAY                  = VCSS_INDEFINITE_DELAY,
    VPNSS_SESSION_EXPIRING                  = VCSS_SESSION_EXPIRING,
    VPNSS_MT_DISCONNECTED_DISABLED          = VCSS_MT_DISCONNECTED_DISABLED,
    VPNSS_MT_DISCONNECTED_TRUSTED_NW        = VCSS_MT_DISCONNECTED_TRUSTED_NW,
    VPNSS_MT_DISCONNECTED_USER_TUNNEL_ACTIVE= VCSS_MT_DISCONNECTED_USER_TUNNEL_ACTIVE,
    VPNSS_MT_DISCONNECTED_LAUNCH_FAILED     = VCSS_MT_DISCONNECTED_LAUNCH_FAILED,
    VPNSS_MT_DISCONNECTED_CONNECT_FAILED    = VCSS_MT_DISCONNECTED_CONNECT_FAILED,
    VPNSS_MT_DISCONNECTED_BAD_VPN_CONFIG    = VCSS_MT_DISCONNECTED_BAD_VPN_CONFIG,
    VPNSS_MT_DISCONNECTED_SW_UP_PENDING     = VCSS_MT_DISCONNECTED_SW_UP_PENDING,
    VPNSS_MTU_ADJUSTMENT_PENDING            = VCSS_MTU_ADJUSTMENT_PENDING
};

/**
 * WMHint
 * provides a hint for the GUI to either minimize or un-minimize.
 */
#if defined(__midl)
[v1_enum] /*serialize as 32 bits*/
#endif
enum WMHint
{
    MINIMIZE,       /**< hint to minimize GUI */
    OPEN,           /**< hint to un-minimize GUI */
    QUIT,           /**< hint that GUI should close.  @see WMHintReason */
    REFRESHHOSTNAMES,/**< hint to refresh the list of secure gateways */
    REFRESHPREFS,   /**< hint to refresh the preferences */
    SHOWCONNECTING,  /**< hint to display "connecting" status */
    CLOSECREDENTIALPOPUP, /**< hint to close the credentials popup */
};


/**
 * WMHintReason
 * provides a reason indicator for the #WMHint
 */
#if defined(__midl)
[v1_enum] /*serialize as 32 bits*/
#endif
enum WMHintReason
{
    SECONDGUISTART, /**< Indicates a second GUI has been launched.  This
                         indicator is used to suggest that the GUI
                         already running be OPENed and that the first one
                         should exit. */
    PROXYREQUEST,   /**< Proxy credential request can be for web-launch or
                         standalone-initiated connections. */
    SERVICEFAILURE, /**< This tag is used when the VPN service
                         is no longer available. */
    DISCONNECT,     /**< Any disconnect notices should be seen by the user. */
    SERVICESTOPPED, /**< This tag will be used in cases where the VPN service
                         has been stopped. */
    CONNECT,        /**< Tag indicating an action to be taken due to connect,
                         for example a request to minimize the UI. */
    REASONUNKNOWN   /**< */
};

/**
 * provides an indication of the type of credential data being requested.
 */
#if defined(__midl)
[v1_enum] /*serialize as 32 bits*/
#endif
enum ConnectPromptType
{
    CERTIFICATE,    /**< Indicates a certificate-only type of connection and
                         would not normally be sent to client unless a
                         post-authentication banner is to be displayed. */
    CREDENTIALS,    /**< Indicates that the user is to be prompted for authentication
                         credentials */
    PROXY,          /**< Indicates that the user is to be prompted for
                         proxy-authentication credentials */
    MANUAL_PKCS12_IMPORT, /**< Indicates that the user is to be prompted for passwords related
                              to PKCS12 import*/
    STATUS,         /**< Indicates that status messages are to be displayed to
                         the user*/
    SINGLESIGNON,   /**< Indicates an embedded browser based single sign-on authentication method is requested. */
    SINGLELOGOUT,   /**< Indicates an embedded browser based single sign-on authentication logout is requested. */
    LEGACY_SINGLESIGNON,   /**< (deprecated) Indicates a browser based single sign-on authentication method is requested. */
};


/**
 * Indicates the prompt or credential type.
 */
#if defined(__midl)
[v1_enum] /*serialize as 32 bits*/
#endif
enum PromptType { Prompt_Input,     /**< label and value. */
                  Prompt_Password,  /**< label and value, indicates user
                                         response should be masked. */
                  Prompt_Banner,    /**< value (the banner) with no label set. */
                  Prompt_Combo,     /**< list with choices options. */
                  Prompt_Header,    /**< label intended as header and with
                                         value. */
                  Prompt_Hidden,    /**< hidden value, should be ignored and
                                         left unchanged in response. */
                  Prompt_CheckBox,  /**< label and value (contrained to true or false) */
                  Prompt_SSO        /**< single sign-on authentication token prompt */
};

#if defined(__midl)
[v1_enum] /*serialize as 32 bits*/
#endif

/* 
 * ***************** !!! ATTENTION !!! ***********************************
 * *
 * * When updating this preference enum, you must ensure that the enum in
 * * vpn/Api/jni/java/Preference.java is also updated.
 * *
 * ***************** !!! ATTENTION !!! ***********************************
 */
enum PreferenceId 
{
    ServiceDisable,             /**< This preference disable the VPN service.  
                                 If more than one profile exists and any one
                                 profile has VPN enabled, then it will be
                                 enabled.  False is the default. */
    CertificateStoreOverride,/**< This preference will trigger an alternate 
                                 authentication sequence in the API. The 
                                 preference is only settable by an 
                                 administrator. */
    CertificateStore,       /**< This preference indicates which Windows certificate 
                                 store AnyConnect should look in for client certificates.
                                 The options are All, Machine and User with a default of All.
                                 The preference is only settable by an administrator. */
    CertificateStoreMac,    /**< This preference indicates which macOS keychain
                                 AnyConnect should look in for client certificates. 
                                 The options are All, System and Login with a default of All. 
                                 The preference is only settable by an administrator. */
    CertificateStoreLinux,  /**< This preference indicates which Linux certificate
                                 store AnyConnect should look in for client certificates. 
                                 The options are All, Machine and User with a default of All. 
                                 The preference is only settable by an administrator. */
    ShowPreConnectMessage,  /**< The ShowPreConnectMessage preference gives the
                                 administrator the ability to display an AnyConnect 
                                 startup banner message. The message will appear 
                                 only once per AnyConnect program start. The  
                                 preference is only settable by an 
                                 administrator. */
    AutoConnectOnStart,     /**< This preference allows the user to select 
                                 whether to establish a connection automatically
                                 on startup or not. */
    MinimizeOnConnect,      /**< This preference allows the user to select if
                                 the GUI should minimize when the connection is
                                 established */
    LocalLanAccess,         /**< This preference will provide a mechanism where 
                                 the user can disable access to their Local LAN. */
    DisableCaptivePortalDetection, /**<This preference will provide a mechanism where
                                   the user can disable captive portal detection.*/
    AutoReconnect,          /**< First control of the reconnect behavior. If the 
                                 client becomes disconnected for any reason, a 
                                 reconnect attempt is made.   */
    AutoReconnectBehavior,  /**< Second control of the reconnect behavior. When
                                 coming out of suspend/hibernate/standby mode. 
                                 Options are disconnect on suspend and reconnect 
                                 after suspend. */
    SuspendOnConnectedStandby,  /**< This setting allows to control whether the VPN tunnel
                                     is suspended when the system enters the Connected Standby 
                                     mode. It applies only to Windows 8 and above. */
    UseStartBeforeLogon,    /**< This preference allows an administrator to 
                                 control the use of the Start Before Logon 
                                 feature. The preference can be set to true (on) 
                                 or false (off). */
    AutoUpdate,             /**< Once the Downloader has loaded the profile, it 
                                 can check the AutoUpdate preference to see if 
                                 updates are either disabled or enabled */
    RSASecurIDIntegration,  /**< This preference will enable the administrator 
                                 and possibly end user to select the preferred 
                                 method of managing their SDI PIN and PASSCODE 
                                 interactions. Options are Automatic (default), 
                                 SoftwareTokens and HardwareTokens. */
    WindowsLogonEnforcement,/**< This preference allows an administrator to
                                 control if more than one user may be logged into
                                 the client PC during the VPN connection (Windows
                                 only). */
    WindowsVPNEstablishment,/**< This preference allows an administrator to
                                 control whether or not remote users may initiate
                                 a VPN connection (Windows only). */
    LinuxLogonEnforcement,  /**< This preference allows an administrator to
                                 control if more than one user may be logged into
                                 the client PC during the VPN connection (Linux 
                                 only). */
    LinuxVPNEstablishment,  /**< This preference allows an administrator to
                                 control whether or not remote users may initiate
                                 a VPN connection (Linux only). */
    ProxySettings,          /**< This preference allows an administrator to
                                 control how user's proxy setups are handled.*/
    AllowLocalProxyConnections, /**< This preference allows the administrator to control
                                 whether to allow establishing a connection through
                                 a local proxy. */
    PPPExclusion,           /**< This preference allows an administrator to control
                                 the policy used to exclude routes to
                                 PPP servers when connecting over L2TP or PPTP.
                                 Options are Automatic (default), Disable,
                                 and Override. */
    PPPExclusionServerIP,   /**< When PPPExclusion is set to Manual,
                                 the value of this preference allows an
                                 end user to specify the address of a
                                 PPP server that should be excluded
                                 from tunnel traffic. */
    AutomaticVPNPolicy,     /**< This preference allows an administrator to 
                                 define a policy to automatically manage when a 
                                 VPN connection should be started or stopped. */
    TrustedNetworkPolicy,   /**< This preference allows an administrator to 
                                 define a policy for users in trusted networks.
                                 The options are: Disconnect or DoNothing. */
    UntrustedNetworkPolicy, /**< This preference allows an administrator to 
                                 define a policy for users in untrusted networks.
                                 The options are: Connect or DoNothing. */
	BypassConnectUponSessionTimeout, /**< This preference allows an administrator 
                                 the ability to instruct the client to bypass the 
                                 automatic connection retry after a VPN session timeout. */
    TrustedDNSDomains,      /**< This preference defines a list of comma 
                                 separated DNS suffixes that a network interface
                                 in a trusted network might have. */
    TrustedDNSServers,      /**< This preference defines a list of comma 
                                 separated DNS servers that a network interface
                                 in a trusted network might have. */
    TrustedHttpsServerList,  /**< This preference defines a list of comma separated
                                  https servers reachable only via a trusted network.*/
    AlwaysOn,               /**< This preference governs VPN reestablishment after
                                 interruptions */
    ConnectFailurePolicy,   /**< This preference gives the network administrator 
                                 the ability to dictate the network access allowed
                                 by the client endpoint device following a VPN
                                 connection establishment failure. It is a component
                                 of AlwaysOn */
    AllowCaptivePortalRemediation, /**< This preference gives the network administrator
                                    the ability to dictate the network access 
                                    allowed by the client endpoint device following
                                    a VPN connection establishment failure it is a
                                    component of AlwaysOn */
    CaptivePortalRemediationTimeout, /**< This preference allows the network administrator
                                     the ability to impose a time limit for captive portal 
                                     remediation when the ConnectFailurePolicy value is Closed
                                     It is a component of AlwaysOn */
    ApplyLastVPNLocalResourceRules, /**< This preference gives the network administrator 
                                       the ability to allow split routes and firewall rules 
                                       to be applied following a VPN connection establishment
                                       failure when the ConnectFailurePolicy value is Closed
                                       It is a component of AlwaysOn */
    AllowVPNDisconnect,     /**< During Always On, this specifies that the user is allowed to
                                 disconnect the VPN session. */
    AllowedHosts,           /**< During Always On, user has access to the specified hosts
                                 when VPN is disconnected. */
    EnableScripting,        /**< This preference allows an administrator to 
                                 enable scripting (on connect or on
                                 disconnect). */
    TerminateScriptOnNextEvent,   /**< This preference dictates whether or not
                                       AnyConnect will terminate a running script
                                       process if a transition to another
                                       scriptable event occurs. */
    EnablePostSBLOnConnectScript, /**< This preference is used to control whether
                                       or not the OnConnect script will be launched
                                       from the desktop GUI when a tunnel has been
                                       established via SBL. */
    AutomaticCertSelection,   /**< This preference dictates whether or not to disable
                                   the default automatic certificate selection for user
                                   certificates. If disabled, a certificate selection dialog is
                                   displayed. This only applies if the GUI is enabled
                                   and not SBL. This only applies to Windows (not WinMobile). */
    RetainVpnOnLogoff,        /**< First control of the logoff behavior. This preference allows
                                   an administrator to control if the VPN is terminated or retained
                                   after user logs off.*/
    UserEnforcement,          /**< Second control of the logoff behavior. When the VPN connection has
                                   been retained after user logged off. Controls what user can log in 
                                   and keep the VPN connection. Options are same user only and any user. */
    DeviceLockRequired,           /**< This preference indicates whether or not 
                                       a Windows Mobile device must be configured
                                       with a password or PIN prior to establishing
                                       a VPN connection. This configuration is 
                                       only valid on Windows Mobile devices that
                                       use the Microsoft Default Local 
                                       Authentication Provider (LAP). */
    DeviceLockMaximumTimeoutMinutes,   /**< When set to a non-negative number, 
                                            this preference specifies the maximum
                                            number of minutes a device can be 
                                            inactive before device lock takes 
                                            into effect. (WM5/WM5AKU2+) */
    DeviceLockMinimumPasswordLength,   /**< When set to a non-negative number, 
                                            this preference specifies that any 
                                            PIN/password used for device lock 
                                            must be equal to or longer than
                                            the specified value, in characters.
                                            This setting must be pushed down to
                                            the mobile device by syncing with 
                                            an Exchange server before it can be 
                                            enforced. (WM5AKU2+) */
    DeviceLockPasswordComplexity,      /**< This preference checks whether or 
                                            not the password belongs to one of
                                            three subtypes: alpha, pin, strong */
    EnableAutomaticServerSelection,    /**< Automatic server selection will 
                                            automatically select the optimal 
                                            secure gateway for the endpoint */
    AutoServerSelectionImprovement,    /**< During a reconnection attempt after
                                            a system resume, this setting 
                                            specifies the minimum  estimated
                                            performance improvement required to
                                            justify transitioning a user to a new server 
                                            This value represents percentage in 0..100 */
    AutoServerSelectionSuspendTime,    /**< During a reconnection attempt after
                                            a system resume, this specifies the
                                            minimum time a user must have been 
                                            suspended in order to justify a new
                                            server selection calculation. Unit is hours */
    AuthenticationTimeout,             /**< Time, in seconds, that the client waits
                                            for authentication to be completed.*/
    SafeWordSofTokenIntegration,  /**< This preference will enable the administrator and possibly
                                       the end user to enable SafeWord SofToken integration.
                                       Options are Enabled (true) and Disabled (false - default). */
    AllowIPsecOverSSL,                      /**< if 'true' then tunneling of IPSEC over SSL
                                            is made possible with help from the ASA.
                                        */
    ClearSmartcardPin,                 /**< This preference controls whether the smartcard pin
                                            will be cleared on a successful connection*/
    IPProtocolSupport,                 /**< This preference controls which protocol(s) will be 
                                            allowed for the connection*/
    CaptivePortalRemediationBrowserFailover, /**< This preference is applicable to enhanced captive portal 
                                                  remediation and specifies whether the user is allowed to
                                                  opt for an external browser for remediation, as opposed to 
                                                  the AnyConnect browser. */
    AllowManualHostInput,              /**< This preference specifies whether the user
                                            is allowed to type a new hostname in the VPN
                                            edit box. */
    BlockUntrustedServers,             /**< This preference specifies whether the user wants
                                            to allow for connections to secure gateways with
                                            certificate errors. */
    PublicProxyServerAddress,          /**< This preference specifies the public proxy server
                                            address to be used. This number is in the format
                                            ServerAddr:ServerPort (ex. 101.89.85.444:8080)
                                            or just the FQDN. */
    CertificatePinning,                /**< This preference specifies whether Certificate Pinning
                                            check should be performed during server certificate
                                            verification. */
    UnknownPreference
}; 


/** 
 * Indicates the scope of the preferences contained in a PreferenceInfo object 
 */
#if defined(__midl)
[v1_enum] /*serialize as 32 bits*/
#endif
enum PreferenceScope    
{
    User,               /**< Indicates that the preferences were set by a user */
    Global,             /**< Indicates that the preferences are global */
    UserAndGlobal       /**< Indicates that we have both user and global preferences */
};

/** 
 * Indicates the client mode of operation. Unlike tunneling mode or other 
 * mutually exclusive modes, client operating modes are independent settings,
 * several of which can be turned on simultaneously.  
 */
#if defined(__midl)
[v1_enum] /*serialize as 32 bits*/
#endif
enum OperatingMode
{
    FIPS                     = (1 << 0), /**< Indicates that the client is 
                                              running in FIPS mode. */
    StartBeforeLogon         = (1 << 1), /**< Indicates that the client is 
                                              running in Start Before Login 
                                              mode. */
    GUI                      = (1 << 2), /**< Indicates that the client is 
                                              a GUI client. */
    TrustedNetworkDetection  = (1 << 3), /**< Indicates that a Trusted Network
                                              Detection policy is enabled for
                                              the client. */
    AlwaysOnVpn              = (1 << 4), /**< Indicates that the Always On 
                                              policy is enabled for the client. */
    NetworkIssue             = (1 << 5), /**< For user notifications only.
                                              Indication by API to the UI that
                                              there is a network condition. */
    Quarantined              = (1 << 6), /**< Indicates that the VPN session is being 
                                              Quarantined by the secure gateway. */
    AutomaticHeadendSelection= (1 << 7), /**< Indicates that Automatic Headend
                                              is enabled. */
    DisconnectAllowed        = (1 << 8), /**< Indicates that the user is allowed
                                              to disconnect the VPN based on 
                                              policy. */
    VPNDisabled              = (1 << 9), /**< Indicates that the VPN service is
                                              to be marked as disabled. */
    SCEPMode                 = (1 << 10), /**< Indicates that the client is
                                               performing a SCEP cert enrollment. */
    OnTrustedNetwork         = (1 << 11), /**< Indicates that at last check, the
                                               client detected that it was on
                                               a trusted network. */
    ManualHostInputAllowed   = (1 << 12), /**< Indicates that the user is allowed
                                               to add a new host by typing its name
                                               in the VPN edit box. */
    ErrorSuppressed          = (1 << 13), /**< Indicates a connection error has
                                               been returned fronm the agent, but
                                               was suppressed to warning to 
                                               prevent popup dialog in the UI. */
    StrictMode               = (1 << 14), /**< Indicates that the client is 
                                               running in strict certificate trust mode. */
    CLI                      = (1 << 15), /**< Indicates that the client is
                                               a CLI client. */
    Management               = (1 << 16)  /**< Indicates that the client is strictly
                                               used for initiating a management tunnel. */
};

/**
 * Indicates the last error seen by the API in this connection attempt.
 */
#if defined(__midl)
[v1_enum] /*serialize as 32 bits*/
#endif
enum VPNError
{
    VPNSuccess = 0,                      /**< No error has occurred. */
    VPNError_Generic_FatalError = 1,     /**< An error of unknown type has
                                              occured */

    VPNError_Connection_Error = 100,     /**< An unknown connection error has
                                              occured, such as bad hostname, bad
                                              group, etc. */
    VPNError_Connection_InvalidGroupURL, /**< Invalid Group URL specified in the 
                                              server address */

    VPNError_Network_Error = 200,        /**< An unknown network error has
                                              occured, such as DNS resolution
                                              error, unable to open socket,
                                              routing error, captive portal,
                                              etc. */

    VPNError_Authentication_Error = 300, /**< An unknown user authenication
                                              error has occured. */

    VPNError_Authentication_DAP_Terminate, /**< Access Denied: Your system does 
                                                not meet policy requirements (DAP). */

    VPNError_ClientCertificate_UnknownError = 400, /**< An unknown client
                                                        certificate error has
                                                        occured. */
    VPNError_ClientCertificate_Missing,  /**< A client certificate is required
                                              but no client certificate has been
                                              found on the system. */
    VPNError_ClientCertificate_Expired,  /**< The client certificate has expired */
    VPNError_ClientCertificate_NotYetValid, /**< The client certificate is not
                                                 yet valid. */

    VPNError_ServerCertificate_UnknownError = 500, /**< An unknown error has
                                                        occured when validating
                                                        the server certificate. */
    VPNError_ServerCertificate_Expired,    /**< The server certificate has
                                                expired. */
    VPNError_ServerCertificate_NotYetValid /**< The server certificate is not
                                                yet valid. */
};

enum CertAuthMode
{
    CertAuth_Automatic, /**< Will try each available certificate in succession
                             until authentication is obtained or we run out of 
                             available certificates */
    CertAuth_Disabled,  /**< Will disable Certificate Based Authentication */
    CertAuth_Manual     /**< Will only use preconfigured certificate to attempt
                             Certificate Based Authentication */
};

enum CertificateType
{
    CertificateType_Client,
    CertificateType_SystemTrusted,
    CertificateType_Server
};

enum CertConfirmReason
{
    CertConfirmReason_Unspecified,
    CertConfirmReason_NameMismatch,
    CertConfirmReason_Expired,
    CertConfirmReason_NotValidYet,
    CertConfirmReason_UntrustedSource,
    CertConfirmReason_InvalidUse,
    CertConfirmReason_Revoked,
    CertConfirmReason_Malformed,
    CertConfirmReason_NotFipsCompliant,
    CertConfirmReason_InvalidateDate,
    CertConfirmReason_SignatureAlgorithm,
    CertConfirmReason_KeySize
};

enum UserResponseError
{
    UserResponseError_None,
    UserResponseError_Unspecified,
    UserResponseError_BadServerCert,
    UserResponseError_SsoNavigation,
    UserResponseError_SsoGettingCookie,
    UserResponseError_SsoTimeout,
    UserResponseError_SsoMissingDependency,
    UserResponseError_SsoClientCertRequest,
    UserResponseError_SsoAborted,
};

#if defined(__midl)
[v1_enum] /*serialize as 32 bits*/
#endif
enum ClientType
{
    ClientType_GUI,
    ClientType_GUI_SBL,
    ClientType_CLI,
    ClientType_MGMT
};

#endif // _APISTDHEADER_
