'''
Created on Aug 4, 2016
Copyright (c) 2016 by Cisco Systems, Inc.
All rights reserved.
@author: aahluwalia
'''

from devpkg.devicemodel import DeviceModel, Features
from devpkg.base.dmlist import DMList
from devpkg.base.dmobject import DMObject
from devpkg.connector import Vifs, Vlan, EncapAss
from shared_config import SharedConfigs

from inline_set import InlineSetList
from interface import NGIPSInterfaceConfigs
from fmc.acpolicy import AccessPolicy
from fmc.security_zone import SecurityZones
from fmc.network_object_group import NetworkObjectGroups
from bridge_group_interface import BridgeGroupInterfaces
from etherchannel import NGIPSEtherChannelConfigs
from fmc.config_keeper import ConfigKeeper
import devpkg.utils.util

class FTDDeviceModel(DeviceModel):
    '''
    This is the representation of the device configuration in IFC model. The structure should be the same as
    device_specification.xml. Namely, at the top level, we have the
        - MFunc: the function configuration. In our case, it can be "NGIPS/Firewall".
        - MGrpCfg: the configuration shared by different functions.
        - MDevCfg: the configuration shared by all groups.
    The order to register the components is important, that is the order to translate configuration
    from IFC format to device format.

    NOTE:
    The configuration structure of the device_specification.xml is based on "Insieme Service Management Integration"
    version 0.2. It is not quite in the final state yet, and is very likely to change.
    '''
    
    def __init__(self, device = None, interfaces = None, features = Features.vnsAll, is_audit = False, is_etherchannel = False):
        '''
        @param device: dict
            to identify the device, passed in from device_script APIs
        @param interfaces: dict
            physical interfaces names passed in from device_script APIs
        @param features: bit-mask from Features
            to indicate what part of configurations to generate
        @param is_audit: boolean
            to indicate if this is used in a xxxxAudit device script API
        '''
        DMObject.__init__(self, ifc_key = FTDDeviceModel.__name__)
        self.device = device
        self.interfaces = interfaces
        self.is_audit = is_audit
        self.probe = ConfigKeeper.get_global(devpkg.utils.util.get_top_uuid_from_device(device))

        'Stuff defined under vnsClusrterCfg > vnsDevCfg section of device_specification.xml'
        """
        if features & Features.vnsDevCfg:
            self.register_child(FailoverConfig())
        """
            
            
        """
        All the stuff defined in vnsClusterCfg section, except those in vnsDevCfg sub-section,
        of device_specification.xml'
        """


        'Child for vsnGrpCfg element'
        if features & Features.vnsMFunc:
            'The following items are passed to vnsMFunc at runtime'
            self.register_child(Vifs())
            self.register_child(DMList('VLAN', Vlan))
            self.register_child(DMList('ENCAPASS', EncapAss))

            'vnsMGrpCfg under which there is vnsMFunc'
            self.register_child(SharedConfigs())

            'Configuration from vnsMDevCfg'
            'This needs to be generated after connectors are processed'
            self.register_child(SecurityZones(probe=self.probe))
            
            # Based on configuration, determine to register NGIPSEtherChanneConfig or NGIPSInterfaceConfig
            if (is_etherchannel):
                self.register_child(NGIPSEtherChannelConfigs(self.probe))
            else:
                self.register_child(NGIPSInterfaceConfigs(self.probe))
            self.register_child(InlineSetList(self.probe))
            self.register_child(BridgeGroupInterfaces(probe=self.probe))
            self.register_child(NetworkObjectGroups(probe=self.probe))
            self.register_child(AccessPolicy(self.probe))
