'''
Created on Apr 12, 2015

@author: Puneet Garg

Copyright (c) 2015 by Cisco Systems, Inc.
All rights reserved.

Classes used APIC API call implementations for a device.
'''

import copy
import json
import traceback
import time

from devpkg.utils.util import asciistr
from devpkg.base.command_interaction import CommandInteraction
from devpkg.base.command_service import CommandService
from ftd.ftddevicemodel import FTDDeviceModel as DeviceModel
from devpkg.devicemodel import Features
from devpkg.devicemodelservice import DeviceModelService
import devpkg.utils.env as env
from devpkg.utils.state_type import State
from devpkg.utils.errors import ConnectionError, IFCConfigError, DeviceConfigError, FMC429Error
from devpkg.utils.errors import DeviceBusyError, FaultCode
from devpkg.utils.util import get_config_root_key, sleep
from devpkg.utils.util import hide_passwords
from devpkg.utils.util import normalize_faults
from devpkg.utils.util import pretty_dict
from devpkg.utils.util import get_top_uuid_from_device, get_access_policy_from_config
import re
from fmc.parsers import dispatch_executor, param_executor, delete_device_object_executor, parse_response_for_target, is_cifs_etherchannel
from fmc.probe import Probe
from fmc.config_maker import Config_Maker, Config_Helper
from fmc.config_keeper import ConfigKeeper
from devpkg.base.dmobject import DMObject
from fmc.selective_probe import SelectiveProbe
from devpkg.base.command_dispatch import CommandDispatch
import devpkg.utils.util
from fmc.parsers import device_object_executor
from ftd.interface import NGIPSInterfaceConfig, interface_executor
from devpkg.base import command_executor

class Status(object):
    'Possible value for "state" in the return value'
    SUCCESS =   0 # thing is fine and dandy
    TRANSIENT = 1 # Temporary failure. Script Engine will retry the configuration by calling device script API again
    PERMANENT = 2 # Configuration failure - could be due to invalid configuration parameter or feature unsupported on device etc.  Script will not retry
    AUDIT =     3 # Script can request to trigger an Audit

def get_config_dict(argv):
    '''Return the last dict object in a list of parameters, which is the configuration parameter.
    Assuming the last dictionary item in the list is it.
    '''
    dicts = filter(lambda x: isinstance(x, dict), argv)
    if len(dicts) > 0:
        return dicts[-1]
    else:
        return None            
            
def get_interfaces_dict(argv):
    '''Return the last dict object in a list of parameters, which is the interfaces parameter
    Assuming the first dictionary item in the list with two dictionary times is it.
    '''
    dicts = filter(lambda x: isinstance(x, dict), argv)
    if len(dicts) == 2:
        return dicts[0]
    else:
        return None            
            
def dump_config_arg(f):
    ''' Decorator used to dump out device , interfaces and configuration argument passed to an API
    '''

    def trace(*argv):
        message = []

        message.append("***** API Called: %s" % f.__name__)

        device = argv[0]
        message.append("[Device argument]\n%s" %
                       json.dumps(hide_passwords(copy.deepcopy(device)),
                                  indent=3))

        interfaces = get_interfaces_dict(argv[1:])
        if interfaces:
            message.append("[Interfaces argument]\n%s" % pretty_dict(interfaces))

        config  = get_config_dict(argv[1:])
        if config:
            message.append("[Configuration argument]\n%s" % pretty_dict(config))

        env.debug('\n'.join(message))

        result = f(*argv)
        if isinstance(result, dict):
            env.debug("[Result of %s]\n%s" % (f.__name__,  result))

        return result
    return trace

def exception_handler(f):
    ''' Decorator used to handle exceptions, only used for xxxModify and xxxAudit APIs for now
    '''
    def handler(*argv, **kwargs):
        config  = get_config_dict(argv[1:])
        root = get_config_root_key(config)
        result = {'state': Status.SUCCESS}
        try:
            ret = f(*argv, **kwargs)
            result = ret if ret else result
        except ConnectionError as e:
            result['faults'] = [(root, FaultCode.CONNECTION_ERROR, asciistr(e))]
            result['state'] = Status.PERMANENT
        except IFCConfigError as e:
            result['faults'] = e.fault_list
            result['state'] = Status.PERMANENT
        except DeviceConfigError as e:
            result['faults'] = e.fault_list
            result['state'] = Status.PERMANENT
        except DeviceBusyError as e:
            result['faults'] = [(root, FaultCode.BUSY_ERROR, asciistr(e))]
            result['state'] = Status.TRANSIENT
        except FMC429Error as e:
            result['faults'] = [(root, FaultCode.FMC_429_ERROR, asciistr(e))]
            result['state'] = Status.PERMANENT
        except Exception as e:
            result['faults'] = [(root, FaultCode.UNEXPECTED_ERROR,
                                "Unexpected exception: " + asciistr(e) + '\n' + 
                                traceback.format_exc())]
            result['state'] = Status.PERMANENT
        finally:
            if 'faults' in result:
                result['faults'] = normalize_faults(result['faults'], config)
            return result
    return handler

'ToDo: CSCvd20931 - Comment out the code block for now to improve the coder coverage since it is not executed in current implementation. vnsMGrpCfg is empty from current device_specification_ftd.xml'
# def get_interface_info(device, interface_names):
#     clii = CommandInteraction('InterfaceStatistics')
#     clii.command = "Devices"
#     clii.param_command = "fmc_config/v1/domain/<domainUUID>/devices/devicerecords/<deviceUUID>/fpinterfacestatistics?expanded=true&name="
#     clii.command_executor = interface_health_executor
#     intrerfaceIndex = 0;
#     interfaceKeys = {}
#     for name, vlan in interface_names:
#         if vlan:
#             int_name = name + '.' + str(vlan)
#         else:
#             int_name = name
#         if not clii.params.has_key(int_name):
#             clii.params[int_name] = CommandParam(param_key = 'interface', param_value = name, parent = clii, param_formatter = 'name', param_command = 'fmc_config/v1/domain/<domainUUID>/devices/devicerecords/<deviceUUID>/fpphysicalinterfaces', command_executor = None)
#             interfaceKeys[int_name] = intrerfaceIndex
#             intrerfaceIndex = intrerfaceIndex + 1
#         
#     executer = CommandService(device, [clii], dispatch_executor)
#     return (executer.execute(False), interfaceKeys)

@dump_config_arg
@exception_handler
def deviceValidate( device,version ):
    def return_version_mismatch():
        result = {}
        result['state'] = Status.PERMANENT
        result['faults'] = [(get_config_root_key({}), FaultCode.UNSUPPORTED_DEVICE, "Device Version Mismatch. Device not supported by package")]
        return result
    
    #lets check length of dn. It should be less than or equal to 40
    try:
        dn = device['dn']
        dn = DMObject.get_unique_graph_string_from_device_dn(dn)
        if len(dn) >= 40:
            result = {}
            result['faults'] = [(get_config_root_key({}), FaultCode.CONFIGURATION_ERROR, """FMC fields are limited to 48 characters and are saved on FMC as "<Field Value>+<Tenant Name>+<L4-L7 Device Name>".  
                Your current Tenant and Device names combined with delimiters (+) are greater than 40 characters, leaving you with 8 character function profile fields. Please reduce your device or tenant name lengths to accommodate this limit.""")]
            result['state'] = Status.PERMANENT
            return result
    except:
        pass
    #Lets check version info based on the version info that we have
    
    if device.has_key("manager"):
        env.debug("Version Sent: %s" % version)
        clii = CommandInteraction("VersionInfo")
        clii.param_command = "fmc_platform/v1/info/serverversion"
        clii.command_executor = param_executor
        clii.response_parser = parse_response_for_target
        clii.param_value = None
        clii.response_parser_arg = {"target" : "serverVersion"}
        executor = CommandService(device, [clii])
        try:
            executor.execute(False)
        except DeviceConfigError as e: #there was a fault.
            result = {}
            result['faults'] = e.fault_list
            result['state'] = Status.PERMANENT
            return result
        except:
            result = {}
            result['state'] = Status.PERMANENT
            result['faults'] = [(get_config_root_key({}), FaultCode.CONNECTION_ERROR, "Cannot login to device")]
            return result
        server_version = clii.param_uuid
        if not re.match(version, server_version): #there was a version mismatch
            return return_version_mismatch()
        return {'faults': [], 'state':0, 'version':server_version}

    return {
            'faults': [], 'state': 0,'version': '1.0'
            }

@dump_config_arg    
def deviceModify( device,interfaces,configuration):
    return {
            'state': 0,'faults': [],'health': []
            }

@dump_config_arg
def deviceAudit( device,interfaces,configuration ):
    return {
            'state': 0,'faults': [],'health': []
            }

@dump_config_arg    
def deviceHealth( device,interfaces,configuration ):
    """
    if device.has_key("manager"):
        #ngipss, connectors = get_all_connectors(configuration)
        result = {}
        try:
            interface_names = [interface[2] for interface in interfaces.iterkeys()]
            interface = []
            for intername in interface_names:
                interface.append((intername, None))
            cli_results = get_interface_info(device, interface)
            interfaceKeys = cli_results[1]
            cli_results = cli_results[0]
        except Exception as e:
            result['state'] = Status.TRANSIENT
            result['faults'] = [([], FaultCode.CONNECTION_ERROR, asciistr(e))]
            return result

        line_status = {}
        for int_name in interface_names:
            line_status[int_name] = 'success' in cli_results[interfaceKeys[int_name]].err_type
        faults = []
        result['health'] = []
        result['state'] = Status.SUCCESS
        up = True
        for int_name in interface_names:
            status = line_status[int_name]
            if not status:
                faults.append(([], FaultCode.LINE_STATUS_ERROR, ''))
            up = up and status
        result['health'].append(([], 100 if up else 0))
        if faults:
            result['faults'] = faults
        return result
    """
    return {'state': 0}

@dump_config_arg
def deviceCounters( device,interfaces,configuration ):
    """
    if device.has_key("manager"):
        result = {'counters': []}
        try:
            interface_names = [interface[2] for interface in interfaces.iterkeys()]
            interface = []
            for intername in interface_names:
                interface.append((intername, None))
            cli_results = get_interface_info(device, interface)
            #interfaceKeys = cli_results[1]
            cli_results = cli_results[0]
            result['state'] = Status.SUCCESS
            for connector, cli_result in zip(interfaces, cli_results):
                path = [connector]
                counters = []
                counters = parse_connector_counters(cli_result.err_msg)
                result['counters'].append((path, counters))
        except Exception as e:
            result['state'] = Status.TRANSIENT
            result['faults'] = [([], FaultCode.CONNECTION_ERROR, asciistr(e))]
        
    
        return result
        
    """
    return {'state': 0}

@dump_config_arg
def clusterModify( device,interfaces,configuration ):
    return modify_operation(device, interfaces, configuration)

@dump_config_arg    
def clusterAudit( device,interfaces,configuration ):
    dispatcher = CommandDispatch(device, dispatch_executor)
    fmc_ip = dispatcher.baseIp
    try:
        devs = device['devs']
        for dev in devs.itervalues():
            if dev['state'] == 3:
                # Both clusterAudit and serviceAduit calls audit_operation and they'd interrupt each other in unexpected way.
                # And instead of waiting for a fixed time which result is unpredictable, let them finish one by one.
                sleep(2) #There so that serviceAudit can get a good head start
                return perform_operation_in_sequence(device, interfaces, configuration, True, Features.vnsClusterCfg)
        raise Exception('We are not running the clusterAudit')
    except:
        env.set_variable(fmc_ip + ':' + env.OPERATION_IN_PROGRESS, "False")
        return {
            'state': 0,'faults': [],'health': []
            }
    
#
# FunctionGroup API
#
@dump_config_arg
def serviceModify( device,configuration ):
    dispatcher = CommandDispatch(device, dispatch_executor)
    fmc_ip = dispatcher.baseIp
    try:
        return perform_operation_in_sequence(device, [], configuration, False, None)
    except:
        env.set_variable(fmc_ip + ':' + env.OPERATION_IN_PROGRESS, "False")

def perform_operation_in_sequence(device, interfaces, configuration, is_audit, features):
    dispatcher = CommandDispatch(device, dispatch_executor)
    fmc_ip = dispatcher.baseIp
    operation_in_progress = env.get_variable(fmc_ip + ':' + env.OPERATION_IN_PROGRESS)
    # This is not expected, but implemented max retries to break a long-hold lock just in case it happened
    max_retries = 2400 # 2400 = 3600 (seconds / hour) * 20 (hours) / 30 ( seconds / iteration), roughly 20 hours
    while operation_in_progress == "True" and max_retries > 0:
        max_retries -= 1
        sleep(30)
        operation_in_progress = env.get_variable(fmc_ip + ':' + env.OPERATION_IN_PROGRESS)
    env.set_variable(fmc_ip + ':' + env.OPERATION_IN_PROGRESS, "True")
    if is_audit:
        ret = audit_operation(device, interfaces, configuration, features)
    else: # modify operation
        ret = modify_operation(device, interfaces, configuration)
        
    env.set_variable(fmc_ip + ':' + env.OPERATION_IN_PROGRESS, "False")
    return ret
    
@dump_config_arg    
def serviceAudit( device,configuration ):
    dispatcher = CommandDispatch(device, dispatch_executor)
    fmc_ip = dispatcher.baseIp
    try:
        return perform_operation_in_sequence(device, [], configuration, True, Features.vnsMFunc)
    except:
        env.set_variable(fmc_ip + ':' + env.OPERATION_IN_PROGRESS, "False")
    
@dump_config_arg
def serviceHealth( device,configuration ):
    """
    ngipss, connectors = get_all_connectors(configuration, DeviceModel)
    
    if connectors:
        result = {}
        try:
            cli_results = get_interface_info(
                device, [connector.get_vif() for connector in connectors])
            interfaceKeys = cli_results[1]
            cli_results = cli_results[0]
        except Exception as e:
            result['state'] = Status.TRANSIENT
            result['faults'] = [([], FaultCode.CONNECTION_ERROR, asciistr(e))]
            return result

        # Build a dictionary of connector line status
        line_status = {}   
        for connector in connectors:
            'Status is True if the line protocol is up'
            name, vlan = connector.get_vif()
            int_name = name + '.' + str(vlan)
            line_status[connector.get_nameif()] = 'success' in cli_results[interfaceKeys[int_name]].err_msg

        faults = []
        result['health'] = []
        result['state'] = Status.SUCCESS
        for ngips in ngipss:
            up = True
            for connector in ngips.iter_connectors():
                status = line_status[connector.get_nameif()]
                if not status:
                    faults.append((connector.get_config_path(),
                                   FaultCode.LINE_STATUS_ERROR,
                                   ''))
                up = up and status
            result['health'].append((ngips.get_config_path(), 100 if up else 0))
        if faults:
            result['faults'] = faults
        return result
    else:
        return {'state': Status.PERMANENT, 'faults': [([], FaultCode.CONFIGURATION_ERROR, 'Service Graph does not have connectors information.')]}
    """
    return {'state': 0}
    
@dump_config_arg
def serviceCounters( device,configuration ):
    """
    result = {'counters': []}

    connectors = get_all_connectors(configuration, DeviceModel)[1]
    if connectors:
        try:
            cli_results = get_interface_info(
                device, [connector.get_vif() for connector in connectors])[0]
            result['state'] = Status.SUCCESS
            for connector, cli_result in zip(connectors, cli_results):
                path = connector.get_config_path()
                counters = []
                counters = parse_connector_counters(cli_result.err_msg)
                result['counters'].append((path, counters))
        except Exception as e:
            result['state'] = Status.TRANSIENT
            result['faults'] = [([], FaultCode.CONNECTION_ERROR, asciistr(e))]
    else:
        result.update({'state': Status.PERMANENT, 'faults': [([], FaultCode.CONFIGURATION_ERROR, 'Service Graph does not have connectors information.')]})

    return result
    """
    return {'state': 0}
#
# EndPoint/Network API
#
@dump_config_arg
def attachEndpoint( device, configuration, endpoints ):
    return {
            'state': 0,
            'faults': [],
            'health': [],
            }
@dump_config_arg    
def detachEndpoint( device, configuration, endpoints ):
    return {
            'state': 0,
            'faults': [],
            'health': [],
            }

@dump_config_arg    
def attachNetwork( device, configuration, networks ):
    return {
        'state': 0,
        'faults': [],
        'health': [],
        }

@dump_config_arg    
def detachNetwork( device, configuration, networks ):
    return {
            'state': 0,
            'faults': [],
            'health': [],
            }

@exception_handler    
def modify_operation(device, interfaces, configuration):
    '''Modify the configuration on a device.
    The configuration dictionary follows the format described above.

    This function is called for create, modify and destroy of graph and its associated functions

    @param device: dict
        a device dictionary
    @param configuration: dict
        configuration in delta form
    @return: Faults dictionary through the exception_handler decorator
    '''
    
    start_time = time.time()
    result = {'state': Status.SUCCESS}

    # Check if the connector is etherchannel or normal interface (physical or sub-interface)
    is_etherchannel = is_cifs_etherchannel(configuration)
    ldev = devpkg.utils.util.get_top_uuid_from_device(device)
    start_time1 = time.time()
    access_policy_name = get_access_policy_from_config(configuration)
    probe = ConfigKeeper.get_global(ldev)
    if isinstance(probe, list):
        probe = Probe(ldev)
    probe.run(device, policy_name=access_policy_name)
    ConfigKeeper.set_global(ldev, probe)
    ConfigKeeper.set_global(ldev + ':' + 'device', device)
    end_time1 = time.time()
    env.debug("time spent in probe: " + str(end_time1 - start_time1) + " seconds")
    devicemodel = DeviceModel(device, interfaces, is_etherchannel=is_etherchannel)

    # Get data for reference purpose only (not for making diff as that's not needed for a modify operation). For example,
    # Security zone reference is needed here because the config from serviceModify doesn't contain security_zone_id but security zone name only,
    # and we need security_zone_id when doing a PUT with the serviceModify config to FMC.
    clis = DeviceModelService(devicemodel).ifc2dm(configuration, devicemodel)
    clis = list(set(clis)) # remove duplicates
    ipv4_from_fmc = []
    generate_ipv4_cli_from_probe(probe, ipv4_from_fmc, ldev)
    clis, ipv4_from_fmc = combine_ipv4_cli_mini_audit(clis, ipv4_from_fmc, ldev)
    # put them together
    clis.extend(ipv4_from_fmc)
    faults = push_modified_clis_to_FMC(device, clis, probe)
    
    if len(faults) > 0:
        result = {}
        result['faults'] = faults
        result['state'] = Status.PERMANENT

    end_time = time.time()
    env.debug("time spent in modify_operation: " + str(end_time - start_time) + " seconds")

    return result

def combine_ipv4_cli_mini_audit(cli_from_apic, ipv4_from_fmc, top_uuid):
    cli_from_apic_to_keep = []
    for cli in cli_from_apic:
        obj_type = cli.get_obj_type()
        if obj_type == "ipv4staticroutes" and cli.state != State.DESTROY:
            ipv4_to_keep_from_fmc = None
            found_exact_same_cli = False
            for route in ipv4_from_fmc:
                (is_same_key, is_same_all) = is_same_ipv4_cli(cli, route, top_uuid)
                if is_same_key:
                    ipv4_to_keep_from_fmc = route
                    if is_same_all:
                        found_exact_same_cli = True
                    break
            if not found_exact_same_cli:
                # keep it even if it's slightly different
                cli_from_apic_to_keep.append(cli)
            if ipv4_to_keep_from_fmc:
                ipv4_from_fmc.remove(ipv4_to_keep_from_fmc)
        else: # Keep all others
            cli_from_apic_to_keep.append(cli)
    return (cli_from_apic_to_keep, ipv4_from_fmc)

def is_same_ipv4_cli(cli, route, top_uuid):
    ifname_from_apic = asciistr(cli.params['interfaceName'].param_value + '_' + top_uuid) if not asciistr(cli.params['interfaceName'].param_value).endswith(top_uuid) else asciistr(cli.params['interfaceName'].param_value)
    ifname_from_fmc = asciistr(route.params['interfaceName'].param_value)

    network_from_apic = asciistr(cli.params['network'].param_value)
    network_from_fmc = asciistr(route.params['network'].param_value)

    gateway_from_apic = asciistr(cli.params['gateway'].param_value)
    gateway_from_fmc = asciistr(route.params['gateway'].param_value)

    metric_value_from_apic = int(cli.params['metric'].param_value) if cli.params.has_key('metric') else 1
    metric_value_from_fmc = int(route.params['metric'].param_value)

    isTunneled_from_apic = asciistr(cli.params['isTunneled'].param_value).lower() if cli.params.has_key('isTunneled') else 'false'
    isTunneled_from_fmc = asciistr(route.params['isTunneled'].param_value).lower() if route.params.has_key('isTunneled') else 'false'

    is_same_interface = ifname_from_apic == ifname_from_fmc
    is_same_network = network_from_apic == network_from_fmc
    is_same_gateway = gateway_from_apic == gateway_from_fmc
    is_same_metric = metric_value_from_apic == metric_value_from_fmc
    is_same_isTunneled = isTunneled_from_apic == isTunneled_from_fmc

    is_same_key = is_same_interface and is_same_network and is_same_gateway
    is_same_all = is_same_key and is_same_metric and is_same_isTunneled
    return (is_same_key, is_same_all)

def generate_ipv4_cli_from_probe(probe, ipv4_from_fmc, top_uuid):
    for route in probe.config_keeper.get('ipv4staticroutes'):
        interfaceName = asciistr(route['interfaceName']) if route.has_key('interfaceName') else ''
        if not interfaceName.endswith(top_uuid):
            # bypass the route if it belongs to another tenant
            continue
        clii = CommandInteraction('IPv4StaticRoute', model_key='IPv4StaticRoute', probe=probe)
        clii.add_basic_interaction("fmc_config/v1/domain/<domainUUID>/devices/devicerecords/<deviceUUID>/routing/ipv4staticroutes", route['id'], delete_device_object_executor, "", id=route['id'])
        clii.add_data_param(interfaceName, 'interfaceName')
        clii.add_data_param(probe.config_keeper.get_network_value_from_ipv4_route(route), 'network')
        clii.add_data_param(probe.config_keeper.get_host_value_from_ipv4_route(route), 'gateway')
        metric_value = int(route['metricValue']) if route.has_key('metricValue') else 1
        clii.add_data_param(metric_value, 'metric')
        isTunneled_value = route['isTunneled'] if route.has_key('isTunneled') else False
        clii.add_data_param(isTunneled_value, 'isTunneled')
        clii.state = State.DESTROY
        ipv4_from_fmc.append(clii)
        
def push_modified_clis_to_FMC(device, clis, probe):
    # We do deletion before addition and modification
    # For deletion, we delete access rules and subinterfaces first before others
    need_deploy = False
    
    securityzones = [[], [], [], []] # list of SZ in state 0, 1, 2, and 3
    inlinesets = [[], [], [], []] # list of inlinssets in state 0, 1, 2, and 3
    ipv4staticroutes = [[], [], [], []]
    physicalinterfaces_SubInterfaceUUID = [[], [], [], []]
    physicalinterfaces_SecurityZoneInterfaceInlineUUID = [[], [], [], []]
    physicalinterfaces = [[], [], [], []]
    subinterfaces = [[], [], [], []]
    etherchannelinterfaces = [[], [], [], []]
    interfacesecurityzones = [[], [], [], []]
    bridgegroupinterfaces = [[], [], [], []]
    networkgroups = [[], [], [], []]
    accessrules = [[], [], [], []]
    accesspolicies = [[], [], [], []]
    policyassignments = [[], [], [], []]
 
    faults = []
    for cli in clis:
        cli_dic = cli.toDict()
        if cli_dic.has_key('state') and cli_dic['state'] in (1, 2, 3):
            need_deploy = True
            obj_type = cli.get_obj_type()
            state = cli_dic['state']
            if obj_type == "securityzones":
                securityzones[state].append(cli)
            elif obj_type == "inlinesets":
                inlinesets[state].append(cli)
            elif obj_type == "ipv4staticroutes":
                ipv4staticroutes[state].append(cli)
            elif obj_type == "physicalinterfaces_SubInterfaceUUID":
                physicalinterfaces_SubInterfaceUUID[state].append(cli)
            elif obj_type == "physicalinterfaces_SecurityZoneInterfaceInlineUUID":
                physicalinterfaces_SecurityZoneInterfaceInlineUUID[state].append(cli)
            elif obj_type == "physicalinterfaces":
                physicalinterfaces[state].append(cli)
            elif obj_type == "subinterfaces":
                subinterfaces[state].append(cli)
            elif obj_type == "etherchannelinterfaces":
                etherchannelinterfaces[state].append(cli)
            elif obj_type == "interfacesecurityzones":
                interfacesecurityzones[state].append(cli)
            elif obj_type == "bridgegroupinterfaces":
                bridgegroupinterfaces[state].append(cli)
            elif obj_type == "networkgroups":
                networkgroups[state].append(cli)
            elif obj_type == "accesspolicies":
                accesspolicies[state].append(cli)
            elif obj_type == "accessrules":
                accessrules[state].append(cli)
            elif obj_type == "policyassignments":
                policyassignments[state].append(cli)
        elif not cli_dic.has_key('state') and cli_dic['command'] == 'AccessPolicy':
            # We don't want to skip AccessPolicy even if 'state' is missing here
            need_deploy = True
            accesspolicies[0].append(cli)

    # When security zone type is changed, it must be deleted and re-created. If we don't see the create CLI, then it's not the case and it must be some other scenario,
    # like service graph is detached where security zone should be deleted, in this case, the deletion operation should be performed after other cleanup operation is 
    # completed due to usage dependency,
    if len(securityzones[1]) > 0:
        faults += push_cli(device, securityzones[3], probe)
        faults += push_cli(device, securityzones[1], probe)
    faults += push_cli(device, inlinesets[1], probe)
    faults += push_cli(device, physicalinterfaces_SecurityZoneInterfaceInlineUUID[1], probe)
    faults += push_cli(device, physicalinterfaces_SubInterfaceUUID[1], probe)
    faults += push_cli(device, physicalinterfaces[1], probe)
    # Due to a change in FMC 6.3 that enforced the security zone mode being compatible with interface mode, we have to remove the 'securityZone'
    # tag at this step otherwise it'll error-out because security zone is in SWITCHED mode and the subinterface is not in BVI interface yet.
    # The error will look like the following:
    #    {"error":{"category":"FRAMEWORK","messages":[{"description":"Security zone should be in same mode as of Interface mode type...."severity":"ERROR"}}
    # The security zone is in SWITCHED mode because that's the ultimate state it'll be and once security zone is created the mode cannot be modified.
    # The association between security zone and interface will be established later when pushing interfacesecurityzones.
    if len(bridgegroupinterfaces[1]) > 0:
        for sub_intf in subinterfaces[1]:
            if sub_intf.params.has_key('securityZone'):
                del sub_intf.params['securityZone']
    faults += push_cli(device, subinterfaces[1], probe)

    # in case same interfaces are used by the old and new BGI, we need to delete it before adding it
    faults += push_cli(device, bridgegroupinterfaces[3], probe)
    faults += push_cli(device, bridgegroupinterfaces[1], probe)

    faults += push_cli(device, etherchannelinterfaces[1], probe)
    faults += push_cli(device, interfacesecurityzones[1], probe)
    faults += push_cli(device, ipv4staticroutes[1], probe)
    faults += push_cli(device, networkgroups[1], probe)
    faults += push_cli(device, accesspolicies[1], probe)
    faults += push_cli(device, accessrules[1], probe)
    faults += push_cli(device, policyassignments[1], probe)

    # Do modify before delete, e.g, when we modify access rule to remove security zone, that should happen before delete the corresponding security zone
    faults += push_cli(device, securityzones[2], probe)
    faults += push_cli(device, inlinesets[2], probe)
    
    # inlineset needs to be deleted before modifying member interface, for example, we cannot remove the nameif from an interface if it's part of an inlineset
    # this happens when we detach a graph in inline mode where we need to cleanup all attributes of an interface and delete the inlineset itself, and in this
    # case we need to delete the inlineset first, then update the interface with a PUT request.
    faults += push_cli(device, inlinesets[3], probe)
    faults += push_cli(device, physicalinterfaces_SubInterfaceUUID[2], probe)
    # The following call is extra with empty security_zone_id when doing modify, so skip it.
    # faults += push_cli(device, physicalinterfaces_SecurityZoneInterfaceInlineUUID[2])
    faults += push_cli(device, physicalinterfaces[2], probe)
    faults += push_cli(device, subinterfaces[2], probe)
    faults += push_cli(device, etherchannelinterfaces[2], probe)
    faults += push_cli(device, interfacesecurityzones[2], probe)
    faults += push_cli(device, bridgegroupinterfaces[2], probe)
    faults += push_cli(device, ipv4staticroutes[2], probe)
    faults += push_cli(device, networkgroups[2], probe)
    faults += push_cli(device, accessrules[2], probe)
    
    faults += push_cli(device, accessrules[3], probe)
    # The rational to move access policy update after access rule deletion is: when there are a lot of rules to be deleted and
    # it exceeded the 15 min service call timeout, the next round of service call can still find the rules (via the policy description) to be deleted. 
    # In another word, if the description is changed after the rules are cleaned up, the system won't be able to find it afterwards.
    faults += push_cli(device, accesspolicies[0], probe)
    faults += push_cli(device, accesspolicies[2], probe)
    faults += push_cli(device, policyassignments[2], probe)
    
    faults += push_cli(device, ipv4staticroutes[3], probe)
    faults += push_cli(device, subinterfaces[3], probe)
    faults += push_cli(device, etherchannelinterfaces[3], probe)
    faults += push_cli(device, physicalinterfaces_SubInterfaceUUID[3], probe)
    if len(securityzones[1]) == 0: # defer security zone deletion operation till now, if it's not the case of changing security zone type.
        faults += push_cli(device, securityzones[3], probe)
    faults += push_cli(device, physicalinterfaces_SecurityZoneInterfaceInlineUUID[3], probe)
    faults += push_cli(device, physicalinterfaces[3], probe)
    faults += push_cli(device, networkgroups[3], probe)
    faults += push_cli(device, accesspolicies[3], probe)
    
    if need_deploy:
        try:
            executer = CommandService(device, [], dispatch_executor)
            executer.execute(True, True)
        except FMC429Error as fmc429error:
            raise FMC429Error(fmc429error.message)
        except Exception as ex:
            if hasattr(ex, 'fault_list'):
                faults += ex.fault_list
    return faults

def push_cli(device, cli, probe):
    faults = []
    if len(cli) > 0:
        try:
            executer = CommandService(device, cli, dispatch_executor)
            executer.execute(False)
            if cli[0].get_obj_type() in ('securityzones'):
                # update probe as other object may depend on them
                ldev = devpkg.utils.util.get_top_uuid_from_device(device)
                obj_type = cli[0].get_obj_type()
                selective_probe = SelectiveProbe(ldev, obj_type)
                selective_probe.run(device)
        except FMC429Error as fmc429error:
            raise FMC429Error(fmc429error.message)
        except Exception as ex:
            if hasattr(ex, 'fault_list'):
                faults += ex.fault_list
    return faults

@exception_handler            
def audit_operation(device, interfaces, configuration, features = Features.vnsNone):
    ''' Device script is expected to fetch the device configuration and reconcile with the
    configuration passed by IFC.

    In this call IFC will pass entire device configuration across all graphs. The device
    should insert/modify any missing configuration on the device. It should remove any extra
    configuration found on the device.

    @param device: dict
        a device dictionary
    @param interfaces: list of strings
        a list of interfaces for this device.
        e.g. [ 'E0/0', 'E0/1', 'E1/0', 'E1/0' ]
    @param configuration: dict
        Entire configuration on the device.
    @param features: bit-mask from Features
        to indicate what part of configurations to generate
    @return: Faults dictionary through the exception_handler decorator

    '''

    start_time = time.time()
    result = {'state': Status.SUCCESS}

    # Check if the connector is etherchannel or normal interface (physical or sub-interface)
    is_etherchannel = is_cifs_etherchannel(configuration)
    ldev = devpkg.utils.util.get_top_uuid_from_device(device)
    if ldev == '' and features != Features.deleteAll:
        return
    start_time1 = time.time()
    access_policy_name = get_access_policy_from_config(configuration)
    probe = ConfigKeeper.get_global(ldev)
    if isinstance(probe, list): # probe is not initialized before
        probe = Probe(ldev)
    probe.run(device, policy_name=access_policy_name)
    ConfigKeeper.set_global(ldev, probe)
    ConfigKeeper.set_global(ldev + ':' + 'device', device)
#     probe.config_keeper.set(ldev + ':' + 'device', device)
    end_time1 = time.time()
    env.debug("time spent in probe: " + str(end_time1 - start_time1) + " seconds")
    devicemodel = DeviceModel(device, interfaces, features, is_audit=True, is_etherchannel=is_etherchannel)
    devicemodelservice =  DeviceModelService(devicemodel)

    # The following call generates the CLIs based on the APIC configuration 'configuration'.
    # Note: as the 2nd parameter is None, this function generates a delta_cfg against an empty FMC config rather than the current FMC config.
    # As a result, here clis is essentially the whole APIC configuration that needs to be pushed to FMC.
    clis_apic = devicemodelservice.generate_dm_delta_cfg(configuration, None, device, interfaces, None, features)

    start_time2 = time.time()
    config_maker = Config_Maker(probe, ldev)
    config = config_maker.create_config()
    # Above is to get all the current FMC configuration based on the probe result
    Config_Helper.edit_all_config_state(config, State.DESTROY)
    # Above is to mark all the FMC configuration as 'to be deleted'
     
    # When tenant is deleted, the configuration value will be {}, in this case we need to figure out if the interface is etherchannel from querying the device
    if configuration == {} or len(configuration.values()) == 0 or configuration.values()[0]['value'] == {}:
        is_etherchannel = is_etherchanne_interface(device, config, interfaces)
    devicemodel = DeviceModel(device, interfaces, is_etherchannel=is_etherchannel)
    clis_fmc = DeviceModelService(devicemodel).ifc2dm(config, devicemodel)
    end_time2 = time.time()
    env.debug("time spent in generating clis_fmc: " + str(end_time2 - start_time2) + " seconds")
    # Above is to generate all the 'to-be-deleted' CLIs based on 'config', which is all the current FMC configurations 
    
    faults = calculate_diff_and_push_to_FMC(device, configuration, probe, clis_apic, clis_fmc, ldev)
    end_time = time.time()
    
    env.debug("time spent in audit_operation: " + str(end_time - start_time) + " seconds")

    if len(faults) > 0:
        result = {}
        result['faults'] = faults
        result['state'] = Status.PERMANENT

    return result

def get_keys_from_deleted_bvi_CLI(bvi, probe):
    # get the bvi interfaces from probe by bvi name. Should be only one member(selected interfaces) from the BVI
    bvi_interfaces = probe.find_all_with_param_regex("name", str(bvi.get_name()))
    if len(bvi_interfaces) > 0 and bvi_interfaces[0].has_key('selectedInterfaces'):
        return bvi_interfaces[0]['selectedInterfaces']
    else:
        return []

def get_keys_from_added_bvi_CLI(bvi):
    if bvi.params.has_key('selectedInterfaces'):
        return bvi.params['selectedInterfaces'].get_value()
    else:
        return []

def get_member_interface_of_BVI_to_be_updated(bvis_to_be_added, bvi_to_be_deleted, probe):
    # find if the same bvi_to_be_deleted exists in bvis_to_be_added, if so,
    # this means the bvi is deleted and then added back, likely the case of changing bvi_id

    # Step 1: find the common interfaces of bvi_to_be_deleted and bvi_to_be_added.
    deleted_bvi_keys = get_keys_from_deleted_bvi_CLI(bvi_to_be_deleted, probe)
    common_interfaces = []
    for bvi in bvis_to_be_added:
        added_bvi_keys = get_keys_from_added_bvi_CLI(bvi)
        for added_bvi_key in added_bvi_keys:
            for deleted_bvi_key in deleted_bvi_keys:
                if added_bvi_key['name'] == deleted_bvi_key['name']:
                    common_interfaces.append(deleted_bvi_key)
                    break

    # Step 2: create CLI and put it to either physicalinterfaces_to_be_modified, etherchannels_to_be_modified, or subinterfaces_to_be_updated accordingly
    clis_to_update = []
    for selected_intf in common_interfaces:

        intf_clii = CommandInteraction(selected_intf['type'], model_key=selected_intf['type'], probe=probe)
        interface_type = 'physicalinterfaces' if selected_intf['type'] == 'PhysicalInterface' else 'etherchanelinterfaces'

        # if vlan is supported(physical interface or virtual with vlan truncking)
        if '.' in selected_intf['name']:
            interface_type = 'subinterfaces'
            url = "fmc_config/v1/domain/<domainUUID>/devices/devicerecords/<deviceUUID>/" + interface_type
            intf_clii.add_basic_interaction(url, selected_intf['name'].split(".")[0], interface_executor, NGIPSInterfaceConfig.GENERAL_INTERFACE)
            intf_clii.idParam.param_value['vlanId'] = selected_intf['name'].split('.')[1]
            intf_clii.add_data_param('SubInterface', 'type')
        else:
            url = "fmc_config/v1/domain/<domainUUID>/devices/devicerecords/<deviceUUID>/" + interface_type
            intf_clii.add_basic_interaction(url, selected_intf['name'], device_object_executor, "")

        intf_clii.add_data_param(True, 'enabled')
        securityZone = {}
        securityZone['type'] = 'SecurityZone'
        security_zone_id = probe.config_keeper.get_security_zone_id_from_interface_name(selected_intf['name'])
        if security_zone_id != '':
            securityZone['id'] = security_zone_id
            intf_clii.add_data_param(securityZone, 'securityZone')
        clis_to_update.append(intf_clii)

    return clis_to_update

def calculate_diff_and_push_to_FMC(device, configuration, probe, clis_apic, clis_fmc, ldev):
    accessrules_to_be_deleted = []
    accesspolicies_to_be_deleted = []
    subinterfaces_to_be_deleted = []
    bridgegroupinterfaces_to_be_deleted = []
    ipv4staticroutes_to_be_deleted = []
    other_clis_to_be_deleted = []
    
    securityzones_to_be_added = []
    inlinesets_to_be_added = []
    physicalinterfaces_to_be_added = []
    physicalinterfaces_to_be_modified = []
    etherchannelinterfaces_to_be_added = []
    etherchannelinterfaces_to_be_modified = []
    subinterfaces_to_be_added = []
    subinterfaces_to_be_updated = []
    ipv4staticroutes_to_be_added = []
    interfacesecurityzones_to_be_added = []
    bridgegroupinterfaces_to_be_added = []
    networkgroups_to_be_added = []
    accesspolicies_to_be_added = []
    accessrules_to_be_added = []
    policyassignments = []

    # The following is to calculate the diff of addition.
    start_time4 = time.time()
    addition_count = 0
    for cli_apic in clis_apic:
        if not cli_apic.cli_in_probe(cli_apic, probe, ldev):
            addition_count += 1
            obj_type = cli_apic.get_obj_type()
            if obj_type == "securityzones":
                securityzones_to_be_added.append(cli_apic)
            elif obj_type == "inlinesets":
                inlinesets_to_be_added.append(cli_apic)
            elif obj_type in ("physicalinterfaces", "physicalinterfaces_SubInterfaceUUID", "physicalinterfaces_SecurityZoneInterfaceInlineUUID"):
                if cli_apic.state == 1:
                    # This cli_apic is to add ifname to the interface
                    physicalinterfaces_to_be_added.append(cli_apic)
                elif cli_apic.state == 2:
                    # This cli_apic is to apply all fields to the interface
                    physicalinterfaces_to_be_modified.append(cli_apic)
            elif obj_type == "etherchannelinterfaces":
                if cli_apic.state == 1:
                    etherchannelinterfaces_to_be_added.append(cli_apic)
                elif cli_apic.state == 2:
                    etherchannelinterfaces_to_be_modified.append(cli_apic)
            elif obj_type == "subinterfaces":
                subinterfaces_to_be_added.append(cli_apic)
            elif obj_type == "ipv4staticroutes":
                ipv4staticroutes_to_be_added.append(cli_apic)
            elif obj_type == "interfacesecurityzones":
                interfacesecurityzones_to_be_added.append(cli_apic)
            elif obj_type == "bridgegroupinterfaces":
                bridgegroupinterfaces_to_be_added.append(cli_apic)
            elif obj_type == "networkgroups":
                networkgroups_to_be_added.append(cli_apic)
            elif obj_type == "accesspolicies":
                accesspolicies_to_be_added.append(cli_apic)
            elif obj_type == "accessrules":
                accessrules_to_be_added.append(cli_apic)
            elif obj_type == "policyassignments":
                policyassignments.append(cli_apic)
    end_time4 = time.time()
    env.debug("time spent in calculating diff add: " + str(end_time4 - start_time4) + " seconds")

    # The following is to calculate the diff of deletion.
    start_time3 = time.time()
    deletion_count = 0
    for cli_fmc in clis_fmc:
        if not CommandInteraction.cli_in_list(cli_fmc, clis_apic, ldev):
            deletion_count += 1
            if cli_fmc.get_obj_type() == 'accesspolicies':
                accesspolicies_to_be_deleted.append(cli_fmc)
            elif cli_fmc.get_obj_type() == 'accessrules':
                accessrules_to_be_deleted.append(cli_fmc)
            elif cli_fmc.get_obj_type() == 'subinterfaces':
                subinterfaces_to_be_deleted.append(cli_fmc)
            elif cli_fmc.get_obj_type() == 'ipv4staticroutes':
                ipv4staticroutes_to_be_deleted.append(cli_fmc)
            elif cli_fmc.get_obj_type() == "bridgegroupinterfaces":
                bridgegroupinterfaces_to_be_deleted.append(cli_fmc)
                # 1. When bridge group interface needs to be added, in the case of modifying BVI
                #       Will update member interfaces of this BVI accordingly depending on the member interface type 
                # 2. When there is no bridge group interface to be added, most likely in the case of deleting graph/tenant,
                #       No need to update member interfaces
                if not bridgegroupinterfaces_to_be_added:
                    continue
                else:
                    clis_to_update = get_member_interface_of_BVI_to_be_updated(bridgegroupinterfaces_to_be_added, cli_fmc, probe)
                    for cli_to_update in clis_to_update:
                        if cli_to_update.command == 'PhysicalInterface':
                            physicalinterfaces_to_be_modified.append(cli_to_update)
                        elif cli_to_update.command == 'EtherChannel':
                            etherchannelinterfaces_to_be_modified.append(cli_to_update)
                        elif cli_to_update.command == 'SubInterface':
                            subinterfaces_to_be_updated.append(cli_to_update)
            elif cli_fmc.get_obj_type() == 'etherchannelinterfaces':
                # etherchannel interfaces cannot be deleted, so continue
                continue
            else:
                other_clis_to_be_deleted.append(cli_fmc)
    end_time3 = time.time()
    env.debug("time spent in generating diff delete: " + str(end_time3 - start_time3) + " seconds")

    '''
    This is clearing FSMC configs command.
    clear_clis = devicemodelservice.generate_clear_device_cfg_commands(clis)
    clis = clear_clis.append(clis)
    '''

    # The following is to push changes to FMC, first deletion, then addition.

    # Allow audit process to complete and report fault at the end
    faults = []

    # delete order should consider dependency: for example, accessrules should be deleted before accesspolicies
    #          BVI should be deleted before subinterfaces in case they are assigned to BVI
    faults = push_config_to_device(device, accessrules_to_be_deleted, dispatch_executor, faults, probe=probe)
    faults = push_config_to_device(device, accesspolicies_to_be_deleted, dispatch_executor, faults, probe=probe)
    faults = push_config_to_device(device, ipv4staticroutes_to_be_deleted, dispatch_executor, faults, probe=probe)
    faults = push_config_to_device(device, bridgegroupinterfaces_to_be_deleted, dispatch_executor, faults, probe=probe)
    faults = push_config_to_device(device, subinterfaces_to_be_deleted, dispatch_executor, faults, probe=probe)
    faults = push_config_to_device(device, other_clis_to_be_deleted, dispatch_executor, faults, probe=probe)

    # This is ROUTED mode case, we create security zones first. The rational behind this is, sub-interface creation will fail if security zone is in SWITCHED mode and we create it first.
    if len(interfacesecurityzones_to_be_added) == 0:
        if len(securityzones_to_be_added) > 1: # Use bulk API
            faults = push_config_to_device(device, securityzones_to_be_added, dispatch_executor, faults, True, probe=probe)
            # After insert security zones using bulk, we need to update the probe data accordingly
            start_selective_probe = time.time()
            selective_probe = SelectiveProbe(ldev, 'securityzones')
            selective_probe.run(device)
            end_selective_probe = time.time()
            env.debug("time spent for this selective probe: " + asciistr(end_selective_probe - start_selective_probe) + " seconds")
        else:
            faults = push_config_to_device(device, securityzones_to_be_added, dispatch_executor, faults, False, probe=probe)
    # Now, adding new changes
    faults = push_config_to_device(device, subinterfaces_to_be_added, dispatch_executor, faults, probe=probe)
    start_selective_probe = time.time()
    selective_probe = SelectiveProbe(ldev, 'subinterfaces')
    selective_probe.run(device)
    end_selective_probe = time.time()

    # bridge group interface may use sub-interface, so we prepare sub-interface above first
    faults = push_config_to_device(device, bridgegroupinterfaces_to_be_added, dispatch_executor, faults, probe=probe)
    faults = push_config_to_device(device, subinterfaces_to_be_updated, dispatch_executor, faults, probe=probe)

    # This is BVI case, we create subinterfaces first, securityzones next. The relationship between them will be established by interfacesecurityzones_to_be_added
    if (len(interfacesecurityzones_to_be_added)) > 0:
        if len(securityzones_to_be_added) > 1: # Use bulk API
            faults = push_config_to_device(device, securityzones_to_be_added, dispatch_executor, faults, True, probe=probe)
            # After insert security zones using bulk, we need to update the probe data accordingly
            start_selective_probe = time.time()
            selective_probe = SelectiveProbe(ldev, 'securityzones')
            selective_probe.run(device)
            end_selective_probe = time.time()
            env.debug("time spent for this selective probe: " + asciistr(end_selective_probe - start_selective_probe) + " seconds")
        else:
            faults = push_config_to_device(device, securityzones_to_be_added, dispatch_executor, faults, False, probe=probe)

    # This is to add ifname to physical interface, which is required when adding inlineset below
    # as only named interface is eligible to be added into inlineset. 
    faults = push_config_to_device(device, physicalinterfaces_to_be_added, dispatch_executor, faults, probe=probe)

    faults = push_config_to_device(device, etherchannelinterfaces_to_be_added, dispatch_executor, faults, probe=probe)
    faults = push_config_to_device(device, ipv4staticroutes_to_be_added, dispatch_executor, faults, probe=probe)
    # Create inlineset would modify the interface mode, which is required in order to apply other interface fields to the interface.
    faults = push_config_to_device(device, inlinesets_to_be_added, dispatch_executor, faults, probe=probe)

    # Here we apply all fields to physical/etherchannel interface, which requires that interface mode matches the
    # mode in security zone, and in the case of INLINE mode, the physical interface mode would have been
    # adjusted by applying the 'inlinesets_to_be_added' in prior step.
    faults = push_config_to_device(device, etherchannelinterfaces_to_be_modified, dispatch_executor, faults, probe=probe)
    faults = push_config_to_device(device, physicalinterfaces_to_be_modified, dispatch_executor, faults, probe=probe)
    faults = push_config_to_device(device, interfacesecurityzones_to_be_added, dispatch_executor, faults, probe=probe)
    faults = push_config_to_device(device, networkgroups_to_be_added, dispatch_executor, faults, probe=probe)
    faults = push_config_to_device(device, accesspolicies_to_be_added, dispatch_executor, faults, probe=probe)
    if len(accessrules_to_be_added) > 1:
        faults = push_config_to_device(device, accessrules_to_be_added, dispatch_executor, faults, True, probe=probe)
    else:
        faults = push_config_to_device(device, accessrules_to_be_added, dispatch_executor, faults, False, probe=probe)
    faults = push_config_to_device(device, policyassignments, dispatch_executor, faults, probe=probe)

    if deletion_count + addition_count > 0:
        try:
            # There are changes, commit
            executor = CommandService(device, [], dispatch_executor)
            executor.execute(True, True)
        except FMC429Error as fmc429error:
            raise FMC429Error(fmc429error.message)
        except Exception as ex:
            if hasattr(ex, 'fault_list'):
                faults += ex.fault_list

    return faults

def push_config_to_device(device, config_list, dispatch_executor, faults, using_bulk = False, probe=None):
    if len(config_list) > 0:
        try:
            executor = CommandService(device, config_list, dispatch_executor, using_bulk)
            executor.execute(False)
            if config_list[0].get_obj_type() in ('securityzones'):
                # update probe as other object may depend on them
                ldev = devpkg.utils.util.get_top_uuid_from_device(device)
                obj_type = config_list[0].get_obj_type()
                selective_probe = SelectiveProbe(ldev, obj_type)
                selective_probe.run(device)
        except FMC429Error as fmc429error:
            raise FMC429Error(fmc429error.message)
        except Exception as ex:
            if hasattr(ex, 'fault_list'):
                faults += ex.fault_list
    return faults

def is_etherchanne_interface(device, config, interfaces):
    devicemodel = DeviceModel(device, interfaces, is_etherchannel=True)
    clis_audit = DeviceModelService(devicemodel).ifc2dm(config, devicemodel)
    for cli in clis_audit:
        if cli.command == 'EtherChannelInterface':
            return cli.name.startswith('Port-channel')
    return False
