

'''
Created on Apr 12, 2015

@author: Puneet Garg

Copyright (c) 2015 by Cisco Systems, Inc.
All rights reserved.

Classes used for device command executor invocation.
'''
from devpkg.utils.util import asciistr
from devpkg.utils.errors import ConnectionError, DeviceConfigError, DeviceBusyError
from devpkg.utils.errors import FaultCode, FMC429Error
from devpkg.base.command_dispatch import CommandDispatch, TokenObjectHelper
import fmc.parsers
from devpkg.utils.util import connection_exception_handler
from devpkg.base.command_executor import CommandExecutor
import inspect
from devpkg.base.requestexecutor.login_executor import LoginExecutor
from devpkg.base.command_interaction import CommandInteraction
import devpkg.utils.env as env
from requests.auth import HTTPBasicAuth
import requests
from devpkg.utils.util import sleep
from random import randint
import json
import devpkg.utils.util
import fmc.config_keeper
import time


class CommandService:
    """
    This class is used to run a batch of clis
    """
    def __init__(self, device, commands, dispatch_executer = None, using_bulk = False):
        """
        @param device: the device json
        @param commands: the list of clis
        @param dispatch_executor: legacy code. Is supposed to set how the login will work. Currently uses LoginExecutor 
        """
        self.device = device
        self.probe = fmc.config_keeper.ConfigKeeper.get_global(devpkg.utils.util.get_top_uuid_from_device(device))
        #self.commands = commands
        self.dispatch_executer = dispatch_executer
        self.using_bulk = using_bulk
        """
        Need to compare each command so that there is only 1 command of same data and type.
        This is to prevent execution of a cli that would do the exact same thing.
        """
        commands_dicts = [c.toDict() for c in commands] #generating an array of dictionaries for each command
        """
        example of a command:
        a CommandInteraction class that has attributes:
        command_executor - function
        params - OrderedDict of other CommandInteractions
        etc...
        
        example of a command_dict:
        a dictionary that has:
        key: value where value is a dictionary or a literal
        """
        commands_to_execute_list = [] #array to store after the comparison
        commands_to_execute_dict = [] #array to store the comparing object
        for command, command_dict in zip(commands, commands_dicts): 
            if not command_dict in commands_to_execute_dict: #if command_dict is not in commands_to_execute_dict. Then new data.
                commands_to_execute_list.append(command)
                commands_to_execute_dict.append(command_dict)
            else:
                #we have to see if command has something in commands_to_execute_list
                if command.is_pointer_in_cli_list_from_params(commands_to_execute_list):
                    ind = commands_to_execute_dict.index(command_dict)
                    del commands_to_execute_dict[ind]
                    del commands_to_execute_list[ind]
                    commands_to_execute_list.append(command)
                    commands_to_execute_dict.append(command_dict)
        
        try:
            
            for cli in commands_to_execute_list:
                cli.remove_pointers_not_in_cli_list_from_params(commands_to_execute_list)

            self.commands = commands_to_execute_list
        except:
            self.commands = commands
        
    
        
    @connection_exception_handler    
    def execute(self, apply_changes=True, hard_error=False):
        """
        runs all of the clis as well as logging into the machine
        @param apply_changes: if set to True will run DeployDevice in the end
        @param hard_error: if set to True will return an exception as soon as we hit an error instead of waiting until the end
        """
        dispatcher = CommandDispatch(self.device, self.dispatch_executer)#create the dispatcher object
        token = TokenObjectHelper.get_token(dispatcher)
        if token is None: #never logged into this device before
            login = LoginExecutor(dispatcher)
            errs = login.execute()
        else:
            start_time = time.time()
            device_uuid = CommandExecutor(dispatcher, CommandInteraction("Device UUID"), self.probe, fmc.parsers.get_device_uuid)
            errs = device_uuid.execute()
            end_time = time.time()
            env.debug("time spent in getting device_uuid=" + str(end_time - start_time) + " seconds")
        if errs != None: #possible errors during login
            faults = []
            for err in errs:
                if err is None: #false alarm
                    continue
                # Check if the ASA is busy with a previous configuration and only report one error
                if err.err_msg.startswith('Command Ignored, Configuration in progress...'):
                    raise DeviceBusyError(err.err_msg)
                if 'conn_error' in err.err_type:
                    raise ConnectionError(err.err_msg)
                faults.append((err.model_key, FaultCode.CONFIGURATION_ERROR, err.err_msg))
                
                raise DeviceConfigError(faults)
        
        def dispatch(clis):
            'deliver a list of CommandInteraction, and return list errors if any'
            if inspect.isclass(clis.command_executor):
                executor = clis.command_executor(dispatcher, clis)
            else:
                executor = CommandExecutor(dispatcher, clis, self.probe, clis.command_executor)
            results = executor.execute()
            if results is None:
                errs = None
            else:
                errs = filter(lambda x: x != None and x.err_msg != None and len(x.err_msg.strip()) > 0, results)
            return errs
        
        def get_bulk_payload(commands):
            payload = []
            for cli in commands:
                executor = CommandExecutor(dispatcher, cli, self.probe, cli.command_executor)
                if hasattr(cli, 'params'):
                    for paramValue in cli.params.values():
                        # In the case of access rule bulk, we need to fill-up the 'newComments' here if it's empty.
                        # Note: param_formatter 'newComments' is available in access rule only
                        if paramValue.param_formatter == 'newComments' and paramValue.param_uuid == '':
                            paramValue.param_uuid = []
                            # Adding 'ACI_GENERATED' prefix as the rule is being added from ACI
                            paramValue.param_uuid.append(':::ACI_GENERATED;ACI:' + paramValue.param_value + ':')
                            break
                payload.append(json.loads(fmc.parsers.get_json(executor)))
            return payload
        
        def make_bulk_request(commands):
            # when using_bulk is True, all the commands must be of the same type, currently only security zones, url must be the same
            command_executor = CommandExecutor(dispatcher, commands[0], self.probe, commands[0].command_executor)
            TokenObjectHelper.get_token(command_executor.dispatch)
            url = command_executor.command_holder.get_url(domainUUID=command_executor.dispatch.domain_uuid) # e.g.: 'fmc_config/v1/domain/e276abec-e0f2-11e3-8169-6d9ed49b625f/object/securityzones/<SecurityZoneUUID>'
            adjusted_url = '/'.join(url.split('/')[:-1]) # remove the last part, i.e. /<SecurityZoneID>, result: 'fmc_config/v1/domain/e276abec-e0f2-11e3-8169-6d9ed49b625f/object/securityzones'
            bulk_url = command_executor.dispatch.url + adjusted_url + '?bulk=true'
            payload = get_bulk_payload(commands)
            env.debug('Requesting POST to URL ' + bulk_url)
            env.debug('Payload is ' + json.dumps(payload))
            header = {'Content-Type': 'application/json', 'x-auth-access-token': command_executor.dispatch.auth_token}
            timeout = 120 #recomended by FMC team
            execption_hit_count = 0
            EXCEPTION_RETRY_COUNT = 10
            while True:
                try:
                    response = requests.post(bulk_url, data=json.dumps(payload), headers=header, timeout=timeout, verify=False)
                    break
                except requests.exceptions.ReadTimeout:
                    execption_hit_count += 1
                    if execption_hit_count <= EXCEPTION_RETRY_COUNT:
                        env.debug('Timed out will try again')
                        sleep(randint(1,10))
                    else:
                        raise
                
            env.debug('response: ' + asciistr(response))            
            
            return (fmc.parsers.parse_response(command_executor, response, None, url, payload, 'POST'), response)
            
            
        
        faults = []
        successes = []
        errs = [None]
        dispatch_exceptions = []
        applied_changes = 0
        
        if self.using_bulk:
            # make bulk request in batch if the number of commands is too big
            batch_size = 25
            start_index = 0
            while True:
                end_index = start_index + batch_size
                subset = self.commands[start_index:end_index]
                start_index = end_index
                if len(subset) == 0:
                    break
                else:
                    make_bulk_request(subset)
        else:
            for i, temp_cli in enumerate(self.commands) :#for each cli
                if self.commands.__len__() == i + 1:
                    dispatcher.is_last = True #not actually used anywhere. Just set in case it becomes useful in the future
                try:
                    errs = dispatch(temp_cli)
                except FMC429Error as fmc429error:
                    raise fmc429error(fmc429error.message)
                except Exception as e:
                    # Ignore error first and report it at the end, so that other good configuration may get the chance to be pushed to FMC.
                    env.debug("Found Exception, but continue so we can apply the rest of the commands")
                    if hasattr(e, 'fault_list'):
                        faults.extend(e.fault_list)
                    continue
            
                if errs and errs.__len__() > 0:
                    for err in errs:
                        if err is None:
                            continue
                        # Check if the ASA is busy with a previous configuration and only report one error
                        if err.err_msg.startswith('Command Ignored, Configuration in progress...'):
                            #session_logout(dispatcher)
                            raise DeviceBusyError(err.err_msg)
                        if 'conn_error' in err.err_type:
                            #session_logout(dispatcher)
                            raise ConnectionError(err.err_msg)
                        if 'success' not in err.err_type:
                            faults.append((err.model_key, FaultCode.CONFIGURATION_ERROR, err.err_msg))
                            if hard_error:
                                raise DeviceConfigError(faults)
    
                        else:
                            successes.append(err)
                applied_changes += 1
                if applied_changes > 25:
                    applied_changes = 0
                    self.apply_change(dispatcher, faults, apply_changes, hard_error)                
                        
            # Commit configuration changes to sensor.
        self.apply_change(dispatcher, faults, apply_changes, hard_error)                
        
        if faults.__len__() > 0:
            if len(dispatch_exceptions) > 0:
                faults.append((err.model_key, FaultCode.UNEXPECTED_ERROR, asciistr(dispatch_exceptions)))
            raise DeviceConfigError(faults)

        if successes.__len__() > 0:
            if len(dispatch_exceptions) > 0:
                faults = []
                faults.append((err.model_key, FaultCode.UNEXPECTED_ERROR, asciistr(dispatch_exceptions)))
                raise DeviceConfigError(faults)
            return successes       
        
    @connection_exception_handler    
    def apply_change(self, dispatcher, faults, apply_changes=True, hard_error=False):
        if dispatcher.is_apply_changes and not faults and apply_changes:
            errs = [None]
            appyChangesExecutor = CommandExecutor(dispatcher, CommandInteraction("Apply Changes"), self.probe, fmc.parsers.apply_changes_executor_handler)
            errs = appyChangesExecutor.execute()
            if errs and errs.__len__() > 0 and errs[0] is not None:
                for err in errs:
                    if err and 'success' not in err.err_type:
                        faults.append((err.model_key, FaultCode.CONFIGURATION_ERROR, err.err_msg))            
