'''
Created on Feb 25, 2015

@author: pgarg

Copyright (c) 2015 by Cisco Systems

'''
from collections import OrderedDict

#from devpkg.base.command_interaction import CommandInteraction, command_param_factory
import devpkg.base.command_interaction
from devpkg.utils.state_type import Type, State
from devpkg.utils.type2key import type2key
from devpkg.base.validators import Validator
from devpkg.utils.errors import FaultCode
#from devpkg.utils.util import normalize_param_dict
import devpkg.utils.util
import devpkg, re

class DMObject(object):
    '''
    Device Model Object.
    This is base class of device model Object representation i.e. string, list, interface. It defines base implementation required by Device Model Object.
    This can be extended to implement specific Object and it's behavior.
    
    '''

    NOT_MODIFIABLE_ERROR = "Modification of the object not allowed. Please delete the old instance, and create the new instance instead."

    def __init__(self, ifc_key = "", dm_key= None, probe=None):
        """
        Constructor
        """
        #the components as dictionary of dmobject.
        self.children = OrderedDict()
        #the key to identify this dmobject in the children of its parent in the IFC model.
        self.ifc_key = ifc_key
        #the key to identify this dmobject for the device configuration object. It could be the attribute name in a device.
        self.dm_key = dm_key
        #parent DMObject
        self.parent = None
        #consider recursive state
        self.recursive_state = None
        #cli for data
        self.cli = None
        #dependent on is being used casually. Lets make it an actual variable
        self.dependent_on = None
        
        #Variables that are used but not instantiated
        self.command_executor = None
        self.delete_executor = None
        self.probe = probe

    def __iter__(self):
        'Support iterating over the children'
        return self.children.itervalues()

    def get_key(self):
        """
        @return:
            the key for this object in the IFC model.
        The value of the key should be the name  of the IFC's vnsParam or vnsFolder element
        """
        return self.ifc_key

    'ToDo: CSCvd20931 - Comment out the code block for now to improve the coder coverage. The code is not executed for current implementation.'
#     def get_dm_key(self):
#         """
#         @return:
#             the string of the device configuration object to identify this object in the IFC model.
#         The value of the key could be the attribute name in a device.
#         """
#         return self.dm_key

    def register_child(self, dmobj):
        """
        Add a dmobject as a child of this dmobject.

        @param dmobj:
            input, a dmobject
        """
        self.children[dmobj.get_key()] = dmobj
        dmobj.parent = self
 
    'ToDo: CSCvd20931 - Comment out the code block for now to improve the coder coverage. The code is not executed for current implementation.'        
#     def unregister_child(self, dmobj):
#         """
#         Remove a dmobject child from children.
#         """
#         del self.children[dmobj.get_key()]        
        

    def get_top(self):
        """
        Recursively traverse from child to top parent object, this is
        useful when access to other object's contents is required
        
        @return:
            the top parent object which is know as the device model
        """
        if (self.parent == None):
            return self
        else:
            return self.parent.get_top()

    'ToDo: CSCvd20931 - Comment out the code block for now to improve the coder coverage. The code is not executed for current implementation.'
#     def get_parent(self):
#         '@return: the parent object'
#         return self.parent
# 
#     def get_ancestor_by_class(self, ancestor_class):
#         """
#         Recursively traverse from child to top parent object looking for an
#         ancestor which is an instance of the specified class.
#         If found, return the object, else return None
#         """
#         if (self.parent == None or isinstance(self.parent, ancestor_class)):
#             return self.parent
#         else:
#             return self.parent.get_ancestor_by_class(ancestor_class)

    def get_child(self, key):
        """
        @return:
            the child for a given key.
        @param key:
            input, a string representing the IFC key of the child dmobject,
            or a triplet:  type, key, instance
        """
        if isinstance(key, tuple):
            kind, key, instance = key
            """
            For certain types of entries in the configuration parameter, the key
            may not corresponding to our internal key.
            Example:
                (Type.ENCAP, '', '3')
            we look up the internal key for it here.
            """
            key = type2key.get(kind, key)
        return self.children[key] if key in self.children else None

    'ToDo: CSCvd20931 - Comment out the code block for now to improve the coder coverage. The code is not executed for current implementation.'
#     def get_child_by_dm_key(self, dm_key):
#         """
#         @return:
#             the child for a given device key.
#         @param dm_key:
#             input, a string representing the device key of the child dmobject.
#  
#         """
#         for child in self.children:
#             if dm_key.startswith(child.get_dm_key()):
#                 return child
#         return None
# 
#     def get_children_count(self):
#         """
#         @return:
#             the number of children in this dmobject.
#         """
#         return len(self.children)


    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        '''Store the ifc configuration for this object. It is used to generate device configuration on
        on calling ifc2dm method.

        @param delta_ifc_key: tuple
            (type, key, instance)
            Notice here that this delta_ifc_key is not the delta_ifc_key for self.children, but rather than delta_ifc_key
            for IFC delta dictionary.
            "key" field corresponds to the ifc_key to self.children dictionary.
        @param delta_ifc_cfg_value: dict
            {
            'state': state,
            'device': device,
            'connector': connector,
            'value': scalar value or config dictionary
            }
        '''

        #the following two attributes will be used by ifc2dm method to perform ifc2dm translation
        self.delta_ifc_key = delta_ifc_key
        self.delta_ifc_cfg_value = delta_ifc_cfg_value
        if not self.children:
            return

        cfg = delta_ifc_cfg_value.get('value')
        if not isinstance(cfg, dict): #terminal node
            return

        for delta_ifc_key, value in cfg.iteritems():
            child = self.get_child(delta_ifc_key)
            if child:
                child.populate_model(delta_ifc_key, value)
            #else:
                #self.log('DMObject.populateModel: %s not found' % name)

    'ToDo: CSCvd20931 - Comment out the code block for now to improve the coder coverage. The code is not executed for current implementation.'
#     def get_ifc_delta_cfg_ancestor(self):
#         '''@return the ancestor DMObject that has ifc_delta_cfg container for this DMObject.
#         '''
#         result = self.parent
#         while result:
#             if result.has_ifc_delta_cfg():
#                 return result
#             result = result.parent
# 
#     def create_missing_ifc_delta_cfg(self):
#         '''For diff_ifc_dm method to work, we need each DMObject container
#         to have proper IFC dictionary entry.
#         @warning: This method is invoked by translator delta engine. Please do no override
#                   it in derived class unless you really know what you are doing.
#         @precondition: populate_model has been called.
#         '''
#         if not self.children:
#             return
#         if  not self.has_ifc_delta_cfg():
#             self.delta_ifc_key = Type.FOLDER, self.ifc_key, ''
#             self.delta_ifc_cfg_value = {'state': State.NOCHANGE, 'value': {}}
#             ancestor = self.get_ifc_delta_cfg_ancestor()
#             if ancestor:
#                 ancestor.delta_ifc_cfg_value['value'][self.delta_ifc_key] =  self.delta_ifc_cfg_value
#         for child in self.children.values():
#             child.create_missing_ifc_delta_cfg()

    def ifc2dm(self, no_dm_cfg_stack,  dm_cfg_list):
        '''
        Generate device configuration from IFC configuration delta. Assume that
        the device model has already populated with IFC delta configuration
        using populate_model method.

        One uses (self.delta_ifc_key, self.delta_ifc_cfg_value) to generate device
        configuration.

        @param no_dm_cfg_stack: stack
            input/output, stack of device calls to contain delete calls
        @param dm_cfg_list: list
            input/output, list of device configurations
        '''
        for child in self.children.values():
            child.ifc2dm(no_dm_cfg_stack, dm_cfg_list)

    def has_ifc_delta_cfg(self):
        '''Determine if there is IFC delta configuration for this class
        @return: boolean
            True if there is IFC delta configuration, otherwise False.
        '''
        if not hasattr(self, 'delta_ifc_cfg_value'):
            return False;
        if not hasattr(self, 'delta_ifc_key'):
            return False;
        if self.delta_ifc_cfg_value == None or self.delta_ifc_key == None:
            return False;
        return True

    def generate_command(self, dm_cfg_list, cfg, **keywords):
        '''Append the given cfg to dm_cfg_list.
        @param dm_cfg_list: list
        @param cfg: str
        @param keywords: parameters to pass to CommandInteraction

        '''
        'Checks to create a new command object or add attributes into existing command object.'
        if dm_cfg_list.__len__() > 0:
            clii = dm_cfg_list[0]
            if clii.probe is None and self.probe:
                clii.probe = self.probe
            clii.params[self.ifc_key] = self.generate_command_param(clii, **keywords)
        else:
            clii = devpkg.base.command_interaction.CommandInteraction(cfg, model_key=self.parent.get_config_path(), probe=self.probe)
            clii.command = self.parent.dm_key
            clii.params[self.ifc_key] = self.generate_command_param(clii, **keywords)
            dm_cfg_list.append(clii)
            
    def generate_command_param(self, clii, **keywords):
        """
        Used internally
        generates the correct CommandParam
        """
        action = self.get_action()
        if action == State.DESTROY and self.is_removable:
            executor = self.delete_executor
            
            if executor:
                value = self.get_value()#we will need to know these values
            else:
                value = self.defaults#these should be null values

        else:
            executor = self.command_executor
            value = self.get_value()
        return devpkg.base.command_interaction.command_param_factory(param_key = self.ifc_key, param_value = value, parent = clii, param_formatter = self.dm_key, param_command = self.dm_gen_template, command_executor = executor, **keywords)            

    def get_config_path(self):
        '@return the IFC configuration path to this object'
        if self.parent:
            result = self.parent.get_config_path()
        else:
            result = []
        if self.has_ifc_delta_cfg():
            result.append(self.delta_ifc_key)
        return result

    def get_value(self):
        '@return configuration value for this object'
        if self.has_ifc_delta_cfg():
            if 'value' in self.delta_ifc_cfg_value:
                value = self.delta_ifc_cfg_value['value']
            elif 'target' in self.delta_ifc_cfg_value:
                value = self.delta_ifc_cfg_value['target']
            else:
                return None
            return devpkg.utils.util.normalize_param_dict(value)

    def get_state(self):
        '@return configuration state for this object'
#         return self.delta_ifc_cfg_value['state'] if self.has_ifc_delta_cfg() else None

        def get_state_from_independants(dependant_str):
            """
            @param dependant_str: a xpath str that points to the independant object
            @return: the state
            """
            independant = self.find(dependant_str)
            tstate = State.NOCHANGE
            if independant != None:
                tstate = independant.get_state() #recursive call
                if tstate == State.NOCHANGE or tstate == None: #these are low priority states
                    tstate = independant.get_state_recursive()
            if tstate == State.DESTROY:
                tstate = State.MODIFY #we do this since if an independant structure is being deleted we just need to modify this object and nothing else
            return tstate
        
        state = self.delta_ifc_cfg_value['state'] if self.has_ifc_delta_cfg() else None
        
        if (state == State.NOCHANGE or state is None) and self.dependent_on is not None:
            #self.dependent_on can be an array if it is dependant on multiple things
            if isinstance(self.dependent_on, list):
                for independant in self.dependent_on:
                    if state == State.NOCHANGE or state == None:
                        state = get_state_from_independants(independant)
                    else:
                        return state
            else:
                state = get_state_from_independants(self.dependent_on)
        
        if state == State.NOCHANGE or state == None:
            return self.get_top_state_recursive()
        else:
            return state
    
    def get_state_recursive(self):
        'Combine the state of self and all of its children'
        if self.children:
            state = self.get_state()
            if state == State.MODIFY:
                # No need to check anymore
                return state

            result = State.NOCHANGE if state == None else state
            for child in self.children.values():
                state = child.get_state_recursive()
                if state != None:
                    if result == State.NOCHANGE:
                        result = state
                    elif result != state:
                        result = State.MODIFY
                    if result == State.MODIFY:
                        # No need to check anymore
                        return result
        else:
            return self.get_state()

    def set_state(self, state):
        'Set the state for this object'
        self.delta_ifc_cfg_value['state'] = state

    def validate_configuration(self):
        '''Check the validity of the IFC configuration for this object.
        @return list of faults
        '''
        result = []
        if isinstance(self, Validator):
            error = self.validate(self.get_value())
            self.report_fault(error, result)

        for child in self.children.values():
            child_result = child.validate_configuration()
            self.report_fault(child_result, result)
        return result

    'ToDo: CSCvd20931 - Comment out the code block for now to improve the coder coverage. The code is not executed for current implementation.'
#     def generate_fault(self, msg, attribute_key = None):
#         """
#         @param msg: str
#             The error message.
#         @param attribute_key: str or list of str
#             The key of a configuration attribute.
#         @return fault, which is a (path, code, str) tuple.'
#         """
#         def get_ifc_key_path(config, attribute_key):
#             '@return IFC key path from normal key path of an attribute'
#             if not attribute_key:
#                 return
#             if isinstance(attribute_key, str):
#                 key_tuple = filter(lambda x: x[1]==attribute_key, config.keys())[0]
#                 return key_tuple
#             'attribute_key is a list of str'
#             key_tuple = get_ifc_key_path(config, attribute_key[0])
#             result = [key_tuple]
#             if len(attribute_key) > 1:
#                 result.extend(get_ifc_key_path(config[key_tuple]['value'], attribute_key[1:]))
#             return result
# 
#         path = self.get_config_path()
#         if not attribute_key:
#             return (path, FaultCode.CONFIGURATION_ERROR, msg)
#         ifc_path = get_ifc_key_path( self.delta_ifc_cfg_value['value'], attribute_key)
#         if isinstance(ifc_path, tuple):
#             ifc_path = [ifc_path]
#         path.extend(ifc_path)
#         return (path, FaultCode.CONFIGURATION_ERROR, msg)

    def report_fault(self, error, result):
        if isinstance(error, str):
            result.append(self.generate_fault(error))
        elif isinstance(error, tuple):
            result.append(error)
        elif isinstance(error, list):
            result.extend(error)

    def get_top_state_recursive(self):
        """
        Recursively traverse from child to top parent object, this is
        useful when access to other object's contents is required
        
        @return:
            the top parent object which is know as the device model
        """
        if isinstance(self.parent, devpkg.devicemodel.DeviceModel) or isinstance(self.parent, devpkg.devicemodel.DMList):
#             return self.get_state_recursive()
            return self.recursive_state
            
        else:
            return self.parent.get_top_state_recursive()
        
    @staticmethod
    def get_unique_graph_string_from_device_dn(string):
        """
        Gets the unique string
        """
        dn = string.split('/')
        tenant = dn[1]
        dev = dn[2]
        tenant = tenant[3:] #remove the first 3 characters (tn-)
        dev = dev[8:] #remove the first 8 characters (lDevVip-)
        return devpkg.utils.util.asciistr(tenant + "_" + dev)
    
    def get_top_uuid(self):
        """
        Gets the DN and gets the 
        @return: the LDevID from the top. a string
        """
        top = self.get_top()
        if not top.has_ifc_delta_cfg():
            return None
        dn = top.get_device_dn()
        try:
            return DMObject.get_unique_graph_string_from_device_dn(dn)
        except:
            return None
    
    def get_unique_graph_string_append(self):
        return "_%s" % self.get_top_uuid()
    
    def does_string_have_unique_graph_string(self, string):
        """
        Check to see if there is a specialized string based on regex
        """
        if re.match("_.+?_.+", string):
            return True
        return False
    
    def find(self, xpath):
        '''
        goes through DMObjects to find the DMOobject in the name
        roughly uses the xpath methodology
        @param xpath: the xpath
        @return: a DMObject 
        '''
        try:
            path_arr = xpath.split('/')
            if len(path_arr) == 1:
                return self.get_child(path_arr[0])
            path = path_arr.pop(0)
            if path == '':#top
                return self.get_top().find('/'.join(path_arr))
            return self.get_child(path).find('/'.join(path_arr))
        except:
            return None
    
    def add_cli_to_array(self, clii, array):
        '''
        adds the cli to the array and sets the cli to its own cli
        @param clii: the clii
        @param array: an array
        @param always_append: if set to True will always append regardless of state
        '''
        self.cli = clii
        self.cli.state = self.get_state()
        #we need to make sure that all the modify calls happen first
        if self.get_state() == State.DESTROY:
            for i in range(len(array)):
                if array[i].state == State.DESTROY:
                    array.insert(i, self.cli)
                    return
            array.insert(0, self.cli)
        else:
            for i in range(len(array)):
                if array[i].state == State.DESTROY:
                    array.insert(i, self.cli)
                    return
            array.append(self.cli)
