'''
Created on Feb 17, 2015
Copyright (c) 2016 by Cisco Systems, Inc.
All rights reserved.
@author: pgarg

Classes used for REST generation and parsing
'''
import json, traceback
import devpkg.utils.env as env
import devpkg.utils.util
from collections import OrderedDict
from fmc.config_keeper import ConfigKeeper

class CommandInteraction(object):
    '''
    Information about a REST used for generation and parsing

    '''
    """
    Some static variables
    """
    DO_NOT_ADD_TO_PUT_POST_JSON = "DONOTADD"
    probe = None
    physicalinterfaces = None
    securityzones = None
    accesspolicies = None
    accessrules = None
    subinterfaces = None
    bridgegroupinterfaces = None
    interfacesecurityzones = None
    
    class CommandInteractionEncoder(json.JSONEncoder):
        def default(self, v):
            types = {
                'CommandInteraction': lambda v: devpkg.utils.util.asciistr(v.get_value()),
                'CommandParam'      : lambda v: devpkg.utils.util.asciistr(v.get_value()),
                'CommandParamValue' : lambda v: devpkg.utils.util.asciistr(v.get_value()),
                'CommandParamAppend': lambda v: devpkg.utils.util.asciistr(v.get_value())
            }
            vtype = type(v).__name__
            if vtype in types:
                return types[type(v).__name__](v)
            else:
                return json.JSONEncoder.default(self, v)
    
    def __init__(self, command, model_key=None, mode_command=None, mode_response_parser=None,
                 response_parser=None, response_parser_arg=None, error_parser=None, params={}, command_executor=None, probe=None):
        '''
        The command parameter is a string containing the CLI.

        The optional model_key parameter is the key from the device model for
            the object that generated this CLI.  It will be used to report
            errors for this CLI that can be associated with the object that
            caused the errors.

        The optional response_parser parameter is a function that will be called
            to parse the response to the command.  If the response should be
            ignored, then it should be set to None.

        The optional response_parser_arg parameter is a dictionary that will be the arguments
            to send to the response_parser.
            
        The optional error_parser is a function that will be called when an unknown  error happens.
        
        params is always set to {} and is here for any code that still uses it in any other way.
        
        The optional command_executer is a function that will be executed in execute.
        
        
        There are variables here from CommandParam as well and are there if they are called by parser.
        '''
        self.command = command
        self.model_key = model_key
        self.response_parser_arg = response_parser_arg
        self.response_parser = response_parser
        self.error_parser = error_parser
        self.params = OrderedDict()
        
        self.mode_command = mode_command
        self.mode_response_parser= mode_response_parser
        
        self.parent = None
        self.param_command = None
        self.command_executor = command_executor
        self.probe = probe
        self.param_value = None
        self.param_uuid = None
        self.removable = OrderedDict()
        self.rest_operation = None
                
    def add_basic_interaction(self, commandBase, name, executor, defaultData, throwError=False, request_exception=[], id=None, mustBeNew=False, getByNameInactive=False):
        """
        Adds the basic find by name and Put/Post/Delete functionality that most Interactions use.
        @param name: is the name for what we are looking for
        @param commandBase: is the url that forms the basis for all of the Get and Put/Post commands
        @param executor: is the executor that will be used for the Put/Post
        @param defaultData: is the data that will be used as a framework when making a Post command
        @param throwError: Optional -- should the get request throw an error. Should be set to True if this interaction cannot Post    
        @param request_exception: an array that states what requests can not be done. Of strings PUT, POST, DELTE, and/or GET
        @param id: the id of the cli. if this is set, dataParam will never be set since the call will never be made.
        @param mustBeNew: if we find it we throw an error
        """
        from fmc.parsers import param_executor, parse_response_for_target_and_store_json, parse_response_for_target, get_by_name_and_store
        self.exception_request_type = request_exception
        self._set_default(defaultData)
        self.name = name
        self.response_parser = parse_response_for_target
        self.response_parser_arg = {"target": "id"}
        
        
        self.dataParam = self.add_data_param("", "STORE", addToJson=False)
        if id:
            self.params['GET'] = self.add_data_param(id, 'idParam', addToJson=False)
        elif name and not getByNameInactive:
            self.params['GET'] = CommandParam(param_key="get_id_%s" % self.command, param_value={"name":name, "expanded":"true", "limit":"10000"}, 
                                           parent=self, param_formatter="%sUUID" % self.command, command_executor=param_executor, response_parser=parse_response_for_target_and_store_json, 
                                           response_parser_arg={"target":"id", "store":self.dataParam}, param_command=commandBase, throw_error=throwError, param_req_formater=CommandInteraction.DO_NOT_ADD_TO_PUT_POST_JSON)
        elif name and getByNameInactive:
            self.params['GET'] = CommandParam(param_key="get_id_%s" % self.command, param_value={"expanded":"true", "limit":"10000"}, 
                                           parent=self, param_formatter="%sUUID" % self.command, command_executor=param_executor, response_parser=get_by_name_and_store, 
                                           response_parser_arg={"name": name, "target":"id", "store":self.dataParam}, param_command=commandBase, throw_error=throwError, param_req_formater=CommandInteraction.DO_NOT_ADD_TO_PUT_POST_JSON)
        else:
            self.params['GET'] = CommandParam(param_key="get_id_%s" % self.command, param_value= {"expanded":"true", "limit":"10000"}, 
                                           parent=self, param_formatter="%sUUID" % self.command, command_executor=param_executor, response_parser=parse_response_for_target_and_store_json, 
                                           response_parser_arg={"target":"id", "store":self.dataParam}, param_command=commandBase, throw_error=throwError, param_req_formater=CommandInteraction.DO_NOT_ADD_TO_PUT_POST_JSON)
        self.idParam = self.params['GET']
        if mustBeNew:
            self.idParam.throw_error = False #otherwise error if we don't and error if we do
            self.idParam.postexecute = item_already_exists_error
        if self.idParam.throw_error:
            #we will add a secondary check since this means that if we dont get any info that we throw an error
            self.idParam.postexecute = item_does_not_exist_error
        self.param_command = commandBase + "/<%s>" % self.params['GET'].param_formatter
        self.command_executor = executor
    
    def add_param(self, param_key, param_value, command_executor, param_command, param_formatter = None, param_req_formater = None, response_parser = None, throw_error = True, response_parser_arg = None, error_parser=None, param_type = None):
        """
        This is the same as as adding a param manually except that you dont have to set the parent and there are more required params.
        It is stored in the same key as the param_key. so if the param_key is "bacon", it will be stored in self.params["bacon"] = CommandParam(...)
        @return: the param that was made
        """
        self.params[param_key] = command_param_factory(typeToMake=param_type, param_key=param_key, param_value=param_value, command_executor=command_executor, param_command=param_command, param_formatter=param_formatter, parent=self,
                                                             param_req_formater=param_req_formater, response_parser=response_parser, throw_error=throw_error, response_parser_arg=response_parser_arg, error_parser=error_parser)
        return self.params[param_key]
    
    def add_data_param(self, data, key, addToJson=True, paramType=None):
        """
        This is here when you want to add non-executing params.
        @param data: the data that you want to store
        @param key: this is going to being everything else. Not only is this how you will get the data (self.params[key])
            but it is also the param_formatter and the param_key.
        @param addToJson: Do you want this data to be added to the json during a Put/Post command.  
        @param paramType: The type of param that you want to add. If None is the basic CommandParam class
        @return: the param that was made
        """
        paramReqFormatter = None
        if not addToJson:
            paramReqFormatter = CommandInteraction.DO_NOT_ADD_TO_PUT_POST_JSON
        self.params[key] = command_param_factory(typeToMake=paramType, param_key=key, param_value=data, param_formatter=key, param_req_formater=paramReqFormatter, parent=self, probe=self.probe)
        self.params[key].param_uuid = data
        return self.params[key]
    
    def change_param_command(self, newCommandBase):
        """
        There are sometimes when the Get command and the Put/Post command are different.
        Use this to change the Put/Post to something else after you execute add_basic_interaction.
        """
        self.param_command = newCommandBase + "/<%s>" % self.params['GET'].param_formatter
        self.dataParam = self.params['DEFAULT']
    
    def get_name(self):
        """
        Will either get the name of this class or the parent class going up.
        """
        if hasattr(self, "name"):
            return self.name
        if self.parent:
            return self.parent.get_name()
        return None
        
    def set_param_device_uuid(self):
        """
        Sometimes you need the device UUID.
        This command will create a param that has paramaters in it in such a way that parsers when making its json
        will set the param_uuid of this param to the device UUID
        """
        self.params['deviceUUID'] = CommandParam(param_key="device_uuid", param_formatter=None, param_value="<deviceUUID>", parent=self, param_req_formater=CommandInteraction.DO_NOT_ADD_TO_PUT_POST_JSON)
    
    def get_param_device_uuid(self):
        """
        Will get the param that was created in set_param_device_uuid.
        It will return None otherwise
        """
        try:
            return self.params['deviceUUID']
        except Exception as e:
            env.debug("Unexpected exception: " + devpkg.utils.util.asciistr(e) + '\n' + traceback.format_exc()) 
            return None
    
    def set_param_device_name(self):
        """
        Sometimes you need the device name.
        This command will create a param that has parameters in it in such a way that parsers when making its json
        will set the param_uuid of this param to the device name
        """
        self.params['deviceName'] = CommandParam(param_key="device_name", param_formatter=None, param_value="<deviceNAME>", parent=self, param_req_formater=CommandInteraction.DO_NOT_ADD_TO_PUT_POST_JSON)
    
    def get_param_device_name(self):
        """
        Will get the param that was created in set_param_device_name.
        It will return None otherwise
        """
        try:
            return self.params['deviceName']
        except Exception as e:
            env.debug("Unexpected exception: " + devpkg.utils.util.asciistr(e) + '\n' + traceback.format_exc())
            return None
    
    def set_get_to_end(self):
        """
        Sets the get function to the end of the calls before the main executor call
        """
        del self.params['GET']
        self.params['GET'] = self.idParam
    
    def should_add_to_json(self):
        """
        Internal and used by parsers.
        """
        if self.param_req_formater != None:
            return False
        if self.param_uuid is None:
            return False
        if self.param_formatter is None:
            return False
        return True
    
    def does_id_exist(self):
        """
        checks to see if the id from the original Get by name exists.
        """
        try:
            if self.dataParam.param_uuid is None or self.dataParam.param_uuid == "":
                return False
            else:
                return True
        except:
            if self.param_uuid == None or self.param_uuid == "":
                return False
            else:
                return True
            

    def _set_default(self, defaultData):
        """
        Internal.
        sets the default json that will be the frame for the Post.
        """
        try:
            if not isinstance(defaultData, dict):
                defaultData = json.loads(defaultData)
            self.defaultData = self.add_data_param(defaultData, "DEFAULT", False)
            self.defaultData.param_uuid = self.defaultData.param_value
        except:
            pass
    
    def get_value(self):
        return self.param_uuid
    
    def get_url(self, new=False, url=None, **kwargs):
        """
        Helper function that gets you the url based on parameters sent and parameters in the url.
        Uses the param_command to make a url by mactching the  <string> with the paramter names.
        EX:  url = api/<some_data>/internal/<other_data>/location/<id>
            paramters = some_data = 'datadata', other_data = 'somethingelse', id = 1234
            output: api/datadata/internal/somethingelse/location/1234
        this also check the param_formaters of all of its children and children of it's parent to fill the gap
        EX: url = api/<some_data>/internal/<other_data>/location/<id>
            paramters = some_data = 'datadata', other_data = 'somethingelse'
            has child of CommandParam that has param_key = id and param_uuid = 4321
            output: api/datadata/internal/somethingelse/location/4321
        @param new: Is this a new url? Used for making Posts.
        @param url: will use the url in url instead of an internal one 
        """
        if url:
            temp_url = '%s' % devpkg.utils.util.asciistr(url) #we want a copy and not a pointer
        elif hasattr(self, "dm_key_command"):
            temp_url = '%s' % devpkg.utils.util.asciistr(self.dm_key_command)
        else:
            temp_url = '%s' % devpkg.utils.util.asciistr(self.param_command)
        if new and temp_url[-1] == ">": #last string is an id that does not exist we must remove it
            index_new_start = temp_url.rfind('<')
            temp_url = temp_url[:-(len(temp_url)-index_new_start)]

        for key, val in kwargs.iteritems():
            if temp_url.find("<%s>" % key) > 0:
                if val is not None and val is not "":
                    temp_url = str.replace(temp_url, "<%s>" % key, val)
                        
        for child in self.params.itervalues():
            if temp_url.find("<%s>" % child.param_formatter) > 0:
                if child.param_uuid is not None or child.param_uuid != "":
                    # In the case of "acUUID", we need to get the id from probe in order to work around an issue in legacy code where we got empty tag
                    if (devpkg.utils.util.asciistr(child.param_uuid) != ''):
                        temp_url = str.replace(temp_url, "<%s>" % child.param_formatter, devpkg.utils.util.asciistr(child.param_uuid))
                    elif child.param_formatter == "acUUID":
                        ac_id = self.probe.config_keeper.get_id_from_probe(devpkg.utils.util.asciistr(child.param_uuid.param_value['name']), devpkg.utils.util.asciistr(child.param_uuid.command)) if self.probe is not None else ''
                        temp_url = str.replace(temp_url, "<%s>" % child.param_formatter, ac_id)
        
        if temp_url.find('<') > 0 or temp_url.find('<') > 0:
            if self.parent:
                try:
                    temp_url = self.parent.get_url(url=temp_url)
                except:
                    pass
            elif self.get_obj_type() == 'securityzones':
                sz_id = self.probe.config_keeper.get_id_from_probe(self.get_name(), 'securityzones') if self.probe is not None else ''
                temp_url = temp_url.replace('<SecurityZoneUUID>', sz_id)
            else:
                # If id is subinterface and is not found, try to get it from probe, e.g. 'fmc_config/v1/..../subinterfaces/<SubInterfaceUUID>'
                subinterface_id = self.probe.config_keeper.get_id_from_probe(devpkg.utils.util.asciistr(self.get_name()), 'subinterfaces') if self.probe is not None else ''
                temp_url = temp_url.replace('<SubInterfaceUUID>', subinterface_id)
                
        return  temp_url
        
    def add_removable(self, key, value=None):
        """
        Add something as removable. Will be removed from the json afterwards.
        Takes things out quite cleanly and efficiantly. Be a bit careful when using.
        @param key: the key where the json will check for. This is the root key to check for.
        @param value: the value that will be removed along with all of its "siblings"
        EX: add_removable("interfaces", "Eth1") and the graph looks like:
        {
            "name"  : "SZName",
            "id"    : "1235",
            "interfaces" : {
                            [<data>],[{"name" : "Eth1", "id":"456"}, {"name" : "Eth2", "id":"668"}]
                            },
            "other":{<some other data>}
            
        }
        after the execution will look like:
        {
            "name"  : "SZName",
            "id"    : "1235",
            "interfaces" : {
                            [<data>],[{"name" : "Eth2", "id":"668"}]
                            },
            "other":{<some other data>}
            
        }
        Removed:
        {"name" : "Eth1", "id":"456"}
        
        If there is multiple "Eth1" they will also be removed
        """
        if self.removable.has_key(key):
            self.removable[key].append(value)
        else:
            self.removable[key] = [value]
        
    def is_value_in_removable(self, key, value):
        """
        Internal Checks if a certain key value pair exist in removable
        """
        if self.removable.has_key(key):
            try:
                for v in self.removable[key]:
                    if is_string_in_json(v, value):
                        return True
            except:
                return True
        return False
    
    def remove_value_from_dict(self, key, value, dictionary):
        """
        Internal and used by parser. Does the actual removing
        @param key: the parent key
        @param value: the value that will be removed along with all of its "siblings"
        @param dictionary: where we will remove from
        """
        
        def remove_data(where_in, n, parent):
            """
            Removal is done based on what type of data structure the parent was
            """
            try:
                if where_in == "list":
                    parent.remove(n)
                elif where_in == "dict":
                    for key, val in parent.iteritems():
                        if val == n:
                            del parent[key]
            except:
                pass
        
        try:
            if not self.is_value_in_removable(key, value):#is in removable
                return
            #will remove data from the dictinary parrallel to it and all data below
            
            for v in self.removable[key]:#for each value that will be deleted for this value
                if v is None: #we just remove the key
                    del dictionary[key]
                    continue
                key_to_delete_from = dictionary[key] #we start with this point
                node = None
                road = [(key_to_delete_from, "dict", dictionary)]
                """
                road is the que used for this Breadth First Search.
                it always has 3 points of data (where we are now, the type of data structure that we are in now in a string, the parent)
                """
                while len(road) > 0:
                    node, where_in, parent = road.pop()
                    if isinstance(node, list)and v in node: #if we find the value in the node that is a list
                        remove_data(where_in, node, parent)
                    elif isinstance(node, dict) and v in node.itervalues():#if we find the value in the node that is a dictionary
                        remove_data(where_in, node, parent)
                    else: #based on the type that node is we keep on adding it to road to maintain our BFS
                        if isinstance(node, list):
                            for i in node:
                                road.insert(0,(i, "list", node))
                        elif isinstance(node, dict):
                            for i in node.itervalues():
                                road.insert(0, (i, "dict", node))
        except:
            return
    
    
    def toDict(self):
        """
        @return: dict
        return dictionary version of this with some data removed so that it can be easily compared.
        """
        def convert_to_dict(value): #is this a function. we just want the name
            if callable(value):
                return 
            elif isinstance(value, dict): #for dictionaries we have a function call to get the right value
                return ordered_dict_with_param_to_dict(value)
            elif isinstance(value, list):
                return list_with_param_to_list_with_dict(value)
            else:
                try:
                    return value.toDict()
                except:
                    return value
                
        def ordered_dict_with_param_to_dict(ordered_dict):
            """
            Takes a dicionary with mixed data and returns a dictionary with static values
            """
            return_dict = {}
            for k, v in ordered_dict.iteritems():
                if v is None: #object is None and we will not add it to our return dictionary
                    continue
                return_dict[k] = convert_to_dict(v)
            return return_dict
        
        def list_with_param_to_list_with_dict(lst):
            """
            Handles a list
            """
            return_lst = []
            for v in lst:
                if v is None: #object is None and we will not add it to our return dictionary
                    continue
                return_lst.append(convert_to_dict(v))
            return return_lst
            
        self_dict = self.__dict__
        skip_list = ['model_key', 'dataParam', 'defaultData', 'idParam', 'parent']
        ret_dict = {}
        for key, value in self_dict.iteritems():
            if value is None: #object is None and we will not add it to our return dictionary
                continue
            if key in skip_list: #key is in skip_list and will not be in the dictionary
                continue
            ret_dict[key] = convert_to_dict(value)
        
        return ret_dict
    
    def remove_pointers_not_in_cli_list_from_params(self, cli_list):
        """
        @param cli_list: list of clis
        @return: None
        removes all pointers that are not in the cli list. Theese pointer's parents are not going ot be run 
        and are not going to be populated.
        """
        for key, param in self.params.iteritems():
            #we need to traverse the param_value and find all CommandExecutor Objects
            for dat in CommandInteraction.json_to_list(param.param_value):
                if isinstance(dat, CommandInteraction): #this is a pointer
                    remove = True
                    tt = dat
                    while not tt is None:
                        if tt in cli_list: 
                            remove = False
                            break
                        tt = tt.parent
                    if remove:        
                        self.add_removable(key, dat)
                        
    def is_pointer_in_cli_list_from_params(self, cli_list):
        """
        @param cli_list: list of clis
        @return: True or False
        checks if there is a pointer in this object that is in the cli list
        """
        for key, param in self.params.iteritems():
            #we need to traverse the param_value and find all CommandExecutor Objects
            for dat in CommandInteraction.json_to_list(param.param_value):
                if isinstance(dat, CommandInteraction): #this is a pointer
                    exists = False
                    tt = dat
                    while not tt is None:
                        if tt in cli_list: 
                            exists = True
                            break
                        tt = tt.parent
                    if exists:        
                        return True
        return False
    
    def parse_response(self, response, **kwargs):
        'Parses the response to this CLI command'
        """
        if not self.cmd_response_parser:
            return response
        return self.cmd_response_parser(response)
        """
        if self.response_parser:
            return self.response_parser(response, **kwargs)
        
    def postexecute(self, executor):
        return [None]

    def __str__(self):
        return self.command
    
    def process(self, dispatch):
        return None
    
    def __equal__(self, cli_fmc, ldev):
        obj_type = self.get_obj_type()
        IPV4ROUTE = 'ipv4staticroutes'
        if self.get_url(ldev) != cli_fmc.get_url(ldev):
            if obj_type == IPV4ROUTE:
                if cli_fmc.get_obj_type() != IPV4ROUTE:
                    return False
            else:
                return False
        if obj_type == 'securityzones':
            if self.get_name() == cli_fmc.get_name():
                self_interface_mode = self.params['interfaceMode'].param_value if self.params.has_key('interfaceMode') else ''
                other_interface_mode = self.probe.config_keeper.get_mode_from_securityzone_name(cli_fmc.get_name())
                return self_interface_mode == other_interface_mode
        elif obj_type == 'accesspolicies':
            return self.get_name() == cli_fmc.get_name()
        elif obj_type == 'accessrules':
            return self.get_name() == cli_fmc.get_name()
        elif obj_type in ('physicalinterfaces', 'physicalinterfaces_SubInterfaceUUID', 'physicalinterfaces_SecurityZoneInterfaceInlineUUID'):
            # physical interface should always be present
            return True
        elif obj_type == 'inlinesets':
            return self.get_name() == cli_fmc.get_name()
        elif obj_type == 'bridgegroupinterfaces':
            return self.is_same_bvi(cli_fmc)
        elif obj_type == 'subinterfaces':
            return self.get_name() == cli_fmc.get_name()
        elif obj_type == IPV4ROUTE:
            return self.is_same_ipv4_static_route(cli_fmc, ldev)
        else:
            return False

    def is_same_ipv4_static_route(self, other, ldev):
        self_ifname = self.params['interfaceName'].param_value + '_' + ldev if self.params.has_key('interfaceName') else ''
        self_network = self.params['network'].param_value if self.params.has_key('network') else ''
        self_gateway = self.params['gateway'].param_value if self.params.has_key('gateway') else ''
        self_metric = int(self.params['metric'].param_value) if self.params.has_key('metric') and self.params['metric'].param_value is not None else 1
        self_isTunneled = str(self.params['isTunneled'].param_value) if self.params.has_key('isTunneled') else 'false'
        
        other_ifname = other.params['interfaceName'].param_value if other.params.has_key('interfaceName') else ''
        other_network = other.params['network'].param_value if other.params.has_key('network') else ''
        other_gateway = other.params['gateway'].param_value if other.params.has_key('gateway') else ''
        other_metric = int(other.params['metric'].param_value) if other.params.has_key('metric') and other.params['metric'].param_value is not None else 1
        other_isTunneled = str(other.params['isTunneled'].param_value).lower() if other.params.has_key('isTunneled') else 'false'
        
        is_same_ifname = self_ifname == other_ifname
        is_same_network = self_network == other_network
        is_same_gateway = self_gateway == other_gateway
        is_same_metric = self_metric == other_metric
        is_same_isTunneled = self_isTunneled == other_isTunneled

        return is_same_ifname and is_same_network and is_same_gateway and is_same_metric and is_same_isTunneled

    def is_same_bvi(self, cli_fmc):
        is_same = False

        # In the case of transparent BVI, the 'other' BVI may not contain all the detailed info to make an accurate decision,
        # so we get the corresponding BVI from the probe data to make the comparison instead where all detailed info is available.
#         all_bvi_from_FMC = self.probe.config_keeper.get('bridgegroupinterfaces')
#         for each_bvi in all_bvi_from_FMC:
        all_bvi_from_FMC = self.probe.config_keeper.get('bridgegroupinterfaces')
        found_bvi_in_probe = False
        for each_bvi in all_bvi_from_FMC:
            if each_bvi['name'] == cli_fmc.get_name():
                cli_fmc = each_bvi
                found_bvi_in_probe = True
                break
        if not found_bvi_in_probe:
            return False
        if cli_fmc['name'] == self.get_name():
            # This is the exact one we'd like to compare

            # Compare the logical name, IP address, and selected interfaces
            #if self.params.has_key('ifname') and self.params['ifname'].get_value()
            if not CommandInteraction.has_same_bvi_logical_name(cli_fmc, self):
                return False
            if not CommandInteraction.has_same_IPV4(cli_fmc, self):
                return False

            SELECTED_INTERFACES = 'selectedInterfaces'
            if self.params.has_key(SELECTED_INTERFACES) and cli_fmc.has_key(SELECTED_INTERFACES):
                if len(self.params[SELECTED_INTERFACES].toDict()['param_value']) == len(cli_fmc[SELECTED_INTERFACES]):
                    for self_intf in self.params[SELECTED_INTERFACES].toDict()['param_value']:
                        found_in_other = False
                        for other_intf in cli_fmc[SELECTED_INTERFACES]:
                            if self_intf['name'] == other_intf['name']:
                                found_in_other = True
                                break
                        if not found_in_other:
                            return False
                    return True
            elif not self.params.has_key(SELECTED_INTERFACES) and not cli_fmc.has_key(SELECTED_INTERFACES):
                is_same = True
        return is_same

    def get_obj_type(self):
        url = self.get_param_command() # e.g. 'fmc_config/v1/domain/<domainUUID>/object/securityzones/<SecurityZoneUUID>'
        tag_list = url.split('/')
        major = tag_list[-2] if url else None  # e.g. 'securityzones'
        minor = tag_list[-1] if url else None  # e.g. '<SecurityZoneUUID>'
        if major == 'securityzones' and minor == '<SecurityZoneInterfaceBridgeGroupUUID>':
            return 'interfacesecurityzones'
        elif major == 'physicalinterfaces' and minor == '<SubInterfaceUUID>':
            return 'physicalinterfaces_SubInterfaceUUID'
        elif major == 'physicalinterfaces' and minor == '<SecurityZoneInterfaceInlineUUID>':
            return 'physicalinterfaces_SecurityZoneInterfaceInlineUUID'
        elif major == 'subinterfaces' and minor == '<IPv4StaticRouteUUID>':
            return 'ipv4staticroutes'
        else:
            return major
        
    def get_param_command(self):
        return self.param_command

    def get_command(self):
        return self.command  # e.g. 'AccessPolicy', 'SecurityZone'

    @staticmethod
    def cli_in_list(cli_fmc, clis_apic, ldev):
        for cli_apic in clis_apic:
            if cli_apic.__equal__(cli_fmc, ldev):
                return True
        return False

    @staticmethod
    def is_same_gateway(route, host_value):
        (host_name, host_id) = ConfigKeeper.get_host_from_hosts_by_value(host_value)
        if route['gateway'].has_key('object'):
            is_same_gateway = route['gateway']['object']['name'] == host_name
        else:
            is_same_gateway = route['gateway']['literal']['value'] == host_value
        return is_same_gateway
    
    def cli_in_probe(self, cli, probe, ldev):
        if isinstance(probe, list) and len(probe) == 0:
            return False
        cli_type = cli.get_obj_type()
        if cli_type == 'accesspolicies':
            for policy in probe.accesspolicies:
                if policy['name'] == cli.get_name():
                    if policy.has_key('description') and cli.toDict()['params']['description']['param_value'] in policy['description']:
                        return True
            return False
            
        elif cli_type == 'accessrules':
            for rule in probe.accessrules:
                if rule['name'] == cli.get_name():
                    # access policy value maybe stored directly
                    ac_uuid_param_uuid = cli.toDict()['params']['acUUID']['param_uuid']
                    access_policy_of_cli = ac_uuid_param_uuid['param_value']['name'] if isinstance(ac_uuid_param_uuid, dict) and ac_uuid_param_uuid.has_key('param_value') and ac_uuid_param_uuid['param_value'].has_key('name') else devpkg.utils.util.asciistr(ac_uuid_param_uuid)
                    access_policy_of_rule = rule['metadata']['accessPolicy']['name'] if rule.has_key('metadata') else access_policy_of_cli
                    if access_policy_of_rule == access_policy_of_cli:
                        return CommandInteraction.has_equal_zones(cli, rule)
            return False
            
        elif cli_type == 'securityzones':
            for securityzone in probe.securityzones:
                if securityzone['name'] == cli.get_name():
                    # We need to check interfaceMode also
                    if securityzone.has_key('interfaceMode') and cli.toDict()['params'].has_key('interfaceMode') and \
                        devpkg.utils.util.asciistr(securityzone['interfaceMode']) != devpkg.utils.util.asciistr(cli.toDict()['params']['interfaceMode']['param_uuid']):
                        continue
                    else:
                        return True
            return False    
        elif cli_type in ('physicalinterfaces', 'physicalinterfaces_SubInterfaceUUID', 'physicalinterfaces_SecurityZoneInterfaceInlineUUID'):
            return self.has_same_interface(probe.physicalinterfaces, cli, ldev)
        elif cli_type == 'etherchannelinterfaces':
            return self.has_same_interface(probe.etherchannelinterfaces, cli, ldev)
        elif cli_type == 'ipv4staticroutes':
            if not cli.params.has_key('network'):
                return False
            network_value = cli.params['network'].param_value
            (network_name, network_id) = probe.config_keeper.get_network_from_networks_by_value(network_value)
            if not network_name:
                # See FMC defect: CSCvj36576 - Not able to create network with /32 prefix.
                # On FMC, a /32 network is considered a Host rather than a network, so if a network cannot
                # be found above, we might be able to find it in the hosts bucket.
                if network_value is not None and network_value.endswith('/32'):
                    host_value_of_network, prefix = network_value.split('/')
                    (host_name, host_id) = probe.config_keeper.get_host_from_hosts_by_value(host_value_of_network)
                    if not host_name:
                        return False
                    else:
                        # Found network_name from the hosts table
                        network_name = host_name
                else:
                    return False
            host_value = cli.params['gateway'].param_value
            nameif = cli.params['interfaceName'].param_value + '_' + ldev
            (host_name, host_id) = probe.config_keeper.get_host_from_hosts_by_value(host_value)
            for route in probe.config_keeper.get('ipv4staticroutes'):
                is_same_nameif = devpkg.utils.util.asciistr(nameif) == devpkg.utils.util.asciistr(route['interfaceName'])
                if not is_same_nameif:
                    continue
                is_same_gateway = False
                if route['gateway'].has_key('object'):
                    is_same_gateway = devpkg.utils.util.asciistr(route['gateway']['object']['name']) == devpkg.utils.util.asciistr(host_name)
                else:
                    is_same_gateway = devpkg.utils.util.asciistr(route['gateway']['literal']['value']) == devpkg.utils.util.asciistr(host_value)
                is_same_network = devpkg.utils.util.asciistr(network_name) == devpkg.utils.util.asciistr(route['selectedNetworks'][0]['name'])

                # optional parameters: isTunneled(default: False), metic_value(default: 1)
                cli_isTunneled_value = True if cli.params.has_key('isTunneled') and str(cli.params['isTunneled'].param_value).lower() == 'true' else False
                probe_isTunneled_value = True if route.has_key('isTunneled') and str(route['isTunneled']).lower() == 'true' else False
                is_same_isTunneled = cli_isTunneled_value == probe_isTunneled_value

                is_same_metric = False
                if cli_isTunneled_value:
                    # isTunneled is used only for default route where metric is not needed, so assume is_same_metric True
                    is_same_metric = True
                else:
                    cli_metric_value = int(cli.params['metric'].param_value) if cli.params.has_key('metric') and cli.params['metric'].param_value is not None else 1
                    probe_metric_value = route['metricValue'] if route.has_key('metricValue') and route['metricValue'] is not None else 1
                    is_same_metric = cli_metric_value == probe_metric_value
                if is_same_nameif and is_same_network and is_same_gateway and is_same_metric and is_same_isTunneled:
                    return True
            return False
        elif cli_type == 'subinterfaces':
            for subinterface in probe.subinterfaces:
                if devpkg.utils.util.asciistr(subinterface['name']) + '.' + str(subinterface['vlanId']) == cli.get_name():
                    if not subinterface.has_key('securityZone') or not subinterface.has_key('ifname'):
                        return False
                    if not cli.toDict()['params'].has_key('ifname'):
                        if subinterface.has_key('ifname') and devpkg.utils.util.asciistr(subinterface['ifname']).strip() != '':
                            return False
                    has_same_ifname = devpkg.utils.util.asciistr(subinterface['ifname']) == devpkg.utils.util.asciistr(cli.toDict()['params']['ifname']['param_value'])
                    has_same_ipv4 = False
                    if not subinterface.has_key('ipv4') and not cli.toDict().has_key('ipv4'):
                        has_same_ipv4 = True
                    elif subinterface.has_key('ipv4') and cli.toDict()['params']['ipv4']['param_uuid'].has_key('static') and \
                       str(subinterface['ipv4']['static']['netmask']) == str(cli.toDict()['params']['ipv4']['param_uuid']['static']['netmask']) and \
                       str(subinterface['ipv4']['static']['address']) == str(cli.toDict()['params']['ipv4']['param_uuid']['static']['address']) and \
                       str(subinterface['MTU']) == str(cli.toDict()['params']['MTU']['param_value']):
                        has_same_ipv4 = True
                    if not has_same_ifname or not has_same_ipv4:
                        return False
                    # check if security zone is the same
                    sz_id = probe.config_keeper.get_id_from_securityzones(cli.toDict()['params']['securityZone']['param_value']['id']['param_value']['name'])
                    if sz_id == str(subinterface['securityZone']['id']):
                        if subinterface.has_key('activeMACAddress') and cli.toDict()['params'].has_key('activeMACAddress'):
                            if str(subinterface['activeMACAddress']).lower() == str(cli.toDict()['params']['activeMACAddress']['param_value']).lower():
                                return True
                        elif not subinterface.has_key('activeMACAddress') and not cli.toDict()['params'].has_key('activeMACAddress'):
                            return True
                    else:
                        return False
            return False
        elif cli_type == 'bridgegroupinterfaces':
            for bgi in probe.bridgegroupinterfaces:
                if bgi.has_key('bridgeGroupId') and str(bgi['bridgeGroupId']) == cli.toDict()['params']['bridge_group_id']['param_value']:
                    if not CommandInteraction.has_same_bvi_logical_name(bgi, cli):
                        return False
                    if not CommandInteraction.has_same_IPV4(bgi, cli):
                        return False
                    SELECTED_INTERFACES = 'selectedInterfaces'
                    if bgi.has_key(SELECTED_INTERFACES) and cli.params.has_key(SELECTED_INTERFACES):
                        if len(bgi[SELECTED_INTERFACES]) == len(cli.params[SELECTED_INTERFACES].get_value()):
                            for bgi_intf in bgi[SELECTED_INTERFACES]:
                                for cli_intf in cli.params[SELECTED_INTERFACES].get_value():
                                    if bgi_intf['name'] == cli_intf['name']:
                                        return True
                    elif not bgi.has_key(SELECTED_INTERFACES) and not cli.params.has_key(SELECTED_INTERFACES):
                        return True
            return False
        elif cli_type == 'inlinesets':
            for inlineset in probe.inlinesets:
                if type(cli) == CommandInteraction:
                    if inlineset['name'] == cli.get_name():
                        if cli.params.has_key('mtu') and isinstance(cli.params['mtu'], CommandParam) and cli.params['mtu'].toDict().has_key('param_value') \
                                and str(cli.params['mtu'].toDict()['param_value']) == str(inlineset['mtu']):
                            return True
                        elif cli.params.has_key('mtu') and not isinstance(cli.params['mtu'], CommandParam) and str(cli.params['mtu'].toDict()['param_value']) == str(inlineset['mtu']):
                            return True
                        elif not cli.params.has_key('mtu'):
                            return True
                else:
                    if inlineset['name'] == cli.get_name():
                        return True
            return False
        elif cli_type == 'interfacesecurityzones':
            for interfacesecurityzone in probe.interfacesecurityzones:
                if interfacesecurityzone['name'] == cli.get_name():
                    return True
            # We may find it in the securitzones bucket
            for interfacesecurityzone in probe.securityzones:
                if interfacesecurityzone['name'] == cli.get_name():
                    return True

        return False

    @staticmethod
    def has_same_bvi_logical_name(bgi, cli):
        # if the logical name is empty from FMC side -- configKeeper
        is_bgi_ifname_empty = True if not bgi.has_key('ifname') or (bgi.has_key('ifname') and len(bgi['ifname']) == 0) else False

        # if the ifname is empty from APIC configs
        is_cli_ifname_empty = True if not cli.params.has_key('ifname') or (cli.params.has_key('ifname') and len(cli.params['ifname'].get_value()) == 0) else False

        if is_bgi_ifname_empty and is_cli_ifname_empty:
            return True
        elif not is_bgi_ifname_empty and not is_cli_ifname_empty and devpkg.utils.util.asciistr(bgi['ifname']) == cli.params['ifname'].get_value():
            return True

        return False

    def has_same_interface(self, intf_list, cli, ldev):
        for intf in intf_list:
            if intf['name'] == cli.get_name():
                # We skip the comparison on MTU value of inline member interface since we can not change it through REST API starting from FMC6.2.3
                # FMC error if trying to change the MTU value: Cannot change MTU as the Interface is member of an InlineSet
                if intf.has_key('ifname') and cli.toDict()['params'].has_key('ifname') and devpkg.utils.util.asciistr(intf['ifname']) == devpkg.utils.util.asciistr(cli.toDict()['params']['ifname']['param_value']) and \
                   ( (intf.has_key('mode') and str(intf['mode']) == 'INLINE') or \
                   intf.has_key('MTU') and cli.toDict()['params'].has_key('MTU') and str(intf['MTU']) == str(cli.toDict()['params']['MTU']['param_value'])):
                    # further check ipv4
                    same_ipv4 = CommandInteraction.has_same_IPV4(intf, cli)
                    if same_ipv4:
                        # now check security zone
                        if not intf.has_key('securityZone') and not cli.toDict()['params'].has_key('securityZone'):
                            return True
                        elif intf.has_key('securityZone') and cli.toDict()['params'].has_key('securityZone'):
                            sz_id = self.probe.config_keeper.get_id_from_securityzones(cli.toDict()['params']['securityZone']['param_value']['id']['param_value']['name'])
                            if intf.has_key('securityZone') and sz_id == str(intf['securityZone']['id']):
                                if intf.has_key('activeMACAddress') and cli.toDict()['params'].has_key('activeMACAddress'):
                                    if str(intf['activeMACAddress']).lower() == str(cli.toDict()['params']['activeMACAddress']['param_value']).lower():
                                        return True
                                elif not cli.toDict()['params'].has_key('activeMACAddress') and not cli.toDict()['params'].has_key('activeMACAddress'):
                                    return True
                        elif intf.has_key('securityZone') and cli.toDict()['params'].has_key('int_security_zone'):
                            sz_name = cli.toDict()['params']['int_security_zone']['param_value']['security_zone']
                            if not sz_name.endswith(ldev):
                                sz_name += '_' + ldev
                            sz_id = self.probe.config_keeper.get_id_from_securityzones(sz_name)
                            if intf.has_key('securityZone') and sz_id == str(intf['securityZone']['id']):
                                if intf.has_key('activeMACAddress') and cli.toDict()['params'].has_key('activeMACAddress'):
                                    if str(intf['activeMACAddress']).lower() == str(cli.toDict()['params']['activeMACAddress']['param_value']).lower():
                                        return True
                                elif not cli.toDict()['params'].has_key('activeMACAddress') and not cli.toDict()['params'].has_key('activeMACAddress'):
                                    return True
                elif cli.get_obj_type() == 'physicalinterfaces_SecurityZoneInterfaceInlineUUID' and \
                   ( (intf.has_key('mode') and str(intf['mode']) == 'INLINE') or \
                   intf.has_key('MTU') and cli.toDict()['params'].has_key('MTU') and str(intf['MTU']) == str(cli.toDict()['params']['MTU']['param_value'])):
                    # further check ipv4
                    same_ipv4 = CommandInteraction.has_same_IPV4(intf, cli)
                    if same_ipv4:
                        # now check security zone
                        if not intf.has_key('securityZone') and not cli.toDict()['params'].has_key('securityZone'):
                            return True
                        elif intf.has_key('securityZone') and cli.toDict()['params'].has_key('securityZone'):
                            sz_id = self.probe.config_keeper.get_id_from_securityzones(cli.toDict()['params']['securityZone']['param_value']['id']['param_value']['name'])
                            if intf.has_key('securityZone') and sz_id == str(intf['securityZone']['id']):
                                if intf.has_key('activeMACAddress') and cli.toDict()['params'].has_key('activeMACAddress'):
                                    if str(intf['activeMACAddress']).lower() == str(cli.toDict()['params']['activeMACAddress']['param_value']).lower():
                                        return True
                                elif not cli.toDict()['params'].has_key('activeMACAddress') and not cli.toDict()['params'].has_key('activeMACAddress'):
                                    return True
                        elif intf.has_key('securityZone') and cli.toDict()['params'].has_key('int_security_zone'):
                            sz_name = cli.toDict()['params']['int_security_zone']['param_value']['security_zone']
                            if not sz_name.endswith(ldev):
                                sz_name += '_' + ldev
                            sz_id = self.probe.config_keeper.get_id_from_securityzones(sz_name)
                            if intf.has_key('securityZone') and sz_id == str(intf['securityZone']['id']):
                                if intf.has_key('activeMACAddress') and cli.toDict()['params'].has_key('activeMACAddress'):
                                    if str(intf['activeMACAddress']).lower() == str(cli.toDict()['params']['activeMACAddress']['param_value']).lower():
                                        return True
                                elif not cli.toDict()['params'].has_key('activeMACAddress') and not cli.toDict()['params'].has_key('activeMACAddress'):
                                    return True
                else:
                    return False
        return False

    @staticmethod
    def has_same_IPV4(intf, cli):
        is_intf_ipv4_empty = False
        if not intf.has_key('ipv4') or not intf['ipv4'].has_key('static') or len(intf['ipv4']['static']) == 0:
            is_intf_ipv4_empty = True

        is_cli_ipv4_empty = False
        # for BridgeGroupInterface, ipv4 value is stored under params:ipv4:param_uuid:static: {'netmask': '24', 'address': '1.1.2.3'}
        if not cli.toDict()['params'].has_key('ipv4') or not cli.toDict()['params']['ipv4'].has_key('param_uuid') or not cli.toDict()['params']['ipv4']['param_uuid'].has_key('static') or \
                len(cli.toDict()['params']['ipv4']['param_uuid']['static']) == 0:
            is_cli_ipv4_empty = True

        if is_intf_ipv4_empty and is_cli_ipv4_empty:
            return True
        elif not is_intf_ipv4_empty and not is_cli_ipv4_empty and \
                str(intf['ipv4']['static']['netmask']) == str(cli.toDict()['params']['ipv4']['param_uuid']['static']['netmask']) and \
                str(intf['ipv4']['static']['address']) == str(cli.toDict()['params']['ipv4']['param_uuid']['static']['address']):
            return True

        return False

    @staticmethod
    def get_security_zone_from_cli(cli, is_src=True):
        key = 'sourceZones' if is_src else 'destinationZones'
        sz_list = []
        if cli.toDict()['params'].has_key(key):
            obj_list = cli.toDict()['params'][key]['param_uuid']['objects']
            for sz in obj_list:
                sz_list.append(devpkg.utils.util.asciistr(sz['name']))
        return sz_list

    @staticmethod
    def get_security_zone_from_probe(rule, is_src=True):
        key = 'sourceZones' if is_src else 'destinationZones'
        sz_list = []
        if rule.has_key(key):
            obj_list = rule[key]['objects']
            for sz in obj_list:
                sz_list.append(devpkg.utils.util.asciistr(sz['name']))
        return sz_list

    @staticmethod
    def has_equal_zones(cli, rule):
        cli_src_zone = CommandInteraction.get_security_zone_from_cli(cli, True)
        cli_dest_zone = CommandInteraction.get_security_zone_from_cli(cli, False)
        rule_src_zone = CommandInteraction.get_security_zone_from_probe(rule, True)
        rule_dest_zone = CommandInteraction.get_security_zone_from_probe(rule, False)
        return set(cli_src_zone) == set(rule_src_zone) and set(cli_dest_zone) == set(rule_dest_zone)

    def get_operation(self):
        return self.rest_operation
    
    def set_operation(self, operation):
        self.rest_operation = operation

    @staticmethod
    def json_to_list(param, lst=None):
        if lst is None:
            lst = []
        if isinstance(param, list):
            for n in param:
                lst = CommandInteraction.json_to_list(n, lst)
        elif isinstance(param, dict):
            for k, n in param.iteritems():
                lst.append(k)
                lst = CommandInteraction.json_to_list(n, lst)
        else:
            lst.append(param)
        return lst
        
    
class CommandResult:
    'Result from a CLI that has been sent to the ASA'

    ERROR = 1
    INFO = 2
    WARNING = 3

    def __init__(self, cli, err_type, err_msg, model_key):
        self.cli = cli
        self.err_type = err_type
        self.err_msg = err_msg
        self.model_key = model_key


class CommandParam(CommandInteraction):
    'REST request parameter'

    def __init__(self, param_key, param_value, parent = None, param_formatter = None,  param_command = None, param_req_formater = None, command_executor = None, response_parser = None, throw_error = True, response_parser_arg = None, error_parser=None, probe=None):
        """
        CommandParam can be used in many different ways. It is a CommandInteraction but it has an init that is 
        vastly different and the main usage is to organize data that is and will be there.
        
        @param param_key: the "name" of the CommandParam. Doesnt do anything else besides that but it is good 
            for debug.
        @param param_value: the present value of CommandParam. This should be set on creation to something.
            this variable is used differently depending on how the CommandParam is going to be used. Mostly
            this is the data that you have or that you need to give to get the data you need.
            EG: InlineSet has a param_value set as self.ifc_key that happens to be the name of the inlins set
            whose ID it needs. Other times its just data.
        @param parent: THe parent has to be another CommandInteraction type object. This is optional but is strongly recomended to be added.
        @param param_formatter: a string that many other objects like parser uses. It matches string in url or in jsons
        @param param_command: the url in where the command will be sent
        @param param_req_formatter: if it is None will add this param to the Put/Post json
        @param command_executor: the executor to execute
        @param response_parser: the function that will be called to parse the response of a successful command
        @param response_parser_arg: the arguments that the response_parser might need
        @param error_parser: the function that will be called in case of an unsuccessful command
        @param throw_error: If True, if an error occurs will abort the interaction.
        """
        if parent:
            CommandInteraction.__init__(self, parent.command, model_key=parent.model_key, probe=probe)
        else:
            self._command_param_to_command_interaction_setup(parent)
        self.param_key = param_key
        self.param_formatter = param_formatter

        self.param_value = param_value
        self.param_command = param_command
        self.param_req_formater = param_req_formater
        self.command_executor = command_executor
        self.response_parser = response_parser
        self.parent = parent
        self.param_uuid = ""
        self.throw_error = throw_error
        self.response_parser_arg = response_parser_arg
        self.error_parser = error_parser
        self.probe = probe
        
    def _command_param_to_command_interaction_setup(self, parent):
        """
        Sets the CommandParam object so that it can be treated like a CommandInteraction
        adds the needed variables and data.
        """
        try:
            self.command = parent.command
            self.model_key = parent.model_key
        except:    
            self.command = None
            self.model_key = None
        self.params = OrderedDict()
        self.command_executor = None
        
    def process(self, dispatch):
        return None
        
    def __str__(self):
        return devpkg.utils.util.asciistr(self.get_value())
    
def true_false_to_json(data):
    if data == '1':
        return 'true'
    elif data == '0':
        return 'false'
    return data

def ignore_info_response_parser(response):
    'Ignores INFO response, otherwise returns original'
    return None if not response or 'INFO' in response.upper() else response

def ignore_response_parser(response):
    'Ignores any response'
    return None

def ignore_warning_response_parser(response):
    'Ignores WARNING response, otherwise returns original'
    return None if not response or 'WARNING' in response.upper() else response
    
def command_param_factory(typeToMake=None, *args, **kwargs):
    """
    Used Internally
    A basic factory that makes various command_params based on what they must be made
    @param typeToMake: the ParamCommand that is goiing to be made. Must be the class and not a string
    """
    if typeToMake is None:
        return CommandParam(*args, **kwargs)
    else:
        try:
            return typeToMake(*args, **kwargs)
        except Exception as e:
            env.debug("Unexpected exception: " + devpkg.utils.util.asciistr(e) + '\n' + traceback.format_exc()) #should not get here. Check code if we do
            env.debug("in command_param_factory %s is not a correct type" % devpkg.utils.util.asciistr(typeToMake))

def item_already_exists_error(command_interaction, *args, **kwargs):
    """
    item should not exist
    """
    if not command_interaction.command_holder.does_id_exist():
        return [None]
    msg= "Item with name %s already exists. Please choose a different name or delete the current item" % command_interaction.command_holder.get_name()
    errmsg = CommandResult(cli=command_interaction.command_holder.command, err_type='error', err_msg=msg, model_key=command_interaction.command_holder.model_key)
    command_interaction.command_holder.throw_error = True
    return [errmsg]

def item_does_not_exist_error(command_interaction, *args, **kwargs):
    """
    if item does not exist throw an error
    """
    if command_interaction.command_holder.does_id_exist():
        return [None]
    if len(args) > 0:
        msg = devpkg.utils.util.asciistr(args)
    else:
        msg = "Can't find configured %s with Value %s." % (command_interaction.command_holder.command, command_interaction.command_holder.get_name())
    errmsg = CommandResult(cli=command_interaction.command_holder.command, err_type='error', err_msg=msg, model_key=command_interaction.command_holder.model_key)
    command_interaction.command_holder.throw_error = True
    return [errmsg]

def is_string_in_json(string, json_param):
    """
    If we find a string in the json or not
    @param string: the string that we are looking for
    @param json_param: the json class that we are looking for it in
    """
    json_string = json.dumps(json_param, cls=CommandInteraction.CommandInteractionEncoder)
    if string in json_string:
        return True
    return False
