/**
 * Top-level script for recreating the Persistent storage tables,
 * table constraints (e.g. primary keys), triggers AND stored
 * procedures for the OT ELE model state persistence via Oracle
 *
 * Model type manipulating methods mirror what's in InterfaceModelTypeUtil.java
 */

/**
 * Declaration OT memory footprint-specific types, constants AND routines
 */
CREATE OR REPLACE PACKAGE ot_memfp IS
    c_part CONSTANT INTEGER := 16;
    uninit_seq_num CONSTANT INTEGER := -1;

    /**
     * @return sequence number partition representative
     */
    FUNCTION sequence_num_part(model_type IN INTEGER) RETURN INTEGER;

    /**
     * @return lowest # item in partition
     */
    FUNCTION low_part(model_type IN INTEGER) RETURN INTEGER;

    /**
     * @return highest # item in partition
     */
    FUNCTION high_part(model_type IN INTEGER) RETURN INTEGER;

    /**
     * @return move a model_type into a new partition as identified 
     * by the new_seq_num model type
     */ 
    FUNCTION move_part(new_seq_num IN INTEGER, old_model_type IN INTEGER) RETURN INTEGER;

    /**
     * stored procedure to change node id. If new_node_id already exists,
     * the procedure exits
     * @return -1 if new_node_id already exists / 0 if runs 100% to completion
     */
    PROCEDURE change_node_id(old_node_id IN INTEGER, new_node_id IN INTEGER, status OUT INTEGER);

    /**
     * procedure to update the sequence number. The procedure takes
     * care of the conversion of model type to sequence # model type
     */
    PROCEDURE update_seq_num(nid IN INTEGER, mtype IN INTEGER, seq_num INTEGER);

    /**
     * stored procedure to update the sequence number. The procedure takes
     * care of the conversion of model type to sequence # model type
     */
    PROCEDURE get_seq_num(nid IN INTEGER, mtype IN INTEGER, result IN OUT INTEGER);

    /**
     * stored procedure to delete the entire partition identified by mtype
     */
    PROCEDURE del_partition(nid IN INTEGER, mtype IN INTEGER, rowcount OUT INTEGER);

    /**
     * creates an empty blob row (if one does not already exist)
     */
    PROCEDURE insert_empty_blob(nid IN INTEGER, mtype IN INTEGER, ndx IN INTEGER);

END;
/

/* -----------------------------------------------------------------------*/

DECLARE
    table_name VARCHAR(32);
    pk_name VARCHAR(32);
    details VARCHAR(512);
    phy_crit VARCHAR(512);

    /**
     * Centralized logging
     * @param str - user string to be logged
     */
    PROCEDURE log(str IN VARCHAR2) IS
    BEGIN
    	DBMS_OUTPUT.PUT_LINE(str);
    END log;

    /**
     * Convert PSQL BOOLEAN type to a SQL parameter string
     */
    FUNCTION boolean_to_varchar2(flag IN BOOLEAN) RETURN VARCHAR2 IS
    BEGIN
        IF flag = TRUE THEN
           RETURN 'true';
        ELSE
           RETURN 'false';
        END IF;
    END boolean_to_varchar2;

    /**
     * Utility method to help check for the existence of user tables,
     * constraints, etc.
     * @return true iff there is a row in system table: <tableName> with
     * scolumn <colName> with value <criteria>
     */
    FUNCTION doesExist(criteria IN VARCHAR2, tableName IN VARCHAR2, colName IN VARCHAR2) RETURN BOOLEAN IS
        cnt INTEGER;
        exist BOOLEAN;
        str VARCHAR(128);
        crit VARCHAR(32);

    BEGIN
        crit := UPPER(criteria); -- all schema names are presumed stored in upper-case
        str := 'SELECT COUNT(*) FROM ' || tableName || ' WHERE ' || colName || ' = :crit';
        EXECUTE IMMEDIATE str INTO cnt USING IN crit;
        exist := cnt > 0;
        RETURN exist;
    END doesExist;

    /**
     * Utility method to re-create a table.
     * If the table already existed it's dropped AND any associated integrity constraints are dropped as well
     */
    PROCEDURE recreateTable(table_name IN VARCHAR2, record_typedef IN VARCHAR2, opt_phy_crit IN VARCHAR2) IS
        systable CONSTANT VARCHAR(32) := 'user_tables';
        syscol CONSTANT VARCHAR(32) := 'TABLE_NAME';
        banner CONSTANT VARCHAR(32) := '*** table ';
        exist BOOLEAN;

    BEGIN
        exist := doesExist(table_name, systable, syscol);
        log(banner || table_name || ' already exists? ' || boolean_to_varchar2(exist));
        IF exist THEN
            BEGIN
                -- drop the table since it exists. Cascade the drop to any integrity constraints...
                EXECUTE IMMEDIATE 'DROP TABLE ' || table_name || ' CASCADE CONSTRAINTS';
            END;
        END IF;

        EXECUTE IMMEDIATE 'CREATE TABLE ' || table_name || ' (' || details || ' )' || ' ' || opt_phy_crit;

        IF (exist) THEN
            log(banner || table_name || ' re-created...');
        ELSE
            log(banner || table_name || ' created...');
        END IF;
        log('');
    END recreateTable;

    /**
     * Utility method to help re-create a LOB column's criteria
     * This allows things like the (BLOB) column chunk size to be set
     */
    FUNCTION recreateLobCriteria(table_name IN VARCHAR2, col_name IN VARCHAR2, details IN VARCHAR2) RETURN VARCHAR IS

    BEGIN
        RETURN 'LOB (' || col_name || ') STORE AS ( ' || details || ' )';
    END recreateLobCriteria;

    /**
     * Utility method to re-create a primary key.
     * If the key already existed it's dropped
     */
    PROCEDURE recreatePKConstraint(pk_name IN VARCHAR2, table_name IN VARCHAR2, keys IN VARCHAR2) IS
        systable CONSTANT VARCHAR(32) := 'ALL_CONSTRAINTS';
        syscol CONSTANT VARCHAR(32) := 'CONSTRAINT_NAME';
        banner CONSTANT VARCHAR(32) := '*** constraint ';
        exist BOOLEAN;

    BEGIN
        exist := doesExist(pk_name, systable, syscol);
        log(banner || pk_name || ' already exists? ' || boolean_to_varchar2(exist));
        IF exist THEN
            BEGIN
                -- drop the existing constraint
                 EXECUTE IMMEDIATE 'ALTER TABLE ' || table_name || ' DROP CONSTRAINT ' || pk_name;
            END;
        END IF;

        -- add the new constraint
        EXECUTE IMMEDIATE 'ALTER TABLE ' || table_name || ' ADD CONSTRAINT ' || pk_name || ' PRIMARY KEY (' || keys || ' )';

        IF (exist) THEN
            log(banner || pk_name || ' re-created...');
        ELSE
            log(banner || pk_name || ' created...');
        END IF;
        log('');
    END recreatePKConstraint;

BEGIN
    -- 454 product-line model instance state table
    -- Attribute data is represented in the data stream
    -- Works across 454 product type AND versions that support
    -- model state persistence
    --
    table_name := 'ne_model_state';

    details :=
        ' node_id      INT       NOT NULL,' || -- node id
        ' model_type   INT       NOT NULL,' || -- model classname mapped to a unique model class type + added user data - if needed (e.g. slot id)
        ' model_index  INT       NOT NULL,' || -- model tindex
        ' data         BLOB      DEFAULT empty_blob()' -- all attribute state
        ;

    -- keep the blob chunk size down as small as possible
    phy_crit :=
        'CHUNK 512'
    ;

    recreateTable(table_name, details, recreateLobCriteria(table_name, 'data', phy_crit));

    -- primary key for this table consists of the following...
    pk_name := table_name || '_pk';    -- convention is table_name + "_pk"
    details := 'node_id, model_type, model_index';
    recreatePKConstraint(pk_name, table_name, details);


    --  454 product-line model 'type' state
    --  Primary attribute support is a model-type wide sequence number
    --
    table_name := 'ne_model_type';

    details :=
        'node_id        INT        NOT NULL,' || -- node id
        'model_type     INT        NOT NULL,' || -- model classname mapped to a unique model class type + opt user data (e.g. slot id)
        'signature      BLOB,'                || -- can be used to store class-portion (i.e. ObjectStreamClass part) of the object stream
        'sequence_num   INT'                     -- if you want to maintain a single one for the whole object
        ;
    recreateTable(table_name, details, '');
    -- primary key for this table consists of the following...
    pk_name := table_name || '_pk';    -- convention is table_name + "_pk"
    details := 'node_id, model_type';
    recreatePKConstraint(pk_name, table_name, details);

    EXCEPTION
        -- global exception handler
        WHEN OTHERS THEN
           DECLARE
               error_code NUMBER := SQLCODE;
               error_msg VARCHAR(512) := SQLERRM;
           BEGIN
               -- log errors to DBMS output
               DBMS_OUTPUT.PUT_LINE('err: ' || TO_CHAR(error_code) || ' : ' || error_msg);
           END;
END;
/

/**
 * ne_model_state INSERT trigger
 * integrity trigger to insure there's always/only a model type entry in the
 * ne_model_type table when there's some corresponding ne_model_state
 */
CREATE OR REPLACE TRIGGER ne_model_state_ins_trig
AFTER INSERT
ON ne_model_state
FOR EACH ROW
DECLARE
    seq_num_model_type INTEGER;
BEGIN
    seq_num_model_type := ot_memfp.sequence_num_part(:NEW.model_type);
    INSERT INTO ne_model_type (node_id, model_type, sequence_num)
    VALUES (:NEW.node_id, seq_num_model_type, -1);
EXCEPTION
    WHEN DUP_VAL_ON_INDEX THEN NULL; -- ignore err if a corr. model_type already exists
END ne_model_state_ins_trig;
/

/**
 * ne_model_type DELETE trigger
 * integrity trigger to insure there's always/only a model type entry in the
 * ne_model_type table when there's some corresponding ne_model_state
 */
CREATE OR REPLACE TRIGGER ne_model_type_del_trig
BEFORE DELETE OR UPDATE
ON ne_model_type
FOR EACH ROW
DECLARE
    old_low INTEGER;
    old_high INTEGER;
    new_model_type INTEGER;
BEGIN
    old_low := ot_memfp.low_part(:OLD.model_type);
    old_high := ot_memfp.high_part(:OLD.model_type);
    IF DELETING THEN
        -- delete corr. model state instances
        DELETE ne_model_state where
        node_id = :OLD.node_id
        AND old_low <= model_type
        AND model_type <= old_high;
    ELSE
        -- update node_id/model_type (partition) for corr. model state instances
        UPDATE ne_model_state
            SET node_id = :NEW.node_id,
                model_type = ot_memfp.move_part(:NEW.model_type, model_type)
	WHERE node_id = :OLD.node_id
        AND old_low <= model_type
        AND model_type <= old_high;
    END IF;
END ne_model_type_del_trig;
/

/**
 * Body OT memory footprint-specific types, constants AND routines
 */
CREATE OR REPLACE PACKAGE BODY ot_memfp IS

    /**
     * @return sequence number partition representative
     */
    FUNCTION sequence_num_part(model_type IN INTEGER) RETURN INTEGER IS
        result INTEGER;
    BEGIN
        result := c_part * FLOOR(model_type / c_part);
        RETURN result;
    END sequence_num_part;

    /**
     * @return lowest # item in partition
     */
    FUNCTION low_part(model_type IN INTEGER) RETURN INTEGER IS
        result INTEGER;
    BEGIN
        result := sequence_num_part(model_type);
        RETURN result;
    END low_part;

    /**
     * @return highest # item in partition
     */
    FUNCTION high_part(model_type IN INTEGER) RETURN INTEGER IS
        result INTEGER;
    BEGIN
        result := low_part(model_type) + c_part - 1;
        RETURN result;
    END high_part;

    /**
     * @return move a model_type into a new partition as identified by the new_seq_num
     * model type
     */
    FUNCTION move_part(new_seq_num IN INTEGER, old_model_type IN INTEGER) RETURN INTEGER IS
        result INTEGER;
    BEGIN
        result := c_part * ROUND(new_seq_num / c_part, 0) + (old_model_type MOD c_part);
        RETURN result;
    END move_part;

    /**
     * stored procedure to change node id. If new_node_id already exists,
     * the procedure exits
     * @return -1 if new_node_id already exists / 0 if runs 100% to completion
     */
   PROCEDURE change_node_id(old_node_id IN INTEGER, new_node_id IN INTEGER, status OUT INTEGER) IS
        exist BOOLEAN;
        tmp INTEGER;
        ok CONSTANT INTEGER := 0;
        no_change CONSTANT INTEGER := 1;
        new_id_already_exists CONSTANT INTEGER := -1;
   BEGIN
        status := no_change;
        IF (old_node_id = new_node_id) THEN
            RETURN;
        END IF;
        tmp := NULL;
        -- new node id should not already exist
        -- ...test for this
        BEGIN
            SELECT 1 INTO tmp FROM ne_model_type
                WHERE EXISTS (SELECT 1 FROM ne_model_type WHERE node_id = new_node_id);
        EXCEPTION
            WHEN NO_DATA_FOUND THEN NULL; -- ignore 'SELECT INTO' except. when no rows selected
            WHEN TOO_MANY_ROWS THEN tmp := 1; -- set tmp to 1 'SELECT INTO' except. when >1 rows selected
        END;
        exist := NOT tmp IS NULL;
        status := new_id_already_exists;
        IF (NOT exist) THEN
            UPDATE ne_model_type
                SET node_id = new_node_id
                WHERE node_id = old_node_id;
            tmp := SQL%ROWCOUNT;
            IF (tmp = 0) THEN
                status := no_change;
            ELSE
                status := ok;
            END IF;
        END IF;
    END;

    /**
     * procedure to update the sequence number. The procedure takes
     * care of the conversion of model type to sequence # model type
     * @return corr. update count i.e. # of rows effected
     */
    PROCEDURE update_seq_num(nid IN INTEGER, mtype IN INTEGER, seq_num INTEGER) IS
        seq_num_model_type INTEGER;
        tmp INTEGER;
    BEGIN
        seq_num_model_type := sequence_num_part(mtype);
        BEGIN
            UPDATE ne_model_type SET sequence_num = seq_num
                WHERE node_id = nid AND model_type = seq_num_model_type;
            tmp := SQL%ROWCOUNT;
            IF tmp = 0 THEN
                INSERT INTO ne_model_type (node_id, model_type, sequence_num)
                    VALUES (nid, seq_num_model_type, seq_num); -- try inserting instead
            END IF;
        END;
    END;

    /**
     * stored procedure to update the sequence number. The procedure takes
     * care of the conversion of model type to sequence # model type
     */
    PROCEDURE get_seq_num(nid IN INTEGER, mtype IN INTEGER, result IN OUT INTEGER) IS
    	seq_num_model_type INTEGER;
    BEGIN
    	seq_num_model_type := sequence_num_part(mtype);
        BEGIN
            SELECT sequence_num INTO result FROM ne_model_type
               WHERE node_id = nid AND model_type = seq_num_model_type;
        EXCEPTION
    	    WHEN NO_DATA_FOUND THEN NULL; -- ignore 'SELECT INTO' except. when no rows selected
        END;
    END;

    /**
     * stored procedure to delete the entire partition identified by mtype
     # last-changed seq # is reset for the assoc. slot
     */
    PROCEDURE del_partition(nid IN INTEGER, mtype IN INTEGER, rowcount OUT INTEGER) IS
        low INTEGER;
        high INTEGER;
    BEGIN
        low  := low_part(mtype);
        high := high_part(mtype);
        UPDATE ne_model_type
            SET sequence_num = uninit_seq_num
            WHERE node_id = nid AND low <= model_type AND model_type <= high;
        DELETE FROM ne_model_state
            WHERE node_id = nid AND low <= model_type AND model_type <= high;
        rowcount := SQL%ROWCOUNT;
    END;

    /**
     * creates an empty blob (if one does not already exist)
     */
    PROCEDURE insert_empty_blob(nid IN INTEGER, mtype IN INTEGER, ndx IN INTEGER) IS
    BEGIN
        BEGIN
            INSERT into ne_model_state (node_id, model_type, model_index, data)
                VALUES (nid, mtype, ndx, empty_blob());
        EXCEPTION
            -- global exception handler
            WHEN DUP_VAL_ON_INDEX THEN NULL; -- ignore err if a corr. row already exists
        END;
    END insert_empty_blob;

--BEGIN -- pkg body
END;
/
