"""
    1. Searches directory structure and obtains ALL modified feature template pairs
    2. Creates change objects for every change identified
    3. Sanitizes change objects into organized JSON Input that can be consumed by migration.py

    Author: Shreyas Ramesh
    Email: shrerame@cisco.com
"""

# ----------------- #
#  Generic Imports  #
# ----------------- #

import os
import sys
import json
import copy
import numbers
from pprint import pprint
from collections import deque, OrderedDict
from operator import itemgetter

# ------------------- #
#  Global Parameters  #
# ------------------- #

### NO GLOBAL PARAMETERS DEFINED

class GenerateJSONInput():

    def __init__(self):
        self.listOfMigrationDicts = []


    def initialzevEdgeModelDirectory(self, pathToNMS, vManageVersion):
        """
        1. Initializes path to NMS for vEdge schema
        2. Accepts (from) vManage version; This is the version of vManage you
           want to migrate templates from. This version of vManage does not have
           feature templates split between cEdges and vEdges. Eg: 19.2
        3. Initializes search for vEdge static JSONs (schema files)
        """
        self.pathTovEdgeModelNMS = pathToNMS
        self.vEdgevManageVersion = vManageVersion
        self.vEdgeSearch = SearchTemplatePairsInDir(pathToNMS)


    def initialzecEdgeModelDirectory(self, pathToNMS, vManageVersion):
        """
        1. Initializes path to NMS for cEdge schema
        2. Accepts (to) vManage version; This is the version of vManage you
           want to migrate templates to. This version of vManage has
           feature templates split between cEdges and vEdges. Eg: 20.1
        3. Initializes search for cEdge static JSONs (schema files)
        """
        self.pathTocEdgeModelNMS = pathToNMS
        self.cEdgevManageVersion = vManageVersion
        self.cEdgeSearch = SearchTemplatePairsInDir(pathToNMS)


    def initializeDictInInputJSON(self):
        """
        Initializes the dictionary to be used to store all schema diffs
        between one (from) vManage version and another (to) vManage version
        """
        self.dictForTemplateMigration = dict()
        self.dictForTemplateMigration["fromvManageVersion"] = self.vEdgevManageVersion
        self.dictForTemplateMigration["tovManageVersion"] = self.cEdgevManageVersion
        self.dictForTemplateMigration["templateTypeList"] = []


    def createDictInputToMigrator(self):
        """
        Loop 1: Current directory holding static JSON
        Loop 2: Fetch static JSON (schema) name for vEdge and cEdge

        If both static JSON files are found, move to comparing schema diff
        Append all the diffs to initialized dictionary
        """
        for templateDir in self.vEdgeSearch.listOfTemplateDirs:
            for templatePair in self.vEdgeSearch.getFilePairsFromFeatureTemplateDir(templateDir):
                vedgeJSON = ObjectifyAndSortJSON(os.path.join(self.vEdgeSearch.allTemplatesDir, templateDir, templatePair[0]))
                cedgeJSON = ObjectifyAndSortJSON(os.path.join(self.cEdgeSearch.allTemplatesDir, templateDir, templatePair[1]))
                if vedgeJSON and cedgeJSON:
                    comparer = CompareOrderedJSONs(vedgeJSON.jsonObject, cedgeJSON.jsonObject)
                    comparer.levelwiseDiffStarter()
                    self.buildAndAppendTemplateTypeList(comparer, templatePair)

    def newDefaultValueAlgorithm(self, comparer, defaultInstance):
        # Search for min and max (to and from) range values
        minInstance, maxInstance = None, None
        for instance in comparer.changeDict["min"]:
            if self.fixFieldHierarchyWithPath(instance[0]) == self.fixFieldHierarchyWithPath(defaultInstance[0]):
                minInstance = instance
                break
        for instance in comparer.changeDict["max"]:
            if self.fixFieldHierarchyWithPath(instance[0]) == self.fixFieldHierarchyWithPath(defaultInstance[0]):
                maxInstance = instance
                break

        if minInstance != None and maxInstance != None:
            #We check if ranges exist for this element
            if defaultInstance[1] >= minInstance[2] and defaultInstance[1] <= maxInstance[2]:
                # If old default value conforms to new range bounds, return old default
                return defaultInstance[1]
            else:
                # New default is minimum of (maximum of new min and new default value) and new max)
                return min(max(minInstance[2], defaultInstance[2]), maxInstance[2])
        elif minInstance != None:
            # New minimum in range. Maximum remains the same
            return max(minInstance[2], defaultInstance[2])
        elif maxInstance != None:
            # New maximum in range. Minimum remains the same
            return min(maxInstance[2], defaultInstance[2])
        else:
            # No range available. Proceed with old default value
            return defaultInstance[1]


    def buildAndAppendTemplateTypeList(self, comparer, templateTypePair):
        thisTemplateTypeDict = {}
        thisTemplateTypeDict["fromFeatureName"] = self.vEdgeSearch.dirToTemplateTypeName[templateTypePair[0]][0]
        thisTemplateTypeDict["toFeatureName"] = self.vEdgeSearch.dirToTemplateTypeName[templateTypePair[0]][1]
        thisTemplateTypeDict["listOfTasks"] = []

        for removeInstance in comparer.changeDict["remove"]:
            fixedInstance = self.fixFieldHierarchyWithPath(removeInstance)
            thisTemplateTypeDict["listOfTasks"].append({"operation": "remove",
                                                   "fieldHierarchyList": fixedInstance})

        for defaultInstance in comparer.changeDict["default"]:
            fixedInstance = self.fixFieldHierarchyWithPath(defaultInstance[0])
            newDefaultValue = self.newDefaultValueAlgorithm(comparer, defaultInstance)
            thisTemplateTypeDict["listOfTasks"].append({"operation": "default",
                                                   "fieldHierarchyList": fixedInstance,
                                                   "default": newDefaultValue})

        for minInstance in comparer.changeDict["min"]:
            fixedInstance = self.fixFieldHierarchyWithPath(minInstance[0])
            thisTemplateTypeDict["listOfTasks"].append({"operation": "range",
                                                   "fieldHierarchyList":fixedInstance,
                                                   "rangeMin": minInstance[2]})

        for maxInstance in comparer.changeDict["max"]:
            fixedInstance = self.fixFieldHierarchyWithPath(maxInstance[0])
            thisTemplateTypeDict["listOfTasks"].append({"operation": "range",
                                                   "fieldHierarchyList": fixedInstance,
                                                   "rangeMax": maxInstance[2]})
        self.dictForTemplateMigration["templateTypeList"].append(thisTemplateTypeDict)


    def addDictToMigrationList(self):
        self.listOfMigrationDicts.append(self.dictForTemplateMigration)


    def writeJsonToFile(self, directory, JSONFilename, data):
        with open(os.path.join(directory, JSONFilename), "w+") as JSONFile:
             json.dump(data, JSONFile, indent=4)


    def fixFieldHierarchyWithPath(self, fieldHierarchyList, seperator='/'):
        newFieldHierarchyList = []
        for i in fieldHierarchyList:
            seperatedList = i.split(seperator)
            for element in seperatedList:
                if element:
                    newFieldHierarchyList.append(element)

        return newFieldHierarchyList


class SearchTemplatePairsInDir():

    def __init__(self, dirStructureStart):
        """
        1. Sets path to current directory and static JSONs
        2. dirToFilePairTuples - Defines the templates that need migration. Key
           Value pair where, value is a list of tuples. The first element of
           tuple represents the static JSON of the (from) vManage version. The
           second element of the tuple represents the (to) vManage version.
        3. dirToTemplateTypeName - Each (from) vManage version static JSON needs
           at the very least a templateType change. This is represented in a
           tuple where, the first element is the (from) templateType and the
           second is the (to) templateType.
        """
        self.currentDirPath = os.path.dirname(__file__)
        self.dirStructureStart = dirStructureStart
        self.relativePathToTemplates = "src/main/java/com/viptela/vmanage/server/deviceconfig/template/general/type/"
        self.relativePathToTemplateDefinitions = "src/main/java/com/viptela/vmanage/server/datastore/schema/vmanagedb/FactoryDefaultFeatureTemplates/"
        self.allTemplatesDir = os.path.join(dirStructureStart, self.relativePathToTemplates)
        self.dirToFilePairTuples = {"banner": [("banner.json", "cisco_banner.json")],
                                    "bfd": [("bfd-vedge.json", "cisco_bfd.json")],
                                    "bgp": [("bgp.json", "cisco_bgp.json")],
                                    "dhcpserver": [("dhcp-server.json", "cisco_dhcp_server.json")],
                                    "logging": [("logging.json", "cisco_logging.json")],
                                    "ntp": [("ntp.json", "cisco_ntp.json")],
                                    "omp": [("omp-vedge.json", "cisco_omp.json")],
                                    "ospf": [("ospf.json", "cisco_ospf.json")],
                                    "snmp": [("snmp.json", "cisco_snmp.json")],
                                    "system": [("system-vedge.json", "cisco_system.json")],
                                    "vpn": [("vpn-vedge.json", "cisco_vpn.json"), ("vpn-vedge-interface.json", "cisco_vpn_interface.json"),
                                            ("vpn-vedge-interface-gre.json", "cisco_vpn_interface_gre.json"),
                                            ("vpn-vedge-interface-ipsec.json", "cisco_vpn_interface_ipsec.json")],
                                    "security": [("security-vedge.json", "cisco_security.json")]}
        self.listOfTemplateDirs = self.dirToFilePairTuples.keys()

        self.dirToTemplateTypeName = {"banner.json": ("banner", "cisco_banner"),
                                      "bfd-vedge.json": ("bfd-vedge", "cisco_bfd"),
                                      "bgp.json": ("bgp", "cisco_bgp"),
                                      "dhcp-server.json": ("dhcp-server", "cisco_dhcp_server"),
                                      "logging.json": ("logging", "cisco_logging"),
                                      "ntp.json": ("ntp", "cisco_ntp"),
                                      "omp-vedge.json": ("omp-vedge", "cisco_omp"),
                                      "ospf.json": ("ospf", "cisco_ospf"),
                                      "snmp.json": ("snmp", "cisco_snmp"),
                                      "system-vedge.json": ("system-vedge", "cisco_system"),
                                      "vpn-vedge.json": ("vpn-vedge", "cisco_vpn"),
                                      "vpn-vedge-interface.json": ("vpn-vedge-interface", "cisco_vpn_interface"),
                                      "vpn-vedge-interface-gre.json": ("vpn-vedge-interface-gre", "cisco_vpn_interface_gre"),
                                      "vpn-vedge-interface-ipsec.json": ("vpn-vedge-interface-ipsec", "cisco_vpn_interface_ipsec"),
                                      "security-vedge.json": ("security-vedge", "cisco_security")}


    def getFilePairsFromFeatureTemplateDir(self, templateDirName):
        for fileTuple in self.dirToFilePairTuples[templateDirName]:
            yield fileTuple


class CompareOrderedJSONs():
    def __init__(self, json1, json2):
        """
        Initializes changes dictionary holding diff for the current vEdge and
        cEdge static JSON template pair.

        changeDict - Consists of 4 types of diffs: remove, default, min and max
        """
        self.vedgeJSON = json1
        self.cedgeJSON = json2
        self.changeDict = {}


    def levelwiseDiffStarter(self):
        """
        Initialize each diff type to a list
        pathSoFar - Used to track current position in the YANG tree
        """
        pathSoFar = []
        self.changeDict = {"remove": [], "default": [], "min": [], "max": []}
        self.levelwiseDiff(pathSoFar, self.vedgeJSON, self.cedgeJSON)


    def levelwiseDiff(self, pathSoFar, json1Val, json2Val):
        """
        !!!CRITICAL BUSINESS LOGIC!!!
        1. If current vEdge element is a list, then
            1. If first value in vEdge list is a dictionary, then
                1. If dataPath element is present in vEdge element, then
                    1. Create a dictionary with key (key, /dataPath/../..) and value as index.
                   Otherwise, create a dictionary with key (key) and value as the index.
                2. If dataPath element in cEdge element, then
                    1. Create a dictionary with key (key, /dataPath/../..) and value as index.
                   Otherwise, create a dictionary with key (key) and value as the index.
                3. Perform set difference between the vEdge and cEdge keys
                4. Perform set intersection between the vEdge and cEdge keys
                5. If set difference has elements at current level, then there was a removal of
                   leaf in the current level.
                6. For each element in the set intersection, continue levelwiseDiff
            2. If first value in list is a string, then
                1. Find diffence between the vEdge and cEdge list

        2. Otherwise, if current vEdge element is a dictionary or  OrderedDict, then
            1. Find the set difference and set intersection between keys in vEdge dict and cEdge dict
            2. If there are removals, capture the keys removed as well as the path so far
            3. For each key in the set intersection,
                1. Make a copy of the path traversed so far
                2. Traverse each key

        3. Otherwise, the elements are regular strings
            1. If the vEdge and cEdge values differ, then pass values to funtion handling updates
               (default, min and max)
        """
        if isinstance(json1Val, list):
            # Can be either a list of dictionaries, or a list of values
            if json1Val and (isinstance(json1Val[0], dict) or isinstance(json1Val, OrderedDict)):
                json1KV, json2KV = {}, {}
                for idx, dict_temp in enumerate(json1Val):
                    if "dataPath" in dict_temp.keys():
                        json1KV[(dict_temp["key"], '/'.join(dict_temp["dataPath"]))] = idx
                    else:
                        json1KV[dict_temp["key"]] = idx
                for idx, dict_temp in enumerate(json2Val):
                    if "dataPath" in dict_temp.keys():
                        json2KV[(dict_temp["key"], '/'.join(dict_temp["dataPath"]))] = idx
                    else:
                        json2KV[dict_temp["key"]] = idx

                setDifference = set(json1KV.keys()).difference(json2KV.keys())
                setIntersection = set(json1KV.keys()).intersection(json2KV.keys())

                if setDifference:
                    for removedKey in setDifference:
                        self.convertRemovalToFieldHierarchyList(pathSoFar, removedKey)

                for key in setIntersection:
                    pathSoFarCopy = self.makeCopyAndAppend(pathSoFar, key)
                    self.levelwiseDiff(pathSoFarCopy, json1Val[json1KV[key]], json2Val[json2KV[key]])

            elif json1Val and isinstance(json1Val[0], str):
                # Normal list of strings
                setDifference = set(json1Val).difference(json2Val)
                setIntersection = set(json1Val).intersection(json2Val)

                if setDifference:
                    for removedKey in setDifference:
                        self.convertRemovalToFieldHierarchyList(pathSoFar, removedKey)

        elif isinstance(json1Val, dict) or isinstance(json1Val, OrderedDict):
            keys1 = json1Val.keys()
            keys2 = json2Val.keys()

            setDifference = set(keys1).difference(keys2)
            setIntersection = set(keys1).intersection(keys2)

            if setDifference:
                for removedKey in setDifference:
                    self.convertRemovalToFieldHierarchyList(pathSoFar, removedKey)

            for key in setIntersection:
                pathSoFarCopy = self.makeCopyAndAppend(pathSoFar, key)
                self.levelwiseDiff(pathSoFarCopy, json1Val[key], json2Val[key])
        else:
            # Check if value has changed
            if json1Val != json2Val:
                self.convertUpdateToFieldHierarchyList(pathSoFar, json1Val, json2Val)


    def makeCopyAndAppend(self, listToAppendInto, valToAppend):
        """
        Append value into list if list exists. If not, return a new list with
        just the value appended into it
        """
        if listToAppendInto:
            copyList = listToAppendInto[:]
            copyList.append(valToAppend)
            return copyList
        return [valToAppend]


    def convertRemovalToFieldHierarchyList(self, pathSoFar, removedKey):
        """
        1. Iterate through each node in the YANG tree
            1. If the element is either `fields` or `children`, then do nothing
            2. If element is a tuple, then first element is the child node of the second
               element, which is the parent path
            3. Otherwise, append the path as it a single path
        2. At the end, the fieldHierarchyList is the YANG data path that was removed
        """
        fieldHierarchyList = []
        temp = pathSoFar[:]
        temp.append(removedKey)

        for element in temp:
            if element == "fields" or element == "children":
                continue
            if isinstance(element, tuple):
                # First value is key, second is dataPath.
                # First append dataPath then key.
                if element[1]:
                    fieldHierarchyList.append(element[1])
                fieldHierarchyList.append(element[0])
            else:
                fieldHierarchyList.append(element)

        self.changeDict["remove"].append(fieldHierarchyList)


    def convertUpdateToFieldHierarchyList(self, pathSoFar, updateFrom, updateTo):
        """
        1. If just the name changes, then add name change to changeDict
        2. For each element in the path list,
            1. If element is fields or children, then skip
            2. If element is a tuple, then update fieldHierarchyList
            3. Otherwise, if current node in the YANG tree is an update,
                1. If current node is min and the minimum has increased, then add to changeDict
                2. If current node is max and the maximum has decreased, then add to changeDict
                3. If current node is default, then append vEdge value to currentDict
        """
        fieldHierarchyList = []

        # Update name changes
        if len(pathSoFar) == 1 and pathSoFar[0] == "name":
            self.changeDict["name"] = (updateFrom, updateTo)
        else:
            # Update range and default changes
            for element in pathSoFar:
                if element == "fields" or element == "children":
                    continue
                if isinstance(element, tuple):
                    # First value is key, second is dataPath.
                    # First append dataPath then key.
                    if element[1]:
                        fieldHierarchyList.append(element[1])
                    fieldHierarchyList.append(element[0])
                else:
                    # Update range changes
                    if element == "min":
                        if updateTo > updateFrom:
                            self.changeDict["min"].append((fieldHierarchyList, updateFrom, updateTo))
                        break
                    elif element == "max":
                        if updateTo < updateFrom:
                            self.changeDict["max"].append((fieldHierarchyList, updateFrom, updateTo))
                        break
                    elif element == "default":
                        self.changeDict["default"].append((fieldHierarchyList, updateFrom, updateTo))
                        break


class ObjectifyAndSortJSON():
    def __init__(self, filePath):
        self.obejectPairsHook = OrderedDict
        self.filePath = filePath
        self.jsonObject = self.sortOrderedDict(self.readJsonFromFile(self.filePath))


    def readJsonFromFile(self, filePath):
        if not os.path.exists(filePath):
            return None
        with open(filePath) as JSONFile:
            return json.load(JSONFile)


    def writeJsonToFile(self, directory, JSONFilename, data):
        with open(os.path.join(directory, JSONFilename), "w+") as JSONFile:
             json.dump(data, JSONFile)


    def sortOrderedDict(self, jsonDict):
        if jsonDict is None:
            return None
        res = OrderedDict()
        for k, v in sorted(jsonDict.items()):
            if isinstance(v, dict):
                res[k] = sortOrderedDict(self, v)
            else:
                res[k] = v
        return res


if __name__ == "__main__":
    generatedInstance = GenerateJSONInput()
    generatedInstance.initialzevEdgeModelDirectory("./19.3/nms", "19.3")
    generatedInstance.initialzecEdgeModelDirectory("./20.1/nms", "20.1")
    generatedInstance.initializeDictInInputJSON()
    generatedInstance.createDictInputToMigrator()
    generatedInstance.addDictToMigrationList()
    generatedInstance.writeJsonToFile(os.path.dirname(__file__), "JSONInput.json", generatedInstance.listOfMigrationDicts)
