"""
    Template Migration - Tool to migrate vEdge-cEdge combined vManage templates to cEdge only vManage templates (20.1 onwards).
"""

from session import session
import os
import json
import argparse
import requests
from requests.exceptions import ReadTimeout
from requests.packages.urllib3.exceptions import HTTPError,InsecureRequestWarning
from pathlib import Path
import sys
from datetime import datetime
import random
from collections import OrderedDict
import migration
import logging
import re
from getpass import getpass

__copyright__    = "Copyright (c) 2019-2020 Cisco Systems, Inc. and/or its affiliates"
__author__       = "Shreyas Ramesh, Shayan Ahmed"
__email__        = "shrerame@cisco.com, shayahme@cisco.com"

idNameMapping = dict()
cwd = os.getcwd()
logDir = os.path.join(cwd,"logs")
inputDir = os.path.join(cwd,"Input")
dataDir = os.path.join(cwd,"Data")
featureDir= os.path.join(dataDir,'Feature/')
masterDir = os.path.join(dataDir,"Master/")
pythonScript = os.path.join(cwd,"migration.py")
sessionScript = os.path.join(cwd,"session.py")
supportedDevicesandTemplates = os.path.join(inputDir,"supported_cedges_templates.json")
globalTemplate = os.path.join(inputDir,"globaltemplate.json")
migratedTemplates = os.path.join(dataDir,"migratedFeatureTemplates/")
migratedDevice = os.path.join(dataDir,"migratedDeviceTemplates/")
logFile = os.path.join(logDir,"template-migration.log")
dryRunFile = os.path.join(dataDir,"dryrun.txt")
deviceTemplateClass = "cedge"
TemplateTypeMapping = {"banner":"cisco_banner",
                      "bfd-vedge":"cisco_bfd",
                      "bgp":"cisco_bgp",
                      "dhcp-server":"cisco_dhcp_server",
                      "logging":"cisco_logging",
                      "ntp":"cisco_ntp",
                      "omp-vedge":"cisco_omp",
                      "ospf":"cisco_ospf",
                      "snmp":"cisco_snmp",
                      "system-vedge":"cisco_system",
                      "vpn-vedge":"cisco_vpn",
                      "vpn-vedge-interface":"cisco_vpn_interface",
                      "vpn-vedge-interface-gre":"cisco_vpn_interface_gre",
                      "vpn-vedge-interface-ipsec":"cisco_vpn_interface_ipsec",
                      "security-vedge":"cisco_security"}
globalTemplateMapping ={
                        "templateId":None,
                        "templateType":"cedge_global"
                       }
supportedTemplates = set()
supportedDevices = set()
templateNameRegex = r'(?=^.{1,128}$)[^&<>! "]+$'
DEVICE_CHANGE_LOG_PATH = os.path.join(logDir, "device_template.log")
deviceLogger = migration.setup_logger("migration.FACILITATOR", logFile)
updateLogger = migration.setup_logger("migration.UPDATE", DEVICE_CHANGE_LOG_PATH)
GLOBALTEMPLATEDEAFAULT = "Factory_Default_Global_CISCO_Template"
aaaFound = False
aaaType = "aaa"
regex = re.compile(templateNameRegex)

def buildFeatureTemplateSet(masterJson,idSet, masterMigration=False):
    for template in masterJson:
        if 'templateId' in template.keys() and 'templateType' in template.keys() and template['templateType'] in supportedTemplates:
            idSet.add(template['templateId'])
            masterMigration=True
        if template["templateType"] == aaaType:
            global aaaFound
            aaaFound = True
        if 'subTemplates' in template.keys():
            result = buildFeatureTemplateSet(template['subTemplates'],idSet,masterMigration)
            idSet = result[0]
            masterMigration = result[1]
    return (idSet,masterMigration)
def updateMasterTemplate(masterJson,featureIdMapping, masterName,globalFound = False):
    featureJson = []
    for template in masterJson:
        try:
            if 'templateId' in template.keys() and 'templateType' in template.keys():
                oldId = template['templateId']
                templateType = template['templateType']
                if templateType == "cedge_global":
                    globalFound = True
                if 'subTemplates' in template.keys():
                    template['subTemplates'],globalFound = updateMasterTemplate(template['subTemplates'],featureIdMapping, masterName,globalFound)
                if oldId in featureIdMapping.keys() and featureIdMapping[oldId] is not None:
                    template['templateId'] = featureIdMapping[oldId]
                    updateLogger.info("{} has been attached to {}".format(idNameMapping[featureIdMapping[oldId]],masterName))
                if templateType in TemplateTypeMapping.keys():
                    templateType = template['templateType']
                    template['templateType'] = TemplateTypeMapping[templateType]
                    updateLogger.info("{} has been updated to {} in {}".format(templateType,TemplateTypeMapping[templateType], masterName))
                featureJson.append(template)
        except Exception as e:
            deviceLogger.error("updateMasterTemplate:%s"% e)
            print("migrating device template has failed. Ending template migration")
            sys.exit(0)
    return (featureJson,globalFound)
def writeJsonToFile(dir,name,Json):
    try:
        with open("%s/%s"%(dir,name),'w') as fp:
            json.dump(Json,indent=4, fp= fp)
        print("saving %s in %s" %(name,dir))
    except Exception as e:
        deviceLogger.error("writeJsonToFile:%s"% e)
def mkDir(name):
    if Path(name).exists() is False:
        os.system("mkdir %s"%name)
def cleanDir(dir):
    if Path(dataDir).exists:
        os.system("rm -rf %s" % dir)
def acquireAllFeatureTemplates(session):
    try:
        return session.get_all_feature_templates()["data"]
    except Exception as e:
        print("Failed to get feature templates. Ending Template migration now.")
        deviceLogger.error("acquireAllFeatureTemplates:%s" % e)
        sys.exit(0)
def preCheck(ip,usr,password,prefix,tv,fv,ji,d,port):
    try:
        if Path(dataDir).exists:
            cleanDir(featureDir)
            cleanDir(masterDir)
            cleanDir(migratedDevice)
            cleanDir(migratedTemplates)
        assert regex.match(prefix), "Prefix does not match template naming conventions. Template name can be a maximum of 128 characters and must not contain &,<,>,!,\",\\, and/or spaces. This also needs to hold true when adding the prefix to the name."
        mkDir(dataDir)
        if d == migratedTemplates:
            mkDir(d)
        assert os.path.isdir(d), "%s is not a proper directory or does not exist." % d
        with open(ji, "r") as f:
            inputFile = json.load(f,object_pairs_hook=OrderedDict)
            found = False
            for j in inputFile:
                if "squashedVmanageVersions" not in j.keys() and "tovManageVersion" not in j.keys():
                    raise KeyError("The %s is missing the required information or has been changed" % ji)
                if tv==j["tovManageVersion"] and fv in j["squashedVmanageVersions"]:
                    found = True
            assert found is True,"The version conversion is either invalid or not supported at this time"
        with open(pythonScript) as p:
            pass
        with open(sessionScript) as s:
            pass
        with open(supportedDevicesandTemplates) as s:
            j = json.load(s,object_pairs_hook=OrderedDict)
            supportedDevices.update(j['cedges'])
            supportedTemplates.update(j['oldTemplates'])
        with open(globalTemplate) as g:
            pass
        mkDir(featureDir)
        mkDir(masterDir)
        return session(ip,usr,password,port)
    except AssertionError as e:
        print(e)
        deviceLogger.error("precheck:%s"% e)
        sys.exit(0)
    except IOError as e:
        print(e)
        deviceLogger.error("precheck:%s"% e)
        sys.exit(0)
    except KeyError as e:
        deviceLogger.error("precheck:%s"% e)
        sys.exit(0)
    except (HTTPError, ReadTimeout) as e:
        deviceLogger.error("precheck:%s"% e)
        print("Something went wrong during authentication. Check your internet connection, ip address, and your credentials")
        sys.exit(0)
    except Exception as e:
        print("Cannot continue with template migration.")
        deviceLogger.error("precheck:%s"% e)
        sys.exit(0)
def findTemplate(session,name,device_template):
    featuretemplates = []
    if device_template:
        featuretemplates = session.get_all_device_templates()['data']
    else:
        featuretemplates = acquireAllFeatureTemplates(session)
    for template in featuretemplates:
        if template["templateName"] == name:
            return template["templateId"]
    return None

def postTemplate(session,body,device_template=False,retry=0):
    templateId = None
    if "templateId" in body.keys():
        templateId = body["templateId"]
    try:
        if 'templateId' in body.keys():
            del body['templateId']
        if "@rid" in body.keys():
            del body["@rid"]
        if "createdOn" in body.keys():
            del body["createdOn"]
        if "lastUpdatedOn" in body.keys():
            del body["lastUpdatedOn"]
        if "gTemplateClass" in body.keys():
            body["gTemplateClass"] = "cedge"
        response = json.loads(session.post_request(body,device_template).content,object_pairs_hook=OrderedDict)
        assert "error" not in response.keys(),response["error"]["message"]+": "+response["error"]["details"]
        assert "templateId" in response.keys(),"template id not found in response for %s template post" % body["templateName"]
        return response['templateId']
    except AssertionError as e:
        deviceLogger.error("posttemplate:%s"% e)
        print("Failed to upload template %s" % (body["templateName"]))
        return None
    except ReadTimeout as e:
        deviceLogger.error(e)
        id = findTemplate(session,body["templateName"],device_template)
        if id is not None:
            return id
        if id is None and retry < 1:
            body["devicesAttached"] = 10
        if retry > 0 and retry < 3:
             body["devicesAttached"]*=10
        if retry == 2:
            return None
        retry+=1
        deviceLogger.info("Retrying the upload for template %s." % body["templateName"])
        return postTemplate(session,body,device_template,retry)
    except HTTPError as e:
        print("Template migration failed because of internet connection issues.\nTemplate migration will end.")
        deviceLogger.error("posttemplate:%s"% e)
        sys.exit(0)
    except Exception as e:
        print("Failed to upload template %s " % (body["templateName"]))
        deviceLogger.error("posttemplate:%s\ndevice template post:%s" % (e,device_template))
        return None


def pushMigratedFeatureTemplates(session,migratedTemplate,oldNewIdMapping):
    for root, dirs, files in os.walk(migratedTemplate):
        num = 1
        for filename in files:
            with open(os.path.join(root,filename)) as migratedFile:
                migratedJson = json.load(migratedFile,object_pairs_hook=OrderedDict)
                oldId = migratedJson['templateId']
                if oldId in oldNewIdMapping.keys():
                    id = postTemplate(session,migratedJson)
                    if id is not None:
                        oldNewIdMapping[oldId] = id
                        updateLogger.info("New template {} has been uploaded to vmanage.".format(migratedJson['templateName']))
                        idNameMapping[id] = migratedJson['templateName']
                    else:
                        print("A feature template has failed to upload to vmanage.")
                        print('Attempting to restore vmanage back to its original state and ending template migration')
                        cleanUp(idMapping=oldNewIdMapping,session=session)
                        sys.exit(0)
    return oldNewIdMapping
def getMasterTemplate(session,m,masterTemplateSet):
    try:
        masterTemplate = session.get_request("template/device/object/%s"%m['templateId'],True)
        assert 'generalTemplates' in masterTemplate.keys(), "Unable to acquire master template %s" % m['templateName']
        migrationInformation = buildFeatureTemplateSet(masterTemplate['generalTemplates'],masterTemplateSet)
        masterTemplateSet = migrationInformation[0]
        if migrationInformation[1]:
            masterTemplate['templateId'] = m['templateId']
            writeJsonToFile(masterDir,m['templateId']+".json",masterTemplate)
        return migrationInformation
    except Exception as e:
        print("could not save master template.")
        deviceLogger.error("getMasterTemplate:%s"% e)
    return masterTemplateSet
def findGlobalDefaultTemplate(prefix,session,idMapping):
    id = None
    try:
        id = findTemplate(session,GLOBALTEMPLATEDEAFAULT,False)
        globalTemplateMapping["templateId"] = id
    except Exception as e:
        deviceLogger.error("findGlobalDefaultTemplate:%s"%e)

def deviceTemplateMigration(prefix,idMapping,session):
    deviceTemplateMapping = dict()
    for root, dirs, files in os.walk(masterDir):
        for filename in files:
            try:
                with open(os.path.join(root,filename)) as f:
                    j = json.load(f,object_pairs_hook=OrderedDict)
                    j['templateName'] = prefix + j['templateName']
                    featureJson, globalFound = updateMasterTemplate(j['generalTemplates'],idMapping, j['templateName'])
                    j['generalTemplates']=featureJson
                    j['factoryDefault'] = False
                    j["templateClass"] = deviceTemplateClass
                    #response = json.loads(session.post_request(j,device_template=True).content)
                    if globalFound is False and globalTemplateMapping['templateId'] is not None:
                        j['generalTemplates'].append(globalTemplateMapping)
                    oldId = j['templateId']
                    j['templateId'] = postTemplate(session,j,True,0)
                    if j['templateId'] is None:
                        raise Exception('Could not upload %s to vmanage.' % j['templateName'])
                    deviceTemplateMapping[oldId] = j['templateId']
                    writeJsonToFile(migratedDevice,prefix+j['templateId']+".json",j)
            except IOError as e:
                print("Could not a migrate device template. Attempting to revert vmanage to its original state.")
                deviceLogger.error("deviceTemplateMigration:%s"% e)
                if len(deviceTemplateMapping.values) > 0:
                    cleanUpDevice(deviceTemplateMapping,session)
                cleanUp(idMapping=idMapping,session=session)
                print("Templates have been deleted. Ending template migration.")
                writeJsonToFile(dataDir,prefix+"deviceTemplates.json",deviceTemplateMapping)
                sys.exit(0)
            except Exception as e:
                print("Could not complete template migration. Attempting to restore vmanage to its original state before ending template migration.")
                deviceLogger.error("deviceTemplateMigration:%s"% e)
                writeJsonToFile(dataDir,prefix+"deviceTemplates.json",deviceTemplateMapping)
                if len(deviceTemplateMapping.values()) > 0:
                    cleanUpDevice(deviceTemplateMapping,session)
                cleanUp(idMapping=idMapping,session=session)
                sys.exit(0)
    writeJsonToFile(dataDir,prefix+"deviceTemplates.json",deviceTemplateMapping)

def getFeatureTemplates(id,dir,session):
    try:
        featureJson = session.get_request("template/feature/object/%s"%id,True)
        if 'factoryDefault' in featureJson and featureJson['factoryDefault'] is True:
            return None
        writeJsonToFile(dir,featureJson['templateId']+".json",featureJson)
        idNameMapping[id] = featureJson['templateName']
        return featureJson
    except Exception as e:
        print("Failed to acquire Feature template")
        deviceLogger.error("getFeatureTemplates:%s"% e)
        print("Template migration will end because it cannot acquire a feature template.")
        sys.exit(0)

    return None
def filterMasterTemplates(session,m,idSet):
    try:
        deviceType = m['deviceType']
        assert m['factoryDefault'] is False, "%s will not be migrated because it is factory default template" % m['templateName']
        assert  deviceType in supportedDevices, "Master Template %s with device type %s does not need to be migrated"% (m['templateName'],deviceType)
        assert m['configType'] == "template", "%s is a cli template and does not need migration." % m['templateName']
        return getMasterTemplate(session,m,idSet)
    except AssertionError as e:
        deviceLogger.info("filterMasterTemplates:%s"% e)
    except KeyError as e:
        deviceLogger.error("filterMasterTemplates:%s"% e)
        print("Failed to acquire master template %s. Ending template migration." % m['templateName'])
        sys.exit(0)
    return (idSet,False)

def generateMapping(prefix,session):
    idMapping = dict()
    with open("Data/feature.json") as f:
        featureTemplates = json.load(f,object_pairs_hook=OrderedDict)["data"]
        for root, dirs, files in os.walk(migratedTemplates):
            counter = 0
            for filename in files:
                with open(os.path.join(root,filename)) as f:
                    for template in featureTemplates:
                        if template["templateName"] + ".json" == filename:
                            counter+=1
                            break
    return idMapping

def cleanUp(**kwargs):
    print("Attempting to delete migrated feature templates.")
    try:
        idMapping = None
        session = None
        if 'prefix' in kwargs.keys():
            with open(os.path.join(dataDir,kwargs['prefix']+"idmapping.json")) as f:
                idMapping = json.load(f,object_pairs_hook=OrderedDict)
        elif 'idMapping' in kwargs.keys():
            idMapping = kwargs['idMapping']
        if 'session' in kwargs.keys():
            session = kwargs['session']
        else:
            session = session(kwargs['ip'],kwargs['usr'],kwargs['password'],kwargs['port'])
        for id in idMapping.values():
            if id is not None:
                session.delete_request('template/feature', id)
        print('Template migration has terminated.\n')
    except IOError as e:
        print('idmapping json for prefix %s not found. This JSON is needed for delete the templates.\n' % kwargs['prefix'])
        deviceLogger.error("cleanUp:%s"% e)
        sys.exit(0)
    except (HTTPError,ReadTimeout) as e:
        print("Could not connnect to vmanage to delete templates.\n")
        print("Template migration will end.\n")
        deviceLogger.error("cleanUp:%s"% e)
        sys.exit(0)

def cleanUpDevice(idMapping,session):
    print("Attempting to delete migrated device Templates.")
    try:
        for id in idMapping.values():
            session.delete_request('template/device', id)
    except HTTPError as e:
        print("Could not connect to vmanage to delete device template templates. Template migration will end now.")
        deviceLogger.error("cleanUpDevice:%s"% e)
        sys.exit(0)
    except Exception as e:
        print("Could not delete device templates. Template migration will end now.")
        deviceLogger.error("cleanUpDevice:%s"% e)
        sys.exit(0)

def generateDryRunFormat(masterTemplate,featureIdMapping, output):
    for template in masterTemplate:
        if 'templateId' in template.keys() and 'templateType' in template.keys() and template['templateType'] in supportedTemplates:
            featureTemplate = featureIdMapping[template['templateId']]
            output = output+"%-135s\t%20s\t%s\n"%(featureTemplate['templateName'],template['templateType'],template['templateId'])
        if 'subTemplates' in template.keys():
            output = generateDryRunFormat(template['subTemplates'],featureIdMapping, output)
    return output

def acquireFeatureTemplateIds(session):
    idSet = set()
    masterTemplates = session.get_all_device_templates()['data']
    migrationNeeded = False
    masterCounter = 0
    assert len(masterTemplates) > 0, "No master templates were found."
    for m in masterTemplates:
        idSet, migrationNeeded = filterMasterTemplates(session,m,idSet)
        if migrationNeeded:
            masterCounter +=1
    print("\n\n%s feature templates and %s device templates need to be migrated.\n\n" % (len(idSet),masterCounter))
    return idSet

def migrateFeatureTemplates(session,tv,ji,d,prefix,idSet):
    pythonCMD = "python2.7 %s migrate %s %s %s %s %s -d %s -p %s"% (pythonScript,'all',tv,featureDir,ji,logFile,d,prefix)
    os.system(pythonCMD)
    print("Feature templates have been migrated.")
    idMapping = dict()
    for id in idSet:
        idMapping[id] = None
    print('Pushing migrated feature templates to vmanage.')
    idMapping = pushMigratedFeatureTemplates(session,d,idMapping)
    mkDir(migratedDevice)
    findGlobalDefaultTemplate(prefix,session, idMapping)
    writeJsonToFile(dataDir,prefix+"idmapping.json",idMapping)
    return idMapping

def dryRun(**kwargs):
    session = preCheck(**kwargs)
    featureIdSet = acquireFeatureTemplateIds(session)
    idDefinitionMapping = dict()
    for id in featureIdSet:
        idDefinitionMapping[id] = getFeatureTemplates(id,featureDir,session)
    output = "%-135s\t%-20s\t%s\n" % ("template name","template type","template id")
    output = output + "%-135s\t%20s\t%s\n" % (135*"_",20*"_",36*"_")
    for root, dirs, files in os.walk(masterDir):
        for file in files:
            with open(os.path.join(masterDir,file)) as f:
                masterTemplate = json.load(f,object_pairs_hook=OrderedDict)
                output = output + ("%-135s\t%20s\t%s\n" % (masterTemplate['templateName'],"-", masterTemplate['templateId']))
                output = output + generateDryRunFormat(masterTemplate['generalTemplates'],idDefinitionMapping,"") + "\n"
    with open(dryRunFile, "w+") as f:
        f.write(output)
    print("Please check %s for the dry-run output." % dryRunFile)

def checkCharacterLimit(prefix,dir):
    msg = "illegal template name found:%s\n"
    illegal = False
    for root, dirs, files in os.walk(dir):
        for file in files:
            with open(os.path.join(dir,file)) as f:
                template = json.load(f,object_pairs_hook=OrderedDict)
                templateName =template['templateName']
                migratedTemplateName = prefix + templateName
                if regex.search(migratedTemplateName) is None:
                    print(msg % migratedTemplateName)
                    deviceLogger.error("checkCharacterLimit:{} has an illegal name.".format(migratedTemplateName))
                    illegal = True
    return illegal


def verifyNameCollosion(session,prefix,dir,device=False):
    templates = []
    if device is not True:
        templates = acquireAllFeatureTemplates(session)
    else:
        templates = session.get_all_device_templates()['data']
    NameSet = set()
    namecollosions = set()
    for template in templates:
        NameSet.add(template["templateName"])
    for root, dirs, files in os.walk(dir):
        for file in files:
            try:
                with open(os.path.join(dir,file)) as f:
                    featureTemplates = json.load(f,object_pairs_hook=OrderedDict)
                    name = featureTemplates['templateName']
                    if prefix+name in NameSet:
                        namecollosions.add(name)
            except IOError as e:
                deviceLogger.error("verifyNameCollosion:%s"% e)
    if len(namecollosions) > 0:
        msg = "The following template names will result in name collisions with your prefix:"
        for name in namecollosions:
            msg+="\n\t"+name
        print(msg)
        raise Exception("verifyNameCollosion:%s"% msg)


def migrateTemplates(ip,usr,password,prefix,tv,fv,ji,d,port):
    try:
        session = preCheck(ip,usr,password,prefix,tv,fv,ji,d,port)
        idSet = acquireFeatureTemplateIds(session)
        for id in idSet:
            getFeatureTemplates(id,featureDir,session)
        print("\n\nChecking naming convention of templates")
        masterillegal = checkCharacterLimit(prefix,masterDir)
        featureillegal = checkCharacterLimit(prefix,featureDir)
        assert masterillegal is False and featureillegal is False, "Feature/device template has illegal naming convention. Template name can be a maximum of 128 characters and must not contain &,<,>,!,\",\\, and/or spaces."
        print("The names for the templates follow the vmanage standard.")
        print("Checking for possible name collisions for device templates.")
        verifyNameCollosion(session,prefix,masterDir,True)
        print("Check for possible name collisions for feature templates.")
        verifyNameCollosion(session,prefix,featureDir)
        print("No potential name collisions were detected.")
        idMapping = migrateFeatureTemplates(session,tv,ji,d,prefix,idSet)
        print('Attempting to migrate device templates')
        deviceTemplateMigration(prefix,idMapping,session)
        if globalTemplateMapping['templateId'] is None:
            print('\nCould not attach global template to device template. Please add the Factory global template on to your migrated device template.')
        print("\n\nYour device and feature templates have successfully migrated and have been uploaded to vmanage.")
        print("Look over %s for changes that were done to your feature templates.\nLook over %s for changes done to your device templates.\n\n" % (migration.MIGRATION_CHANGE_LOG_PATH,DEVICE_CHANGE_LOG_PATH))
        global aaaFound
        if aaaFound == True:
            print("At least one of your device templates contains a AAA feature template. Please create and attach a Cisco AAA feature template to these devices templates.")
    except AssertionError as e:
        print(e)
        deviceLogger.error("migrateTemplates:%s"% e)
    except (HTTPError,ReadTimeout) as e:
        print("Something went wrong while trying to connect to vmange. Verify the ip address, credentials, and internet connection")
        print("template migration will cease since connection to vmanage is lost.")
        deviceLogger.error("migrateTemplates:%s"% e)
    except Exception as e:
        print("Template migration has failed.")
        deviceLogger.error("migrateTemplates:%s"% e)

def main():
    requests.packages.urllib3.disable_warnings(InsecureRequestWarning)
    baseurl = "https://%s:9912/%s"
    templatePath = "dataservice/template/device/object/%s"
    argparser = argparse.ArgumentParser(description=__doc__)
    argparser.add_argument('--version', action='version',
                        version='%(prog)s {version}'.format(version=migration.__version__))
    argparser.add_argument("-ip",type=str,help="ip address of the vmanage.",required=True)
    #argparser.add_argument("-user",type=str, help="user name to access vmanage",required=True)
    #argparser.add_argument("-password",type=str,help="password to access vmanage with the provided user name",required=True)
    argparser.add_argument("-fv",type=str,help="the version to convert from",required=True)
    argparser.add_argument("-tv",type=str,help="the version to convert the templates to",required=True)
    argparser.add_argument("-prefix", type=str, default="cisco_",help="the prefix to put for you templates. Default is master.")
    argparser.add_argument("-port", type=str,help="The port to connect to in order to access vmanage.",default=443)
    argparser.add_argument("-cleanUp", dest='clean', action='store_const',default=None,const=cleanUp,help="Deletes feature templates that were migrated. Used for testing purposes")
    argparser.add_argument("-dryRun", dest='apply', action='store_const',default=migrateTemplates,const=dryRun,help="Displays all the device templates and feature templates that will be migrated.")
    arg, unknown = argparser.parse_known_args()
    if len(unknown) > 0:
        print("The follow unknown arguments are found:")
        for u in unknown:
            print(u)
        print('Use -h to find the acceptable arguments.')
        sys.exit(0)
    usr = getpass("Username:")
    password=getpass("Password:")
    ip = arg.ip
    if arg.clean is not None:
        session = preCheck(ip=ip,usr=usr,password=password,prefix=arg.prefix,tv=arg.tv,fv=arg.fv,ji=os.path.join(cwd,"Input","JSONInput.json"),d=migratedTemplates,port = arg.port)
        try:
            with open(os.path.join(dataDir,arg.prefix+"deviceTemplates.json")) as f:
                idMapping = json.load(f,object_pairs_hook=OrderedDict)
                cleanUpDevice(idMapping,session)
        except IOError as e:
            print('Cannont find device template mapping proceeding with feature template deletion.')
        arg.clean(prefix=arg.prefix,session=session)

        if arg.apply is not dryRun:
            sys.exit(0)

    arg.apply(ip=ip,usr=usr,password=password,prefix=arg.prefix,tv=arg.tv,fv=arg.fv,ji=os.path.join(cwd,"Input","JSONInput.json"),d=migratedTemplates,port = arg.port)

if __name__== "__main__": main()
