'''
Created on Aug 26, 2015

@author: jeffryp

Copyright (c) 2015 by Cisco Systems, Inc.
All rights reserved.

Classes used for AAA server group configuration.
'''

import re

from asaio.cli_interaction import ignore_error_response_parser
from base.compositetype import CompositeType
from base.dmlist import DMList
from base.dmobject import DMObject
from base.simpletype import SimpleType
from state_type import State, Type
from structured_cli import StructuredCommand
from utils.util import filter_first, ifcize_param_dict, normalize_ip_address
from utils.util import set_cfg_state
from validators import Validator

class AAAServer(CompositeType, Validator):
    '''
    AAA server configuration
    '''

    CLI_PATTERN = None

    def __init__(self, instance):
        super(AAAServer, self).__init__(
            ifc_key=instance,
            asa_gen_template='aaa-server %(server_tag)s (%(interface)s) host %(host)s')
        self.register_child(Key())

    def create_asa_key(self):
        return self.get_cli()

    def diff_ifc_asa(self, cli):
        if not self.has_ifc_delta_cfg():
            self.delta_ifc_key = (Type.FOLDER, self.parent.ifc_key, self.ifc_key)
            self.delta_ifc_cfg_value = {
                'state': State.DESTROY,
                'value': ifcize_param_dict(self.parse_cli(cli))}
            ancestor = self.get_ifc_delta_cfg_ancestor()
            if ancestor:
                ancestor.delta_ifc_cfg_value['value'][self.delta_ifc_key] = self.delta_ifc_cfg_value
        elif self.get_state() != State.DESTROY:
            if self.is_the_same_cli(cli):
                self.set_action(State.NOCHANGE)
            else:
                self.set_action(State.MODIFY)
            if isinstance(cli, StructuredCommand):
                for cmd in cli.sub_commands:
                    translator = self.get_child_translator(cmd)
                    if translator:
                        translator.diff_ifc_asa(cmd)

    def get_cli(self):
        value = self.get_value()
        # Normalize host IP address
        value['host'] = normalize_ip_address(value['host'])
        if not 'interface' in value:
            value['interface'] = 'management'
        value['server_tag'] = self.parent.get_server_tag()
        return self.asa_gen_template % value

    @classmethod
    def get_cli_pattern(cls):
        if not cls.CLI_PATTERN:
            cls.CLI_PATTERN = re.compile('aaa-server (\S+) \(([^)]*)\) host (\S+)')
        return cls.CLI_PATTERN

    def ifc2asa(self, no_asa_cfg_stack,  asa_cfg_list):
        if self.has_ifc_delta_cfg():
            if self.get_state() == State.DESTROY:
                # When deleting a non-existent AAA server, the ASA will return
                # an error with an empty response.
                self.generate_cli(no_asa_cfg_stack,
                                  'no ' + self.get_cli(),
                                  response_parser=ignore_error_response_parser)
            else:
                cli_list = []
                no_cli_stack = []
                super(AAAServer, self).ifc2asa(no_cli_stack, cli_list)
                if cli_list or no_cli_stack:
                    no_asa_cfg_stack.extend(no_cli_stack)
                    asa_cfg_list.extend(cli_list)
                elif self.get_state() == State.CREATE:
                    # There is no sub-command configuration so need to generate
                    # the AAA server CLI
                    self.generate_cli(asa_cfg_list, self.get_cli())

    def is_the_same_cli(self, cli):
        if isinstance(cli, basestring):
            if cli != self.get_cli():
                return False
        elif cli.command != self.get_cli():
            return False
        sub_commands = [s.strip() for s in str(cli).split('\n')[1:]]
        return set(sub_commands) ==  set(self.get_sub_commands())

    def parse_cli(self, cli):
        if isinstance(cli, StructuredCommand):
            cli = cli.command
        m = self.get_cli_pattern().match(cli.strip())
        if m:
            result = {}
            result['interface'] = m.group(2)
            result['host'] = m.group(3)
            return result

    def remove_deleted(self):
        "Remove any child's configuration that is to be deleted"
        for child in self:
            if child.get_state() == State.DESTROY:
                del child.delta_ifc_key
                del child.delta_ifc_cfg_value

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        """
        If it is to DESTROY interface connecting the server in a MODIFY operation,
        we will restore it to the default 'management'.
        """
        if delta_ifc_cfg_value['state'] != State.MODIFY:
            return CompositeType.populate_model(self, delta_ifc_key, delta_ifc_cfg_value)
        intf_cfg_key = filter_first(lambda (t, k, n): k == 'interface', delta_ifc_cfg_value['value'])
        if intf_cfg_key and delta_ifc_cfg_value['value'][intf_cfg_key]['state'] == State.DESTROY:
            delta_ifc_cfg_value['value'][intf_cfg_key]['state'] = State.MODIFY
            delta_ifc_cfg_value['value'][intf_cfg_key]['value'] = 'management'
        return CompositeType.populate_model(self, delta_ifc_key, delta_ifc_cfg_value)

class AAAServerGroup(DMObject):
    '''
    AAA server group configuration
    '''

    CLI_PATTERN = None

    def __init__(self, instance):
        super(AAAServerGroup, self).__init__(instance)
        self.response_parser = self.aaa_server_group_response_parser
        self.mini_audit_command = ('show running-config aaa-server ' + self.ifc_key)

    @staticmethod
    def aaa_server_group_response_parser(response):
        if response:
            if 'does not exist' in response:
                return
            return response

    def diff_ifc_asa(self, cli):
        if not self.has_ifc_delta_cfg():
            self.delta_ifc_key = (Type.FOLDER, self.parent.ifc_key, self.ifc_key)
            self.delta_ifc_cfg_value = {'state': State.DESTROY, 'value': {}}
            ancestor = self.get_ifc_delta_cfg_ancestor()
            if ancestor:
                ancestor.delta_ifc_cfg_value['value'][self.delta_ifc_key] = self.delta_ifc_cfg_value
        else:
            self.set_state(State.MODIFY)

    def get_cli(self):
        return 'aaa-server %s protocol radius' % self.ifc_key

    @classmethod
    def get_cli_pattern(cls):
        if not cls.CLI_PATTERN:
            cls.CLI_PATTERN = re.compile('aaa-server (\S+) protocol radius')
        return cls.CLI_PATTERN

    def get_server_tag(self):
        # The server-tag is the instance
        return self.ifc_key

    def get_translator(self, cli):
        if self.is_my_cli(cli):
            for child in self:
                result = child.get_translator(cli)
                if result:
                    return result
    
            result = AAAServer(cli)
            self.register_child(result)
            return result

    def ifc2asa(self, no_asa_cfg_stack,  asa_cfg_list):
        if (filter_first(lambda child: child.get_state() == State.MODIFY, self)
                and not self.get_top().is_audit):
            # Remove any entries to be deleted
            for key, child in self.children.items():
                if child.get_state() == State.DESTROY:
                    del(self.children[key])
                else:
                    child.remove_deleted()
            # Change the remaining entries to CREATE
            for child in self.children.itervalues():
                set_cfg_state(child.delta_ifc_cfg_value, State.CREATE)
            self.mini_audit()

        state = self.get_state()
        if state == State.DESTROY:
            self.generate_cli(no_asa_cfg_stack, 'no ' + self.get_cli())
        elif state in (State.CREATE, State.MODIFY):
            if state == State.CREATE:
                self.generate_cli(asa_cfg_list, self.get_cli())
            for child in self:
                child.ifc2asa(no_asa_cfg_stack, asa_cfg_list)

    def is_my_cli(self, cli):
        if isinstance(cli, StructuredCommand):
            cli = cli.command
        return AAAServer.get_cli_pattern().match(cli.strip())

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        self.delta_ifc_key = delta_ifc_key
        self.delta_ifc_cfg_value = delta_ifc_cfg_value

        cfg = delta_ifc_cfg_value.get('value')
        if isinstance(cfg, dict):
            for key, value in cfg.iteritems():
                instance = key[2]
                child = self.get_child(instance)
                if not child:
                    child = AAAServer(instance)
                    self.register_child(child)
                child.populate_model(key, value)

class AAAServerGroups(DMList):
    '''
    A list of AAA server groups
    '''

    def __init__(self):
        super(AAAServerGroups, self).__init__(AAAServerGroup.__name__,
                                              AAAServerGroup, 'aaa-server')

    def get_translator(self, cli):
        if isinstance(cli, StructuredCommand):
            cli = cli.command

        # AAA server group command
        m = AAAServerGroup.get_cli_pattern().match(cli.strip())
        if m:
            server_tag = m.group(1)
            group = self.get_child(server_tag)
            if not group:
                group = self.child_class(server_tag)
                self.register_child(group)
            return group

        # AAA server command
        m = AAAServer.get_cli_pattern().match(cli.strip())
        if m:
            server_tag = m.group(1)
            group = self.get_child(server_tag)
            if group:
                return group.get_translator(cli)

class Key(SimpleType):
    'AAA server key configuration'

    def __init__(self):
        super(Key, self).__init__('key', 'key')

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        if self.has_ifc_delta_cfg():
            state = self.get_state()
            if state in (State.CREATE, State.MODIFY):
                self.generate_cli(asa_cfg_list, self.get_cli())
            elif state == State.DESTROY:
                self.generate_cli(no_asa_cfg_stack, 'no ' + self.asa_key)

