'''
Created on Aug 27, 2013

@author: tsnguyen
Copyright (c) 2013-2015 by Cisco Systems
'''

from translator.base.dmobject import DMObject
from translator.base.dmboolean import DMBoolean
from translator.base.simpletype import SimpleType
from translator.state_type import State
from translator.validators import UnsupportedOnMultiContextASAValiator

class AdvancedThreatDetection(DMObject):
    '''
    Advanced Thread Detection
    Supports the following CLI:

    threat-detection statistics access-list
    threat-detection statistics host [number-of-rate {1 | 2 | 3}]
    threat-detection statistics port [number-of-rate {1 | 2 | 3}]
    threat-detection statistics protocol [number-of-rate {1 | 2 | 3}]

    threat-detection statistics tcp-intercept [rate-interval <minutes>]
        [burst-rate <attacks_per_sec>]
        [average-rate <attacks_per_sec>]
    '''

    THREAT_DETECTION_STATISTICS = "threat-detection statistics"
    THREAT_DETECTION_STATISTICS_TCP_INTERCEPT = "threat-detection statistics tcp-intercept"
    THREAT_DETECTION_STATISTICS_ACCESS_LIST = "threat-detection statistics access-list"

    def __init__(self):
        DMObject.__init__(self, AdvancedThreatDetection.__name__)
        self.register_child(ThreatDetectStatisticsAccessList())
        self.register_child(TcpIntercept('AdvancedThreatDetectionTcpIntercept',
                                          self.THREAT_DETECTION_STATISTICS_TCP_INTERCEPT))
        for opt in ['host', 'port', 'protocol']:
            self.register_child(NumberOfRateOption('AdvancedThreatDetection' +
                                                   opt.title(), self.THREAT_DETECTION_STATISTICS + ' ' + opt))

class ThreatDetectStatisticsAccessList(DMBoolean, UnsupportedOnMultiContextASAValiator):
    def __init__(self):
        DMBoolean.__init__(self, "access_list",
                           AdvancedThreatDetection.THREAT_DETECTION_STATISTICS_ACCESS_LIST)

class NumberOfRateOption(SimpleType, UnsupportedOnMultiContextASAValiator):
    ''' Class to support CLI
    threat-detection statistics host [number-of-rate {1 | 2 | 3}]
    threat-detection statistics port [number-of-rate {1 | 2 | 3}]
    threat-detection statistics protocol [number-of-rate {1 | 2 | 3}]
    '''
    def __init__(self, name, asa_key):
        SimpleType.__init__(self, name, asa_key,
                            asa_gen_template = asa_key + ' number-of-rate %(number_of_rate)s',
                            defaults= {'number_of_rate': 1})

    def get_cli(self):
        return self.asa_key if self.get_state() == State.DESTROY\
               else SimpleType.get_cli(self)

    def is_the_same_cli(self, cli):
        if cli == self.asa_key:
            defalut_number = self.defaults['number_of_rate']
            return int(self.get_value().get('number_of_rate', defalut_number)) == defalut_number
        return SimpleType.is_the_same_cli(self, cli)

class TcpIntercept(SimpleType):
    ''' Class to support CLI:
    threat-detection statistics tcp-intercept [rate-interval <minutes>]
        [burst-rate <attacks_per_sec>]
        [average-rate <attacks_per_sec>]
        '''
    def __init__(self, name, asa_key):
        SimpleType.__init__(self, name, asa_key,
                            asa_gen_template = asa_key + 
                                               ' rate-interval %(rate_interval)s'
                                               ' burst-rate %(burst_rate)s'
                                               ' average-rate %(average_rate)s',
                            defaults = {'rate_interval': 30,
                                        'average_rate': 200,
                                        'burst_rate':   400})

    def get_cli(self):
        return self.asa_key if self.get_state() == State.DESTROY\
               else SimpleType.get_cli(self)

    def is_the_same_cli(self, cli):
        if cli == self.asa_key:
            ifc_config = {}
            for key in self.defaults:
                ifc_config[key] = self.get_value().get(key, self.defaults[key])
            return ifc_config == self.defaults
        return SimpleType.is_the_same_cli(self, cli)
