'''
Created on Jan 14, 2015

@author: jeffryp

Copyright (c) 2015 by Cisco Systems, Inc.
All rights reserved.

Classes used for OSPF process configuration.
'''

import re

from asaio.cli_interaction import ignore_response_parser
from common import Redistribute, RouterCommon, RouterID
from translator.base.compositetype import CompositeType
from translator.base.dmlist import DMList
from translator.base.dmobject import DMObject
from translator.base.simpletype import SimpleType
from translator.state_type import State, Type
from translator.structured_cli import convert_to_structured_commands
from translator.structured_cli import StructuredCommand
from utils.util import filter_first
from utils.util import normalize_network_address

class Area(RouterCommon, SimpleType):
    'Area configuration'

    AREA_TYPES = ('nssa', 'stub')
    CLI_PATTERN = None

    def __init__(self, instance, ipv6):
        super(Area, self).__init__(instance)
        self.ipv6 = ipv6

        self.register_child(AreaCost())
        if not ipv6:
            self.networks = Networks()
            self.register_child(self.networks)

    def create_missing_ifc_delta_cfg(self):
        if not self.has_ifc_delta_cfg():
            self.value = {'state': State.DESTROY}
            self.parent.get_value()[self.get_key()] = self.value
        for child in self:
            child.create_missing_ifc_delta_cfg()

    def diff_ifc_asa(self, cli):
        if not self.has_ifc_delta_cfg():
            self.value = self.parse_cli(cli)
            self.set_state(State.DESTROY)
            self.parent.get_value()[self.get_key()] = self.value
        elif self.get_state() != State.DESTROY:
            if self.is_the_same_cli(cli):
                self.set_state(State.NOCHANGE)
            else:
                # Save a copy of the old type
                self.value['old_config'] = self.parse_cli(cli)
                self.set_state(State.MODIFY)

    @classmethod
    def find_area_id(cls, cli, ipv6):
        '''
        If this is an area related CLI, return the area ID
        '''
        m = cls.get_cli_pattern().match(cli)
        if m:
            return int(m.group(1))
        area_id = AreaCost.find_area_id(cli)
        if area_id != None:
            return area_id
        if not ipv6:
            return Network.find_area_id(cli)

    def get_area_id(self):
        return self.get_key()

    def get_cli(self):
        area_type = self.get_type()
        if area_type not in self.AREA_TYPES:
            return self.get_cli_type_prefix(area_type)

        cli = []
        cli.append(self.get_cli_type_prefix(area_type))
        if area_type == 'nssa' and not self.value.get('redistribute', False):
            cli.append(' no-redistribution')
        if not self.value.get('summary', False):
            cli.append(' no-summary')
        return ''.join(cli)

    @classmethod
    def get_cli_pattern(cls):
        if not cls.CLI_PATTERN:
            cls.CLI_PATTERN = re.compile(
                'area (\d+)(?: (nssa|stub)(?: (no-redistribution))?(?: (no-summary))?)?$')
        return cls.CLI_PATTERN

    def get_cli_prefix(self):
        return 'area %s' % self.get_key()

    def get_cli_type_prefix(self, area_type):
        cli = []
        cli.append(self.get_cli_prefix())
        if area_type in self.AREA_TYPES:
            cli.append(' ' + area_type)
        return ''.join(cli)

    def get_translator(self, cli):
        if self.is_my_cli(cli):
            return self
        for child in self:
            result = child.get_translator(cli)
            if result:
                return result

    def get_type(self):
        return self.value.get('type', '')

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        state = self.get_state()
        if state == State.DESTROY:
            # Before removing an area, the associated networks need to be
            # removed.  The other area related configuration will be removed by
            # the ASA when the area is removed.
            #
            # For OSPFv3, an area needs to be removed from an interface
            # before it can be removed from the router process.  To do this,
            # put the no CLI on the asa_cfg_list, not the no_asa_cfg_stack.
            area_no_list = asa_cfg_list if self.ipv6 else no_asa_cfg_stack
            self.generate_cli(area_no_list,
                              'no ' + self.get_cli_prefix(),
                              response_parser=area_response_parser)
            if not self.ipv6:
                self.networks.ifc2asa_with_mode(no_asa_cfg_stack, asa_cfg_list,
                                                self.mode_command)
            return

        if state != State.NOCHANGE:
            area_type = self.get_type()
            no_redistribute = not self.value.get('redistribute', False)
            no_summary = not self.value.get('summary', False)
    
            old_config = self.value.get('old_config', {})
            old_no_redistribute = not old_config.get('redistribute', False)
            old_no_summary = not old_config.get('summary', False)
            old_type = old_config.get('type', '')
    
            if state == State.MODIFY:
                # To change the type of an area, the old one must be removed first.
                if old_type != area_type and old_type in self.AREA_TYPES:
                    self.generate_cli(asa_cfg_list,
                                      'no ' + self.get_cli_type_prefix(old_type))
    
            # For IPv6, don't generate CLI for normal type
            if self.should_generate_cli():
                if state == State.CREATE:
                    self.generate_cli(asa_cfg_list, self.get_cli())
                elif state == State.MODIFY:
                    # This algorithm is based on the ASDM OSPFv3 code.
                    #
                    # To remove an individual option in the 'area' CLI, a 'no'
                    # CLI with just that option is used.  Sending a new 'area'
                    # CLI to replace the existing one is not supported.
                    # However, adding an option can be done by sending a new
                    # 'area' CLI to replace the existing one.
                    updateRedistribution = False
                    updateSummary = False
                    if old_type == area_type:
                        if area_type == 'nssa':
                            if old_no_redistribute != no_redistribute:
                                if not no_redistribute:
                                    self.generate_cli(
                                        asa_cfg_list,
                                        'no ' + self.get_cli_type_prefix(area_type)
                                            + ' no-redistribution')
                                else:
                                    updateRedistribution = True
                        if old_no_summary != no_summary:
                            if not no_summary:
                                self.generate_cli(
                                    asa_cfg_list,
                                    'no ' + self.get_cli_type_prefix(area_type)
                                        + ' no-summary')
                            else:
                                updateSummary = True
                    if (old_type != area_type or updateRedistribution or
                            updateSummary):
                        self.generate_cli(asa_cfg_list, self.get_cli())

        # For IPv4, the area related configuration needs to be generated after
        # the area is created or modified.  IPv6 uses the normal order of
        # commands; All the deletes are before the adds/modifies 
        if self.ipv6:
            self.ifc2asa_with_mode(no_asa_cfg_stack, asa_cfg_list,
                                   self.mode_command)
        else:
            cli_list = []
            no_cli_stack = []
            self.ifc2asa_with_mode(no_cli_stack, cli_list,
                                   self.mode_command)
            asa_cfg_list.extend(reversed(no_cli_stack))
            asa_cfg_list.extend(cli_list)

    def is_my_cli(self, cli):
        return self.get_cli_pattern().match(cli)

    def parse_cli(self, cli):
        m = self.get_cli_pattern().match(cli)
        if m:
            result = {}
            result['type'] = m.group(2)
            result['redistribute'] = (m.group(3) != 'no-redistribution')
            result['summary'] = (m.group(4) != 'no-summary')
            return result

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        self.value = delta_ifc_cfg_value
        for child in self:
            child.populate_model(None, self.value)

    def should_generate_cli(self):
        # Don't generate area CLI for normal type if IPv6
        if self.get_type() not in self.AREA_TYPES and self.ipv6:
            return False
        return True

class AreaCost(RouterCommon, SimpleType):
    'Area cost configuration'

    CLI_PATTERN = None
    DEFAULT = 1

    def __init__(self):
        super(AreaCost, self).__init__('areaCost')

    def diff_ifc_asa(self, cli):
        if not self.has_ifc_delta_cfg():
            self.value = self.parse_cli(cli)
            self.set_state(State.DESTROY)
            self.parent.get_value()[self.get_key()] =  self.value
        elif self.get_state() != State.DESTROY:
            # When there is a default cost in the ASA configuration, then there
            # will be no normal area configuration.  Change the state of the
            # area from CREATE to NOCHANGE
            area = self.parent
            if area.get_state() == State.CREATE:
                area.set_state(State.NOCHANGE)

            if self.is_the_same_cli(cli):
                self.set_state(State.NOCHANGE)
            else:
                self.set_state(State.MODIFY)

    @classmethod
    def find_area_id(cls, cli):
        '''
        If this is an area cost CLI, return the area ID
        '''
        m = cls.get_cli_pattern().match(cli)
        if m:
            return int(m.group(1))

    def get_area_id(self):
        return self.parent.get_area_id()

    def get_cli(self):
        return 'area %s default-cost %s' % (self.get_area_id(),
                                            self.value[self.get_key()])

    @classmethod
    def get_cli_pattern(cls):
        if not cls.CLI_PATTERN:
            cls.CLI_PATTERN = re.compile('area (\d+) default-cost (\d+)')
        return cls.CLI_PATTERN

    def get_translator(self, cli):
        if isinstance(cli, StructuredCommand):
            cli = cli.command
        m = self.get_cli_pattern().match(cli.strip())
        if m:
            return self

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        # Cost is only supported for NSSA and stub areas
        if (self.has_ifc_delta_cfg() and
                self.parent.get_type() in Area.AREA_TYPES):
            state = self.get_state()
            if state in (State.CREATE, State.MODIFY):
                self.generate_cli(asa_cfg_list, self.get_cli())
            elif state == State.DESTROY:
                self.generate_cli(no_asa_cfg_stack, 'no ' + self.get_cli(),
                                  response_parser=area_response_parser)

    def parse_cli(self, cli):
        m = self.get_cli_pattern().match(cli)
        if m:
            return {self.get_key(): int(m.group(2))}

class Areas(RouterCommon, DMList):
    'Container for Area objects'

    def __init__(self):
        super(Areas, self).__init__('areas', Area)

    def create_missing_ifc_delta_cfg(self):
        if not self.has_ifc_delta_cfg():
            self.value = {}
            self.parent.get_value()[self.get_key()] = self.value
        for child in self:
            child.create_missing_ifc_delta_cfg()

    def get_translator(self, cli):
        if isinstance(cli, basestring):
            area_id = Area.find_area_id(cli, self.is_ipv6())
            if area_id != None:
                area = self.get_child(area_id)
                if not area:
                    area = self.child_class(area_id, self.is_ipv6())
                    self.register_child(area)
                    area.create_missing_ifc_delta_cfg()
                return area.get_translator(cli)

    def is_ipv6(self):
        return self.parent.is_ipv6()

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        if 'areas' in delta_ifc_cfg_value:
            self.value = delta_ifc_cfg_value['areas']
            for area_id, value in self.value.iteritems():
                if isinstance(value, dict):
                    area = self.get_child(area_id)
                    if not area:
                        area = self.child_class(area_id, self.is_ipv6())
                        self.register_child(area)
                    area.populate_model(None, value)

            if not self.is_ipv6():
                self.update_networks()

    def remove_deleted(self):
        'Remove any area that is to be deleted'
        for area in self.children.values():
            if area.get_state() == State.DESTROY:
                self.remove(area)

    def update_networks(self):
        # Use the OSPF interface configuration along with the associated
        # interface IPv4 address to populate the networks.  The state of the
        # area cannot be used since it is the merged state of all the
        # configuration for that area.  Instead, use the area state from the
        # OSPF interface configuration.
        for interface in self.get_top().get_child('Interface'):
            area_id = interface.get_ospf_area_id()
            area = area_id != None and self.get_child(area_id)
            if area:
                addr_dict = interface.get_ipv4_address_dict()
                area_state = interface.get_ospfvif_state()
                if area_state != None and area.get_state() == State.CREATE:
                    # Networks need to be created when the area is created
                    area_state = State.CREATE
                if addr_dict and area_state != None:
                    addr_state = addr_dict['state']
                    if not ((area_state == State.CREATE and
                            addr_state == State.DESTROY) or
                            (area_state == State.DESTROY and
                            addr_state == State.CREATE)):
                        if area_state != State.NOCHANGE:
                            addr_dict['state'] = area_state
                        area.networks.add_network(addr_dict)

class CombinedProcess(RouterCommon, DMObject):
    '''
    The combined IPv4 and IPv6 OSPF processes
    '''

    ASA_CFG_KEY = 'OspfProcess#'

    def __init__(self, processID):
        super(CombinedProcess, self).__init__(processID)
        self.ospf_devs = []

        self.register_child(OspfProcess('ipv6', ipv6=True))
        self.register_child(OspfProcess('ipv4'))

    def create_missing_ifc_delta_cfg(self):
        for child in self:
            child.create_missing_ifc_delta_cfg()
        self.value['translated'] = True

    def diff_ifc_asa(self, cli):
        if not self.has_ifc_delta_cfg():
            self.delta_ifc_key = Type.OSPFDEV, '', cli
            self.value = {'state': State.MODIFY,
                          'enable': True,
                          'processID': self.get_process_id()}
            ancestor = self.get_ifc_delta_cfg_ancestor()
            if ancestor:
                ancestor.delta_ifc_cfg_value['value'][self.delta_ifc_key] =  self.value
            self.get_top().delta_ifc_cfg_value[self.get_asa_cfg_key()] = self.value

        for child in self.children.itervalues():
            translator = child.get_translator(cli)
            if translator:
                translator.diff_ifc_asa(cli)
                return

    def get_address_family(self, area_id):
        # Combined address family of all the OSPF interface configuration with
        # the specified area ID
        address_family = set()
        for encapass in self.get_top().get_child('ENCAPASS'):
            ospfvif = encapass.delta_ifc_cfg_value.get('OspfVIfCfg')
            if ospfvif:
                for key, value in ospfvif.iteritems():
                    if (isinstance(key, tuple) and key[0] == Type.OSPFVENCAPASC
                            and 'area' in value and value['area'] == area_id
                            and 'addressFamily' in value):
                        address_family.update(value['addressFamily'])
        return address_family

    def get_asa_cfg_key(self):
        return self.ASA_CFG_KEY + self.get_key()

    def get_process_id(self):
        return self.get_key()

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        if not hasattr(self, 'value'):
            self.delta_ifc_key = delta_ifc_key
            self.value = self.get_top().delta_ifc_cfg_value.setdefault(
                self.get_asa_cfg_key(), {})
        self.ospf_devs.append(delta_ifc_cfg_value)

    def populate_passive_interface(self, interface, value):
        # Only supported for OSPFv3
        ospfv3 = self.get_child('ipv6')
        if ospfv3:
            ospfv3.populate_passive_interface(interface, value)

    def update_references(self):
        if 'translated' not in self.value:
            for ospf_dev in self.ospf_devs:
                if ospf_dev.get('state') == State.DESTROY:
                    # If an area is being destroyed, there may not be any
                    # interfaces using it, so delete both the IPv4 and IPv6
                    # areas.
                    address_family = set(('ipv4', 'ipv6'))
                else:
                    address_family = self.get_address_family(ospf_dev.get('area'))
                translate_ifc_cfg(self.value, ospf_dev, address_family)

        for child in self.children.itervalues():
            child.populate_model(None, self.value)

class Network(RouterCommon, SimpleType):
    '''
    Network configuration

    Only used for IPv4.
    '''

    CLI_PATTERN = None

    def __init__(self, instance):
        super(Network, self).__init__(instance)

    def create_asa_key(self):
        return self.get_cli()

    def diff_ifc_asa(self, cli):
        if not self.has_ifc_delta_cfg():
            self.value = self.parse_cli(cli)
            self.set_state(State.DESTROY)
            self.parent.get_value().append(self.value)
        elif self.get_state() != State.DESTROY:
            if self.is_the_same_cli(cli):
                self.set_state(State.NOCHANGE)
            else:
                self.set_state(State.MODIFY)

    @classmethod
    def find_area_id(cls, cli):
        '''
        If this is a network CLI, return the area ID
        '''
        m = cls.get_cli_pattern().match(cli)
        if m:
            return int(m.group(3))

    def get_area_id(self):
        return self.parent.get_area_id()

    def get_cli(self):
        area = self.get_area_id()
        ipaddress = self.value['ipaddress']
        netmask = self.value['netmask']
        return 'network %s %s area %s' % (ipaddress, netmask, area)

    @classmethod
    def get_cli_pattern(cls):
        if not cls.CLI_PATTERN:
            cls.CLI_PATTERN = re.compile('network (\S+) (\S+) area (\d+)')
        return cls.CLI_PATTERN

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        if self.has_ifc_delta_cfg():
            state = self.get_state()
            if state in (State.CREATE, State.MODIFY):
                self.generate_cli(asa_cfg_list, self.get_cli())
            elif state == State.DESTROY:
                self.generate_cli(no_asa_cfg_stack, 'no ' + self.get_cli(),
                                  response_parser=area_response_parser)

    def parse_cli(self, cli):
        m = self.get_cli_pattern().match(cli)
        if m:
            result = {}
            result['ipaddress'] = m.group(1)
            result['netmask'] = m.group(2)
            result['area'] = int(m.group(3))
            return result

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        self.value = delta_ifc_cfg_value
        self.value['ipaddress'] =  normalize_network_address(self.value['ipaddress'],
                                                             self.value['netmask'])

    @staticmethod
    def make_key(network):
        netmask = str(network['netmask'])
        ipaddress = normalize_network_address(network['ipaddress'], netmask)
        return ipaddress + '/' + netmask

    def set_state_recursive(self, state):
        # This call only occurs during mini-audit to set the state to CREATE.
        # Don't create a Network if it is being destroyed.
        if self.get_state() != State.DESTROY:
            super(Network, self).set_state_recursive(state)

class Networks(RouterCommon, DMList):
    '''
    Container for Network objects

    Only used for IPv4.
    '''

    def __init__(self):
        super(Networks, self).__init__('networks', Network)

    def add_network(self, network):
        key = Network.make_key(network)
        if key not in self.children:
            child = self.child_class(key)
            self.register_child(child)
            child.populate_model(None, network)
            self.get_value().append(child.get_value())

    def create_missing_ifc_delta_cfg(self):
        if not self.has_ifc_delta_cfg():
            self.value = []
            self.parent.get_value()[self.get_key()] = self.value

    def get_area_id(self):
        return self.parent.get_area_id()

    def get_translator(self, cli):
        if isinstance(cli, basestring):
            if Network.get_cli_pattern().match(cli):
                for network in self.children.itervalues():
                    result = network.get_translator(cli)
                    if result:
                        return result

                # Only destroy a Network during a full audit or when the Area is
                # being destroyed.  For a mini-audit, not all of the interface
                # configuration is available.
                if (self.parent.get_state() == State.DESTROY or
                        self.get_top().is_audit):
                    result = self.child_class(cli)
                    self.register_child(result)
                    return result

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        if 'networks' in delta_ifc_cfg_value:
            if not self.has_ifc_delta_cfg():
                self.value = delta_ifc_cfg_value['networks']
            for network in self.value:
                key = Network.make_key(network)
                if key not in self.children:
                    child = self.child_class(key)
                    self.register_child(child)
                    child.populate_model(None, network)

class OspfProcess(RouterCommon, CompositeType):
    'OSPF process configuration'

    def __init__(self, ifc_key, ipv6=False):
        super(OspfProcess, self).__init__(ifc_key)
        if ipv6:
            self.asa_key = 'ipv6 router ospf'
            self.ip_key = 'ipv6'
            self.mini_audit_command = 'show running-config ipv6'
        else:
            self.asa_key = 'router ospf'
            self.ip_key = 'ipv4'
            self.mini_audit_command = 'show running-config router ospf'
        self.asa_gen_template = self.asa_key + ' %s'
        self.ipv6 = ipv6
        self.areas = Areas()
        self.router_id = RouterID('rtrId', 'router-id')

        self.register_child(self.router_id)
        self.register_child(self.areas)
        if ipv6:
            self.register_child(PassiveInterfaces())

        subnets = not ipv6
        self.register_child(Redistribute('connected', subnets))
        self.register_child(Redistribute('static', subnets))
        self.register_child(Redistribute('bgp', subnets))

    def create_missing_ifc_delta_cfg(self):
        if not self.has_ifc_delta_cfg():
            self.value = {'state': State.MODIFY, 'enable': True}
            self.parent.get_value()[self.get_key()] = self.value
        if 'redistribute' not in self.value:
            self.value['redistribute'] = {}
        for child in self:
            child.create_missing_ifc_delta_cfg()

    def diff_ifc_asa(self, cli):
        if not self.has_ifc_delta_cfg():
            self.value = {'state': State.DESTROY}
            self.parent.value[self.ip_key] =  self.value
        elif self.get_state() != State.DESTROY:
            self.set_state(State.MODIFY)
            if isinstance(cli, StructuredCommand):
                for cmd in cli.sub_commands:
                    translator = self.get_child_translator(cmd)
                    if translator:
                        translator.diff_ifc_asa(cmd)

    def get_cli(self):
        return self.asa_gen_template % self.get_process_id()

    def get_process_id(self):
        return self.parent.get_process_id()

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        if self.has_ifc_delta_cfg():
            if (self.get_state() == State.DESTROY or
                    not self.value.get('enable', False)):
                self.generate_cli(no_asa_cfg_stack,
                                  'no ' + self.get_cli(),
                                  response_parser=ignore_response_parser)
            else:
                if (self.get_state() == State.MODIFY and
                        not self.get_top().is_audit):
                    self.areas.remove_deleted()
                    self.set_state_recursive(State.CREATE)
                    self.mini_audit()

                # If the router ID has changed, the old OSPF process needs to be
                # removed and added back with the new router ID.
                if self.router_id.get_state() == State.MODIFY:
                    self.areas.remove_deleted()
                    self.set_state_recursive(State.CREATE)
                    self.generate_cli(no_asa_cfg_stack,
                                      'no ' + self.get_cli(),
                                      response_parser=ignore_response_parser)

                self.ifc2asa_with_mode(no_asa_cfg_stack, asa_cfg_list,
                                       self.get_child_mode_command())

    def is_ipv6(self):
        return self.ipv6

    def is_the_same_cli(self, cli):
        # Handle the case for no sub-commands
        if isinstance(cli, basestring):
            return SimpleType.is_the_same_cli(self, cli)
        return super(OspfProcess, self).is_the_same_cli(cli)

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        if self.ip_key in delta_ifc_cfg_value:
            self.value = delta_ifc_cfg_value[self.ip_key]
            for child in self.children.itervalues():
                child.populate_model(None, self.value)

            # Added networks may have changed the composite state
            if self.get_state() == State.NOCHANGE:
                self.set_state(self.get_state_recursive())

    def populate_passive_interface(self, interface, value):
        passive_interfaces = self.get_child('passive')
        if passive_interfaces:
            passive_interfaces.populate_passive_interface(interface, value)

    def read_asa_config(self):
        asa_clis = super(OspfProcess, self).read_asa_config()
        if asa_clis:
            asa_clis = convert_to_structured_commands(asa_clis.split('\n'))
            mine = filter_first(lambda x: isinstance(x, StructuredCommand) and
                                x.command.startswith(self.asa_key),  asa_clis)
            if mine:
                return str(mine)

class OspfProcesses(DMList):
    '''
    Container for OSPF processes
    '''

    CLI_PATTERN = None

    def __init__(self):
        super(OspfProcesses, self).__init__('OspfProcess', CombinedProcess)

    def get_active_process(self):
        'Returns the active OSPF process'

        for ospf in self.children.itervalues():
            if ospf.has_ifc_delta_cfg() and ospf.get_state() != State.DESTROY:
                return ospf

    @classmethod
    def get_cli_pattern(cls):
        if not cls.CLI_PATTERN:
            cls.CLI_PATTERN = re.compile('(?:ipv6 )?router ospf (\d+)')
        return cls.CLI_PATTERN

    def get_translator(self, cli):
        if isinstance(cli, StructuredCommand):
            cli = cli.command
        m = self.get_cli_pattern().match(cli.strip())
        if m:
            processID = m.group(1)
            process = self.get_child(processID)
            if not process:
                process = self.child_class(processID)
                self.register_child(process)
            return process

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        if not 'processID' in delta_ifc_cfg_value:
            # Default is 1 on the APIC
            delta_ifc_cfg_value['processID'] = 1
        processID = str(delta_ifc_cfg_value['processID'])
        process = self.get_child(processID)
        if not process:
            process = self.child_class(processID)
            self.register_child(process)
        process.populate_model(delta_ifc_key, delta_ifc_cfg_value)

class PassiveInterface(RouterCommon, SimpleType):
    'Passive interface configuration'

    def __init__(self, instance):
        super(PassiveInterface, self).__init__(instance, 'passive-interface')

    def parse_cli(self, cli):
        return {}

    def get_cli(self):
        return self.asa_gen_template % self.get_key()

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        self.value = delta_ifc_cfg_value

    def set_state_recursive(self, state):
        # This call only occurs during mini-audit to set the state to CREATE.
        # Don't create a PassiveInterface if it is being destroyed.
        if self.get_state() != State.DESTROY:
            super(PassiveInterface, self).set_state_recursive(state)

class PassiveInterfaces(RouterCommon, DMList):
    'Collection of passive interfaces'

    CLI_PATTERN = None

    def __init__(self):
        super(PassiveInterfaces, self).__init__('passive',
                                                PassiveInterface,
                                                'passive-interface')

    def create_missing_ifc_delta_cfg(self):
        if not self.has_ifc_delta_cfg():
            self.value = {}
            self.parent.get_value()[self.get_key()] = self.value

    def get_cli_pattern(self):
        if not self.CLI_PATTERN:
            self.CLI_PATTERN = re.compile(self.asa_key + ' (.+)')
        return self.CLI_PATTERN

    def get_translator(self, cli):
        if isinstance(cli, StructuredCommand):
            cli = cli.command
        m = self.get_cli_pattern().match(cli.strip())
        if m:
            key = m.group(1)
            child = self.get_child(key)
            if not child:
                child = self.child_class(key)
                self.register_child(child)
            return child

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        if 'passive' in delta_ifc_cfg_value:
            if not self.has_ifc_delta_cfg():
                self.value = delta_ifc_cfg_value['passive']
            for key, value in self.value.iteritems():
                if isinstance(value, dict) and key not in self.children:
                    child = self.child_class(key)
                    self.register_child(child)
                    child.populate_model(None, value)

        # Populate the passive interfaces from the interface configuration
        for interface in self.get_top().get_child('Interface'):
            intf_ospfv3 = interface.get_ospfv3()
            passive = (intf_ospfv3.get_value() and
                       intf_ospfv3.get_value().get('passive'))
            if passive:
                self.populate_passive_interface(interface.nameif, passive)
            if (intf_ospfv3.get_state() == State.MODIFY and
                    self.parent.get_state() == State.NOCHANGE):
                # Force the OSPF Process to do a mini_audit by
                # setting it's state to MODIFY.  This will pick up
                # any changes to the passive interface.
                self.parent.set_state(State.MODIFY)

    def populate_passive_interface(self, interface, value):
        if interface not in self.children:
            child = self.child_class(interface)
            self.register_child(child)
            child.populate_model(None, value)

def area_response_parser(response):
    '''
    CLI response parser for area related commands
    '''

    if 'Specified area is not configured' in response:
        return None
    return response

def ospf_process_audit_init(configuration):
    '''
    Transformer to initialize the configuration for audit. It sets all the
    translated dictionaries to a state of CREATE.
    '''

    def set_state(cfg):
        if isinstance(cfg, dict):
            if 'state' in  cfg:
                cfg['state'] = State.CREATE
            for value in cfg.itervalues():
                set_state(value)
        elif isinstance(cfg, list):
            for value in cfg:
                set_state(value)

    ospf_keys = filter(
        lambda key:
        isinstance(key, basestring) and key.startswith(CombinedProcess.ASA_CFG_KEY),
        configuration.values()[0].keys())
    for ospf_key in ospf_keys:
        set_state(configuration.values()[0][ospf_key])
    return configuration

def translate_ifc_cfg(asa_cfg, ifc_cfg, address_family):
    '''
    Translate from the IFC model to the ASA model and create any missing
    configuration. The ASA model has a one-to-one correspondence to the ASA
    command hierarchy.
    '''
    def make_dict(asa_cfg, ifc_cfg, key, state):
        if key in ifc_cfg:
            asa_cfg.setdefault(key, {key: ifc_cfg[key]})
            merge_state(asa_cfg[key], state)

    def merge_state(asa_cfg, new_state):
        if 'state' in asa_cfg:
            states = set((asa_cfg['state'], new_state))
            if len(states) > 1:
                # The new state and current state are different
                asa_cfg['state'] = State.MODIFY
        else:
            asa_cfg['state'] = new_state

    state = ifc_cfg['state']
    merge_state(asa_cfg, state)

    for ip_key in address_family:
        is_ipv6 = (ip_key == 'ipv6')
        ip_value = asa_cfg.setdefault(ip_key, {})
        merge_state(ip_value, state)

        # Areas
        areas = ip_value.setdefault('areas', {})
        if 'area' in ifc_cfg:
            area = areas.setdefault(int(ifc_cfg['area']), {})
            merge_state(area, state)
            area_type = ifc_cfg.get('areaType')
            area.setdefault('type',
                            area_type if area_type in ('nssa', 'stub') else '')
            for ctrl in ifc_cfg.get('areaCtrl', []):
                if ctrl in ('redistribute', 'summary'):
                    area.setdefault(ctrl, True)

            # Area cost
            make_dict(area, ifc_cfg, 'areaCost', state)

            # Networks
            if not is_ipv6:
                area.setdefault('networks', [])

        # Enable
        ip_value.setdefault('enable', ifc_cfg.get('enable', False))

        # Passive interfaces
        if is_ipv6:
            ip_value.setdefault('passive', {})

        # Redistributions
        if 'redistribute' in ifc_cfg:
            redistribute = ip_value.setdefault('redistribute', {})
            for protocol in ifc_cfg['redistribute']:
                if protocol in ('bgp', 'connected', 'static'):
                    redistribute.setdefault(protocol, {})
                    merge_state(redistribute[protocol], state)

        # Router ID
        make_dict(ip_value, ifc_cfg, 'rtrId', state)
