'''
Created on Sep 13, 2017

@author: dli
'''
import re
from translator.base.dmobject import DMObject
from translator.base.compositetype import Description
from translator.base.simpletype import SimpleType
from translator.base.dmlist import DMList
from translator.state_type import State, Type

#Name of the built-in user group
GROUP_NAME='SNMPv3UserGroup'

class SNMP(DMObject):
    '''
    Model after SNMP configuration
    '''
    def __init__(self):
        DMObject.__init__(self, SNMP.__name__, 'snmp-server')
        self.register_child(SimpleType('contact', 'snmp-server contact'))
        self.register_child(Location())
        self.register_child(Group())
        self.register_child(DMList('User', User, re.compile('^snmp-server user (\S+) ' + GROUP_NAME)))
        self.register_child(HostList())

class Location(Description):
    '''Model after 'snmp-server location' command.
    It's behavior is identical to Description object
    '''
    def __init__(self):
        SimpleType.__init__(self, 'location', 'snmp-server location')

    def parse_cli(self, cli):
        result = Description.parse_cli(self, cli)
        return result[result.index(' ')+1:]

class Group(SimpleType):
    '''Model after 'snmp-server group SNMPv3Group v3 priv' command,
    This is built-in, not exposed to the user. It is issued if when first creating new user.
    '''
    def __init__(self):
        SimpleType.__init__(self, '__Group__', 'snmp-server group '+ GROUP_NAME + ' v3 priv')

    def update_references(self):
        '''
        Generate the CLI only if all users are newly created, delete the CLI if all users are deleted.
        '''
        if not self.has_ifc_delta_cfg() and self.parent.has_ifc_delta_cfg():
            state = self.determine_state()
            if state == -1:
                return
            self.delta_ifc_key = (Type.PARAM, self.ifc_key, GROUP_NAME)
            self.delta_ifc_cfg_value = {'state': state, 'value': {}}
            self.parent.delta_ifc_cfg_value['value'][self.delta_ifc_key] = self.delta_ifc_cfg_value

    def get_cli(self):
        return self.asa_key

    def determine_state(self):
        """
        The sate depends on the whether there is any user configured.
        Return -1 if there is no need to handle the config, or the proper state
        """
        user_list = self.parent.get_child('User')
        states = map(lambda child: child.get_state(), user_list.children.values())
        if not states:
            return -1
        if filter(lambda state: state == State.CREATE, states):
            result = State.CREATE
        elif len(filter(lambda state: state == State.DESTROY, states)) == len(states):
            result = State.DESTROY
        else:
            result = State.NOCHANGE
        return result

class User(SimpleType):
    '''Model after 'snmp-server user ...' command
    '''
    def __init__(self, name):
        asa_key = 'snmp-server user ' + name + ' ' + GROUP_NAME + ' v3'
        SimpleType.__init__(self, name, asa_key,
                            asa_key +  ' auth md5 %(AuthenticationPassword)s priv aes %(AesSize)s %(PrivacyPassword)s',
                            defaults={'AesSize': '128'})

    def get_cli(self):
        return self.asa_key if self.get_state() == State.DESTROY else SimpleType.get_cli(self)

    def create_delta_ifc_key(self, cli):
        return (Type.PARAM, self.parent.ifc_key, cli.split()[2])

    def is_the_same_cli(self, cli):
        '''Override the default to only check the prefix, as the CLI is stored in a format
        different than what we generated from ASA-DP. The passwords are in encrypted format in
        the output of 'show run'.
        '''
        return cli.startswith(self.get_asa_key())

class HostList(DMList):
    def __init__(self):
        DMList.__init__(self, 'Host', Host, re.compile('^snmp-server host management .+ version 3'),
                        mini_audit_command = 'show run snmp-server | grep host management')

class Host(SimpleType):
    '''Model after 'snmp-server host ...' command
    '''
    def __init__(self, name):
        SimpleType.__init__(self, name,
                            asa_gen_template='snmp-server host %(interface)s %(IP)s poll version 3 %(user)s',
                            defaults={'interface': 'management'})

    def create_asa_key(self):
        return self.get_cli()

    def parse_cli(self, cli):
        '''
        Override the default implementation to take care of unsupported SNMPv3 entry, such as
        ''snmp-server host management 1.1.1.1 trap version 3 joe udp-port 345'
        '''
        result = SimpleType.parse_multi_parameter_cli(self, cli, 'snmp-server host %(interface)s %(IP)s')
        '''
        To delete a snmp-server host, 'no snmp-server host <interface> <ip-address>' should work, but
        there is an ASA bug CSCvh48662, which fails it. We have to issue 'no' on the complete CLI to work around
        this bug.
        '''
        if result:
            result[(Type.PARAM, 'cli_seen_on_asa','')] = {'state': State.NOCHANGE, 'value': cli}
        return result

    def get_cli(self):
        '''
        Override the default implementation to take care of unsupported SNMPv3 entry, such as
        ''snmp-server host management 1.1.1.1 trap version 3 joe udp-port 345'
        '''
        if self.get_state() == State.DESTROY:
            cli = self.get_value().get('cli_seen_on_asa')
            if cli:
                return cli
        return SimpleType.get_cli(self)
