'''
Created on March 26, 2015

@author: dli

Copyright (c) 2015 by Cisco Systems
'''

from translator.base.dmobject import DMObject
from translator.base.dmboolean import DMBoolean
from translator.base.simpletype import SimpleType
from translator.base.dmlist import DMList
from translator.validators import UnsupportedOnMultiContextASAValiator
from translator.structured_cli import StructuredCommand
from translator.state_type import State
from utils.util import ifcize_param_dict

class BasicThreatDetection(DMObject, UnsupportedOnMultiContextASAValiator):
    '''
    Basic Threat Detection
    Supports the following CLIs:

    - threat-detection basic-threat
    - threat-threat-detection rate ...

    '''

    def __init__(self):
        DMObject.__init__(self, BasicThreatDetection.__name__)
        self.register_child(DMBoolean("basic_threat", 'threat-detection basic-threat'))
        drops = [# ifc_key                                  asa_key           default values
                 ('BasicThreatDetectionRateAclDrop',       'acl-drop',        (( 400,  800), ( 320,  640))),
                 ('BasicThreatDetectionRateBadPacketDrop', 'bad-packet-drop', (( 100,  400), (  80,  320))),
                 ('BasicThreatDetectionRateConnLimitDrop', 'conn-limit-drop', (( 100,  400), (  80,  320))),
                 ('BasicThreatDetectionRateDosDrop',       'dos-drop',        (( 100,  400), (  80,  320))),
                 ('BasicThreatDetectionRateFwDrop',        'fw-drop',         (( 400, 1600), ( 320, 1280))),
                 ('BasicThreatDetectionRateIcmpDrop',      'icmp-drop',       (( 100,  400), (  80,  320))),
                 ('BasicThreatDetectionRateInspectDrop',   'inspect-drop',    (( 400, 1600), ( 320, 1280))),
                 ('BasicThreatDetectionRateInterfaceDrop', 'interface-drop',  ((2000, 8000), (1600, 6400))),
                 ('BasicThreatDetectionRateSynAttack',     'syn-attack',      (( 100,  200), (  80,  160))),
                 ]
        for ifc_key, asa_key, defaults in drops:
            self.register_child(RateList(ifc_key, 'threat-detection rate ' + asa_key, defaults))

    def validate_configuration(self):
        if self.has_ifc_delta_cfg() and self.get_value():
            return DMObject.validate_configuration(self)

class RateList(DMList):
    'A helper DMList class that generates asa command templates for its children'
    def __init__(self, ifc_key, asa_key, defaults):
        DMList.__init__(self, ifc_key, ThreadDetectionRate, asa_key)
        self.defaults = {600:  {'average_rate': defaults[0][0], 'burst_rate': defaults[0][1]},
                         3600: {'average_rate': defaults[1][0], 'burst_rate': defaults[1][1]}}

    def register_child(self, dmobj):
        DMList.register_child(self, dmobj)
        reason = self.asa_key.split()[-1]
        dmobj.asa_key_template = 'threat-detection rate ' + reason  + ' rate-interval %(rate_interval)s'
        dmobj.asa_gen_template = dmobj.asa_key_template + ' average-rate %(average_rate)s burst-rate %(burst_rate)s'

    def is_my_cli(self, cli):
        """
        Override the default implementation to catch 'no ' form of the command.
        Need to do this to handle the case where the default rate command is disabled.
        """
        if not self.asa_key:
            return False
        if isinstance(cli, basestring):
            command = cli.strip()
        elif isinstance(cli, StructuredCommand):
            command =  cli.command.strip()
        result = command.startswith(self.asa_key)
        if result:
            return result

class ThreadDetectionRate(SimpleType):
    'model after "thread-detection rate command"'
    def __init__(self, instance):
        SimpleType.__init__(self, ifc_key = instance)

    def create_asa_key(self):
        return self.asa_key_template % self.get_value()

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        """
        Override the default implementation to handle the deletion of default rate command.
        A default rate command is one whose rate is 600 or 3600.
        On receiving request to delete a default rate, we won't issue the 'no' command but set its value
        to its default.

        The 'no' command for default rate will actually disable it, which we do not support it in
        device package.
        """
        if self.has_ifc_delta_cfg() and self.get_state() == State.DESTROY\
           and self.is_default_rate_interval():
            self.delta_ifc_cfg_value['state'] = State.CREATE
            self.delta_ifc_cfg_value['value'] = self.get_default_value()
        return SimpleType.ifc2asa(self, no_asa_cfg_stack, asa_cfg_list)

    def is_default_rate_interval(self):
        return int(self.get_value()['rate_interval']) in (600, 3600)

    def get_default_value(self):
        result = self.get_value()
        rate_interval = int(result['rate_interval'])
        for rate in ('average_rate', 'burst_rate'):
            result[rate] = self.parent.defaults[rate_interval][rate]
        return ifcize_param_dict(result)
