'''
Created on Feb 2, 2015

@author: jeffryp

Copyright (c) 2015 by Cisco Systems, Inc.
All rights reserved.

Common functions and classes used by both BGP and OSPF routing.
'''

import re

from translator.base.dmlist import DMList
from translator.base.simpletype import SimpleType
from translator.state_type import State

class RouterCommon(object):
    'Common functions used in router SimpleType classes'

    def diff_ifc_asa(self, cli):
        if not self.has_ifc_delta_cfg():
            self.value = self.parse_cli(cli)
            self.set_state(State.DESTROY)
            self.parent.get_value()[self.get_key()] =  self.value
        elif self.get_state() != State.DESTROY:
            if self.is_the_same_cli(cli):
                self.set_state(State.NOCHANGE)
            else:
                self.set_state(State.MODIFY)

    def get_config_path(self):
        return self.parent.get_config_path()

    def get_state(self):
        if hasattr(self, 'value') and isinstance(self.value, dict):
            return self.value.get('state')

    def set_state(self, state):
        if hasattr(self, 'value') and isinstance(self.value, dict):
            self.value['state'] = state

    def set_state_recursive(self, state):
        self.set_state(state)
        for child in self:
            child.set_state_recursive(state)

    def get_value(self):
        if hasattr(self, 'value'):
            return self.value

    def has_ifc_delta_cfg(self):
        return hasattr(self, 'value')

    def ifc2asa_with_mode(self, no_asa_cfg_stack, asa_cfg_list, mode_command):
        for child in self:
            if isinstance(child, DMList):
                for c in child:
                    c.mode_command = mode_command
            else:
                child.mode_command = mode_command
            child.ifc2asa(no_asa_cfg_stack, asa_cfg_list)

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        if self.get_key() in delta_ifc_cfg_value:
            self.value = delta_ifc_cfg_value[self.get_key()]

class Redistribute(RouterCommon, SimpleType):
    'Redistribute configuration'

    def __init__(self, protocol, subnets=False):
        super(Redistribute, self).__init__(
            ifc_key=protocol,
            asa_key='redistribute ' + protocol)
        self.subnets = subnets

    def diff_ifc_asa(self, cli):
        if not self.has_ifc_delta_cfg():
            self.value = {'state': State.DESTROY}
            self.parent.get_value()['redistribute'][self.get_key()] = self.value
        elif self.get_state() != State.DESTROY:
            self.set_state(State.NOCHANGE)

    def get_cli(self):
        cli = self.get_cli_prefix()
        if self.subnets:
            return cli + ' subnets'
        if self.get_key() == 'ospf':
            cli += ' match internal external 1 external 2'
        return cli

    def get_cli_prefix(self):
        cli = self.asa_key
        if self.get_key() == 'bgp':
            # Add the ASN of the BGP process to the CLI
            return cli + (' %s' % get_bgp_asn(self.get_top()))
        elif self.get_key() == 'ospf':
            # Add the ASN of the BGP process to the CLI
            return cli + (' %s' % get_ospf_process_id(self.get_top()))
        return cli

    def has_ifc_delta_cfg(self):
        if super(Redistribute, self).has_ifc_delta_cfg():
            if self.get_key() == 'bgp':
                # Ensure that BGP is configured
                if get_bgp_asn(self.get_top()) == None:
                    return False
            elif self.get_key() == 'ospf':
                # Ensure that OSPF is configured
                if get_ospf_process_id(self.get_top()) == None:
                    return False
            return True

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        if self.has_ifc_delta_cfg():
            state = self.get_state()
            if state in (State.CREATE, State.MODIFY):
                self.generate_cli(asa_cfg_list, self.get_cli())
                if self.get_key() == 'ospf':
                    self.generate_cli(asa_cfg_list, 'bgp redistribute-internal')
            elif state == State.DESTROY:
                self.generate_cli(no_asa_cfg_stack,
                                  'no ' + self.get_cli_prefix())
                if self.get_key() == 'ospf':
                    self.generate_cli(no_asa_cfg_stack, 'no bgp redistribute-internal')

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        if 'redistribute' in delta_ifc_cfg_value:
            redistribute = delta_ifc_cfg_value['redistribute']
            if self.get_key() in redistribute:
                self.value = redistribute[self.get_key()]

class RouterID(RouterCommon, SimpleType):
    'Router ID configuration'

    CLI_PATTERN = None

    def __init__(self, ifc_key, asa_key):
        super(RouterID, self).__init__(
            ifc_key=ifc_key,
            asa_key=asa_key,
            asa_gen_template=asa_key + ' %(rtrId)s')

    def diff_ifc_asa(self, cli):
        if not self.has_ifc_delta_cfg():
            self.value = self.parse_cli(cli)
            self.set_state(State.DESTROY)
            self.parent.value[self.ifc_key] =  self.value
        elif self.get_state() != State.DESTROY:
            if self.is_the_same_cli(cli):
                self.set_state(State.NOCHANGE)
            else:
                self.set_state(State.MODIFY)

    def get_cli_pattern(self):
        if not self.CLI_PATTERN:
            self.CLI_PATTERN = re.compile(self.asa_key + ' (\S+)')
        return self.CLI_PATTERN

    def parse_cli(self, cli):
        m = self.get_cli_pattern().match(cli)
        if m:
            result = {}
            result['rtrId'] = m.group(1)
            return result

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        if not hasattr(self, 'value') and self.ifc_key in delta_ifc_cfg_value:
            self.value = delta_ifc_cfg_value[self.ifc_key]

class SingleIntParam(RouterCommon):
    '''
    For a CLI that has a single Int parameter

    Each sub-class must define its own:
        CLI_PATTERN = {}
    This is so CLI_PATTERN will be unique to that class.
    '''

    def get_cli_pattern(self):
        ip_key = self.parent.ip_key
        if ip_key not in self.CLI_PATTERN:
            self.CLI_PATTERN[ip_key] = re.compile(self.asa_key + ' (\d+)')
        return self.CLI_PATTERN[ip_key]

    def parse_cli(self, cli):
        m = self.get_cli_pattern().match(cli)
        if m:
            result = {}
            result[self.get_key()] = int(m.group(1))
            return result

def get_bgp_asn(device_model):
    'Returns the autonomous system number of the active BGP process'

    bgps = device_model.get_child('BGP')
    if bgps:
        return bgps.get_asn()

def get_ospf_process(device_model):
    'Returns the active OSPF process'

    processes = device_model.get_child('OspfProcess')
    if processes:
        return processes.get_active_process()

def get_ospf_process_id(device_model):
    'Returns the process ID of the active OSPF process'

    process = get_ospf_process(device_model)
    if process:
        return process.get_process_id()
