'''
Created on Jan 20, 2015

@author: jeffryp

Copyright (c) 2015 by Cisco Systems, Inc.
All rights reserved.

Classes used for OSPF interface configuration.
'''

from itertools import ifilter
import re

from common import get_ospf_process_id
from common import RouterCommon, SingleIntParam
from asaio.cli_interaction import ignore_response_parser
from translator.base.compositetype import CompositeType
from translator.base.dmlist import DMList
from translator.base.dmobject import DMObject
from translator.base.simpletype import SimpleType
from translator.state_type import State, Type
from translator.structured_cli import StructuredCommand
from translator.validators import Validator

class Area(RouterCommon, SimpleType):
    '''
    Area configuration

    CLI generation is only supported for IPv6.  IPv4 uses the area ID.
    '''

    CLI_PATTERN = None

    def __init__(self):
        super(Area, self).__init__('area', 'ipv6 ospf')
        self.asa_gen_template = self.asa_key + ' %(processID)s area %(area)s'

    def get_area_id(self):
        if self.has_ifc_delta_cfg():
            return self.value[self.get_key()]

    def get_cli(self):
        if 'processID' not in self.value:
            self.value['processID'] = get_ospf_process_id(self.get_top())
        return super(Area, self).get_cli()

    def get_cli_pattern(self):
        if not self.CLI_PATTERN:
            self.CLI_PATTERN = re.compile(self.asa_key + ' (\d+) area (\d+)')
        return self.CLI_PATTERN

    def has_ifc_delta_cfg(self):
        if super(Area, self).has_ifc_delta_cfg():
            return get_ospf_process_id(self.get_top()) != None

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        if self.is_ipv6():
            super(Area, self).ifc2asa(no_asa_cfg_stack, asa_cfg_list)

    def is_ipv6(self):
        return self.parent.is_ipv6()

    def is_my_cli(self, cli):
        if self.is_ipv6():
            return self.get_cli_pattern().match(cli)

    def parse_cli(self, cli):
        m = self.get_cli_pattern().match(cli)
        if m:
            result = {}
            result['processID'] = int(m.group(1))
            result[self.get_key()] = int(m.group(2))
            return result

class AuthenticationKey(RouterCommon, SimpleType):
    '''
    Authentication key configuration

    This is only supported for IPv4.  Leading and trailing spaces are ignored.
    '''

    CLI_PATTERN = None

    def __init__(self):
        super(AuthenticationKey, self).__init__('authKey',
                                                'ospf authentication-key')
        self.asa_gen_template = self.asa_key + ' %(authKey)s'

    def get_cli_pattern(self):
        if not self.CLI_PATTERN:
            self.CLI_PATTERN = re.compile(self.asa_key + ' (.+)')
        return self.CLI_PATTERN

    def parse_cli(self, cli):
        m = self.get_cli_pattern().match(cli)
        if m:
            result = {}
            result[self.get_key()] = m.group(1)
            return result

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        if self.get_key() in delta_ifc_cfg_value:
            auth_key = delta_ifc_cfg_value[self.get_key()]
            auth_key[self.get_key()] = auth_key[self.get_key()].strip()
            if auth_key[self.get_key()]: # Ignore empty string
                self.value = auth_key

class AuthenticationType(RouterCommon, SimpleType):
    '''
    Authentication type configuration

    This is only supported for IPv4.
    '''

    CLI_PATTERN = None

    def __init__(self):
        super(AuthenticationType, self).__init__('authType',
                                                 'ospf authentication')

    def get_cli(self):
        auth_type = self.get_value()[self.get_key()]
        if auth_type == 'message-digest':
            return self.asa_key + ' ' + auth_type
        else:
            return self.asa_key

    def get_cli_pattern(self):
        if not self.CLI_PATTERN:
            self.CLI_PATTERN = re.compile(self.asa_key +
                                          '(?: (message-digest))?')
        return self.CLI_PATTERN

    def parse_cli(self, cli):
        m = self.get_cli_pattern().match(cli)
        if m:
            result = {}
            auth_type = m.group(1)
            result[self.get_key()] = 'message-digest' if auth_type else 'Plain'
            return result

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        if self.get_key() in delta_ifc_cfg_value:
            auth_type = delta_ifc_cfg_value[self.get_key()]
            if auth_type[self.get_key()] in ('message-digest', 'Plain'):
                self.value = auth_type

class Cost(SingleIntParam, SimpleType):
    'Cost configuration'

    CLI_PATTERN = {}

    def __init__(self, ipv6):
        super(Cost, self).__init__('cost',
                                   'ipv6 ospf cost' if ipv6 else 'ospf cost')
        self.asa_gen_template = self.asa_key + ' %(cost)s'

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        if self.get_key() in delta_ifc_cfg_value:
            cost = delta_ifc_cfg_value[self.get_key()]
            if cost[self.get_key()] != 0: # Ignore cost of 0
                self.value = cost

class DeadInterval(SingleIntParam, SimpleType):
    'Dead interval configuration'

    CLI_PATTERN = {}

    def __init__(self, ipv6):
        super(DeadInterval, self).__init__(
            'deadIntvl',
            'ipv6 ospf dead-interval' if ipv6 else 'ospf dead-interval')
        self.asa_gen_template = self.asa_key + ' %(deadIntvl)s'

class HelloInterval(SingleIntParam, SimpleType):
    'Hello interval configuration'

    CLI_PATTERN = {}

    def __init__(self, ipv6):
        super(HelloInterval, self).__init__(
            'helloIntvl',
            'ipv6 ospf hello-interval' if ipv6 else 'ospf hello-interval')
        self.asa_gen_template = self.asa_key + ' %(helloIntvl)s'

class MD5Key(RouterCommon, SimpleType):
    '''
    MD5 key configuration

    This is only supported for IPv4.  Leading and trailing spaces are ignored.
    '''

    def __init__(self, instance):
        super(MD5Key, self).__init__(instance, 'ospf message-digest-key')
        self.asa_gen_template = self.asa_key + ' %(md5KeyId)s md5 %(md5Password)s'

    def diff_ifc_asa(self, cli):
        if not self.has_ifc_delta_cfg():
            self.value = self.parse_cli(cli)
            self.set_state(State.DESTROY)
            self.parent.get_value().append(self.value)
        elif self.is_the_same_cli(cli):
            self.set_state(State.NOCHANGE)
        else:
            self.set_state(State.MODIFY)

    def get_no_cli(self):
        return ('no ' + self.asa_key + ' %(md5KeyId)s') % self.value

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        if self.has_ifc_delta_cfg():
            state = self.get_state()
            if state in (State.DESTROY, State.MODIFY):
                self.generate_cli(no_asa_cfg_stack,
                                  self.get_no_cli(),
                                  response_parser=ignore_response_parser)
            if state in (State.CREATE, State.MODIFY):
                self.generate_cli(asa_cfg_list, self.get_cli())

    def parse_cli(self, cli):
        m = self.parent.get_cli_pattern().match(cli)
        if m:
            result = {}
            result['md5KeyId'] = int(m.group(1))
            result['md5Password'] = m.group(2)
            return result

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        self.value = delta_ifc_cfg_value
        self.value['md5Password'] = self.value['md5Password'].strip()

class MD5Keys(RouterCommon, DMList):
    '''
    Container for MD5 keys

    This is only supported for IPv4.
    '''

    CLI_PATTERN = None

    def __init__(self):
        super(MD5Keys, self).__init__('md5Key',
                                      MD5Key,
                                      'ospf message-digest-key')

    def create_missing_ifc_delta_cfg(self):
        if not self.has_ifc_delta_cfg():
            self.value = []
            self.parent.get_value()[self.get_key()] = self.value

    def get_cli_pattern(self):
        if not self.CLI_PATTERN:
            self.CLI_PATTERN = re.compile(self.asa_key + ' (\d+) md5 (.+)')
        return self.CLI_PATTERN

    def get_translator(self, cli):
        if isinstance(cli, StructuredCommand):
            cli = cli.command
        m = self.get_cli_pattern().match(cli.strip())
        if m:
            key = int(m.group(1))
            md5Key = self.get_child(key)
            if not md5Key:
                md5Key = self.child_class(key)
                self.register_child(md5Key)
            return md5Key

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        if self.get_key() in delta_ifc_cfg_value:
            self.value = delta_ifc_cfg_value[self.get_key()]
            for md5Key in self.value:
                key = md5Key['md5KeyId']
                if key not in self.children:
                    child = self.child_class(key)
                    self.register_child(child)
                    child.populate_model(None, md5Key)

class MTUIgnore(RouterCommon, SimpleType):
    'MTU ignore configuration'

    CLI_PATTERN = {}

    def __init__(self, ipv6):
        super(MTUIgnore, self).__init__(
            'mtu-ignore',
            'ipv6 ospf mtu-ignore' if ipv6 else 'ospf mtu-ignore')
        self.asa_gen_template = self.asa_key

    def parse_cli(self, cli):
        return {}

class P2PUnsupportedValidator(Validator):
    def validate(self, value):
        if not self.has_ifc_delta_cfg():
            return
        if self.get_state() in (State.NOCHANGE, State.DESTROY):
            return
        network_type = self.get_value()[self.get_key()]
        if network_type == 'p2p':
            return ('Point-to-point non-broadcast mode is not supported on ASA. ' +
                    'You must remove the p2p configuration from the APIC if the configuration exists')

class NetworkType(RouterCommon, SimpleType, P2PUnsupportedValidator):
    'Network type configuration'

    CLI_PATTERN = {}

    def __init__(self, ipv6):
        super(NetworkType, self).__init__(
            'nwT',
            'ipv6 ospf network' if ipv6 else 'ospf network')

    def get_cli(self):
        network_type = self.get_value()[self.get_key()]
        if network_type == 'p2p':
            network_type = 'point-to-point non-broadcast'
        return self.asa_key + ' ' + network_type

    def get_cli_pattern(self):
        ip_key = self.parent.ip_key
        if ip_key not in self.CLI_PATTERN:
            self.CLI_PATTERN[ip_key] = re.compile(
                self.asa_key + ' (broadcast|point-to-point non-broadcast)')
        return self.CLI_PATTERN[ip_key]

    def parse_cli(self, cli):
        m = self.get_cli_pattern().match(cli)
        if m:
            result = {}
            network_type = m.group(1)
            if network_type == 'point-to-point non-broadcast':
                network_type = 'p2p'
            result[self.get_key()] = network_type
            return result

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        if self.get_key() in delta_ifc_cfg_value:
            network_type = delta_ifc_cfg_value[self.get_key()]
            if network_type[self.get_key()] in ('broadcast', 'p2p'):
                self.value = network_type

class OspfInterface(RouterCommon, CompositeType):
    'OSPF interface configuration'

    def __init__(self, ifc_key, ipv6=False):
        super(OspfInterface, self).__init__(ifc_key)
        self.ipv6 = ipv6
        self.ip_key = 'ipv6' if self.ipv6 else 'ipv4'

        self.area = Area()

        self.register_child(Cost(ipv6))
        self.register_child(RetransmitInterval(ipv6))
        self.register_child(TransmitDelay(ipv6))
        self.register_child(Priority(ipv6))
        self.register_child(HelloInterval(ipv6))
        self.register_child(DeadInterval(ipv6))
        self.register_child(NetworkType(ipv6))
        if not ipv6:
            # These are only supported for IPv4
            self.register_child(AuthenticationKey())
            self.register_child(MD5Keys())
            self.register_child(AuthenticationType())
        self.register_child(MTUIgnore(ipv6))
        self.register_child(self.area)

        self.cli_prefixes = self.get_cli_prefixes()

    def create_missing_ifc_delta_cfg(self):
        if not self.has_ifc_delta_cfg():
            self.value = {'state': State.NOCHANGE}
            self.parent.get_value()[self.ip_key] = self.value
        for child in self:
            child.create_missing_ifc_delta_cfg()

    def get_area_id(self):
        return self.area.get_area_id()

    def get_child_mode_command(self):
        return self.parent.get_child_mode_command()

    def get_ospfvif(self):
        return self.parent.get_ospfvif()

    def get_translator(self, cli):
        if self.is_my_cli(cli):
            for child in self.children.values():
                result = child.get_translator(cli)
                if result:
                    return result

    def ifc2asa(self, no_asa_cfg_stack,  asa_cfg_list):
        if self.has_ifc_delta_cfg():
            child_mode_command = self.get_child_mode_command()
            for child in self.children.itervalues():
                if isinstance(child, DMList):
                    for c in child.children.itervalues():
                        c.mode_command = child_mode_command
                else:
                    child.mode_command = child_mode_command
                child.ifc2asa(no_asa_cfg_stack, asa_cfg_list)

    def is_ipv6(self):
        return self.ipv6

    def is_my_cli(self, cli):
        if isinstance(cli, StructuredCommand):
            cli = cli.command
        return cli.split()[0] in self.cli_prefixes

    @property
    def nameif(self):
        return self.parent.nameif

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        if not hasattr(self, 'value'):
            if self.ip_key in delta_ifc_cfg_value:
                self.delta_ifc_key = delta_ifc_key
                self.value = delta_ifc_cfg_value[self.ip_key]
                for child in self:
                    child.populate_model(None, self.value)

class OspfCombinedInterface(RouterCommon, DMObject):
    'The combined OSPF and OSPFv3 interface configuration'

    def __init__(self):
        super(OspfCombinedInterface, self).__init__(
            OspfCombinedInterface.__name__)
        self.ospf = OspfInterface('OSPF')
        self.ospfv3 = OspfInterface('OSPFv3', ipv6=True)

        self.register_child(self.ospf)
        self.register_child(self.ospfv3)

    def create_missing_ifc_delta_cfg(self):
        if not self.has_ifc_delta_cfg():
            ospfvif = self.get_ospfvif()
            if ospfvif:
                self.value = ospfvif
            else:
                self.value = {'state': State.NOCHANGE}
        for child in self:
            child.create_missing_ifc_delta_cfg()

    def get_child_mode_command(self):
        return self.parent.get_child_mode_command()

    def get_config_path(self):
        encapass = self.parent.get_encapass()
        if encapass:
            result = encapass.get_config_path()
        else:
            result = []
        if self.has_ifc_delta_cfg():
            result.append(self.delta_ifc_key)
        return result

    def get_encapass(self):
        return self.parent.get_encapass()

    def get_ospf_area_id(self):
        return self.ospf.get_area_id()

    def get_ospfv3(self):
        return self.ospfv3

    def get_ospfvif(self):
        'Return the OspfVIfCfg field of the ENCAPASS for this interface'
        encapass = self.parent.get_encapass()
        if encapass and encapass.has_ifc_delta_cfg():
            return encapass.delta_ifc_cfg_value.get('OspfVIfCfg')

    def get_translator(self, cli):
        for child in self:
            result = child.get_translator(cli)
            if result:
                return result

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        if self.get_state() == State.MODIFY:
            self.set_state_recursive(State.CREATE)
            if not hasattr(self, 'mini_audit_command'):
                self.mini_audit_command = ('show running-config ' +
                                           self.mode_command + ' | grep ospf')
            self.mini_audit()
        for child in self:
            child.ifc2asa(no_asa_cfg_stack, asa_cfg_list)

    @property
    def nameif(self):
        return self.parent.nameif

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        # There may be more than one Encap Association using the same Encap.
        # Combine the states of all of them.
        def get_ospfvencapasc_state(encapass):
            ospfvif = encapass.delta_ifc_cfg_value.get('OspfVIfCfg')
            if isinstance(ospfvif, dict):
                for key, value in ospfvif.iteritems():
                    if isinstance(key, tuple) and key[0] == Type.OSPFVENCAPASC:
                        return value.get('state')

        def get_combined_state():
            encapass = self.parent.get_encapass()
            state = get_ospfvencapasc_state(encapass)
            for other in ifilter(lambda other: other != encapass and
                    other.encap == encapass.encap,
                    self.get_top().get_child('ENCAPASS')):
                if get_ospfvencapasc_state(other) != state:
                    state = State.MODIFY
            return state

        if not hasattr(self, 'value'):
            translate_ifc_cfg(delta_ifc_cfg_value, get_combined_state())
            self.delta_ifc_key = delta_ifc_key
            self.value = delta_ifc_cfg_value
            for child in self:
                child.populate_model(delta_ifc_key, self.value)

    def read_asa_config(self):
        'Override the default implementation to strip leading blanks'
        result = super(OspfCombinedInterface, self).read_asa_config()
        if result:
            return '\n'.join(map(str.strip, result.split('\n')))

    def update_references(self):
        ospfvif = self.get_ospfvif()
        if ospfvif:
            ospfvif_key = None
            for key in ospfvif.iterkeys():
                if isinstance(key, tuple) and key[0] == Type.OSPFVENCAPASC:
                    ospfvif_key = key
                    break
            self.populate_model(ospfvif_key, ospfvif)
        super(OspfCombinedInterface, self).update_references()

class Priority(SingleIntParam, SimpleType):
    'Priority configuration'

    CLI_PATTERN = {}

    def __init__(self, ipv6):
        super(Priority, self).__init__(
            'prio',
            'ipv6 ospf priority' if ipv6 else 'ospf priority')
        self.asa_gen_template = self.asa_key + ' %(prio)s'

class RetransmitInterval(SingleIntParam, SimpleType):
    'Retransmit interval configuration'

    CLI_PATTERN = {}

    def __init__(self, ipv6):
        super(RetransmitInterval, self).__init__(
            'rexmitIntvl',
            'ipv6 ospf retransmit-interval' if ipv6 else 'ospf retransmit-interval')
        self.asa_gen_template = self.asa_key + ' %(rexmitIntvl)s'

class TransmitDelay(SingleIntParam, SimpleType):
    'Transmit delay configuration'

    CLI_PATTERN = {}

    def __init__(self, ipv6):
        super(TransmitDelay, self).__init__(
            'xmitDelay',
            'ipv6 ospf transmit-delay' if ipv6 else 'ospf transmit-delay')
        self.asa_gen_template = self.asa_key + ' %(xmitDelay)s'

def ospf_interface_audit_init(configuration):
    '''
    Transformer to initialize the configuration for audit. It sets all the
    translated dictionaries to a state of CREATE.
    '''

    def set_state(cfg):
        if isinstance(cfg, dict):
            if 'state' in  cfg:
                cfg['state'] = State.CREATE
            for value in cfg.itervalues():
                set_state(value)
        elif isinstance(cfg, list):
            for value in cfg:
                set_state(value)

    encapass_keys = filter(lambda (kind, key, instance):
                           kind == Type.ENCAPASS,
                           configuration.values()[0]['value'].keys())
    for encapass_key in encapass_keys:
        encapass = configuration.values()[0]['value'][encapass_key]
        if 'OspfVIfCfg' in encapass:
            set_state(encapass['OspfVIfCfg'])
    return configuration

def translate_ifc_cfg(ifc_cfg, combined_state):
    '''
    Translate from the IFC model to the ASA model and create any missing
    configuration. The ASA model has a one-to-one correspondence to the ASA
    command hierarchy.
    '''

    def make_dict(to_cfg, from_cfg, key):
        if key in from_cfg:
            to_cfg.setdefault(key, {key: from_cfg[key]})

    def set_default_state(cfg, default_state):
        if isinstance(cfg, dict):
            cfg.setdefault('state', default_state)
            for value in cfg.itervalues():
                set_default_state(value, default_state)
        elif isinstance(cfg, list):
            for value in cfg:
                set_default_state(value, default_state)

    for key, ospfv in ifc_cfg.items():
        if isinstance(key, tuple) and key[0] == Type.OSPFVENCAPASC:
            ifc_cfg.setdefault('state', combined_state)
            for ip_key in ('ipv4', 'ipv6'):
                if ip_key in ospfv.get('addressFamily', []):
                    is_ipv6 = (ip_key == 'ipv6')
                    ip_value = ifc_cfg.setdefault(ip_key, {})
                    ip_value.setdefault('state', combined_state)
    
                    # Area
                    make_dict(ip_value, ospfv, 'area')
    
                    # Authentication key
                    if not is_ipv6:
                        make_dict(ip_value, ospfv, 'authKey')
    
                    # Authentication type
                    if not is_ipv6:
                        make_dict(ip_value, ospfv, 'authType')
    
                    # Cost
                    make_dict(ip_value, ospfv, 'cost')
    
                    # Ctrl
                    ctrl = ospfv.get('ctrl', [])
                    if 'mtu-ignore' in ctrl:
                        ip_value.setdefault('mtu-ignore', {})
                    if is_ipv6 and 'passive' in ctrl:
                        ip_value.setdefault('passive', {})
    
                    # Dead interval
                    make_dict(ip_value, ospfv, 'deadIntvl')
    
                    # Hello interval
                    make_dict(ip_value, ospfv, 'helloIntvl')
    
                    # MD5 keys
                    if not is_ipv6 and 'md5Key' in ospfv:
                        md5Keys = ip_value.setdefault('md5Key', [])
                        for md5Key in ospfv['md5Key']:
                            if md5Key not in md5Keys:
                                md5Keys.append(md5Key)
    
                    # Network type
                    nwT = ospfv.get('nwT')
                    if nwT == 'p2p' or (is_ipv6 and nwT == 'broadcast'):
                        make_dict(ip_value, ospfv, 'nwT')
    
                    # Priority
                    make_dict(ip_value, ospfv, 'prio')
    
                    # Retransmit interval
                    make_dict(ip_value, ospfv, 'rexmitIntvl')
    
                    # Transmit delay
                    make_dict(ip_value, ospfv, 'xmitDelay')

    for ip_key in ('ipv4', 'ipv6'):
        ip_value = ifc_cfg.get(ip_key)
        if ip_value:
            set_default_state(ip_value, ip_value['state'])
