'''
Created on Mar 10, 2016

@author: dli

Copyright (c) 2016 by Cisco Systems
'''
import re

from translator.base.dmlist import DMList
from translator.base.compositetype import CompositeType
from translator.state_type import State, Type
from translator.base.simpletype import SimpleType
from utils.util import normalize_param_dict, ifcize_param_dict
from translator.structured_cli import StructuredCommand

class TimeRanges(DMList):
    def __init__(self):
        super(TimeRanges, self).__init__(TimeRange.__name__, TimeRange, 'time-range')

class TimeRange(CompositeType):
    '''
    It is after 'time-range' command. The APIC object is 'TimeRange'
    '''
    def __init__(self, name):
        'name could be the name string of the time-range, or the CLI corresponding to a time-range command on ASA'
        if isinstance(name, StructuredCommand):
            name = name.command
        name = name.split()[-1]
        super(TimeRange, self).__init__(name,
                                        asa_gen_template='time-range %(name)s');
        self.register_child(AbsoluteTimeDate())
        self.register_child(Periodics(DailyPeriod.__name__, DailyPeriod, re.compile('^periodic .+ to \S+$'), name))
        self.register_child(Periodics(WeeklyPeriod.__name__, WeeklyPeriod, re.compile('^periodic .+ to \S+ \S+$'), name))

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        'Override the default implementation to set the value of "name" with instance ID if it is not there yet.'
        name_key = filter(lambda k: k[1] == 'name', delta_ifc_cfg_value['value'].keys())
        #name_key will be there if the config is on ASA but not in IFC
        if not name_key:
            delta_ifc_cfg_value['value'][(Type.PARAM, 'name', 'name')] = {'state': 1, 'value': delta_ifc_key[2]}
        return CompositeType.populate_model(self, delta_ifc_key, delta_ifc_cfg_value)

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        """
        Override the default implementation so that we can generate 'time-range <name>' even when
        it has empty value.
        """
        if not self.has_ifc_delta_cfg():
            return
        sub_cmd_defined = (self.children.values()[0].get_value() != None or
                           len(self.children.values()[1].children) != 0 or
                           len(self.children.values()[2].children) != 0)
        if not sub_cmd_defined:
            return SimpleType.ifc2asa(self, no_asa_cfg_stack, asa_cfg_list)
        return CompositeType.ifc2asa(self, no_asa_cfg_stack, asa_cfg_list)

    def is_the_same_cli(self, cli):
        'Override default implementation to allow empty '
        if isinstance(cli, basestring):
            return cli == self.get_cli() and self.get_sub_commands() == []
        if cli.command != self.get_cli():
            return False
        'check the sub-commands, order is not important'
        sub_commands = [s.strip() for s in str(cli).split('\n')[1:]]
        return set(sub_commands) ==  set(self.get_sub_commands())

    def create_asa_key(self):
        return self.get_cli()

class AbsoluteTimeDate(SimpleType):
    'absolute sub-command'
    def __init__(self):
        super(AbsoluteTimeDate, self).__init__(AbsoluteTimeDate.__name__, 'absolute')

    def get_cli(self):
        def normalize_month(date):
            'convert number to month string'
            num2month = {1: 'January',
                         2: 'February',
                         3: 'March',
                         4: 'April',
                         5: 'May',
                         6: 'June',
                         7: 'July',
                         8: 'August',
                         9: 'September',
                         10: 'October',
                         11: 'November',
                         12: 'December'
                         }
            month =  date['month']
            if month.isdigit():
                date['month'] = num2month.get(int(month), month)
            return date
        def normalize_day(date):
            'make sure 1 comes out as 01, as it is the form stored in ASA'
            date['day'] = '{:#02d}'.format(int(date['day']))
            return date
        def normalize_time(date):
            'make sure 0:0 comes out as 00:00, as it is the form stored in ASA'
            hh,mm = date['time'].split(':')
            date['time'] = '{:#02d}:{:#02d}'.format(int(hh), int(mm))
            return date
        if self.get_state() == State.DESTROY:
            value = self.delta_ifc_cfg_value['value']
            value =  normalize_param_dict(value)
        else:
            value = self.get_value()
        result = self.asa_key
        start = value.get('Start', None)
        end = value.get('End', None)
        if start:
            start_cli = ' start %(time)s %(day)s %(month)s %(year)s' % normalize_time(normalize_day(normalize_month(start)))
        else:
            start_cli = ''
        if end:
            end_cli = ' end %(time)s %(day)s %(month)s %(year)s' % normalize_time(normalize_day(normalize_month(end)))
        else:
            end_cli = ''
        result += start_cli + end_cli
        return result

    def parse_cli(self, cli):
        pattern = 'absolute (?:start (\S+) (\S+) (\S+) (\S+) ?)?(?:end (\S+) (\S+) (\S+) (\S+))?'
        m = re.match(pattern, cli)
        if m:
            result = {}
            if m.group(1):
                result['Start'] = {'time': m.group(1), 'day': m.group(2), 'month': m.group(3), 'year': m.group(4)}
            if m.group(5):
                result['End'] = {'time': m.group(5), 'day': m.group(6), 'month': m.group(7), 'year': m.group(8)}
        else:
            return
        return ifcize_param_dict(result)


class Periodic(SimpleType):
    'periodic sub-command'
    def create_asa_key(self):
        return self.get_cli()

    def get_value(self):
        'Override the default to make sure 0:0 comes out as 0:00, as it is the form stored in ASA'
        def normalize_time(date):
            for  time in ('start_time', 'end_time'):
                hh,mm = date[time].split(':')
                date[time] = '{:d}:{:#02d}'.format(int(hh), int(mm))
            return date
        return normalize_time(SimpleType.get_value(self))

class DailyPeriod(Periodic):
    'periodic sub-command for daily recurrences'
    def __init__(self, instance):
        super(DailyPeriod, self)\
        .__init__(self, instance,
                  asa_gen_template = 'periodic %(day)s %(start_time)s to %(end_time)s')

    def parse_multi_parameter_cli(self, cli, alternate_asa_gen_template=None):
        """
        override the default implementation to take care of the case where 'day' can have multiple words, e.g. 'Monday Sunday'
        """
        pattern = 'periodic (.+) (\S+) to (\S+)'
        m = re.match(pattern, cli)
        if m:
            result = {'day': m.group(1), 'start_time': m.group(2), 'end_time': m.group(3)}
        else:
            return
        return ifcize_param_dict(result)

class WeeklyPeriod(Periodic):
    'periodic sub-command for weekly recurrences'
    def __init__(self, instance):
        super(WeeklyPeriod, self)\
        .__init__(self, instance,
                  asa_gen_template = 'periodic %(start_day)s %(start_time)s to %(end_day)s %(end_time)s')

class Periodics(DMList):
    'A special DMList that does mini audit of periodic sub-commands'
    def __init__(self, name, child_class, asa_key, time_range_name):
        super(Periodics, self).__init__(name, child_class, asa_key, 'show run time-range ' + time_range_name + ' | grep periodic')

    def read_asa_config(self):
        'override the default implementation to remove leading spaces in the CLIs and cache it'
        if not hasattr(self.parent, 'cached_asa_config'):
            clis = DMList.read_asa_config(self)
            self.parent.cached_asa_config =  '\n'.join(map(str.strip, clis.split('\n'))) if clis else None
        return self.parent.cached_asa_config

