'''
Created on Jul 26, 2013

@author: dli

Copyright (c) 2013 by Cisco Systems
'''
from dmobject import DMObject
from utils.util import filter_first
from translator.state_type import State

class DMList(DMObject):
    '''A list of the same type of DMObject's, such ACE's. e.g. an ACL is a DMList of ACE's.
    It is equivalent to DM*Collection in ASDM, with order preserved.
    '''
    def __init__(self, name, child_class, asa_key='', mini_audit_command = None):
        '''
        @param self: DMList instance
            this class instance
        @param name: str
            the key for this configuration from the device specification file.
        @param child_class: class
            the type of elements in the list
        @param asa_key: str
            the string to identify which CLI is for me
        @param mini_audit_command: str
            flag to indicate whether a mini audit is to be performed if any child is in State.MODIFY.
            If None, it won't perform mini audit operation.
            Or it should be a 'show run', string, such as 'show run interface' for gathering ASA config.
        '''
        DMObject.__init__(self, name, asa_key)
        self.child_class = child_class
        self.mini_audit_command = mini_audit_command

    def register_child(self, dmobj):
        '''Override the default implementation to prevent add child of the wrong type
        '''
        if isinstance(dmobj, self.child_class):
            DMObject.register_child(self, dmobj)
            if hasattr(self, 'is_system_context'):
                dmobj.is_system_context = self.is_system_context
        else:
            raise Exception("Child has wrong type")

    def add(self, child):
        '''Add an element from the list
        @param self: DMList instance
            this class instance
        @param child: child_class
        '''
        self.register_child(child)

    def remove(self, child):
        '''Remove an element from the list
        @param self: DMList instance
            this class instance
        @param child: child_class
            element to add
        '''
        if isinstance(child, self.child_class):
            self.remove_by_key(child.get_key())

    def remove_by_key(self, child_key):
        '''Remove an element from the list
        @param self: DMList instance
            this class instance
        @param child_key: key
           the key of the child in the list
        '''
        if child_key in self.children:
            del(self.children[child_key])


    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        '''Override the default implementation to dynamically generate
        child instance from the "instance" value in the delta_ifc_key
        @attention:
            Assuming the child class initializer takes one argument.
            Override this method if this assumption is not right for your particular class.
        '''
        kind, key, instance = delta_ifc_key
        child = self.get_child(instance)
        if not child:
            child = self.child_class(instance) #create a instance of child_class
            self.register_child(child)
        child.populate_model(delta_ifc_key, delta_ifc_cfg_value)

    def create_missing_ifc_delta_cfg(self):
        '''Override the default implementation, because in most use cases,
        DMList is 'transparent' with respect to ifc_delta_cfg
        '''
        if not self.children or self.has_ifc_delta_cfg():
            return

        for child in self.children.values():
            child.create_missing_ifc_delta_cfg()

    def get_translator(self, cli):
        '''Find the DMObject within this DMOBject that can translate a CLI
        @param cli: CLI
        @return:  the DMObject that can translate the cli
        @note:
            If none of its children matches the CLI, it means we need to delete
            the CLI in order to make IFC and ASA in sync.
        '''
        if self.__dict__.get('is_system_context') and self.get_top().is_user_context():
            'no audit operation if the target ASA device is user context'
            return None
        if not self.asa_key:
            '''
             @note: could be a bad idea if we have DMList not directly corresponding to CLIs
            '''
            # not programmed to handling CLI to IFC translation
            return None

        if not self.is_my_cli(cli):
            '''
             short-circuit to avoid further search, could be bad idea if we have
             @note: could be a bad idea if we have DMList not directly corresponding to CLIs
                    one solution is to use a special asa_key for those objects.
            '''
            return

        'Find the translator from its children'
        result = None
        for child in self.children.values():
            if hasattr(self, 'get_translator'):
                result = child.get_translator(cli)
                if result:
                    break
        if result:
            return result
        'Create a child that is to generate "no" cli, by the invocation of child.diff_ifc_asa'
        result = self.child_class(cli)
        self.register_child(result)
        return result

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        """
        Override the default to take care of modification situation, where we need to
        delete the old value
        """
        if self.mini_audit_command and filter_first(lambda child: child.get_state() == State.MODIFY, self):
            # Remove any entries to be deleted
            for key, child in self.children.items():
                if child.get_state() == State.DESTROY:
                    del(self.children[key])
            self.mini_audit()
        DMObject.ifc2asa(self, no_asa_cfg_stack, asa_cfg_list)
