'''
Created on Nov 12, 2013

@author: emilymw
Copyright (c) 2013, 2015 by Cisco Systems
'''

from itertools import imap
import re

from base.simpletype import SimpleType
from base.dmlist import DMList
from base.compositetype import CompositeType
from translator.state_type import State, Type
from translator.base.dmboolean import DMBoolean
from asaio.cli_interaction import ignore_info_response_parser
from translator.structured_cli import StructuredCommand
from utils.util import normalize_ipv6_address, normalize_ipv4_address_and_mask_4_asa
from validators import IPv6AddressValidator, IPv6AddressSubnetValidator
import asaio.cli_interaction as cli_interaction
from translator.validators import InterfaceIPv4AddressValidator

class BridgeGroupIntfs(DMList):
    'Container of BridgeGroupIntf'

    GROUP_NUMBER_PATTERN = None

    def __init__(self):
        super(BridgeGroupIntfs, self).__init__('BridgeGroupIntf',
                                               BridgeGroupIntf,
                                               'interface BVI')

    @classmethod
    def get_group_number_pattern(cls):
        if not cls.GROUP_NUMBER_PATTERN:
            cls.GROUP_NUMBER_PATTERN = re.compile('interface BVI(\d+)')
        return cls.GROUP_NUMBER_PATTERN

    def get_translator(self, cli):
        if isinstance(cli, StructuredCommand):
            cli = cli.command
        m = self.get_group_number_pattern().match(cli.strip())
        if m:
            group_number = m.group(1)
            child = self.get_child(group_number)
            if not child:
                child = self.child_class(group_number)
                self.register_child(child)
            return child

class BridgeGroupIntf(CompositeType):
    'Model for "interface BVI" CLI'
    def __init__(self, group_number):
        super(BridgeGroupIntf, self).__init__(
            group_number,
            asa_gen_template='interface BVI%s');
        self.register_child(IPv4Addr())
        self.register_child(IPv6Addresses())
        self.register_child(IPv6Enable())
        self.register_child(SimpleType('ipv6_nd_reachable_time',
                                       'ipv6 nd reachable-time'))
        self.register_child(SimpleType('ipv6_nd_ns_interval',
                                       'ipv6 nd ns-interval'))
        self.register_child(SimpleType('ipv6_nd_dad_attempts',
                                       'ipv6 nd dad attempts'))
        self.response_parser = ignore_info_response_parser

    def diff_ifc_asa(self, cli):
        if not self.has_ifc_delta_cfg():
            self.delta_ifc_key = (Type.FOLDER, self.parent.ifc_key, self.ifc_key)
            self.delta_ifc_cfg_value = {'state': State.DESTROY, 'value': {}}
            ancestor = self.get_ifc_delta_cfg_ancestor()
            if ancestor:
                ancestor.delta_ifc_cfg_value['value'][self.delta_ifc_key] = self.delta_ifc_cfg_value
        else:
            super(BridgeGroupIntf, self).diff_ifc_asa(cli)

    def get_cli(self):
        return self.asa_gen_template % self.ifc_key

    def validate_configuration(self):
        if (not self.ifc_key.isdigit() or int(self.ifc_key) < 1 or
                int(self.ifc_key) > 100):
            return self.generate_fault("The name cannot be '" + self.ifc_key +
                                       "', it must be an integer between 1 and 100.")

        if self.get_state() in (State.CREATE, State.MODIFY):
            if not any(imap(lambda x: self.get_child(x).has_address(),
                            ('IPv4Address', 'IPv6Address', 'ipv6_enable'))):
                return self.generate_fault(
                    'IPv4 address or IPv6 address must be defined for a bridge group')
        result = []
        for child in self.children.itervalues():
            child_result = child.validate_configuration()
            self.report_fault(child_result, result)
        return result

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        '''Override the default implementation to generate nameif sub-command to allow it to participate in routing
        if the target ASA is router mode. We are running in IRB, Integrated Routing and Bridging, mode.
        '''
        if self.get_state() == State.CREATE and self.get_top().is_router_mode():
            nameif = SimpleType('nameif', 'nameif', response_parser=ignore_info_response_parser)
            nameif.populate_model((Type.PARAM, 'nameif', ''), {'state': State.CREATE, 'value': 'BVI' + self.ifc_key})
            self.register_child(nameif)
        return super(BridgeGroupIntf, self).ifc2asa(no_asa_cfg_stack, asa_cfg_list)

class AddressCommon(object):
    'Has common functionality for addresses'

    def has_address(self):
        return self.has_ifc_delta_cfg() and self.get_state() != State.DESTROY

class IPv4Addr(SimpleType, AddressCommon, InterfaceIPv4AddressValidator):
    'Model after "ip address ipv4_address netmask" sub-command'
    def __init__(self):
        super(IPv4Addr, self).__init__('IPv4Address', 'ip address',
                                       asa_gen_template='ip address %(ipv4_address)s',
                                       response_parser = cli_interaction.ignore_warning_response_parser)

    def get_cli(self):
        'Override the default implementation to take care / delimiter'
        value = self.get_value()
        ipv4address = value.get('ipv4_address')
        if '.' in ipv4address: #ipv4 address
            ipv4address = normalize_ipv4_address_and_mask_4_asa(ipv4address)
            value.update({'ipv4_address': ipv4address})
        if 'ipv4_standby_address' in value:
            return (self.asa_gen_template + ' standby %(ipv4_standby_address)s') % value
        else:
            return self.asa_gen_template % value

    def parse_multi_parameter_cli(self, cli):
        '''
        Override the default implementation in case the CLI does not match asa_gen_template due to optional parameter
        '''

        result = {}

        # Take care of the optional parameters
        tokens = cli.split()

        # The number of tokens must > 2, i.e. 'ip address IP [NETMASK] [standby IP]'
        option = tokens[2:]

        # To be more specific, the valid CLI could be of one the following:
        #     ip address 192.168.2.2
        #     ip address 192.168.2.2 255.255.255.0
        #     ip address 192.168.2.2 255.255.255.0 standby 192.168.2.3
        if len(option) > 2:
            result[(Type.PARAM, 'ipv4_standby_address', '')] = {'state': State.NOCHANGE, 'value': option[2]}
        if len(option) > 1:
            # IPv4 address is in the form of IP/MASK in the device model'
            result[(Type.PARAM, 'ipv4_address', '')] = {'state': State.NOCHANGE, 'value': '/'.join([option[0], option[1]])}
        return result

    def validate(self, value):
        return None if value == None else InterfaceIPv4AddressValidator.validate(self, value.get('ipv4_address'))

class IPv6Addr(SimpleType, AddressCommon):
    'Model after "ipv6 address ipv6_address/prefix " subcommand'
    def __init__(self, instance):
        super(IPv6Addr, self).__init__(instance, 'ipv6 address')

    def create_asa_key(self):
        return self.get_cli()

    def get_cli(self):
        value = self.get_value()
        if not isinstance(value, dict):
            return 'ipv6 address ' + normalize_ipv6_address(value,
                                                            interface_addr=True)
        cli = []
        cli.append('ipv6 address')
        cli.append(normalize_ipv6_address(value.get('ipv6_address_with_prefix'),
                                          interface_addr=True))
        if value.get('ipv6_standby_address'):
            cli.append('standby')
            cli.append(normalize_ipv6_address(value.get('ipv6_standby_address')))
        return ' '.join(cli)

    def parse_multi_parameter_cli(self, cli):
        '''
        Override the default implementation in case the CLI does not match asa_gen_template due to optional parameter
        '''

        result = {}

        # Take care of the optional parameters
        tokens = cli.split()

        # The number of tokens must > 2, i.e. 'ipv6 address IPV6_ADDRESS_WITH_PREFIX [standby IPV6_STANDBY_ADDRESS]'
        option = tokens[2:]

        result[(Type.PARAM, 'ipv6_address_with_prefix', '')] = {'state': State.NOCHANGE, 'value': option[0]}
        if len(option) > 2: # has standby address
            result[(Type.PARAM, 'ipv6_standby_address', '')] = {'state': State.NOCHANGE, 'value': option[2]}
        return result

    def validate_configuration(self):
        if not self.has_ifc_delta_cfg():
            return
        value = self.get_value()
        if not isinstance(value, dict):
            return
        result = []
        address = value.get('ipv6_address_with_prefix')
        error = IPv6AddressSubnetValidator().validate(address)
        if error:
            self.report_fault(error, result, 'ipv6_address_with_prefix')
        standby_address = value.get('ipv6_standby_address')
        if standby_address:
            error = IPv6AddressValidator().validate(standby_address)
            if error:
                self.report_fault(error, result, 'ipv6_standby_address')
        return result

class IPv6Addresses(DMList):
    'Container for IPv6 addresses'

    def __init__(self):
        super(IPv6Addresses, self).__init__('IPv6Address', IPv6Addr,
                                            'ipv6 address')

    def has_address(self):
        return any(imap(lambda x: x.has_address(), self))

class IPv6Enable(DMBoolean, AddressCommon):
    'IPv6 enable configuration'

    def __init__(self):
        super(IPv6Enable, self).__init__('ipv6_enable', 'ipv6 enable', 'enable')
