'''
Copyright (c) 2013 by Cisco Systems, Inc.

@author: emilymw
'''

from asaio.cli_interaction import ignore_info_response_parser
from asaio.cli_interaction import ignore_response_parser
from base.dmlist import DMList
from base.dmobject import DMObject
from base.simpletype import SimpleType
from state_type import State, Type
import translator
import utils.util as util
from utils.util import normalize_network_address, netmask_from_prefix_length,\
    filter_first

class Connector(DMObject):
    '''
    A firewall service can contains two connectors of type external and internal.  A
    Connector in terms of ASA CLI is expressed by a VLAN interface.

    There are several ConnObj (VIF, VLAN, ENCAPASS) objects defined in the global
    configuration under Device Config.  These objects binds the vlan tag with the interface.
    '''

    def __init__(self, instance):
        super(Connector, self).__init__(instance)
        self._interface = None

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        '''
        Populate the Connector configuration
        '''
        self.delta_ifc_key = delta_ifc_key
        self.delta_ifc_cfg_value = delta_ifc_cfg_value
        self.state = delta_ifc_cfg_value['state']
        self.config = self.get_value()
        self.conn_type = self.delta_ifc_key[1]

    @property
    def interface(self):
        if not self._interface and self.has_ifc_delta_cfg():
            intf_config_rel = self.get_intf_config_rel()
            if hasattr(intf_config_rel, 'value'):
                self._interface = (self.get_top().get_child('Interface').
                                   get_child(intf_config_rel.value))
        return self._interface
        
    def get_nameif(self):
        return self.interface.nameif if self.interface else ''

    def get_intf_config_rel(self):
        if self.conn_type == 'external':
            return self.get_ancestor_by_class(translator.firewall.Firewall).get_child('ExIntfConfigRelFolder').get_child('ExIntfConfigRel')
        else:
            return self.get_ancestor_by_class(translator.firewall.Firewall).get_child('InIntfConfigRelFolder').get_child('InIntfConfigRel')

    def update_references(self):
        'Update the corresponding interface with VIF and VLAN information'
        top = self.get_top()
        if self.interface:
            encapass = top.get_child('ENCAPASS').get_child(self.config.get('ENCAPREL'))
            if encapass:
                self.interface.encapass = encapass
                self.interface.vif = top.get_child('VIF').get_child(encapass.vif).value

                encap = top.get_child('VLAN').get_child(encapass.encap)
                self.interface.vlan = encap.tag
                # If the encap is being destroyed, then the interface needs to
                # be destroyed as well
                if encap.get_state() == State.DESTROY:
                    self.interface.set_state(State.DESTROY)

        """
        The reason of calling populate_epg_data here instead of populate_model is because
        populate_epg_data can change the configuration dictionary that populate_model iterates through.
        One cannot iterate through a dictionary whose entries are changing.
        """
        if (self.get_state() != State.NOCHANGE or top.is_audit) and top.get_child('NetworkObjectGroup'):
            self.populate_epg_data()

    def populate_epg_data(self):
        'Populate EPG membership updates to the corresponding NetworkObjectGroup'
        """
        For audit operation, this method will be called twice for the same connector.
        Use 'epg_processed' in the configuration dictionary as a flag, to avoid
        doing it twice to the configuration data.
        """
        if self.delta_ifc_cfg_value.get('epg_processed'):
            return
        for epg, members in self.get_epgs().iteritems():
            self.populate_network_object_group(epg, members)
        self.delta_ifc_cfg_value['epg_processed'] = True

    def get_epgs(self):
        """
        Get all the EPGs concerned by the connector.
        @return a dictionary, keyed by EPG name. The value of each entry is list of members.
        Each member is a tuple of (state, type, value)
        """
        epgs = {}
        for key, value in self.delta_ifc_cfg_value['value'].iteritems():
            if key[0] not in (Type.ENDPOINT, Type.SUBNET):
                continue
            state = value['state']
            network_value = key[2]
            network_type = key[0]
            'epg value will be tenant-profile-epg'
            epg = '-'.join(map(value.get, filter(value.has_key, ('tenant', 'profile', 'epg'))))
            if not epgs.has_key(epg):
                epgs[epg] = []
            epgs[epg].append((state, network_type, network_value))
        return epgs

    def populate_network_object_group(self, epg, members):
        'convert EPG membership update information to the corresponding NetworkObjectGroup'
        def get_entry_key_value(state, network_type, network_value):
            if network_type == Type.ENDPOINT:
                key = 'host_ip_address'
                value = network_value
            else:
                (network_address, prefix_length) = (network_value.split('/'))
                key = 'network_ip_address'
                if '.' in network_address: #IPv4 Address
                    mask = netmask_from_prefix_length(prefix_length)
                    value = normalize_network_address(network_address, mask) + '/' + mask
                else: #IPv6 address
                    value = normalize_network_address(network_address, prefix_length) + '/' + prefix_length
            return ((Type.PARAM, key, value), {'state': state, 'value': value})

        name = self.epg_name2nog_name(epg)
        top = self.get_top()
        config = top.delta_ifc_cfg_value
        nog_key = (Type.FOLDER, 'NetworkObjectGroup', name)
        if not config['value'].has_key(nog_key):
            nog_config = {'state': State.MODIFY, 'value': {}}
            top.delta_ifc_cfg_value['value'][nog_key] = nog_config
        else:
            nog_config = config['value'][nog_key]
        for state, network_type, network_value in members:
            entry_key, entry_value = get_entry_key_value(state, network_type, network_value)
            old_value = nog_config['value'].get(entry_key)
            if not old_value or state != State.NOCHANGE:
                'Make sure we do not override the state with 0 to workaround CSCuv87932.'
                nog_config['value'][entry_key] = entry_value
        nework_object_groups = top.get_child('NetworkObjectGroup')
        nework_object_groups.populate_model(nog_key, nog_config)

    @staticmethod
    def epg_name2nog_name(epg):
        'Make an network object group name from an EPG name'
        return '__$EPG$_%s' % epg

    @staticmethod
    def nog_name2epg_name(name):
        '@return the EPG name from the name of its network object group'
        return name[len('__$EPG$_'):]

    @staticmethod
    def is_epg_nog(nog_name):
        '@return True if the name of an network object group is that of an EPG'
        return nog_name.startswith('__$EPG$_')

class ConnObj(DMObject):
    '''
    This is a base class for all connector objects that are defined under
    Device Config (VIF, VLAN, ENCAPASS, InterfaceConfig).  It overwrite the populate_model()
    and ifc2asa() method. The Connector will handle the generation of CLI
    instead of handling it in these objects.

    Note:  ConnObj are not specified in device specification, they are
    generated by the IFC, except InterfaceConfig which is a relationship binding
    interface to ip address/mask.
    '''
    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        self.delta_ifc_key = delta_ifc_key
        self.delta_ifc_cfg_value = delta_ifc_cfg_value
        self.state = delta_ifc_cfg_value['state']
        self.response_parser = ignore_info_response_parser

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        'Handled by Connector'
        pass

class Vifs(DMList):
    def __init__(self):
        super(Vifs, self).__init__('VIF', Vif, 'interface')

    def is_my_cli(self, cli):
        # Physical interface with no active sub-commands other than 'shutdown' and 'lacp max-bundle'
        if isinstance(cli, basestring):
            return cli.startswith('interface ')
        if not cli.command.startswith('interface '):
            return False
        for sub_command in cli.sub_commands:
            if isinstance(sub_command, basestring):
                if sub_command != 'shutdown' and not sub_command.startswith('lacp max-bundle'):
                    return False
            elif not sub_command.is_no:
                return False
        return True

    def ifc2asa(self, no_asa_cfg_stack,  asa_cfg_list):
        # Only process each physical interface once by keeping track of which have
        # been processed already.
        processed = set()
        for vif in self.iter_vifs():
            if vif.value not in processed:
                vif.ifc2asa(no_asa_cfg_stack, asa_cfg_list)
                processed.add(vif.value)

    def get_translator(self, cli):
        if self.is_my_cli(cli):
            return self

    def diff_ifc_asa(self, cli):
        # More than one Vif may use the same physical interface, so need to
        # process all of them
        for vif in self.iter_vifs():
            vif.diff_ifc_asa(cli)

    def iter_vifs(self):
        for vif in self.children.itervalues():
            yield vif

class Vif(DMObject):
    '''
    (1) In ASA platform, Virtual interface or physical interface that vlan interface will spawn from
    (2) In vASA platform, only generate physical interface
    '''

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        super(Vif, self).populate_model(delta_ifc_key, delta_ifc_cfg_value)
        cif = delta_ifc_cfg_value['cifs'].iteritems().next()[1]
        self.value = util.normalize_interface_name(cif)

    def is_my_cli(self, cli):
        if hasattr(self, 'value'):
            # Physical interface with no active sub-commands
            my_cli = 'interface ' + self.value
            if isinstance(cli, basestring):
                return cli == my_cli
            return cli.command == my_cli

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        if self.get_state() == State.CREATE:
            self.generate_cli(asa_cfg_list,
                              'no shutdown',
                              mode_command='interface '+ self.value,
                              is_system_context = self.get_top().is_multi_mode_asa())

    def diff_ifc_asa(self, cli):
        if not self.is_my_cli(cli):
            return
        if isinstance(cli, basestring):
            self.set_state(State.NOCHANGE)
            return
        'We only take care of shutdown sub-command'
        is_shutdown = filter_first(lambda cmd: str(cmd) == 'shutdown', cli.sub_commands)
        if not is_shutdown:
            self.set_state(State.NOCHANGE)

class Vlan(ConnObj):
    '''
    Describe the VLAN tag
    '''
    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        '''
        Populate model
        '''
        super(Vlan, self).populate_model(delta_ifc_key, delta_ifc_cfg_value)
        self.type = str(delta_ifc_cfg_value.get('type'))
        '''
        hardcode to avoid triggering vxlan code in brownfield because an APIC bug
        which is still generating 1 instead of 0
        '''
        self.type = '0'
        self.tag = delta_ifc_cfg_value.get('tag')

class EncapAss(ConnObj):
    '''
    Present the association of VLAN and VIF
    '''

    def create_missing_ifc_delta_cfg(self):
        if 'OspfVIfCfg' not in self.delta_ifc_cfg_value:
            self.delta_ifc_cfg_value['OspfVIfCfg'] = {'state': State.NOCHANGE}

    def ifc2asa(self, no_asa_cfg_stack,  asa_cfg_list):
        # If this ENCAPASS is destroyed and there are no references to it, then
        # delete the interface it refers to
        if self.get_state() == State.DESTROY:
            top = self.get_top()
            if top.is_vlan_supported():
                hits = filter(lambda x: x.get_encapass() == self,
                             top.get_child('Interface'))
                if not hits:
                    cif = top.get_child('VIF').get_child(self.vif).value
                    vlan = top.get_child('VLAN').get_child(self.encap).tag
                    if top.is_multi_mode_asa():
                        top.get_asa_context().allocate_interface('%s.%s' % (cif, vlan), False)
                    else:
                        cli = 'clear config interface %s.%s' % (cif, vlan)
                        self.generate_cli(no_asa_cfg_stack,
                                          cli,
                                          response_parser=ignore_response_parser, 
                                          is_system_context=top.is_multi_mode_asa())

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        super(EncapAss, self).populate_model(delta_ifc_key, delta_ifc_cfg_value)
        self.vif = delta_ifc_cfg_value['vif']
        self.encap = delta_ifc_cfg_value['encap']

class IntfConfigRel(DMObject):
    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        if not self.has_ifc_delta_cfg() or delta_ifc_cfg_value['state'] == State.CREATE: 
            '''CSCvg23655: modification of an object by two requests: one to create one to destroy.
            '''
            self.delta_ifc_key = delta_ifc_key
            self.delta_ifc_cfg_value = delta_ifc_cfg_value
            self.state = delta_ifc_cfg_value['state']
            'Will deprecate check for target once the target is replaced with value in the initialization'
            if 'target' in delta_ifc_cfg_value:
                self.value = delta_ifc_cfg_value['target']
            else:
                self.value = delta_ifc_cfg_value['value']

class ExIntfConfigRel(IntfConfigRel):
    def __init__(self):
        DMObject.__init__(self, ExIntfConfigRel.__name__)

class InIntfConfigRel(IntfConfigRel):
    def __init__(self):
        DMObject.__init__(self, InIntfConfigRel.__name__)

class ExIntfConfigRelFolder(DMObject):
    'A list of additional interface parameters for external Connectors'

    def __init__(self):
        DMObject.__init__(self, ExIntfConfigRelFolder.__name__)
        self.register_child(ExIntfConfigRel())

class InIntfConfigRelFolder(DMObject):
    'A list of additional interface parameters for internal Connectors'

    def __init__(self):
        DMObject.__init__(self, InIntfConfigRelFolder.__name__)
        self.register_child(InIntfConfigRel())

class IPv6EnforceEUI64(SimpleType):
    'Model after  "ipv6 enforce-eui64 <nameif>'
    def __init__(self):
        super(IPv6EnforceEUI64, self).__init__('ipv6_enforce_eui64',
                                               'ipv6 enforce-eui64')
        self.response_parser = ignore_info_response_parser

    def get_cli(self):
        return 'ipv6 enforce-eui64 ' + self.parent.nameif
