'''
Created on Oct 28, 2013

@author: richauha

Copyright (c) 2013 by Cisco Systems
'''
from translator.base.dmlist import DMList
from translator.base.dmobject import DMObject
from base.simpletype import SimpleType
from translator.base.dmboolean import DMBoolean
import utils.util as util
from translator.state_type import Type, State
from validators import AdminContextValidator

class NTP(DMObject):
    'Container of NTP'
    def __init__(self, is_multi_mode_asa):
        DMObject.__init__(self, NTP.__name__)
        self.register_child(DMList('NTPServer', NTPObj,
                                   asa_key='ntp server',
                                   mini_audit_command='show run ntp | grep ^ntp server'))
        self.register_child(DMList(name='NTPTrustedKey', child_class = NTPTrustedKey,
                                   asa_key = 'ntp trusted-key',
                                   mini_audit_command='show run ntp | grep ^ntp trusted-key'))
        self.register_child(DMBoolean(ifc_key='NTPAuthenticate',
                                      asa_key='ntp authenticate', on_value="enable"))
        self.register_child(DMList(name='NTPAuthenticationKey',  child_class = NTPAuthenticationKey,
                                   asa_key = 'ntp authentication-key',
                                   mini_audit_command='show run ntp | grep ^ntp authentication-key'))
        self.is_system_context = is_multi_mode_asa
        for child in self.children.values():
            child.is_system_context = is_multi_mode_asa

class NTPObj(SimpleType, AdminContextValidator):
    'Model for "ntp server" CLI'

    def __init__(self, name):
        SimpleType.__init__(self, name, asa_gen_template='ntp server %(server)s')

    def get_child_state(self, child_key):
        'Return the state of a child object'
        key_tuples = filter(lambda x: x[1] == child_key, self.delta_ifc_cfg_value['value'].keys())
        if key_tuples:
            return self.delta_ifc_cfg_value['value'][key_tuples[0]]['state']

    def get_cli(self):
        '''Generate the CLI for ntp server config.
        '''
        assert self.has_ifc_delta_cfg()
        config = self.get_value()
        key = config.get('key')
        prefer = config.get('prefer')
        if self.get_state() == State.MODIFY:
            'Take care of the case for removing optional parameters'
            prefer_state = self.get_child_state('prefer')
            key_state = self.get_child_state('key')
            if prefer_state == State.DESTROY:
                prefer = ''
            if key_state == State.DESTROY:
                key = ''
        key = ' key ' + key if key else ''
        prefer = ' prefer ' if prefer == 'enable' else ''
        return ('ntp server %(server)s' % config + key + prefer).strip()

    def create_asa_key(self):
        '''Create the the asa key identifies this object
        @return str
        '''
        assert self.has_ifc_delta_cfg()
        value = self.get_value()
        return 'ntp server %(server)s' % value

    def parse_multi_parameter_cli(self, cli):
        '''
        Override the default implementation in case the CLI does not match asa_gen_template due to optional parameter
        '''
        # Take care of the mandatory parameters
        result = SimpleType.parse_multi_parameter_cli(self, cli, alternate_asa_gen_template = self.asa_gen_template)

        'Take care of the optional parameters'
        tokens = cli.split()

        # The number of mandatory parameters is 3, i.e. 'ntp server 1.1.1.1'
        if len(tokens) == 3:
            return result # no optional parameter

        for name  in ['key', 'prefer']:
            result[(Type.PARAM, name, '')] = {'state': State.NOCHANGE, 'value': ''}

        option = tokens[3:]
        if 'key' in option: # e.g. "ntp server 1.1.1.1 key 11
            result[Type.PARAM, 'key', '']['value'] = option[1]
        if 'prefer' in option: # e.g. "ntp server 1.1.1.1 prefer
            result[Type.PARAM, 'prefer', '']['value'] = 'enable'
        return result

class NTPTrustedKey(SimpleType, AdminContextValidator):
    'Model for "ntp trusted-key" CLI'
    def __init__(self, name):
        SimpleType.__init__(self, name, asa_gen_template='ntp trusted-key %s')

    def create_asa_key(self):
        return self.get_cli()

class NTPAuthenticationKey(SimpleType, AdminContextValidator):
    'Model for "ntp authentication-key" CLI'
    def __init__(self, name):
        SimpleType.__init__(self, name,
                            asa_gen_template='ntp authentication-key %(key_number)s md5 %(key)s')

    def create_asa_key(self):
        return self.get_cli()
