'''
Created on Aug 27, 2013

@author: tsnguyen
'''


from translator.base.simpletype import SimpleType
from translator.base.dmobject import DMObject
from translator.base.dmlist import DMList
from translator.validators import UnsupportedOnMultiContextASAValiator
from translator.basic_threat_detect import RateList
from translator.base.dmboolean import DMBoolean
from translator.state_type import State, Type

class ScanningThreatDetection(DMObject, UnsupportedOnMultiContextASAValiator):
    '''
    Scanning Threat Detection
    Supports the following CLI:

    threat-detection scanning-threat [shun
        [except ip-address ip_address mask]]
    #Note: No support of object-group due to CSCup80986

    threat-detection scanning-threat shun duration seconds
    threat-detection rate scanning-threat
        rate-interval rate_interval average-rate av_rate burst-rate burst_rate
    '''

    THREAT_DETECTION_SCANNING = "threat-detection scanning-threat"
    THREAT_DETECTION_SCANNING_SHUN = "threat-detection scanning-threat shun"
    THREAT_DETECTION_SCANNING_RATE = "threat-detection rate scanning-threat"

    def __init__(self):
        DMObject.__init__(self, ScanningThreatDetection.__name__)
        self.register_child(RateList('ScanningThreatRate',
                                     self.THREAT_DETECTION_SCANNING_RATE, ((5, 10), (4, 8))))
        self.register_child(DMList('ScanningThreatShunExceptIP',
                                   ShunExceptIP,
                                   asa_key=self.THREAT_DETECTION_SCANNING_SHUN + " except ip-address",
                                   mini_audit_command='show run threat-detection scanning-threat | grep shun except ip-address'))
        self.register_child(ShunDuration())
        self.register_child(DMBoolean('shun_status', self.THREAT_DETECTION_SCANNING_SHUN))
        self.register_child(DMBoolean('scanning_threat', self.THREAT_DETECTION_SCANNING))

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        'Override the default implementation to take care of the implicit "threat-detection scanning-threat shun"'
        DMObject.populate_model(self, delta_ifc_key, delta_ifc_cfg_value)
        shun_status = self.get_child('shun_status')
        if shun_status.has_ifc_delta_cfg():
            return
        'destroy shun_status if all other shun parameters are destroyed'
        shun_children = []
        shun_children.extend(self.get_child('ScanningThreatShunExceptIP').children.values())
        shun_children.append(self.get_child('ScanningThreatShunDuration'))
        shun_children = filter(lambda child: child.has_ifc_delta_cfg(), shun_children)
        if not shun_children:
            return
        if all(map(lambda child: child.get_action() == State.DESTROY, shun_children)):
            shun_status.populate_model((Type.PARAM, shun_status.ifc_key, 'dummy'), {'state': State.DESTROY, 'value': 'enable'})

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        'Override the default implementation so that the "no" commands come out in the right order for ASA'
        tmp_no_asa_cfg_stack = []
        DMObject.ifc2asa(self, tmp_no_asa_cfg_stack, asa_cfg_list)
        tmp_no_asa_cfg_stack.reverse()
        no_asa_cfg_stack.extend(tmp_no_asa_cfg_stack)

    def validate_configuration(self):
        if self.has_ifc_delta_cfg() and self.get_value():
            return DMObject.validate_configuration(self)

class ShunDuration(SimpleType):
    ''' Class to support CLI
    threat-detection scanning-threat shun duration time
    '''
    def __init__(self):
        asa_key = ScanningThreatDetection.THREAT_DETECTION_SCANNING_SHUN + ' duration'
        asa_gen_template = asa_key + ' %(duration)s'
        SimpleType.__init__(self, 'ScanningThreatShunDuration', asa_key, asa_gen_template)

class ShunExceptIP(SimpleType):
    ''' Class to support CLI
    threat-detection scanning-threat shun except ip-address ip_address mask
    '''
    def __init__(self, name):
        asa_key= ScanningThreatDetection.THREAT_DETECTION_SCANNING_SHUN + ' except ip-address'
        asa_gen_template = asa_key + " %(except_ip)s %(except_mask)s"
        SimpleType.__init__(self, name, asa_gen_template=asa_gen_template)

    def create_asa_key(self):
        return self.get_cli()
