'''
Created on Sep 16, 2015

@author: jeffryp

Copyright (c) 2015 by Cisco Systems, Inc.
All rights reserved.

Classes used for TrustSec configuration.
'''

from asaio.cli_interaction import ignore_warning_response_parser,\
    ignore_response_parser
from base.dmlist import DMList
from base.dmobject import DMObject
from base.simpletype import SimpleType
from state_type import State
from utils.util import filter_first, normalize_ip_address
from aaa_server_group import AAAServerGroup, AAAServer
from structured_cli import StructuredCommand
from translator.state_type import Type

class CommonDestroy(SimpleType):
    '''
    Overrides ifc2asa to destroy the CLI using the asa_key
    '''

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        if self.has_ifc_delta_cfg():
            state = self.get_state()
            if state in (State.CREATE, State.MODIFY):
                self.generate_cli(asa_cfg_list, self.get_cli())
            elif state == State.DESTROY:
                self.generate_cli(no_asa_cfg_stack, 'no ' + self.asa_key)

class DefaultPassword(CommonDestroy):
    '''
    SXP default password
    '''

    def __init__(self):
        super(DefaultPassword, self).__init__('default_password',
                                              'cts sxp default password')
        self.response_parser = ignore_warning_response_parser

class Enable(SimpleType):
    '''
    SXP enable
    '''

    ASA_KEY = 'cts sxp enable'

    def __init__(self):
        super(Enable, self).__init__('enable', self.ASA_KEY,
                                     asa_gen_template=self.ASA_KEY)

    def get_cli(self):
        return self.asa_gen_template

    def parse_cli(self, cli):
        return ''

class TrustSec(DMObject):
    '''
    Global TrustSec configuration
    '''

    ASA_KEY = 'cts'

    def __init__(self):
        super(TrustSec, self).__init__(TrustSec.__name__, self.ASA_KEY)

        # Must be in the order in which to generate the CLI
        self.register_child(ISEServerGroup())
        self.register_child(CTSServerGroup())
        self.register_child(SXP())
        self.register_child((TrustSecSGTMaps()))

    def get_translator(self, cli):
        if isinstance(cli, basestring) and cli.startswith(self.ASA_KEY):
            return super(TrustSec, self).get_translator(cli)
        return self.children.values()[0].get_translator(cli)

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        """
        'cts server-group' configuration is implicit:
            It is created if ISEServerGroup is created.
            It is destroyed if ISEServerGroup is destroyed.
        """
        super(TrustSec, self).populate_model(delta_ifc_key, delta_ifc_cfg_value)
        ise_server_group = self.children.values()[0]
        if not ise_server_group.has_ifc_delta_cfg():
            return
        cts_server_group = self.children.values()[1]
        if cts_server_group.has_ifc_delta_cfg():
            return
        'Configure cts server-group based on the value of ise_server_group'
        key =  (Type.PARAM, 'server_group', '')
        state = ise_server_group.delta_ifc_cfg_value['state']
        if state == State.MODIFY:
            state = State.NOCHANGE
        value =  {'state': state, 'value': ISEServerGroup.NAME}
        delta_ifc_cfg_value['value'][key] = value
        cts_server_group.populate_model(key, value)

class CTSServerGroup(CommonDestroy):
    def __init__(self):
        super(CTSServerGroup, self).__init__('server_group', 'cts server-group')

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        if self.has_ifc_delta_cfg():
            if self.get_state() == State.DESTROY:
                self.response_parser = ignore_response_parser
        super(CTSServerGroup, self).ifc2asa(no_asa_cfg_stack, asa_cfg_list)

class Peer(SimpleType):
    '''
    SXP connection peer
    '''

    ASA_KEY = 'cts sxp connection peer'
    IFC_KEY = 'peer'

    def __init__(self, instance):
        super(Peer, self).__init__(
            instance,
            asa_gen_template=self.ASA_KEY +
                ' %(ip_address)s password %(password)s mode %(mode)s %(role)s')
        self.response_parser = ignore_warning_response_parser

    def get_cli_prefix(self):
        return (self.ASA_KEY + ' %(ip_address)s') % self.get_value()

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        if self.has_ifc_delta_cfg():
            state = self.get_state()
            if state in (State.DESTROY, State.MODIFY):
                self.generate_cli(no_asa_cfg_stack,
                                  'no ' + self.get_cli_prefix())
            if state in (State.CREATE, State.MODIFY):
                self.generate_cli(asa_cfg_list, self.get_cli())

    def is_my_cli(self, cli):
        return cli.startswith(self.get_cli_prefix())

    def get_value(self):
        'Override the default implementation to normalized IP address format'
        result = SimpleType.get_value(self)
        result['ip_address'] = normalize_ip_address(result['ip_address'])
        return result

class Peers(DMList):
    '''
    A list of SXP connection peers
    '''

    def __init__(self):
        super(Peers, self).__init__(Peer.IFC_KEY, Peer, Peer.ASA_KEY,
                                    'show running-config cts sxp connection')

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        if (not self.get_top().is_audit and
                filter_first(
                    lambda child: child.get_state() == State.MODIFY, self)):
            # Remove any entries to be deleted
            for key, child in self.children.items():
                if child.get_state() == State.DESTROY:
                    del(self.children[key])
            # Change the remaining entries to CREATE
            for child in self:
                child.set_state(State.CREATE)

            self.mini_audit_command = 'show running-config cts sxp connection'
            self.mini_audit()

        for child in self:
            child.ifc2asa(no_asa_cfg_stack, asa_cfg_list)

class SXP(DMObject):
    '''
    SXP configuration
    '''

    ASA_KEY = 'cts sxp'

    def __init__(self):
        super(SXP, self).__init__(SXP.__name__, self.ASA_KEY)

        self.register_child(Enable())
        self.register_child(DefaultPassword())
        self.register_child(CommonDestroy('reconciliation_period',
                                          'cts sxp reconciliation period'))
        self.register_child(CommonDestroy('retry_period',
                                          'cts sxp retry period'))
        self.register_child(Peers())

    def get_translator(self, cli):
        if isinstance(cli, basestring) and cli.startswith(self.ASA_KEY):
            return super(SXP, self).get_translator(cli)

class TrustSecSGTMap(SimpleType):
    '''
    TrustSec SGT mapping of an IP address to an SGT number
    '''

    ASA_KEY = 'cts role-based sgt-map'

    def __init__(self, instance):
        super(TrustSecSGTMap, self).__init__(
            instance,
            asa_gen_template=self.ASA_KEY +
                ' %(ip_address)s sgt %(security_group_tag)s')
        self.response_parser = self.ip_sgt_response_parser

    def is_my_cli(self, cli):
        return cli.startswith((self.ASA_KEY + ' %(ip_address)s ') % self.get_value())

    def get_value(self):
        result = SimpleType.get_value(self)
        if self.get_top().compare_asa_version(9.6, 1, 0) < 0:
            result['ip_address'] = normalize_ip_address(result['ip_address'])
            return result
        '''
            From version 9.6(1), ASA always normalize ip address to have a prefix lengh suffix
            if it does have one already.
            Example:
                1.1.1.1 is stored as 1.1.1.1/32
                1::1 is stored as 1::1/128
        '''
        tokens = result['ip_address'].split('/')
        address = normalize_ip_address(tokens[0])
        if len(tokens) == 1:
            prefix_length = '32' if address.find('.') > 0 else '128'
        else:
            prefix_length = tokens[1]
        result['ip_address'] = '/'.join((address, prefix_length))
        return result

    @staticmethod
    def ip_sgt_response_parser(response):
        if response:
            if 'Failed to remove IP-SGT' in response:
                return
            return response

class TrustSecSGTMaps(DMList):
    '''
    A list of TrustSec SGT maps
    '''

    def __init__(self):
        super(TrustSecSGTMaps, self).__init__(TrustSecSGTMap.__name__,
                                              TrustSecSGTMap,
                                              TrustSecSGTMap.ASA_KEY)


class ISEServerGroup(AAAServerGroup):
    """
    A special, built-in, AAAServerGroup for TrustSec.
    TBD: move inside TrustSec
    """
    NAME = '__$ISEServer$__'

    def __init__(self):
        super(ISEServerGroup, self).__init__(ISEServerGroup.__name__)
        self.mini_audit_command = ('show running-config aaa-server ' + self.NAME)

    def get_cli(self):
        return 'aaa-server %s protocol radius' % self.NAME

    def get_server_tag(self):
        return self.NAME

    def is_my_cli(self, cli):
        if isinstance(cli, StructuredCommand):
            cli = cli.command
        return cli == self.get_cli()

    def get_translator(self, cli):
        if isinstance(cli, StructuredCommand):
            cli = cli.command

        # AAA server group command
        if self.is_my_cli(cli):
            return self

        # AAA server command
        if not cli.startswith('aaa-server ' + self.NAME + ' '):
            return
        m = AAAServer.get_cli_pattern().match(cli.strip())
        if not m:
            return
        for child in self:
            result = child.get_translator(cli)
            if result:
                return result
        result = AAAServer(cli)
        self.register_child(result)
        return result

    def diff_ifc_asa(self, cli):
        if not self.has_ifc_delta_cfg():
            self.delta_ifc_key = (Type.FOLDER, self.ifc_key, '')
            self.delta_ifc_cfg_value = {'state': State.DESTROY, 'value': {}}
            ancestor = self.get_ifc_delta_cfg_ancestor()
            if ancestor:
                ancestor.delta_ifc_cfg_value['value'][self.delta_ifc_key] = self.delta_ifc_cfg_value
        else:
            self.set_state(State.MODIFY)
