'''

ASA Device Script
Copyright (c) 2013 by Cisco Systems, Inc.

All APIs use 'device' param to identify a device to communicate. The Device is a dictionary with following information

device =  {
   'host':   <device ip address>,
   'port': <destion L4 port for communicating with the device>,
   'creds': {'username': useranme, 'password': password}
}


API returns Faults dictionary of the form:
{'status', True, 'health': [], 'faults': [fault]}
where fault is a tuple of (path, code, error_msg),
   path: a list of configuration entry keys leading to the offending configuration object
   error_msg: a str


@see devicemodel for the formation configuration parameter passed to the API's.


Created on Jun 28, 2013

@author: feliu
'''

from collections import Counter
from copy import deepcopy
import pprint
import re
from requests.exceptions import RequestException
import traceback
from asaio.cli_interaction import CLIInteraction, CLIResult
from asaio.dispatch import HttpDispatch
from utils.util import deliver_clis, read_clis, read_asa_version
from utils.util import deliver_sts_table, query_sts_audit_table
from utils.util import get_config_root_key
from utils.util import normalize_faults
from utils.util import get_all_connectors
from utils.util import get_asa_context_information
from utils.errors import ConnectionError, IFCConfigError
from utils.errors import ASACommandError, DeviceParameterError
from utils.errors import ASABusyError, FaultCode
import utils.env as env
from utils.pretty_dict import pretty_str
from translator.cluster import ClusterConfig
from translator.devicemodel import ifc2asa, generate_asa_delta_cfg, Features,\
    DeviceModel
from translator.sts import STS

class Status(object):
    'Possible value for "state" in the return value'
    SUCCESS =   0 # thing is fine and dandy
    TRANSIENT = 1 # Temporary failure. Script Engine will retry the configuration by calling device script API again
    PERMANENT = 2 # Configuration failure - could be due to invalid configuration parameter or feature unsupported on device etc.  Script will not retry
    AUDIT =     3 # Script can request to trigger an Audit

def get_config_dict(argv):
    '''Return the last dict object in a list of parameters, which is the configuration parameter.
    Assuming the last dictionary item in the list is it.
    '''
    dicts = filter(lambda x: isinstance(x, dict), argv)
    if len(dicts) > 0:
        return dicts[-1]
    else:
        return None

def get_interfaces_dict(argv):
    '''Return the last dict object in a list of parameters, which is the interfaces parameter
    Assuming the first dictionary item in the list with two dictionary times is it.
    '''
    dicts = filter(lambda x: isinstance(x, dict), argv)
    if len(dicts) == 2:
        return dicts[0]
    else:
        return None

def dump_config_arg(f):
    ''' Decorator used to dump out device , interfaces and configuration argument passed to an API
    '''

    def trace(*argv):
        message = []

        message.append("***** API Called: %s" % f.__name__)

        device = argv[0]
        message.append("[Device argument]\n%s" % pprint.pformat(device))

        interfaces = get_interfaces_dict(argv[1:])
        if interfaces:
            message.append("[Interfaces argument]\n%s" % pprint.pformat(interfaces))
        env.debug('\n'.join(message))

        config  = get_config_dict(argv[1:])
        if config:
            message = "[Configuration argument]\n%s" % pretty_str(config)
            env.debug(message)

        result = f(*argv)
        if isinstance(result, dict):
            env.debug("[Result of %s]\n%s" % (f.__name__,  pprint.pformat(result)))

        return result

    return trace if dump_config_arg.enabled else f

def is_apic_runtime():
    try:
        import Insieme.Logger as Logger
        return True
    except Exception:
        return False #run in south bound API test

# Starting with the Brazos APIC release, the APIC logs will only be sent to
# debug.log and the ASA device package logs will only be sent to partner.log.
# To ease correlating the configuration dictionary sent to the device package
# with the logging from the device package, the configuration dictionary should
# always be dumped.
dump_config_arg.enabled = True

def exception_handler(f):
    ''' Decorator used to handle exceptions, only used for xxxModify and xxxAudit APIs for now
    '''
    def handler(*argv, **kwargs):
        config  = get_config_dict(argv[1:])
        root = get_config_root_key(config)
        result = {'state': Status.SUCCESS}
        try:
            f(*argv, **kwargs)
        except ConnectionError as e:
            result['faults'] = [(root, FaultCode.CONNECTION_ERROR, str(e))]
            result['state'] = Status.AUDIT
        except IFCConfigError as e:
            result['faults'] = e.fault_list
            result['state'] = Status.PERMANENT
        except ASACommandError as e:
            result['faults'] = e.fault_list
            result['state'] = Status.PERMANENT
        except ASABusyError as e:
            result['faults'] = [(root, FaultCode.BUSY_ERROR, str(e))]
            result['state'] = Status.TRANSIENT
        except DeviceParameterError as e:
            result['faults'] = [(root, FaultCode.DEVICE_PARAMETER_ERROR, str(e))]
            result['state'] = Status.PERMANENT
        except Exception as e:
            result['faults'] = [(root, FaultCode.UNEXPECTED_ERROR,
                                "Unexpected exception: " + str(e) + '\n' +
                                traceback.format_exc())]
            result['state'] = Status.PERMANENT
        finally:
            if 'faults' in result:
                result['faults'] = normalize_faults(result['faults'], config)
            return result
    return handler

@dump_config_arg
def deviceValidate(device,
                   version):
    '''This function validates if the device matches one of the versions
    supported by the device package. This function should do a compare (regular
    expression match) the version fetched from the device with version
    string passed in the param.

    @param device: dict
        a device dictionary
    @param version: str
         a regular expression for the supported versions of the device.
         e.g: '1.0'
     @return: dict
        {'state': <state>, 'version': <version>}
    '''
    env.debug("[Version argument]\n%s" % version)

    result = {}
    asa_version, error = read_asa_version(device)
    if asa_version:
        result['version'] = asa_version
        if re.compile(version).match(asa_version):
            result['state'] = Status.SUCCESS
        else:
            result['state'] = Status.PERMANENT
            message = 'Device running un-supported ASA version %s' % asa_version
            env.debug('deviceValidate: %s' % message)
            # deviceValidate currently ignores faults.  Return them in case
            # they're supported in the future
            result['faults'] = [(get_config_root_key({}),
                                FaultCode.UNSUPPORTED_ASA,
                                message)]
    else: #not able to connect
        result['state'] = Status.TRANSIENT
        env.debug('deviceValidate: %s' % error)
        # deviceValidate currently ignores faults.  Return them in case
        # they're supported in the future
        result['faults'] = [(get_config_root_key({}), FaultCode.CONNECTION_ERROR, error)]
    return result


@dump_config_arg
def deviceModify(device,
                 interfaces,
                 configuration):
    ''' Update global device configuration
    @param device: dict
        a device dictionary
        @warning:  'creds' attribute should be (name, password) pair?!
    @param interfaces: dict of interfaces
        a dictionary of interfaces for this device.
    @param configuration: dict
        Device configuration configured by the user. This is configuration that does not
        depend on a graph/function. Example:
         syslog server IP
         HA peer IP
         HA mode
         Port-Channel
    @return: Faults dictionary
    '''
    return modify_operation(device, interfaces, configuration, Features.vnsDevCfg)

@dump_config_arg
def deviceHealth(device,
                interfaces,
                configuration):
    '''
    This function polls the device. API should return a weighted device health
    score in range (0-100)

    @param device:  dict
        a device dictionary
    @param interfaces:
        A list of the physical interfaces
        The format is:
            {
                (cifType, '', <interface name>) : {
                    'state': <state>,
                    'label': <label>
                },
                ...
            }
    @param configuration: dict
        It contains device configuration. The configuration dictionary follows
        the format described above.
    @return: int
        a weighted device health score in range (0-100)
    Example:
        Poll CPU usage, free memory, disk usage etc.
    '''

    device = deepcopy(device)
    try:
        get_asa_context_information(device)
    except ConnectionError as e:
        result = {}
        result['state'] = Status.TRANSIENT
        result['faults'] = [([], FaultCode.CONNECTION_ERROR, str(e))]
        result['health'] = [([], 0)]
        return result
    model = DeviceModel(device)
    is_multi_mode = model.is_multi_mode_asa()
    if is_multi_mode and model.is_user_context():
        # Cannot access the device interfaces in a user context, so return
        # success
        return {'state': Status.SUCCESS, 'health': [([], 100)]}
    if interfaces:
        result = {}
        try:
            cli_results = get_interface_info(device,
                                             [cif[2] for cif in interfaces],
                                             is_multi_mode)
        except (IOError, RequestException, ConnectionError) as e:
            result['state'] = Status.TRANSIENT
            result['faults'] = [([], FaultCode.CONNECTION_ERROR, str(e))]
            result['health'] = [([], 0)]
            return result

        faults = []
        result['state'] = Status.SUCCESS
        for interface, cli_result in zip(interfaces, cli_results):
            if 'line protocol is up' not in cli_result.err_msg:
                faults.append(([interface], FaultCode.LINE_STATUS_ERROR, ''))
        if faults:
            result['faults'] = faults
            result['health'] = [([], 0)]
            return result
    else:
        result = check_connectivity(device)
        if result['state'] != Status.SUCCESS:
            result['health'] = [([], 0)]
            return result

    # Check cluster to see if it is ready
    cluster_health = ClusterConfig.get_health(device, configuration)
    if cluster_health:
        result['health'] = [cluster_health]
    else:
        result['health'] = [([], 100)]
    return result

@dump_config_arg
def deviceCounters(device,
                   interfaces,
                   configuration):
    '''
    This function is called periodically to report statistics associated with
    the physical interfaces of the device.

    @param device:
        a device dictionary
    @param interfaces:
        A list of the physical interfaces
        The format is:
            {
                (cifType, '', <interface name>) : {
                    'state': <state>,
                    'label': <label>
                },
                ...
            }
    @param configuration: dict
        It contains device configuration. The configuration dictionary follows
        the format described above.
    @return: dict
            The format of the dictionary is as follows
            {
              'state': <state>
              'counters': [(path, counters), ...]
            }

            path: Identifies the object to which the counter is associated.

            counters: {
              'rxpackets': <rxpackets>,
              'rxerrors': <rxerrors>,
              'rxdrops': <rxdrops>,
              'txpackets': <txpackets>,
              'txerrors': <txerrors>,
              'txdrops': <txdrops>
            }
    '''

    result = {'counters': []}
    device = deepcopy(device)
    try:
        get_asa_context_information(device)
    except ConnectionError as e:
        result['state'] = Status.TRANSIENT
        result['faults'] = [([], FaultCode.CONNECTION_ERROR, str(e))]
        return result
    model = DeviceModel(device)
    is_multi_mode = model.is_multi_mode_asa()
    if is_multi_mode and model.is_user_context():
        # Cannot access the device interfaces in a user context, so return
        # success
        result['state'] = Status.SUCCESS
        return result
    if interfaces:
        try:
            cli_results = get_interface_info(device,
                                             [cif[2] for cif in interfaces],
                                             is_multi_mode)
        except (IOError, RequestException, ConnectionError) as e:
            result['state'] = Status.TRANSIENT
            env.debug('deviceCounters: %s' % e)
            # deviceCounters currently ignores faults.  Return them in case
            # they're supported in the future
            result['faults'] = [([], FaultCode.CONNECTION_ERROR, str(e))]
            return result

        result['state'] = Status.SUCCESS
        for cif, cli_result in zip(interfaces, cli_results):
            if cli_result and cli_result.err_type == CLIResult.INFO:
                if cif[2].lstrip().lower().startswith('po'):
                    counters = parse_port_channel_counters(device,
                                                           cli_result.err_msg)
                else:
                    counters = parse_physical_counters(cli_result.err_msg)
                result['counters'].append(([cif], counters))
    else:
        result.update(check_connectivity(device))

    return result

@dump_config_arg
def deviceAudit(device,
                interfaces,
                configuration):
    ''' deviceAudit is called to synchronize configuration between IFC and the device.
    IFC invokes this call when device recovers from hard fault like reboot,
    mgmt connectivity loss etc.

    This call can be made as a result of IFC cluster failover event.

    Device script is expected to fetch the device configuration and reconcile with the
    configuration passed by IFC.

    In this call IFC will pass entire device configuration across all graphs. The device
    should insert/modify any missing configuration on the device. It should remove any extra
    configuration found on the device.

    @param device: dict
        a device dictionary
    @param interfaces: dict of interfaces
        a dictionary of interfaces for this device.
    @param configuration: dict
        Entire configuration on the device.
    @return: Faults dictionary

    '''
    return audit_operation(device, interfaces, configuration, Features.vnsDevCfg)

#
# FunctionGroup API
#
@dump_config_arg
def serviceAudit(device,
                 configuration):
    ''' serviceAudit is called to synchronize configuration between IFC and the device.
    IFC invokes this call when device recovers from hard fault like reboot,
    mgmt connectivity loss etc.

    This call can be made as a result of IFC cluster failover event.

    Device script is expected to fetch the device configuration and reconcile with the
    configuration passed by IFC.

    In this call IFC will pass entire device configuration across all graphs. The device
    should insert/modify any missing configuration on the device. It should remove any extra
    configuration found on the device.

    @param device: dict
        a device dictionary
    @param interfaces: list of strings
        a list of interfaces for this device.
        e.g. [ 'E0/0', 'E0/1', 'E1/0', 'E1/0' ]
    @param configuration: dict
        Entire configuration on the device.
    @return: Faults dictionary

    '''
    return audit_operation(device, [], configuration, Features.vnsMFunc)

@dump_config_arg
def serviceModify(device,
                  configuration):
    '''This function is called when graph is rendered as a result of a EPG attach or when any param
    within the graph is modified.

    The configuration dictionary follows the format described above.

    This function is called for create, modify and destroy of graph and its associated functions

    @param device: dict
        a device dictionary
    @param configuration: dict
        configuration in delta form
    @return: Faults dictionary
    '''
    return modify_operation(device, [], configuration, Features.vnsMFunc)

@dump_config_arg
def serviceHealth(device,
                  configuration):
    '''This function is called periodically to report health of the service function
    on the device.

    @param device:
        a device dictionary
    @param configuration: dict
        It contains device configuration, group configuration for a particular
        graph instance and function configuration. The configuration dictionary follows
        the format described above.
    @return: dict
        It is dictionary of function health represented as an integer within (0-100) range.
        Format of the dictionary is as follows

          { (path): health,  ...}

          path: Is a list identifying a function within the graph. Example
          [ vdev, vgrp, vfunc ]

          vdev - Device Name. Passed in the configuration dictionary
          vgrp - Function Group name passed in the configuration dictionary
          vfunc - function name passed in configuration dictionary

        The health dictionary will contain an entry for each function rendered on the
        device for the given graph
    '''

    firewalls, connectors = get_all_connectors(configuration)
    if connectors:
        result = {}
        try:
            cli_results = get_interface_info(
                device, [connector.get_nameif() for connector in connectors])
        except (IOError, RequestException, ConnectionError) as e:
            result['state'] = Status.TRANSIENT
            result['faults'] = [([], FaultCode.CONNECTION_ERROR, str(e))]
            return result

        # Build a dictionary of connector line status
        line_status = {}
        for connector, cli_result in zip(connectors, cli_results):
            'Status is True if the line protocol is up'
            line_status[connector.get_nameif()] = 'line protocol is up' in cli_result.err_msg

        faults = []
        result['health'] = []
        result['state'] = Status.SUCCESS
        for firewall in firewalls:
            up = True
            for connector in firewall.iter_connectors():
                status = line_status[connector.get_nameif()]
                if not status:
                    faults.append((connector.get_config_path(),
                                   FaultCode.LINE_STATUS_ERROR,
                                   ''))
                up = up and status
            result['health'].append((firewall.get_config_path(), 100 if up else 0))
        if faults:
            result['faults'] = faults
        return result
    else:
        return check_connectivity(device)

@dump_config_arg
def serviceCounters(device,
                    configuration):
    '''
    This function is called periodically to report statistics associated with
    the service functions rendered on the device.

    @param device:
        a device dictionary
    @param configuration: dict
        It contains device configuration, group configuration for a particular
        graph instance and function configuration. The configuration dictionary follows
        the format described above.
    @return: dict
            The format of the dictionary is as follows
            {
              'state': <state>
              'counters': [(path, counters), ...]
            }

            path: Identifies the object to which the counter is associated.
              The path is a list identifying a specific instance of a connector.
              It includes device name, group name, etc. as shown below:

              path = [ vdev, vgrp, vfunc, conn ]

              vdev - Device Name. Passed in the configuration dictionary
              vgrp - Function Group name passed in the configuration dictionary
              vfunc - function name passed in configuration dictionary
              conn - connector name within the function

            counters: {
              'rxpackets': <rxpackets>,
              'rxerrors': <rxerrors>,
              'rxdrops': <rxdrops>,
              'txpackets': <txpackets>,
              'txerrors': <txerrors>,
              'txdrops': <txdrops>
            }
    '''
    result = {'counters': []}

    connectors = get_all_connectors(configuration)[1]
    if connectors:
        try:
            cli_results = get_interface_info(
                device, [connector.get_nameif() for connector in connectors])
            result['state'] = Status.SUCCESS
            for connector, cli_result in zip(connectors, cli_results):
                path = connector.get_config_path()
                counters = parse_connector_counters(cli_result.err_msg)
                result['counters'].append((path, counters))
        except (IOError, RequestException, ConnectionError) as e:
            result['state'] = Status.TRANSIENT
            result['faults'] = [([], FaultCode.CONNECTION_ERROR, str(e))]
    else:
        result.update(check_connectivity(device))

    return result

@dump_config_arg
def clusterAudit(device,
                 interfaces,
                 configuration):
    ''' deviceAudit is called to synchronize configuration between IFC and the device.
    IFC invokes this call when device recovers from hard fault like reboot,
    mgmt connectivity loss etc.

    This call can be made as a result of IFC cluster failover event.

    Device script is expected to fetch the device configuration and reconcile with the
    configuration passed by IFC.

    In this call IFC will pass entire device configuration across all graphs. The device
    should insert/modify any missing configuration on the device. It should remove any extra
    configuration found on the device.

    @param device: dict
        a device dictionary
    @param configuration: dict
        Entire configuration on the device.
    @param interfaces: dict of interfaces
        a dictionary of interfaces for this device.
    @return: boolean
        Success/Failure

    '''
    """
    Take care of CSCvb13121.
    For clusterAudit callout, if the 'vdevs' is empty, it should clean up service parameters on the ASA.
    """
    features = Features.vnsClusterCfg
    vdevs = device.get('vdevs')
    if vdevs == []:
        features |= Features.vnsMFunc
    return audit_operation(device, interfaces, configuration, features)

@dump_config_arg
def clusterModify(device,
                  interfaces,
                  configuration):
    '''This function is called when graph is rendered as a result of a EPG attach or when any param
    within the graph is modified.

    The configuration dictionary follows the format described above.

    This function is called for create, modify and destroy of graph and its associated functions

    @param device: dict
        a device dictionary
    @param configuration: dict
        configuration in delta form
    @param interfaces: dict of interfaces
        a dictionary of interfaces for this device.
    @return: Faults dictionary
    '''
    return modify_operation(device, interfaces, configuration, Features.vnsClusterCfg)


@exception_handler
def modify_operation(device,
                     interfaces,
                     configuration,
                     features):
    '''Modify the configuration on ASA device.
    The configuration dictionary follows the format described above.

    This function is called for create, modify and destroy of graph and its associated functions

    @param device: dict
        a device dictionary
    @param configuration: dict
        configuration in delta form
    @param features: bit-mask from Features
        to indicate what part of configurations to generate
    @return: Faults dictionary through the exception_handler decorator
    '''
    sts = STS()
    device = deepcopy(device)
    get_asa_context_information(device)
    clis = ifc2asa(configuration, device, interfaces, sts, features=features)
    deliver_clis(device, clis)
    is_STS = False
    if is_STS:
        if sts.sts_table:
            deliver_sts_table(device, sts, False)

@exception_handler
def audit_operation(device,
                    interfaces,
                    configuration,
                    features):
    ''' Device script is expected to fetch the device configuration and reconcile with the
    configuration passed by IFC.

    In this call IFC will pass entire device configuration across all graphs. The device
    should insert/modify any missing configuration on the device. It should remove any extra
    configuration found on the device.

    @param device: dict
        a device dictionary
    @param interfaces: list of strings
        a list of interfaces for this device.
        e.g. [ 'E0/0', 'E0/1', 'E1/0', 'E1/0' ]
    @param configuration: dict
        Entire configuration on the device.
    @param features: bit-mask from Features
        to indicate what part of configurations to generate
    @return: Faults dictionary through the exception_handler decorator

    '''
    sts = STS()
    device = deepcopy(device)
    get_asa_context_information(device)
    input_clis = read_clis(device)
    output_clis = generate_asa_delta_cfg(configuration, input_clis, device, interfaces,
                                          sts, features)
    deliver_clis(device, output_clis)
    is_STS = False
    if is_STS:
        query_sts_audit_table(device, sts)
        if sts.sts_table:
            deliver_sts_table(device, sts, True)

CONNECTOR_COUNTER_PATTERN = None
def parse_connector_counters(show_output):
    'Parse the traffic statistics counters in the "show interface" output'
    global CONNECTOR_COUNTER_PATTERN
    if not CONNECTOR_COUNTER_PATTERN:
        CONNECTOR_COUNTER_PATTERN = re.compile(
            '.*\sTraffic Statistics' +
            '.*\s(\d+) packets input' +
            '.*\s(\d+) packets output' +
            '.*\s(\d+) packets dropped',
            re.DOTALL)

    # The ASA doesn't have support for these counters
    result = {
        'rxerrors': 0,
        'txerrors': 0,
        'txdrops': 0
    }

    m = CONNECTOR_COUNTER_PATTERN.match(show_output)
    if m:
        result['rxpackets'] = int(m.group(1))
        result['txpackets'] = int(m.group(2))
        result['rxdrops'] = int(m.group(3))
    else:
        result['rxpackets'] = 0
        result['txpackets'] = 0
        result['rxdrops'] = 0

    return result

PHYSICAL_COUNTER_PATTERN = None
def parse_physical_counters(show_output):
    'Parse the hardware counters in the "show interface" output'
    global PHYSICAL_COUNTER_PATTERN
    if not PHYSICAL_COUNTER_PATTERN:
        PHYSICAL_COUNTER_PATTERN = re.compile(
            '.*\s(\d+) packets input' +
            '.*\s(\d+) input errors' +
            '.*\s(\d+) packets output' +
            '.*\s(\d+) output errors',
            re.DOTALL)

    # The ASA doesn't have support for these counters
    result = {
        'rxdrops': 0,
        'txdrops': 0
    }

    m = PHYSICAL_COUNTER_PATTERN.match(show_output)
    if m:
        result['rxpackets'] = int(m.group(1))
        result['rxerrors'] = int(m.group(2))
        result['txpackets'] = int(m.group(3))
        result['txerrors'] = int(m.group(4))
    else:
        result['rxpackets'] = 0
        result['rxerrors'] = 0
        result['txpackets'] = 0
        result['txerrors'] = 0

    return result

PORT_CHANNEL_MEMBER_PATTERN = None
def parse_port_channel_members(device, show_output):
    '''
    Parse the members in a port channel.

    The example response for 'show interface Po2' is:
            Interface Port-channel2 "", is down, line protocol is down
            Hardware is EtherChannel/LACP, BW 1000 Mbps, DLY 10 usec
            Auto-Duplex(Full-duplex), Auto-Speed(1000 Mbps)
            Input flow control is unsupported, output flow control is off
            Available but not configured via nameif
            MAC address c08c.60e6.9a6c, MTU not set
            IP address unassigned
              Members in this channel:
                  Active: Gi0/6 Gi0/7
                  Inactive: Gi0/4 Gi0/5
    '''
    global PORT_CHANNEL_MEMBER_PATTERN
    if not PORT_CHANNEL_MEMBER_PATTERN:
        PORT_CHANNEL_MEMBER_PATTERN = re.compile(
            '.*Members in this channel' +
            '(?:.*Active:\s+(.*?)\n)?' +
            '(?:.*Inactive:\s+(.*)\n)?',
            re.DOTALL)

    m = PORT_CHANNEL_MEMBER_PATTERN.match(show_output)
    members = []
    if m:
        active = m.group(1)
        if active:
            members.extend(active.split())
        inactive = m.group(2)
        if inactive:
            members.extend(inactive.split())
    return members

def parse_port_channel_counters(device, show_output):
    '''
    Parse the hardware counters for a port channel.

    There is no hardware counter information for a port channel.  Instead, the
    counter information for the members of the port channel needs to be queried
    and summed together.
    '''
    result = Counter(rxpackets=0, rxerrors=0, rxdrops=0, txpackets=0,
                     txerrors=0, txdrops=0)

    members = parse_port_channel_members(device, show_output)

    if members:
        try:
            cli_results = get_interface_info(device, members)
            for cli_result in cli_results:
                if cli_result and cli_result.err_type == CLIResult.INFO:
                    counters = parse_physical_counters(cli_result.err_msg)
                    for key, value in counters.iteritems():
                        result[key] += value
        except (IOError, RequestException):
            # Ignore errors and use 0 for values
            pass
    return dict(result)

def check_connectivity(device):
    'Check if there is connectivity to the device'
    version, error = read_asa_version(device)
    if version:
        return {'state': Status.SUCCESS}
    else:
        return {'state': Status.TRANSIENT,
                'faults': [([], FaultCode.CONNECTION_ERROR, error)]}

def get_interface_info(device, interface_names, use_system_context=False):
    clis = [CLIInteraction('show interface ' + name) for name in interface_names]
    skip_changeto = False
    if use_system_context:
        clis.insert(0, CLIInteraction('changeto system'))
        skip_changeto = True
    else:
        'take care of user-context in multi-context ASA'
        device = deepcopy(device)
        get_asa_context_information(device)
        model = DeviceModel(device)
        if model.is_user_context():
            context_name = model.get_asa_context_name()
            clis.insert(0, CLIInteraction('changeto context ' + context_name))
            skip_changeto = True
    messenger = HttpDispatch(device).make_command_messenger(clis)
    result = messenger.get_results()
    if result and skip_changeto:
        'skip the response to "changeto  ..."'
        result = result[1:]
    return result

