'''
Created on Feb 5, 2015

@author: dli
Copyright (c) 2015 by Cisco Systems

'''
from base.dmlist import DMList
from translator.base.compositetype import CompositeType
from translator.base.simpletype import SimpleType
from state_type import State, Type
import asaio.cli_interaction as cli_interaction
from translator.structured_cli import convert_to_structured_commands,\
    StructuredCommand

class ASAContext(CompositeType):
    '''
    ASAConext represents 'context <name' command in system context of multi-context mode ASA.
    It is used to configure the allocation of interfaces to the context.
    '''
    def __init__(self, context_name):
        self.name = context_name
        CompositeType.__init__(self, ASAContext.__name__, asa_key = 'context ' + context_name)
        self.register_child(DMList('interface_allocation', InterfaceAllocation))
        self.register_child(SensorList(context_name))
        self.is_system_context = True
        for child in self:
            child.is_system_context = True

    def allocate_interface(self, interface_name, allocate = True):
        """
        Allocate the given interface to the context.
        @param interface_name: String.
            The name of the interface.
        @param allocate: boolean
            True if to allocate the interface, otherwise de-allocate
        """
        intfs = self.children.values()[0]
        intf = intfs.get_child(interface_name)
        if not intf:
            intf = InterfaceAllocation(interface_name)
            intfs.register_child(intf)
        ifc_key =   (Type.PARAM, 'interface', interface_name)
        ifc_cfg =   {'state': State.CREATE if allocate else State.DESTROY, 'value': interface_name}
        intf.populate_model(ifc_key, ifc_cfg)

    def allocate_ips_sensor(self, sensor_name, state):
        """
        Allocate the given sensor to the context.
        @param sensor_name: String.
            The name of the ips sensor.
        @param state: one of State.CREATE, State.MODIFY, State.DESTROY
        """
        sensors = self.children.values()[1]
        if not sensor_name:
            return
        sensor = sensors.get_child(sensor_name)
        if not sensor:
            sensor= IpsSensorAllocation(sensor_name)
            sensors.register_child(sensor)
        ifc_key =   (Type.PARAM, 'ips_sensor', sensor_name)
        ifc_cfg =   {'state': state, 'value': sensor_name}
        sensor.populate_model(ifc_key, ifc_cfg)

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        'Override default implementation to skip checking of its own IFC config as it does not have it.'
        '''
        for de-allocating interface, we won't issue the 'no allocate-interface' command if no other context
        uses the interface. We issue 'clear config interface' instead.
        '''
        for intf in self.get_interfaces_to_remove():
            intf.set_state(State.NOCHANGE)
            self.generate_cli(no_asa_cfg_stack, 'clear config interface ' + intf.ifc_key,
                              response_parser=cli_interaction.ignore_response_parser)
        'go through each child to do ifc2asa'
        child_mode_command = self.get_child_mode_command()
        for child in self.children.values():
            child.mode_command = child_mode_command
            if isinstance(child, DMList):
                for c in child.children.values():
                    c.mode_command = self.get_cli()
            child.ifc2asa(no_asa_cfg_stack, asa_cfg_list)

    def get_cli(self):
        return self.asa_key

    def get_translator(self, cli):
        'allow audit operation even if the user context is not admin context'
        if self.is_my_cli(cli):
            return self

    def get_interfaces_to_remove(self):
        '''
        Return list of the interfaces to be de-allocated that is not used by any other context.
        '''
        deallocated_interfaces = filter(lambda intf: intf.get_state() == State.DESTROY,
                                        self.get_child('interface_allocation'))
        if not deallocated_interfaces:
            return []
        all_allocated_interface_names = self.get_all_allocated_interface_names()
        result = []
        for intf in deallocated_interfaces:
            if intf.ifc_key not in all_allocated_interface_names:
                result.append(intf)
        return result

    def get_all_allocated_interface_names(self):
        '''
        Return a list of interface names allocated to other contexts
        '''
        contexts_config = self.query_asa('show run context', 'system').replace('  ', ' ')
        contexts_config = convert_to_structured_commands(filter(lambda line: line not in ['!', ''], contexts_config.split('\n')))
        result = set()
        for cmd in contexts_config:
            if isinstance(cmd, StructuredCommand) and cmd.command != 'context ' + self.name:
                allocate_intf_cmds = filter(lambda sub_cmd:
                                            isinstance(sub_cmd, basestring) and sub_cmd.startswith('allocate-interface '),
                                            cmd.sub_commands)
                if allocate_intf_cmds:
                    intf_names = set(map(lambda line: line.split(' ')[1], allocate_intf_cmds))
                    result = result.union(intf_names)
        return result

class SensorList(DMList):
    def __init__(self, context_name):
        DMList.__init__(self, 'ips_sensor_allocation', IpsSensorAllocation, 'allocate-ips',
                        mini_audit_command='show run context ' + context_name + ' | grep allocate-ips')

    def read_asa_config(self):
        'Override the default implementation to strip leading blanks'
        result = DMList.read_asa_config(self)
        if result:
            return '\n'.join(map(str.strip, result.split('\n')))

    def get_translator(self, cli):
        'allow audit operation even if the user context is not admin context'
        if not self.is_my_cli(cli):
            return
        'Find the translator from its children'
        result = None
        for child in self.children.values():
            result = child.get_translator(cli)
            if result:
                break
        if result:
            return result
        'Create a child that is to generate "no" cli, by the invocation of child.diff_ifc_asa'
        result = self.child_class(cli)
        self.register_child(result)
        return result

class InterfaceAllocation(SimpleType):
    'Model after the sub-command "allocate-interface <interface-name> [visible|invisible]"'
    def __init__(self, name):
        SimpleType.__init__(self, name,  asa_key='allocate-interface',
                            response_parser=cli_interaction.ignore_response_parser,
                            is_system_context=True);

class IpsSensorAllocation(SimpleType):
    'Model after the sub-command "allocate-ips <sensor-name>"'
    def __init__(self, name):
        SimpleType.__init__(self, name,
                            asa_key = 'allocate-ips',
                            response_parser=cli_interaction.ignore_response_parser,
                            is_system_context=True);

    def get_translator(self, cli):
        'Override the default implementation to allow audit operation'
        if self.is_my_cli(cli):
            return self

    def is_my_cli(self, cli):
        return cli == self.get_cli()
