'''
Created on Jun 12, 2013

@author: dli

Copyright (c) 2013 by Cisco Systems
'''
from collections import OrderedDict
import re
from asaio.cli_interaction import CLIInteraction
import utils.env as env
from utils.errors import FaultCode
from utils.util import query_asa
from utils.util import normalize_param_dict
from translator.structured_cli import convert_to_structured_commands
from translator.structured_cli import StructuredCommand
from translator.state_type import Type, State
from translator.type2key import type2key
from translator.validators import Validator

class DMObject(object):
    '''
    dmobject is combination of DMObject, DMHolder in ASDM.
    '''

    # the ASA configuration can be in either CLI format or REST API. It is considered a "static" class variable.
    is_cli_mode = True

    def __init__(self, ifc_key = "", asa_key= None):
        """
        Constructor
        """
        #the components as dictionary of dmobject. It is similar to that in DMHashCollection in ASDM.
        self.children = OrderedDict()
        #the key to identify this dmobject in the children of its parent
        self.ifc_key = ifc_key
        #the key to identify this dmobject for the ASA configuration object. It could be command prefix or re._pattern_type; or attribute name in REST API
        self.asa_key = asa_key
        #parent DMObject
        self.parent = None

    def __iter__(self):
        'Support iterating over the children'
        return self.children.itervalues()

    def get_key(self):
        """
        @return:
            the key for this object in the IFC model. It has similar purpose as getObjectKey() in ASDM.
        The value of the key should be the name  of the IFC's vnsParam or vnsFolder element
        """
        return self.ifc_key

    def get_asa_key(self):
        """
        @return:
            the string of the ASA configuration object to identify this object in the IFC model.
        The value of the key should be the command prefix for ASA CLI,  or the attribute name in ASA REST API.
        """
        return self.asa_key

    def register_child(self, dmobj):
        """
        Add a dmobject as a child of this dmobject.

        @param dmobj:
            input, a dmobject
        """
        self.children[dmobj.get_key()] = dmobj
        dmobj.parent = self
        
    def unregister_child(self, dmobj):
        """
        Remove a dmobject child from children.
        """
        del self.children[dmobj.get_key()]        
        

    def get_top(self):
        """
        Recursively traverse from child to top parent object, this is
        useful when access to other object's contents is required
        
        @return:
            the top parent object which is know as the device model
        """
        if (self.parent == None):
            return self
        else:
            return self.parent.get_top()

    def get_parent(self):
        '@return: the parent object'
        return self.parent

    def get_ancestor_by_class(self, ancestor_class):
        """
        Recursively traverse from child to top parent object looking for an
        ancestor which is an instance of the specified class.
        If found, return the object, else return None
        """
        if (self.parent == None or isinstance(self.parent, ancestor_class)):
            return self.parent
        else:
            return self.parent.get_ancestor_by_class(ancestor_class)

    def get_child(self, key):
        """
        @return:
            the child for a given key.
        @param key:
            input, a string representing the IFC key of the child dmobject,
            or a triplet:  type, key, instance
        """
        if isinstance(key, tuple):
            kind, key, instance = key
            """
            For certain types of entries in the configuration parameter, the key
            may not corresponding to our internal key.
            Example:
                (Type.ENCAP, '', '3')
            we look up the internal key for it here.
            """
            key = type2key.get(kind, key)
        return self.children[key] if key in self.children else None

    def get_child_by_asa_key(self, asa_key):
        """
        @return:
            the child for a given ASA key.
        @param asa_key:
            input, a string representing the ASA key of the child dmobject.
            For CLI mode it is the command prefix.
            For REST API: to be determined."
        """
        for child in self.children:
            if asa_key.startswith(child.get_asa_key()):
                return child
        return None

    def get_children_count(self):
        """
        @return:
            the number of children in this dmobject.
        """
        return len(self.children)


    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        '''Store the ifc configuration for this object. It is used to generate ASA configuration on
        on calling ifc2asa method.

        @param delta_ifc_key: tuple
            (type, key, instance)
            Notice here that this delta_ifc_key is not the delta_ifc_key for self.children, but rather than delta_ifc_key
            for IFC delta dictionary.
            "key" field corresponds to the ifc_key to self.children dictionary.
        @param delta_ifc_cfg_value: dict
            {
            'state': state,
            'device': device,
            'connector': connector,
            'value': scalar value or config dictionary
            }
        '''

        #the following two attributes will be used by ifc2asa method to perform ifc2asa translation
        if not self.has_ifc_delta_cfg() or delta_ifc_cfg_value['state'] == State.CREATE: 
            '''CSCvg23655: modification of an object by two requests: one to create one to destroy.
            And also work around CSCuv87932.
            '''
            self.delta_ifc_key = delta_ifc_key
            self.delta_ifc_cfg_value = delta_ifc_cfg_value
        if not self.children:
            return

        cfg = delta_ifc_cfg_value.get('value')
        if not isinstance(cfg, dict): #terminal node
            return

        for delta_ifc_key, value in cfg.iteritems():
            child = self.get_child(delta_ifc_key)
            if child:
                child.populate_model(delta_ifc_key, value)
            #else:
                #self.log('DMObject.populateModel: %s not found' % name)


    def get_ifc_delta_cfg_ancestor(self):
        '''@return the ancestor DMObject that has ifc_delta_cfg container for this DMObject.
        '''
        result = self.parent
        while result:
            if result.has_ifc_delta_cfg():
                return result
            result = result.parent

    def create_missing_ifc_delta_cfg(self):
        '''For diff_ifc_asa method to work, we need each DMObject container
        to have proper IFC dictionary entry.
        @warning: This method is invoked by translator delta engine. Please do no override
                  it in derived class unless you really know what you are doing.
        @precondition: populate_model has been called.
        '''
        if not self.children:
            return
        if  not self.has_ifc_delta_cfg():
            self.delta_ifc_key = Type.FOLDER, self.ifc_key, ''
            self.delta_ifc_cfg_value = {'state': State.NOCHANGE, 'value': {}}
            ancestor = self.get_ifc_delta_cfg_ancestor()
            if ancestor:
                ancestor.delta_ifc_cfg_value['value'][self.delta_ifc_key] =  self.delta_ifc_cfg_value
        for child in self.children.itervalues():
            child.create_missing_ifc_delta_cfg()

    def ifc2asa(self, no_asa_cfg_stack,  asa_cfg_list):
        '''
        Generate ASA configuration from IFC configuration delta. Assume that
        the device model has already populated with IFC delta configuration
        using populate_model method.

        One uses (self.delta_ifc_key, self.delta_ifc_cfg_value) to generate ASA
        configuration.

        @param no_asa_cfg_stack: stack
            input/output, stack of ASA CLIs to contain delete commands
        @param asa_cfg_list: list
            input/output, list of ASA configurations
        '''
        for child in self.children.itervalues():
            child.ifc2asa(no_asa_cfg_stack, asa_cfg_list)

    def mini_audit(self):
        'This is to handle modify situation where we do not have the old value from IFC'
        asa_clis = self.read_asa_config()
        if not asa_clis:
            return
        self.create_missing_ifc_delta_cfg()
        asa_clis = convert_to_structured_commands(asa_clis.strip().split('\n'))
        translator_mode_command = self.__dict__.get('mode_command')
        for cli in asa_clis:
            translator = self.get_translator(cli)
            if translator:
                translator.diff_ifc_asa(cli)
                translator.mode_command = translator_mode_command

    def read_asa_config(self):
        return self.query_asa(self.mini_audit_command,
                              context = 'system' if self.__dict__.get('is_system_context') else None)

    def is_my_cli(self, cli):
        '''Determine if a CLI matches self.asa_key
        @param cil: str or StructuredCommand
        @return boolean, True if self.asa_key matches CLI, or False
        '''
        if not self.asa_key:
            return False
        if isinstance(cli, basestring):
            command = cli.strip()
        elif isinstance(cli, StructuredCommand):
            # Ignore 'no' commands
            if cli.is_no:
                return False
            command =  cli.command.strip()
        if isinstance(self.asa_key, re._pattern_type):
            return self.asa_key.match(command.strip())
        if len(command) == len(self.asa_key):
            return command == self.asa_key
        return command.startswith(self.asa_key.rstrip() + ' ')

    def get_translator(self, cli):
        '''Find the DMObject within this DMOBject that can translate a CLI
        @param cli: CLI
        @return:  the DMObject that can translate the cli
        '''
        if self.__dict__.get('is_system_context') and self.get_top().is_user_context():
            'no audit operation if the target ASA device is user context'
            return None
        if hasattr(self, 'create_asa_key'):
            self.asa_key = self.create_asa_key()
        if not self.children:
            if self.is_my_cli(cli):
                return self
            else:
                return None
        for child in self.children.itervalues():
            if hasattr(child, 'get_translator'):
                result = child.get_translator(cli)
                if result:
                    return result
        return None


    def has_ifc_delta_cfg(self):
        '''Determine if there is IFC delta configuration for this class
        @return: boolean
            True if there is IFC delta configuration, otherwise False.
        '''
        if not hasattr(self, 'delta_ifc_cfg_value'):
            return False;
        if not hasattr(self, 'delta_ifc_key'):
            return False;
        if self.delta_ifc_cfg_value == None or self.delta_ifc_key == None:
            return False;
        return True

    def generate_cli(self, asa_cfg_list, cli, **keywords):
        '''Append the given cli to asa_cfg_list. It is similar to DMObject.generateCommand(cliHolder, cli) in ASDM.
        @param asa_cfg_list: list
        @param cli: str
        @param keywords: parameters to pass to CLIInteraction

        If a class derived from dmobject has attributes of 'mode_command',
        'response_parser', or 'mode_response_parser', 'is_system_context' they will be used for the
        CLIInteraction parameters.  This can be used to set the mode command for
        all the generated CLI.

        The following example will generate the sub-commands for an interface:
        self.mode_command = 'interface Port-channel2.20'
        self.generate_cli(asa_cfg_list, 'no shutdown')
        self.generate_cli(asa_cfg_list, 'nameif external_Conn1')
        self.generate_cli(asa_cfg_list, 'ip address 20.20.20.20 255.255.255.128')
        '''

        for key in ('mode_command', 'response_parser', 'mode_response_parser', 'is_system_context', 'is_critical'):
            if hasattr(self, key) and key not in keywords:
                keywords[key] = getattr(self, key)
        clii = CLIInteraction(cli, model_key=self.get_config_path(), **keywords)
        asa_cfg_list.append(clii)

    def get_config_path(self):
        '@return the IFC configuration path to this object'
        if self.parent:
            result = self.parent.get_config_path()
        else:
            result = []
        if self.has_ifc_delta_cfg():
            result.append(self.delta_ifc_key)
        return result

    def get_value(self):
        '@return configuration value for this object'
        if self.has_ifc_delta_cfg():
            if 'value' in self.delta_ifc_cfg_value:
                value = self.delta_ifc_cfg_value['value']
            elif 'target' in self.delta_ifc_cfg_value:
                value = self.delta_ifc_cfg_value['target']
            else:
                return None
            return normalize_param_dict(value, self.get_state() == State.MODIFY)

    def get_state(self):
        '@return configuration state for this object'
        return self.delta_ifc_cfg_value['state'] if self.has_ifc_delta_cfg() else None

    def get_state_recursive(self):
        'Combine the state of self and all of its children'
        if self.children:
            state = self.get_state()
            if state == State.MODIFY:
                # No need to check anymore
                return state

            result = state
            for child in self.children.itervalues():
                state = child.get_state_recursive()
                if state != None:
                    if result == None:
                        result = state
                    elif result != state:
                        result = State.MODIFY
                    if result == State.MODIFY:
                        # No need to check anymore
                        break
            return result
        else:
            return self.get_state()

    def set_state(self, state):
        'Set the state for this object'
        self.delta_ifc_cfg_value['state'] = state

    def validate_configuration(self):
        '''Check the validity of the IFC configuration for this object.
        @return list of faults
        '''
        result = []
        if isinstance(self, Validator):
            error = self.validate(self.get_value())
            self.report_fault(error, result)

        for child in self.children.itervalues():
            child_result = child.validate_configuration()
            self.report_fault(child_result, result)
        return result

    def generate_fault(self, msg, attribute_key = None):
        """
        @param msg: str
            The error message.
        @param attribute_key: str or list of str
            The key of a configuration attribute.
        @return fault, which is a (path, code, str) tuple.'
        """
        def get_ifc_key_path(config, attribute_key):
            '@return IFC key path from normal key path of an attribute'
            if not attribute_key:
                return
            if isinstance(attribute_key, str):
                key_tuple = filter(lambda x: x[1]==attribute_key, config.keys())[0]
                return key_tuple
            'attribute_key is a list of str'
            key_tuple = get_ifc_key_path(config, attribute_key[0])
            result = [key_tuple]
            if len(attribute_key) > 1:
                result.extend(get_ifc_key_path(config[key_tuple]['value'], attribute_key[1:]))
            return result

        path = self.get_config_path()
        if not attribute_key:
            return (path, FaultCode.CONFIGURATION_ERROR, msg)
        ifc_path = get_ifc_key_path( self.delta_ifc_cfg_value['value'], attribute_key)
        if isinstance(ifc_path, tuple):
            ifc_path = [ifc_path]
        path.extend(ifc_path)
        return (path, FaultCode.CONFIGURATION_ERROR, msg)

    def get_cli_prefixes(self):
        """
        Return a set of CLI prefixes for this object. It is used to speed up lookup translator for a given CLI.
        A CLI prefix is the first word in a CLI, e.g. 'host' is the CLI prefix for the command 'host <name>'.
        It only concerns with the global level command, not sub-command
        """
        result = set()

        asa_key = None
        if self.asa_key:
            asa_key = self.asa_key
        elif hasattr(self, 'create_asa_key'):
            asa_key = self.create_asa_key()
        if asa_key:
            if isinstance(self.asa_key, str):
                asa_key = self.asa_key
            else: # a regular expression pattern
                asa_key = self.asa_key.pattern
            result.add(asa_key.split()[0])

        for child in self.children.itervalues():
            result = result.union(child.get_cli_prefixes())
        return result

    def report_fault(self, error, result, attribute_key = None):
        if isinstance(error, str):
            result.append(self.generate_fault(error, attribute_key))
        elif isinstance(error, tuple):
            result.append(error)
        elif isinstance(error, list):
            result.extend(error)

    def query_asa(self, query_cmd, context = None):
        """ Get from ASA device the response to some show command
        @param query_cmd: str. e.g. "show run class-map foo"
        @param context: str, the name of a context to run, can be None.
        @return: response string from the device, or None
        @attention:  PLEASE do not use this API to make configuration change on the device
        """
        device = self.get_top().get_device()
        if not device:
            env.debug("DMObject.query_asa: device model has no device attribute")
            return None
        if not context and self.get_top().is_multi_mode_asa():
            context = self.get_top().get_asa_context_name()
        return query_asa(device, query_cmd, context)[0]

    def log(self, msg):
        """
        Print a msg for debugging purpose.
        @param msg: str
            input, a text string
        """
        env.debug(msg)

    def update_references(self):
        '''
        This is called, after the device model is fully populated, to allow an
        object to update its references to other objects or perform any
        operations that depend on the device model being fully populated.
        '''
        for child in self:
            child.update_references()
