'''
Created on Jun 12, 2013

@author: dli
Copyright (c) 2013 by Cisco Systems

'''
from asaio.cli_interaction import format_cli, make_context_aware_cli
from utils.util import set_cfg_state, massage_param_dict, populate_asa_version
from utils.errors import IFCConfigError
from state_type import State

from base.dmobject import DMObject
from base.dmlist import DMList

from shared_config import SharedConfigs

from port_channel_member import PortChannelMembers
from routing.bgp import BGPs, bgp_audit_init
from routing.ospf_interface import ospf_interface_audit_init
from routing.ospf_process import OspfProcesses, ospf_process_audit_init
from trustsec import TrustSec
from rule.access_rule import AccessListList
from rule.nat_rule import NATListList
from timeouts import Timeouts

from logging import LoggingConfig
from failover import FailoverConfig
from cluster import ClusterConfig
from lacp_max_bundle import LACPMaxBundles

from basic_threat_detect import BasicThreatDetection
from advanced_threat_detect import AdvancedThreatDetection
from scanning_threat_detect import ScanningThreatDetection
from netflow_objects import NetFlowObjects

from bridge_group_interface import BridgeGroupIntfs
from connector import Vifs, Vlan, EncapAss
from interface import Interfaces, SameSecurityTraffic

from ns_object_container import NetworkObjects, ServiceObjects
from network_object_group import NetworkObjectGroups
from service_object_group import ServiceObjectGroups
from security_object_group import SecurityObjectGroups
from empty_object import EmptyObjectsList

from ip_audit import IPAudit
from ntp import NTP
from dns import DNS
from smart_call_home import SmartCallHome

from service_policy import GlobalServicePolicyContainer
from translator.asa_context import ASAContext
from translator.tcp_options import GlobalTCPOptions
from translator.arp import Arp
from translator.time_range import TimeRanges
from translator.s2svpn import GlobalS2SVPNConfiguration
from translator.snmp import SNMP
from translator.banner import Banner

class Features(object):
    vnsNone       = 0x0
    vnsClusterCfg = 0x1   # Flag to indicate if the stuff under vnsClusterCfg should be modified
    vnsDevCfg     = 0x2   # Flag to indicate if the configuration under vnsDevCfg folder should be modified
    vnsMFunc      = 0x4   # Flag to indicate if the function configuration should be modified
    vnsAll        = vnsClusterCfg | vnsDevCfg | vnsMFunc

class FunctionMode(object):
    GoTo      = 1
    GoThrough = 2

def ifc2asa(ifc_delta_cfg_dict, device = None, interfaces = None, sts = None, features = Features.vnsAll, is_audit = False):
    '''Generate ASA delta configuration from configuration in IFC delta dictionary form.

    @param ifc_delta_cfg_dict: dict
    for serviceModiy, it is the form
        configuration = {
          (device) : {
              # Device Shared configuration
             (folders): {
                (folders): {
                   (params):
                    ...
                },
                (params):
                ...
             }
             (functionGroup Config): {
                 (connectors):

                 (folders): {
                    (folders): {
                       (params):
                        ...
                    },
                    (params):
                    ...
                 }

                 (function) : {
                     (folders): {
                        (folders): {
                           (params):
                            ...
                        },
                        (params):
                        ...
                     }
                 }
             }
          }

        for deviceModify, it is of the form
        configuration = {
                 (folders): {
                    (folders): {
                       (params):
                        ...
                    },
                    (params):
                    ...
             }
          }


        Configuration Dictionary Format:

            (Type, Key, Instance) : {'state': State, 'device': Device, 'connector': Connector, 'value': Value}
            #
            # Type of attribute
            #
            Type = Insieme.Fwk.Enum(
                DEV=0,
                GRP=1,
                CONN=2,
                FUNC=3,
                FOLDER=4,
                PARAM=5
            )

            #
            # State of the attribute
            #
            State = Insieme.Fwk.Enum(
                NOCHANGE=0,
                CREATE=1,
                MODIFY=2,
                DESTROY=3

            Key: the key of the configuration element given in the device specification file

            Instance: the instance identifier of a configuration object, such the name of an ACL

            Connector: the connector name

            Device: the device name, such as the case in a HA configuration situation.

    @param device: dict
        to identify the ASA device, passed in from device_script APIs
    @param interfaces: dict
        The format is similar to configuration parameter. It is of the following form
        for deviceAudit and deviceModify APIs:
        Interfaces = {
            (Type.CIF, '', <physical interface name>) : {
                      'state': <state>,
                      'label': <label>
            },
          ...
        }

        Example:
        {
            (11, '', 'Gig0_1'): {
                    'state': 1,
                    'label': 'external'
             },
            (11,'', 'Gig0_2'): {
                    'state': 1,
                    'label': 'internal'
            },
            (11, '', 'Gig0_3'): {
                    'state': 1,
                    'label': 'mgmt',
            },
            (11, '', 'Gig0_4'): {
                    'state': 1.
                    'label': 'cluster'
            }
        }

        but for clusterAudit and cluseterModify APIs, it is of the form:
        Interfaces = {
            (Type.LIF, '', <interface long label>) : {
                       'cifs': { <deviceID>: <physical interface name>, ...}
                      'state': <state>,
                      'label': <label>
            },
          ...
        }

        Example:
        {
           (12, '', 'internal'): {
                            'cifs': {
                                    'ASA1': 'Gig0_1',
                                   'ASA2': 'Gig0_1'
                            },
                           'label': 'int',
                           'state': 0},
          (12, '', 'external'): {
                             'cifs': {
                                   'ASA1': 'Gig0_2',
                                  'ASA2': 'Gig0_2'
                            },
                           'label': 'out',
                           'state': 0},
          (12, '', 'utility'): {
                            'cifs': {
                                    'ASA1': 'Gig0_3',
                                   'ASA2': 'Gig0_3'
                            },
                            'label': 'util',
                            'state': 0},
          (12, '', 'failover_link'): {
                            'cifs': {
                                  'ASA1': 'Gig0_4',
                                 'ASA2': 'Gig0_4'
                            },
                            'label': 'linkt',
                            'state': 0}
          },

    @param sts: dict
        to identify the STS external and internal tag
    @param features: bit-mask from Features
        to indicate what part of configurations to generate
    @param is_audit: boolean
        to indicate if this is called by xxxAudit API's
    @return list of CLI's
    '''
    asa = DeviceModel(device, interfaces, features, is_audit)
    'ifc_cfg is a copy of ifc_delta_cfg_dict suited for us.'
    ifc_cfg = massage_param_dict(asa, ifc_delta_cfg_dict)

    '''Since the input dictionary is in arbitrary order, we use two-pass approach to make sure resulting
    CLI list is in the right order given in the device model.
    '''

    #1st pass
    asa.populate_model(ifc_cfg.keys()[0], ifc_cfg.values()[0])
    asa.update_references()
    faults = asa.validate_configuration()
    if faults:
        raise IFCConfigError(faults)

    #2nd pass
    cli_list = []
    no_cli_stack = []
    asa.ifc2asa(no_cli_stack, cli_list)

    #gather result from positive and negative forms for the CLIs
    result = no_cli_stack
    result.reverse() #last in first out for the 'no' CLIs
    result.extend(cli_list)
    if asa.system_context_accessible():
        result = make_context_aware_cli(result, asa.is_admin_context(), asa.get_asa_context_name())
    format_cli(result)

    #update sts table
    if asa.sts_table:
        sts.sts_table = asa.sts_table
    return result

def generate_ifc_delta_cfg(ifc_cfg_dict, asa_clis, device = None, interfaces = None,
                           features = Features.vnsAll):
    '''Compare ifc configuration with asa configuration , and produce
    ifc configuration delta.

    @param asa_clis: list
        input/out list of CLI's read from ASA device
    @param ifc_cfg_dict: dict
        the configuration of ASA in IFC dictionary format. It will be modified
        after the comparison.
    @param device: dict
        to identify the ASA device, passed in from device_script APIs
    @param interfaces: list
        physical interfaces names passed in from device_script APIs
    @param features: bit-mask from Features
        to indicate what part of configurations to generate
    @return dict
        a copy of ifc_cfg_dct but with the state of each entry set to indicate
        if a modification operation or create operation is required.
    '''
    asa = DeviceModel(device, interfaces, features, is_audit=True)

    def create_missing_ifc_delta(ifc_cfg):
        'Transformer to fill in the missing folders. This is important for configuration deletion'
        asa.populate_model(ifc_cfg.keys()[0], ifc_cfg.values()[0])
        asa.update_references()
        'The following call will modify ifc_cfg'
        asa.create_missing_ifc_delta_cfg()
        return ifc_cfg

    def set_cfg_state_2_CREATE(ifc_cfg):
        'Transformer to set the State of each entry to CREATE, the default state of all entries'
        set_cfg_state(ifc_cfg, State.CREATE)
        return ifc_cfg

    'ifc_cfg is a copy of ifc_cfg_dict suited for us.'
    ifc_cfg = massage_param_dict(asa, ifc_cfg_dict,
                                 transformers=[bgp_audit_init,
                                               ospf_process_audit_init,
                                               ospf_interface_audit_init,
                                               set_cfg_state_2_CREATE,
                                               create_missing_ifc_delta])

    '''Since the input dictionary is in arbitrary order, we use two-pass approach to make sure resulting
    CLI list is in the right order given in the device model.
    '''
    for cli in asa_clis:
        translator = asa.get_translator(cli)
        if translator:
            translator.diff_ifc_asa(cli)
    return ifc_cfg

def generate_asa_delta_cfg(ifc_cfg, asa_cfg, device = None, interfaces = None, sts = None,
                           features = Features.vnsAll):
    '''Generate CLIs that represent the delta between given IFC configuration and ASA configuration
    @param ifc_cfg: dict
    @param asa_cfg: list of CLIs
    @param device: dict
        to identify the ASA device, passed in from device_script APIs
    @param interfaces: list
        physical interfaces names passed in from device_script APIs
    @param sts: dict
        to identify the STS external and internal tag
    @param features: bit-mask from Features
        to indicate what part of configurations to generate
    @return: list of CLIs
    '''
    ifc_delta_cfg = generate_ifc_delta_cfg(ifc_cfg, asa_cfg, device, interfaces, features)
    return ifc2asa(ifc_delta_cfg, device, interfaces, sts, features, is_audit=True)

class InterfaceLabel(object):
    'The enumeration of the interface labels defined in device_specifciation.xml'
    External = "ext"
    Internal = "int"
    Mgmt     = "mgmt"
    Utility  = "util"
    FailoverLan  = "lan"   # The LAN based failover connection between the two failover devices
    FailoverLink = "link"  # The link(stateful) failover connection between the two failover devices
    ClusterControLink = "ccl"

class DeviceModel(DMObject):
    '''
    This is the complete representation of the ASA configuration in IFC model. The structure should be the same as
    device_specification.xml. Namely, at the top level, we have the
        - MFunc: the function configuration. In our case, it is "firewall"
        - MGrpCfg: the configuration shared by different functions.
          Not sure what it mean for ASA, since we have only one function now, that is "firewall".
        - MDevCfg: the configuration shared by all groups.
    The order to register the components is important, like in ASDM: that is the order to translate configuration
    from IFC format to ASA format.

    NOTE:
    The configuration structure of the device_specification.xml is based on "Insieme Service Management Integration"
    version 0.2. It is not quite in the final state yet, and is very likely to change.
    '''

    def __init__(self, device = None, interfaces = None, features = Features.vnsAll, is_audit = False):
        '''
        @param device: dict
            to identify the ASA device, passed in from device_script APIs
        @param interfaces: dict
            physical interfaces names passed in from device_script APIs
        @param features: bit-mask from Features
            to indicate what part of configurations to generate
        @param is_audit: boolean
            to indicate if this is used in a xxxxAudit device script API
        '''
        DMObject.__init__(self, ifc_key = DeviceModel.__name__)
        self.device = device
        self.interfaces = interfaces
        self.is_audit = is_audit
        self.sts_table = {}
        self.label2nameif = {} #cache of label to nameif map

        'Stuff defined under vnsClusrterCfg > vnsDevCfg section of device_specification.xml'
        if (features & Features.vnsDevCfg) and not DeviceModel.IS_LITE:
            self.register_child(ClusterConfig())
            self.register_child(FailoverConfig(self.is_multi_mode_asa()))
        """
        All the stuff defined in vnsClusterCfg section, except those in vnsDevCfg sub-section,
        of device_specification.xml'
        """
        if features & Features.vnsClusterCfg:
            self.register_child(PortChannelMembers(self.is_multi_mode_asa()))
            self.register_child(LACPMaxBundles(self.is_multi_mode_asa()))
            if not DeviceModel.IS_LITE:
                self.register_child(LoggingConfig())
                self.register_child(Arp())
                self.register_child(Timeouts())
                self.register_child(TrustSec())
                self.register_child(BasicThreatDetection())
                self.register_child(AdvancedThreatDetection())
                self.register_child(ScanningThreatDetection())
                self.register_child(NetFlowObjects())
                self.register_child(IPAudit())
                self.register_child(GlobalTCPOptions())
                self.register_child(NTP(self.is_multi_mode_asa()))
                self.register_child(DNS())
                self.register_child(SameSecurityTraffic())
                self.register_child(SmartCallHome(self.is_multi_mode_asa()))
                self.register_child(GlobalServicePolicyContainer())
                self.register_child(EmptyObjectsList())
                self.register_child(SNMP())
                self.register_child(Banner())

        'Child for vsnGrpCfg element'
        if features & Features.vnsMFunc:
            'The following items are passed to vnsMFunc at runtime'
            self.register_child(Vifs())
            self.register_child(DMList('VLAN', Vlan))
            self.register_child(DMList('ENCAPASS', EncapAss))

            'Configuration from vnsMDevCfg'
            if not DeviceModel.IS_LITE:
                self.register_child(NetworkObjects());
                self.register_child(ServiceObjects())
                self.register_child(NetworkObjectGroups())
                self.register_child(ServiceObjectGroups())
                self.register_child(SecurityObjectGroups())
                self.register_child(TimeRanges())
                self.register_child(AccessListList())
            else:#for dynamic EPG update in ASA-DP-FI
                self.register_child(NetworkObjectGroups())

            'vnsMGrpCfg under which there is vnsMFunc'
            self.register_child(SharedConfigs())

            'Configuration from vnsMDevCfg'
            'This needs to be generated after connectors are processed'
            if not DeviceModel.IS_LITE:
                self.register_child(GlobalS2SVPNConfiguration())
            self.register_child(BridgeGroupIntfs())
            self.register_child(Interfaces())
            if not DeviceModel.IS_LITE:
                self.register_child(NATListList())
            self.register_child(BGPs())
            self.register_child(OspfProcesses())

            if self.is_multi_mode_asa():
                self.register_child(ASAContext(self.get_asa_context_name()))

    def get_device(self):
        '@return the device parameter from device script API'
        return self.device

    def get_interfaces(self):
        """
        @attention:  It is recommended to use get_nameif instead of calling get_interfaces by applications.
        @return the interfaces parameter from device script API'
        """
        return self.interfaces

    def is_virtual(self):
        '@return True if this device is ASAv, otherwise false'
        return self.device.get('virtual', False) if self.device else False

    def is_router_mode(self):
        '@return True if this device is ASA in router mode, otherwise false'
        mode = self.device.get('funcmode', FunctionMode.GoThrough) if self.device else FunctionMode.GoThrough
        return mode == FunctionMode.GoTo

    def is_vlan_supported(self):
        """
        @return True if VLAN is supported , otherwise false.
        By default, we APIC does not support VLAN for ASAv. However if the flag 'tagPackets' exists in config parameter
        then we support it.
        """
        return not self.is_virtual() or self.delta_ifc_cfg_value.get('tagPackets', False)

    def get_failover_lan_interface(self):
        '''
        Get the interface for LAN based failover connection.
        '''
        return self.get_interface_name(InterfaceLabel.FailoverLan)

    def get_failover_link_interface(self):
        '''
        Get the interface for link(stateful) failover connection.
        '''
        return self.get_interface_name(InterfaceLabel.FailoverLink)

    def get_ClusterControLink_interface(self):
        '''
        Get cluster control link interface.
        '''
        return self.get_interface_name(InterfaceLabel.ClusterControLink)

    def get_mgmt_interface(self):
        return self.get_interface_name(InterfaceLabel.Mgmt)

    def get_interface_name(self, label):
        '@return the physical interface name of a given interface label'
        if not self.interfaces:
            return None
        if not label in (InterfaceLabel.External, InterfaceLabel.Internal,
                         InterfaceLabel.Mgmt, InterfaceLabel.Utility,
                         InterfaceLabel.FailoverLan, InterfaceLabel.FailoverLink,
                         InterfaceLabel.ClusterControLink):
            return None

        hits = filter(lambda item: item[1]['label'] == label, self.interfaces.iteritems())
        if not hits:
            return None
        hit = hits[0]

        if hit[1].has_key('cifs'):
            #hit is of the form {(Type.LIF, '', long_interface_label), {'cifs': {device_id: physical_interface_name, ...} , 'state': 0, 'label': label}}'
            values = hit[1]['cifs'].values()
            if values:
                interface_name = values[0]
            else:#should be rare, but it does happen. CSCup85594
                return None
        else:
            #hit is of the form {(Type.CIF, '', physical_interface_name), {'state': 0, 'label': label}}'
            interface_name = hit[0][2]
        interface_name = interface_name.replace('_', '/')
        return interface_name

    def get_nameif(self, label):
        '@return the nameif of a given interface label'
        if self.label2nameif.has_key(label):# try to lookup the cache
            return self.label2nameif[label]

        interface_name = self.get_interface_name(label)
        if not interface_name:
            self.log('%s not bind to a physical interface.' % label)
            self.label2nameif[label] = None
            return
        query_cmd = "show run interface " + interface_name + " | grep nameif"
        output_cli = self.query_asa(query_cmd)
        if output_cli and output_cli.strip().startswith('nameif '):
            nameif = output_cli.strip().split()[1]
            self.label2nameif[label] = nameif
            return nameif
        else:
            self.log('nameif for interface %s not configured.' % interface_name)
            self.label2nameif[label] = None

    def get_external_nameif(self):
        'short-cut to get nameif for external interface'
        return self.get_nameif(InterfaceLabel.External)

    def get_internal_nameif(self):
        'short-cut to get nameif for internal interface'
        return self.get_nameif(InterfaceLabel.Internal)

    def get_mgmt_nameif(self):
        'short-cut to get nameif for management interface'
        return self.get_nameif(InterfaceLabel.Mgmt)

    def get_utility_nameif(self):
        'short-cut to get nameif for utility interface'
        return self.get_nameif(InterfaceLabel.Utility)

    def iter_groups(self):
        groups = self.get_child('SharedConfig')
        if groups:
            for group in groups.children.itervalues():
                yield group

    def is_multi_mode_asa(self):
        """
        @see util.get_asa_context_information on how asa context information is stored in the device dictionary
        @return True if the targeted ASA is in multiple context mode, or False.
        """
        if not self.device:
            return False
        return self.device.get('asa_context') != None

    def is_fpr_device(self):
        """
        Determine if the device is an ASA blade in a FirePower chassis.
        Out of 'show version | grep ^Hardware':
        On ASA:      "Hardware:   ASA5585-SSP-10, 5969 MB RAM, CPU Xeon 5500 series 2000 MHz, 1 CPU (4 cores)"
        On FPR-2100: "Hardware:   FPR-2110, 6842 MB RAM, CPU MIPS 1200 MHz, 1 CPU (6 cores)"
        On FPR-4100: "Hardware:   FPR4K-SM-24, 116736 MB RAM, CPU Xeon E5 series 2194 MHz, 2 CPUs (48 cores)"
        """
        if not self.device:
            return None
        hardware = self.device.get('hardware')
        if not hardware:
            hardware = self.query_asa('show version | grep ^Hardware')
            if hardware:
                self.device['hardware'] = hardware
        if hardware:
            result =  hardware.split()[1].startswith('FPR')
            return result

    def get_asa_context_name(self):
        '@return the name of the context of ASA in multiple context mode'
        if not self.device:
            return None
        return self.device.get('asa_context').get('context_name')

    def get_all_context_names(self):
        '@return a list of all the context names of ASA in multiple context mode'
        return (self.device.get('asa_all_context_names', []) if self.device
                else [])

    def system_context_accessible(self):
        '@return True if the system context is accessible or False'
        if not self.device:
            return False
        if not self.device.get('asa_context'):
            return False;
        return self.device.get('asa_context').get('system_context_accessible')

    def is_admin_context(self):
        '@return True if the targeted ASA is admin context, or False'
        if not self.device:
            return False
        if not self.device.get('asa_context'):
            return False;
        return self.device.get('asa_context').get('is_admin_context')

    def is_user_context(self):
        '@return True if the targeted ASA is user context (non-admin context), or False'
        if not self.device:
            return False
        if not self.device.get('asa_context'):
            return False;
        return not self.device.get('asa_context').get('is_admin_context')

    def get_asa_context(self):
        return self.get_child('ASAContext')

    def get_asa_version(self):
        if self.device:
            if 'asa_version' not in self.device:
                populate_asa_version(self.device)
            return self.device['asa_version']
        return (0.0, 0, 0)

    def compare_asa_version(self, major, minor=0, build=0):
        '''
        Compare the specified version to the ASA version.

        @param major: float
            the major part of the version.  For 9.3(2)1, it is 9.3.
        @param minor: int
            the minor part of the version.  For 9.3(2)1, it is 2.
        @param build: int
            the build part of the version.  For 9.3(2)1, it is 1.
        @return int
             0 if the ASA version is the same as the specified version
            >0 if the ASA version is greater than the specified version
            <0 if the ASA version is less than the specified version
        '''

        return cmp(self.get_asa_version(), (major, minor, build))

DeviceModel.IS_LITE = True
