'''
Created on Oct 17, 2013
Copyright (c) 2013 by Cisco Systems

@author: dli
'''

from base.dmlist import DMList
from base.dmobject import DMObject
from connector import ExIntfConfigRelFolder, InIntfConfigRelFolder, Connector
from rule.nat_rule import NATPolicy
from state_type import State, Type
from translator.security_policy_assignment import SecurityPolicyAssignment

class Firewalls(DMList):
    'Container for Firewall translator, one per graph'
    def __init__(self):
        super(Firewalls, self).__init__(Firewall.__name__, Firewall)

    def register_child(self, dmobj):
        'Rid of NATPolicy in Firewall for asa-dp lite'
        DMList.register_child(self, dmobj)
        if self.get_top().IS_LITE:
            nat_policy = dmobj.get_child('NATPolicy')
            if nat_policy:
                dmobj.unregister_child(nat_policy)
        else:
            policy_assignment = dmobj.get_child('SecurityPolicyAssignment')
            if policy_assignment:
                dmobj.unregister_child(policy_assignment)

class Firewall(DMObject):
    '''
    This is the function configuration of ASA, assuming the name of "MFunc" element in the device_specifcation is "Firewall".
    Add firewall related configuration objects by call self.register_child(dmobj) in
    the constructor as it is done in DeviceModel.__init(self)__.
    '''

    INTF_CONFIG_ERROR = 'WARNING: %s missing.  This folder is required.'

    def __init__(self, instance):
        DMObject.__init__(self, instance)
        self.register_child(ExIntfConfigRelFolder())
        self.register_child(InIntfConfigRelFolder())
        self.register_child(DMList('CONN', Connector))
        self.register_child(NATPolicy())
        self.register_child(SecurityPolicyAssignment())

    def get_interface(self, connector_name):
        conn = self.get_connector(connector_name)
        if conn:
            return conn.interface

    def get_interfaces(self):
        'Returns a tuple of the external,internal interfaces'

        external = None
        internal = None

        rel = self.get_child('ExIntfConfigRelFolder').get_child('ExIntfConfigRel')
        if hasattr(rel, 'value'):
            external = rel.value
        rel = self.get_child('InIntfConfigRelFolder').get_child('InIntfConfigRel')
        if hasattr(rel, 'value'):
            internal = rel.value
        if external and internal:
            return external,internal

    def get_connector(self, connector_name):
        for conn in self.iter_connectors():
            if hasattr(conn, 'conn_type') and conn.conn_type == connector_name:
                return conn

    def iter_connectors(self):
        for connector in self.get_child('CONN').children.itervalues():
            yield connector

    def create_missing_ifc_delta_cfg(self):
        '''Override the default to take care of the way self.delta_ifc_key is created
        '''
        if  not self.has_ifc_delta_cfg():
            '@todo isolate changes to key creation'
            self.delta_ifc_key = Type.FUNC, Firewall.__name__, self.ifc_key,
            self.delta_ifc_cfg_value = {'state': State.NOCHANGE, 'value': {}}
            ancestor = self.get_ifc_delta_cfg_ancestor()
            if ancestor:
                ancestor.delta_ifc_cfg_value['value'][self.delta_ifc_key] =  self.delta_ifc_cfg_value
        for child in self.children.values():
            child.create_missing_ifc_delta_cfg()

    def validate_configuration(self):
        result = super(Firewall, self).validate_configuration()
        # Check that the interface configuration has been specified
        for intf in ('ExIntfConfigRelFolder', 'InIntfConfigRelFolder'):
            if not self.get_child(intf).has_ifc_delta_cfg():
                self.log(self.INTF_CONFIG_ERROR % intf)
        return result
