'''
Created on Jun 18, 2014

@author: jeffryp

Copyright (c) 2013 by Cisco Systems, Inc.
All rights reserved.
'''

import re

from asaio.cli_interaction import ignore_info_response_parser
from asaio.cli_interaction import ignore_response_parser
from base.dmlist import DMList
from base.dmobject import DMObject
from base.simpletype import SimpleType
from connector import IPv6EnforceEUI64
from interface_config import InterfaceConfig
from logging import LoggingHost
from rule.access_rule import AccessGroups
from service_policy import InterfaceServicePolicyContainer
from state_type import State, Type
from static_route import StaticRoute
from structured_cli import convert_to_structured_commands, StructuredCommand
from tcp_options import TCPOptions
from translator.s2svpn import S2SVPNPolicyList, S2SVPNEnabler
from validators import Validator
from utils.util import set_cfg_state, filter_first,\
    normalize_ipv4_address_and_mask_4_asa, ifcize_param_dict
from snmp import Host as SNMPHost

class Interface(DMObject, Validator):
    '''
    Interface related configuration

    Includes interface sub-commands (e.g. IP address) and configuration that
    refers to an interface (e.g. access group).
    '''

    MY_CLI_PATTERN = None

    def __init__(self, instance):
        super(Interface, self).__init__(instance, asa_key='interface')
        self.interface_config = InterfaceConfig()

        # Must be in the order in which to generate the CLI
        self.register_child(self.interface_config)
        self.register_child(InterfaceLoggingHosts())
        self.register_child(DNSLookup())
        self.register_child(MTU())
        self.register_child(AccessGroups())
        self.register_child(HealthMonitoring())
        self.register_child(IPv6EnforceEUI64())
        self.register_child(StaticRoute())
        self.register_child(TCPOptions())
        self.register_child(InterfaceServicePolicyContainer())
        self.register_child(ICMPRuleList())
        self.register_child(CutThroughProxyList())
        self.register_child(S2SVPNPolicyList())
        self.register_child(S2SVPNEnabler())
        self.register_child(InterfaceSNMPHosts())
        self.cli2dmobject = self.build_cli2dmobject()

    def allow_sub_commands(self):
        '''
        Returns True if sub-commands can be generated.

        There are cases when changes are made to an interface, but there is no
        VIF/VLAN information for the interface (see CSCuw63132 for an example).
        This function is used by commands which are a sub-command of the
        interface to determine if they can be generated.
        '''
        return hasattr(self, 'mode_command')

    def build_cli2dmobject(self):
        'Return a dictionary of CLI prefix to the DMObject that can translate that CLI'
        result = {}
        for child in self.children.itervalues():
            cli_prefixes = child.get_cli_prefixes()
            for cli_prefix in cli_prefixes:
                if result.has_key(cli_prefix):
                    result[cli_prefix].append(child)
                else:
                    result[cli_prefix] = [child]
        return result

    def create_missing_ifc_delta_cfg(self):
        'Override the default to take care of the way self.delta_ifc_key is created'

        if  not self.has_ifc_delta_cfg():
            '@todo isolate changes to key creation'
            self.delta_ifc_key = Type.FOLDER, self.parent.ifc_key, self.ifc_key,
            self.delta_ifc_cfg_value = {'state': State.NOCHANGE, 'value': {}}
            ancestor = self.get_ifc_delta_cfg_ancestor()
            if ancestor:
                ancestor.delta_ifc_cfg_value['value'][self.delta_ifc_key] =  self.delta_ifc_cfg_value

        for child in self.children.values():
            child.create_missing_ifc_delta_cfg()

    def diff_ifc_asa(self, cli):
        # Save the vif and vlan in the value dictionary
        self.delta_ifc_cfg_value.update(self.parse_cli(cli))

        # If there is no connector using this interface, then remove it
        if not self.is_used_by_conn():
            self.delta_ifc_cfg_value['state'] = State.DESTROY
        else:
            top = self.get_top()
            if not top.is_vlan_supported():
                # Update the Vif
                command = cli.command
                vif = top.get_child('VIF').get_translator(command)
                if vif:
                    vif.diff_ifc_asa(command)

            encapass = self.get_encapass()
            if encapass:
                encapass.set_state(State.NOCHANGE)

            self.get_child('InterfaceConfig').diff_ifc_asa(cli)
            self.set_state(self.get_state_recursive())

    def get_encapass(self):
        if hasattr(self, 'encapass'):
            return self.encapass

    def get_ipv4_address_dict(self):
        'Get a dictionary containing the IPv4 address, netmask, and state'
        return self.interface_config.get_ipv4_address_dict()

    def get_ospf_area_id(self):
        return self.interface_config.get_ospf_area_id()

    def get_ospfv3(self):
        return self.interface_config.get_ospfv3()

    def get_ospfvif_state(self):
        return self.interface_config.get_ospfvif_state()

    def get_translator(self, cli):
        if self.is_my_cli(cli):
            return self

        # Get the children that can translate this CLI
        if isinstance(cli, StructuredCommand):
            cli = cli.command
        cli_prefix = cli.split()[0]
        children = self.cli2dmobject.get(cli_prefix, None)
        if children:
            for child in children:
                result = child.get_translator(cli)
                if result:
                    return result

    def is_vlan_id_changed(self):
        '@return True if the VLAN ID for this interface is changed, or False'
        if not self.get_top().is_vlan_supported():
            return #N/A
        encapass = self.get_encapass()
        if not encapass:
            return
        vlan_translator = self.get_top().get_child('VLAN').get_child(encapass.encap)
        assert(vlan_translator.tag == self.vlan)
        return vlan_translator.get_state() == State.MODIFY

    def get_interface_name_by_nameif(self, nameif):
        '@return the name of the physical interface which has the given nameif'
        asa_config = self.query_asa('show run interface')
        if not asa_config:
            return
        asa_config = convert_to_structured_commands(asa_config.strip().split('\n'))
        hit = filter_first(lambda cli: isinstance(cli, StructuredCommand) and
                           filter_first(lambda cmd: str(cmd) == 'nameif ' + nameif, cli.sub_commands),
                           asa_config)
        if hit != None:
            return hit.command.split()[-1]

    def remove_interface_with_old_vlan_id(self, no_asa_cfg_stack):
        """
        Generate CLI to rid of the interface with the old VLAN ID.
        Since we do not have the old VLAN ID, we use nameif instead to look it up from the device instead.
        """
        old_interface_name = self.get_interface_name_by_nameif(self.nameif)
        if not old_interface_name:
            return
        self.generate_cli(no_asa_cfg_stack,
                          'clear config interface ' + old_interface_name,
                          mode_command=None,
                          response_parser=ignore_response_parser,
                          is_system_context = self.get_top().is_multi_mode_asa())

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        # If the ENCAPASS for this interface has changed, then the interface
        # needs to be recreated with the new vlan.
        # So is the case when the VLAN ID is changed
        if self.get_state() != State.CREATE:
            encapass = self.get_encapass()
            # Check if this interface is already configured on the ASA
            if encapass and not self.is_configured_on_asa():
                if encapass.get_state() == State.CREATE:
                    set_cfg_state(self.delta_ifc_cfg_value, State.CREATE)
                if self.is_vlan_id_changed():
                    self.remove_interface_with_old_vlan_id(no_asa_cfg_stack)
                    set_cfg_state(self.delta_ifc_cfg_value, State.CREATE)

        self.syncWithIntfConfigRel()
        vlan_used = self.get_top().is_vlan_supported()
        if not hasattr(self, 'vif'):
            # If there are no connectors using this interface, get the vif and
            # vlan from the value dictionary.
            #
            # This is the case where the interface is on the ASA but on on APIC.
            if self.has_ifc_delta_cfg() and 'vif' in self.delta_ifc_cfg_value:
                self.vif = self.delta_ifc_cfg_value['vif']
                if 'vlan' in self.delta_ifc_cfg_value:
                    self.vlan = self.delta_ifc_cfg_value['vlan']
                    # This is the case where VLAN is not supported for ASAv
                    # but there is interface on the ASA with VLAN. This can
                    # happen if administrator switches the tagPackets flag from true to false
                    # after a graph is deployed.
                    if self.get_state() == State.DESTROY and not vlan_used and self.get_top().is_virtual():
                        vlan_used = True
                elif self.get_state() == State.DESTROY and vlan_used and self.get_top().is_virtual():
                    # This is the case where VLAN is supported for ASAv
                    # but there is interface on the ASA without VLAN. This can
                    # happen if administrator switches the tagPackets flag from false to true
                    # after a graph is deployed.
                    vlan_used = False
            elif self.get_state() == State.MODIFY:
                # Generate interface related commands which are not sub-commands
                super(Interface, self).ifc2asa(no_asa_cfg_stack, asa_cfg_list)
                return
            else:
                # No interface information to generate CREATE/DESTROY commands
                return

        if not vlan_used:
            interface_command = 'interface '+ self.vif
        else:
            if not hasattr(self, 'vlan'):
                # Missing interface information needed to generate commands
                return
            interface_command = 'interface %s.%s' % (self.vif, self.vlan)

        top = self.get_top()
        is_multi_mode = top.is_multi_mode_asa()
        asa_context = top.get_asa_context()
        self.mode_command = interface_command
        tmp_asa_cfg_list = []
        if self.get_state() == State.DESTROY:
            # Clear interface configuration
            if is_multi_mode:
                asa_context.allocate_interface(interface_command.split()[1], False)
            else:
                self.generate_cli(no_asa_cfg_stack,
                                  'clear config ' + interface_command,
                                  mode_command=None,
                                  response_parser=ignore_response_parser)
        elif self.get_state() == State.CREATE:
            if vlan_used:
                if is_multi_mode:
                    asa_context.allocate_interface(interface_command.split()[1])
                self.generate_cli(asa_cfg_list, 'vlan ' + str(self.vlan),
                                  is_system_context = is_multi_mode)
            self.generate_cli(tmp_asa_cfg_list,
                              'nameif ' + self.nameif,
                              response_parser=ignore_info_response_parser)
            # Ensure the all the interface related commands are created when the
            # interface is created.  This is needed for the case where the VLAN
            # has changed.  The old interface is deleted (which also removes all
            # the interface related commands) and then a new interface is
            # created with the new VLAN.
            set_cfg_state(self.delta_ifc_cfg_value, State.CREATE)

        super(Interface, self).ifc2asa(no_asa_cfg_stack, tmp_asa_cfg_list)
        if not tmp_asa_cfg_list:
            return
        if tmp_asa_cfg_list[0].command == 'nameif ' + self.nameif:
            '''Make sure 'bridge-group' sub-command comes before 'nameif', or routed mode ASA 9.7(1) and 9.8(1) won't take it.
            '''
            bridge_group = filter_first(lambda clii: clii.command.startswith('bridge-group'), tmp_asa_cfg_list)
            if bridge_group:
                tmp_asa_cfg_list.remove(bridge_group)
                tmp_asa_cfg_list.insert(0, bridge_group)
        asa_cfg_list.extend(tmp_asa_cfg_list)

    def is_my_cli(self, cli):
        nameif = self.parse_nameif(cli)
        return nameif == self.nameif

    @staticmethod
    def get_interface_ref_name(fw, prefix):
        relation_folder = fw.get_child(prefix + 'IntfConfigRelFolder')
        relation = relation_folder.get_child(prefix + 'IntfConfigRel')
        if hasattr(relation, 'delta_ifc_cfg_value'):
            return relation.delta_ifc_cfg_value.get('target')

    def is_used_by_conn(self):
        'Check if any connector is using this interface'
        for group in self.get_top().iter_groups():
            for fw in group.iter_firewalls():
                if (self.get_interface_ref_name(fw, 'Ex') == self.nameif
                        or self.get_interface_ref_name(fw, 'In') == self.nameif):
                    return True

    @classmethod
    def parse_cli(cls, cli):
        result = {}
        m = cls.get_my_cli_pattern().match(cli.command.strip())
        if m:
            result['vif'] = m.group(1)
            vlan = m.group(2)
            if vlan:
                result['vlan'] = int(vlan)
        return result

    @property
    def nameif(self):
        'The instance name is used for the interface nameif'
        return self.ifc_key

    @classmethod
    def get_my_cli_pattern(cls):
        if not cls.MY_CLI_PATTERN:
            cls.MY_CLI_PATTERN = re.compile('interface (.+?)(?:\.(\d+))?$')
        return cls.MY_CLI_PATTERN

    @classmethod
    def parse_nameif(cls, cli):
        if (isinstance(cli, StructuredCommand)
                and cls.get_my_cli_pattern().match(cli.command.strip())):
            for cmd in cli.sub_commands:
                if isinstance(cmd, basestring) and cmd.startswith('nameif '):
                    nameif = cmd.split()[1]
                    if not nameif.lower().startswith('management'):
                        return nameif
    @classmethod
    def is_sacred_intf_cmd(cls, cli):
        'return True if this a management interface command, or utility interface, which has nameif management-utility'
        'also return True if it is interface used by failover or cluster'
        def is_sacred_interface(cmd):
            if isinstance(cmd, basestring):
                cmd = cmd.strip().lower()
                if cmd.startswith('nameif management'):
                    return True
                sacred_cmds = ['management-only',
                               'description state failover interface',
                               'description clustering interface',
                               'description lan failover interface']
                return cmd in sacred_cmds
        if (isinstance(cli, StructuredCommand) and cls.get_my_cli_pattern().match(cli.command.strip())):
            return any(map(is_sacred_interface, cli.sub_commands))

    def validate(self, value):
        if self.ifc_key.lower().startswith('management'):
            return ("The name cannot be '" + self.ifc_key +
                    "', it must not start with 'management'.")

    def get_my_relations(self):
        'Return list of IntfConfigRel targeting this interface'
        result = []
        for group in self.get_top().iter_groups():
            for fw in group.iter_firewalls():
                for folder in ['ExIntfConfigRelFolder', 'InIntfConfigRelFolder']:
                    rel = fw.get_child(folder).children.values()[0]
                    if rel.has_ifc_delta_cfg() and self.ifc_key == rel.delta_ifc_cfg_value.get('target'):
                        result.append(rel)
        return result

    def is_used_by_new_relation(self):
        'Return true if there is a IntfConfigRel targeting this Interface, and that IntfConfigRel is in CREATE state'
        relations = self.get_my_relations()
        return any(map(lambda rel: rel.get_state() == State.CREATE, relations))

    def is_configured_on_asa(self):
        'Return true if and only if this Interface is already configured on ASA device.'
        vlan_supported = self.get_top().is_vlan_supported()
        if not vlan_supported:
            interface_command = 'interface '+ self.vif
        else:
            interface_command = 'interface %s.%s' % (self.vif, self.vlan)
        show_cmd = 'show run %s | grep nameif %s' % (interface_command, self.ifc_key)
        response = self.query_asa(show_cmd)
        if not response:
            return False
        return response.strip() == 'nameif ' + self.ifc_key

    def syncWithIntfConfigRel(self):
        """
        1. CSCur25935: it is observed that the APIC can have the interface definition in NOCHANGE
        state, but IntfConfigRel is in CREATE state. This leads to missing CLIs.

        2. CSCuu79558 With shared SG, redundant interface ip config causing ospf flap
        This means we need to see if the interface is configured on the ASA before creating it.

        3. CSCuz72495: the interface is in MODIFY state while being bound to IntfConfigRel. The interface
        folder is created in a serviceModify call before being bound to a connector.
        """
        if self.get_top().is_audit:
            return
        if self.get_state() in (State.CREATE, State.DESTROY):
            return
        if self.is_used_by_new_relation() and not self.is_configured_on_asa():
            set_cfg_state(self.delta_ifc_cfg_value, State.CREATE)

class Interfaces(DMList):
    'Container for the Interface object'

    def __init__(self):
        super(Interfaces, self).__init__(Interface.__name__, Interface)
        self.cli_prefixes = Interface('').get_cli_prefixes()

    def register_child(self, dmobj):
        'Rid of the MOs that are not supported by asa-dp lite'
        DMList.register_child(self, dmobj)
        if not self.get_top().IS_LITE:
            return
        for key in ['AccessGroup', 'dns_lookup', 'health_monitoring', 'LoggingHost', 'ServicePolicy',
                    'ICMPRule', 'CutThroughProxy', 'S2SVPNPolicy', 'S2SVPNEnabler']:
            child = dmobj.get_child(key)
            if child:
                dmobj.unregister_child(child)
        dmobj.cli2dmobject = dmobj.build_cli2dmobject()

    def get_translator(self, cli):
        if not self.is_my_cli(cli):
            return False

        if Interfaces.is_interface_cmd(cli):
            if Interface.is_sacred_intf_cmd(cli):
                'Do not touch management nor utility interface! CSCur44552'
                'They are configured out-band.'
                return
            instance = Interface.parse_nameif(cli)
            if not instance:
                'interface without nameif, this is for deletion'
                instance = self.get_uniq_fake_instance()
            interface = self.get_child(instance)
            if interface:
                # Check if the vlan has changed
                if hasattr(interface, 'vif'):
                    vif_vlan = Interface.parse_cli(cli)
                    if ((interface.vif != vif_vlan['vif']) or
                        ('vlan' in vif_vlan and hasattr(interface, 'vlan') and interface.vlan != vif_vlan['vlan']) or
                        ('vlan' in vif_vlan and not self.get_top().is_vlan_supported()) or
                        ('vlan' not in vif_vlan and self.get_top().is_vlan_supported())):
                        # Delete the old interface
                        interface = None
                        instance = self.get_uniq_fake_instance()
            if not interface:
                interface = Interface(instance)
                self.register_child(interface)
                interface.create_missing_ifc_delta_cfg()
            return interface
        else:
            for child in self.children.itervalues():
                if hasattr(child, 'get_translator'):
                    result = child.get_translator(cli)
                    if result:
                        return result

    def is_my_cli(self, cli):
        cli_prefix = cli.split()[0] if isinstance(cli, basestring) else cli.command.split()[0]
        return cli_prefix in self.cli_prefixes

    def get_uniq_fake_instance(self):
        'Get a fake ifc_key for an interface that does not have nameif. This is used for clear operation during audit'
        n = 0
        while True:
            result = '?faked_nameif' + str(n)
            if not self.get_child(result):
                return result
            n += 1

    @classmethod
    def is_interface_cmd(cls, cli):
        """
        return true if the command is an interface command, excluding 'interface bvi' command
        which is handled by BridgeGroupIntfs.
        """
        main_cmd = (cli.command if isinstance(cli, StructuredCommand) else cli).strip()
        return Interface.get_my_cli_pattern().match(main_cmd) and not main_cmd.split()[1].startswith('BVI')

class HealthMonitoring(DMObject):
    'Health monitoring (monitor-interface) configuration'

    def __init__(self):
        super(HealthMonitoring, self).__init__('health_monitoring',
                                               'monitor-interface')

    def diff_ifc_asa(self, cli):
        if not self.has_ifc_delta_cfg():
            # Set the CLI to the default if it isn't already.  Physical
            # interfaces are enabled by default; logical interfaces are disabled
            # by default.
            cli_value = self.parse_cli(cli)
            default = 'enable' if not self.get_top().is_vlan_supported() else 'disable'
            if cli_value != default:
                self.delta_ifc_key = (Type.PARAM, self.ifc_key, '')
                self.delta_ifc_cfg_value = {
                    'state': State.CREATE,
                    'value': default}
                ancestor = self.get_ifc_delta_cfg_ancestor()
                if ancestor:
                    ancestor.delta_ifc_cfg_value['value'][self.delta_ifc_key] =  self.delta_ifc_cfg_value
        elif self.is_the_same_cli(cli):
            self.set_state(State.NOCHANGE)
        else:
            self.set_state(State.MODIFY)

    def get_cli(self):
        return self.asa_key + ' ' + self.parent.nameif

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        # The ASA will remove the health monitoring CLI when the interface is
        # removed
        if (self.has_ifc_delta_cfg() and
                self.parent.get_state() != State.DESTROY):
            state = self.get_state()
            if self.get_value() == 'disable':
                if state in (State.CREATE, State.MODIFY):
                    state = State.DESTROY
                elif state == State.DESTROY:
                    state = State.CREATE
            if state in (State.CREATE, State.MODIFY):
                self.generate_cli(asa_cfg_list, self.get_cli())
            elif state == State.DESTROY:
                self.generate_cli(asa_cfg_list, 'no ' + self.get_cli())

    def is_my_cli(self, cli):
        if isinstance(cli, StructuredCommand):
            cli = cli.command
        return cli.strip() == self.get_cli()

    def is_the_same_cli(self, cli):
        return self.get_value() == self.parse_cli(cli)

    @staticmethod
    def parse_cli(cli):
        return ('disable' if isinstance(cli, StructuredCommand) and
                cli.is_no else 'enable')

class InterfaceLoggingHost(LoggingHost):
    'Interface logging host configuration'

    def __init__(self, instance):
        super(InterfaceLoggingHost, self).__init__(instance)

    def get_value_with_interface(self):
        value = self.get_value()
        value['interface'] = self.parent.nameif
        return value

    def remove_deleted(self):
        'Remove any parameters that are to be deleted'
        cfg_value = self.delta_ifc_cfg_value['value']
        for key, child in cfg_value.items():
            if child.get('state') == State.DESTROY:
                del(cfg_value[key])

class InterfaceLoggingHosts(DMList):
    'Container for the InterfaceLoggingHost object'

    def __init__(self):
        super(InterfaceLoggingHosts, self).__init__('LoggingHost',
                                                    InterfaceLoggingHost,
                                                    'logging host')

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        # The logging host configuration will be removed when the interface is
        # removed
        if self.parent.get_state() != State.DESTROY:
            if (not self.get_top().is_audit and
                    filter_first(
                        lambda child: child.get_state() == State.MODIFY, self)):
                # Remove any entries or parameters to be deleted
                for key, child in self.children.items():
                    if child.get_state() == State.DESTROY:
                        del(self.children[key])
                    else:
                        child.remove_deleted()
                # Change the remaining entries to CREATE
                for child in self:
                    child.set_state(State.CREATE)

                self.mini_audit_command = ('show run logging | grep logging host ' + self.nameif)
                self.mini_audit()

            for child in self:
                child.ifc2asa(no_asa_cfg_stack, asa_cfg_list)

    def is_my_cli(self, cli):
        if isinstance(cli, basestring):
            return cli.startswith(self.asa_key + ' ' + self.nameif)

    @property
    def nameif(self):
        return self.parent.nameif

class InterInterface(SimpleType):
    'Same security traffic inter-interface configuration'

    def __init__(self):
        super(InterInterface, self).__init__(
            'inter_interface',
            'same-security-traffic permit inter-interface')

    def get_cli(self):
        return self.asa_key

class IntraInterface(SimpleType):
    'Same security traffic intra-interface configuration'

    def __init__(self):
        super(IntraInterface, self).__init__(
            'intra_interface',
            'same-security-traffic permit intra-interface')

    def get_cli(self):
        return self.asa_key

class MTU(SimpleType):
    'Interface MTU configuration'

    ASA_KEY = 'mtu'
    CLI_PATTERN = None
    DEFAULT = '1500'

    def __init__(self):
        super(MTU, self).__init__(MTU.__name__, self.ASA_KEY)

    def diff_ifc_asa(self, cli):
        # If there is no MTU configuration and the ASA value is the default,
        # then leave it alone
        if self.has_ifc_delta_cfg() or self.parse_cli(cli) != self.DEFAULT:
            super(MTU, self).diff_ifc_asa(cli)

    def get_cli(self):
        return self.get_cli_prefix() + ' ' + self.get_value()

    def get_cli_pattern(self):
        if not self.CLI_PATTERN:
            self.CLI_PATTERN = re.compile(self.ASA_KEY + ' \S+ (\d+)')
        return self.CLI_PATTERN

    def get_cli_prefix(self):
        return self.ASA_KEY + ' ' + self.parent.nameif

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        # The MTU configuration will be removed when the interface is removed
        if (self.has_ifc_delta_cfg() and
                self.parent.get_state() != State.DESTROY):
            state = self.get_state()
            if state in (State.CREATE, State.MODIFY):
                self.generate_cli(asa_cfg_list, self.get_cli())
            elif state == State.DESTROY:
                self.generate_cli(asa_cfg_list,
                                  self.get_cli_prefix() + ' ' + self.DEFAULT)

    def is_my_cli(self, cli):
        return cli.startswith(self.get_cli_prefix())

    def parse_cli(self, cli):
        if isinstance(cli, basestring):
            m = self.get_cli_pattern().match(cli)
            if m:
                return str(m.group(1))

class SameSecurityTraffic(DMObject):
    'Same security traffic configuration'

    def __init__(self):
        super(SameSecurityTraffic, self).__init__(SameSecurityTraffic.__name__)
        self.register_child(InterInterface())
        self.register_child(IntraInterface())

class DNSLookup(SimpleType):
    'Enable DNS look up from the parent interface'
    def __init__(self):
        SimpleType.__init__(self, 'dns_lookup', 'dns domain-lookup')

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        """
        The actual value entered by user is not used, so we do nothing for State.MODIFY.
        """
        if delta_ifc_cfg_value['state'] == State.MODIFY:
            delta_ifc_cfg_value['state'] = State.NOCHANGE
        """
        No need to issue 'no dns domain-lookup <nameif>' command if the interface is cleared.
        Because 'clear config interface ..' will clear all configurations referencing it.
        Otherwise these commands will be rejected by ASA any way, as the 'dns domain-lookup <nameif>' are already gone.
        CSCuw60753
        """
        if self.parent.get_state() == State.DESTROY:
            delta_ifc_cfg_value['state'] = State.NOCHANGE
        return SimpleType.populate_model(self, delta_ifc_key, delta_ifc_cfg_value)

    def get_cli(self):
        return self.get_asa_key() + ' ' + self.parent.nameif

    def is_my_cli(self, cli):
        return cli == self.get_cli()

class ICMPRuleList(DMList):
    def __init__(self):
        super(ICMPRuleList, self).__init__(ICMPRule.__name__, ICMPRule, 'icmp')

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        '''
        Override the default so that we order the rules according to the ifc_keys
        '''
        if self.parent.get_state() == State.DESTROY:
            return #ICMP commands removed by ASA if the interface is removed
        if filter_first(lambda child: child.get_state() == State.MODIFY, self):
            # mini audit
            for key, child in self.children.items():
                if child.get_state() == State.DESTROY:
                    del(self.children[key])
            self.mini_audit_command = 'show run icmp | grep ' + self.parent.ifc_key
            self.mini_audit()
        # generate the CLI ordered by the ifc_key
        child_values = self.children.values()
        child_values.sort(key=lambda value: value.ifc_key)
        for child in child_values:
            child.ifc2asa(no_asa_cfg_stack, asa_cfg_list)

    def get_translator(self, cli):
        '''
        Override the default implementation to take care of icmp CLI for this interface only
        '''
        if isinstance(cli, basestring) and cli.endswith(' ' + self.parent.ifc_key):
            return DMList.get_translator(self, cli)

class ICMPRule(SimpleType):
    'ICMP permit/deny for the parent interface'
    def __init__(self, name):
        SimpleType.__init__(self, name, 'icmp', 'icmp %(action)s %(source_address)s %(nameif)s')

    def get_value(self):
        result = SimpleType.get_value(self)
        result['nameif'] = self.parent.parent.nameif
        address = result.get('source_address', 'any')
        if address != 'any':
            if '/' in address:
                address = normalize_ipv4_address_and_mask_4_asa(address)
            else:
                address = 'host ' + address
        result['source_address'] = address
        return result

    def is_my_cli(self, cli):
        return cli == self.get_cli()

    def parse_multi_parameter_cli(self, cli, alternate_asa_gen_template=None):
        '''
        CLI format:
          icmp [deny|permit] [any | host <address> | <address> <mask>] <nameif>
        '''
        tokens = cli.split()
        result = {}
        result['action'] = tokens[1]
        if tokens[2] == 'any':
            result['source_address'] = 'any'
        elif tokens[2] == 'host':
            result['source_address'] = tokens[3]
        else:
            result['source_address'] = tokens[2] + '/' + tokens[3]
        return ifcize_param_dict(result)

class CutThroughProxyList(DMList):
    def __init__(self):
        super(CutThroughProxyList, self).__init__(CutThroughProxy.__name__, CutThroughProxy,
                                              'aaa authentication match',
                                              'show run aaa authentication | grep match')

    def get_translator(self, cli):
        if not isinstance(cli, basestring):
            return
        tokens = str(cli).split()
        if tokens[-2] == self.parent.ifc_key:#check for nameif
            return DMList.get_translator(self, cli)

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        if self.parent.get_state() != State.DESTROY:
            #ICMP commands removed by ASA if the interface is removed
            #We do not need to generate explicit CLIs to remove them.
            DMList.ifc2asa(self, no_asa_cfg_stack, asa_cfg_list)

class CutThroughProxy(SimpleType):
    'CLI: aaa authentication match <access_list_name> <nameif> <aaa_server_group>'
    def __init__(self, name):
        SimpleType.__init__(self, name, 
                            'aaa authentication match',
                            'aaa authentication match %(access_list_name)s %(nameif)s %(aaa_server_group)s')

    def get_value(self):
        result = SimpleType.get_value(self)
        result['nameif'] = self.parent.parent.nameif
        result['aaa_server_group'] = result.get('aaa_server_group', 'LOCAL')
        return result

    def is_my_cli(self, cli):
        return cli == self.get_cli()

class InterfaceSNMPHosts(DMList):
    '''Model after SNMP hosts bound to a given connector
    '''
    def __init__(self):
        DMList.__init__(self, 'SNMPHost', InterfaceSNMPHost, 'snmp-server host')

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        # The snmp host configuration will be removed when the interface is
        # removed
        if self.parent.get_state() == State.DESTROY:
            return
        self.mini_audit_command = 'show run snmp-server | grep host ' + self.nameif
        DMList.ifc2asa(self, no_asa_cfg_stack, asa_cfg_list)

    def is_my_cli(self, cli):
        if isinstance(self.asa_key, basestring):
            self.asa_key = re.compile('^snmp-server host ' + self.nameif + ' .+ version 3')
        return DMList.is_my_cli(self, cli)

    @property
    def nameif(self):
        return self.parent.nameif

class InterfaceSNMPHost(SNMPHost):
    '''Model after SNMP host bound to a given connector
    '''
    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        delta_ifc_cfg_value['value'][(Type.PARAM, 'interface', '')] = {'state': self.get_state(), 'value': self.parent.parent.nameif}
        SNMPHost.populate_model(self, delta_ifc_key, delta_ifc_cfg_value)
