'''
Created on Jun 27, 2014

This module provides containers for network objects and service objects.
@author: dli
'''

from collections import OrderedDict
from base.dmobject import DMObject
from base.dmlist import DMList
from network_object import HostObjects, SubnetObjects, RangeObjects, FQDNObjects
from service_object import ICMP4Objects, ICMP6Objects, ProtocolObjects, TCPObjects,UDPObjects

class NSContainer(DMObject):
    '''
    Base class for network objects and service objects container.
    It is introduced so that we can re-use the existing code in network_object and service_object module.

    For APIC, all the network objects of types: HostObject, SubnetObject, RangeObject, FQDNObject, are under
    the same folder. However, in the device script, we have a container for each type of network object, e.g.
    HostObjects contains all instances of type HostObject.

    The same applies to service objects.
    '''
    def __init__(self, ifc_key, child_classes):
        '''
        @param child_classes: list of classes of the objects
        '''
        DMObject.__init__(self, ifc_key=ifc_key)
        for child_class in child_classes:
            self.register_child(child_class())

        "field_name2ifc_key maps 'host_ip_address' to 'HostObject' etc."
        self.field_name2ifc_key = OrderedDict()
        for child in self.children.values():
            instance = child.child_class('dummy')
            field_name = instance.children.keys()[1]
            self.field_name2ifc_key[field_name] = child.ifc_key

        types = tuple(self.field_name2ifc_key.keys())
        self.register_child(InvalidObjects(types))

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        '''Override the default to delegate to the proper DMObject.
        NetworkObject can be one of: HostObject, SubnetObject, RangeObject, FQDNObject.
        ServiceObject can be one of: ICMP4Object, ICMP6Object, ProtocolObject, TCPObject, UDPObject.
        '''
        value = delta_ifc_cfg_value['value']
        if not value:
            return

        if not delta_ifc_key[2]:
            '''
            This is due to delete operation(s) from audit. i.e. the configuration is in ASA but not in APIC,
            so we need to delete it.
            The configuration generated from the diff operation looks something like:
                 { (4, 'NetworkObject', ''): {
                   'state':1,
                    value: {
                       {(5, 'HostObject', 'one'):     {'state':3, 'value': {...}},
                       {(5, 'FqdnObject', 'two'):     {'state':3, 'value': {...}},
                       {(5, 'SubnetObject', 'three'): {'state':3, 'value': {...}},
                       ....
                    }
                   }
                 }
            '''
            'Take care of deletion, multiple deletions possible'
            for child_key, child_value in value.iteritems():
                child = self.get_child(child_key[1])
                child.populate_model(child_key, child_value)
            return

        child_keys = filter(lambda key: value[key] and key[1] != 'name' and key[1] != 'description', value.keys())
        if len(child_keys) == 1:
            key = child_keys[0][1]
            child_key = self.field_name2ifc_key.get(key, key)
            child = self.get_child(child_key)
        else:
            child = self.children[InvalidObjects().ifc_key]
        return child.populate_model(delta_ifc_key, delta_ifc_cfg_value)

    def get_config_path(self):
        'override the default implementation to make DMObject invisible in the fault path '
        return self.parent.get_config_path()

    def create_missing_ifc_delta_cfg(self):
        'Override the default implementation to not create folder, as we are only a container here, like DMList.'
        return

class ServiceObjects(NSContainer):
    def __init__(self):
        NSContainer.__init__(self, 'ServiceObject',
                             (ICMP4Objects, ICMP6Objects, ProtocolObjects, TCPObjects, UDPObjects))

class NetworkObjects(NSContainer):
    def __init__(self):
        NSContainer.__init__(self, 'NetworkObject',
                             (HostObjects, SubnetObjects, RangeObjects, FQDNObjects))

class InvalidObjects(DMList):
    'Container for InvalidObect'
    def __init__(self, types = None):
        '@param types: list of strings to indicate the valid types of objects'
        DMList.__init__(self, 'InvalidObject', InvalidObject)
        self.types = types

class InvalidObject(DMObject):
    '''This used for validation only
    '''
    def is_my_cli(self, cli):
        return False

    def validate_configuration(self):
        'Override the default implementation o signal error.'
        value = self.delta_ifc_cfg_value['value']
        child_keys = filter(lambda key: value[key] and key[1] != 'name' and key[1] != 'description', value.keys())
        child_keys = [ k for (t, k, i) in child_keys]
        if child_keys:
            error = 'Invalid configuration: only one field in %s can be set' % str(child_keys)
        else:
            error = 'Invalid configuration: one of the fields %s must be set' % str(self.parent.types)
        return [self.generate_fault(error)]
