'''
Created on Oct 28, 2013

@author: jeffryp

Copyright (c) 2013 by Cisco Systems, Inc.
All rights reserved.

Classes used for NAT rules.
'''

import re
import copy

from asaio.cli_interaction import ignore_response_parser
from asaio.cli_interaction import ignore_warning_response_parser
import translator
from translator.state_type import State, Type
from translator.base.dmlist import DMList
from translator.base.dmobject import DMObject
from utils.util import filter_first, ifcize_param_dict , set_cfg_state

class NATList(DMList):
    'A list of NAT rules'

    DYNAMIC_CLI_PATTERN = None
    STATIC_CLI_PATTERN = None

    def __init__(self, instance):
        super(NATList, self).__init__(instance, NATRule, 'nat')
        self.clear_translation_state = None
        self.diff_ifc_asa_nat_order = 1
        self.firewall = None
        self.interfaces = None
        self.nat_changed = False

    def create_missing_ifc_delta_cfg(self):
        if not self.has_ifc_delta_cfg():
            self.delta_ifc_key = Type.FOLDER, self.parent.ifc_key, self.ifc_key,
            self.delta_ifc_cfg_value = {'state': State.NOCHANGE, 'value': {}}
            ancestor = self.get_ifc_delta_cfg_ancestor()
            if ancestor:
                ancestor.delta_ifc_cfg_value['value'][self.delta_ifc_key] =  self.delta_ifc_cfg_value

    def diff_ifc_asa(self, cli):
        values = self.parse_cli(cli, self.diff_ifc_asa_nat_order)
        if values:
            if self.get_state() != State.DESTROY:
                self.set_state(State.MODIFY)

            nat = NATRule(cli)
            nat.values = values
            self.diff_ifc_asa_nat_order += 1

            match_nat = self.find_nat(nat)
            if match_nat:
                if self.get_state() != State.DESTROY:
                    match_nat.set_state(State.NOCHANGE)
            else:
                # Delete the NAT if it doesn't exist or is in a different order
                self.register_child(nat)
                nat.diff_ifc_asa(cli)

    def find_nat(self, nat):
        for match_nat in self.children.itervalues():
            if match_nat.values == nat.values:
                return match_nat

    def get_cli_prefix_pattern(self):
        if not self.cli_prefix_pattern:
            self.cli_prefix_pattern = re.compile(self.get_cli_prefix_re())
        return self.cli_prefix_pattern

    def get_cli_prefix_re(self):
        if not self.cli_prefix_re:
            interface_pattern = ('(' + self.interfaces[0] + '|' +
                                 self.interfaces[1] + ')')
            self.cli_prefix_re = ('nat \(' + interface_pattern + ',' +
                                        interface_pattern + '\)')
        return self.cli_prefix_re

    def get_config_path(self):
        '''
        Have to use the path of the template. As the actual instantiation does not exist in the original
        configuration dictionary from APIC. We therefore must use that of the template, which does
        exist there.
        See CSCvb67004.
        '''
        result =  DMList.get_config_path(self)
        template_ifc_key = self.delta_ifc_cfg_value.get('template_ifc_key')
        if template_ifc_key:
            result.pop()
            result.append(template_ifc_key)
        return result

    def get_connector_name(self, interface_name, default):
        if self.interfaces:
            if interface_name == self.interfaces[0]:
                return 'external'
            elif interface_name == self.interfaces[1]:
                return 'internal'
        return default

    def get_interface_name(self, connector_name):
        if self.interfaces:
            if connector_name == 'external':
                return self.interfaces[0]
            elif connector_name == 'internal':
                return self.interfaces[1]
        return ''

    def get_translator(self, cli):
        if (isinstance(cli, basestring) and
                self.get_cli_prefix_pattern().match(cli)):
            return self

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list, firewall=None,
                nat_policy_state=None, interfaces=None):
        '''
        Generate the CLI as follows:
            Create if either NAT policy or NAT list is create
            Destroy if either NAT policy or NAT list is destroy
            Modify if NAT list is modify
        '''

        # If either of the interfaces will be destroyed, then the NAT rules will
        # be removed by the ASA, so no need to destroy them
        if firewall:
                extintf = firewall.get_interface('external')
                intintf = firewall.get_interface('internal')
                if (extintf and extintf.get_state() == State.DESTROY or
                    intintf and intintf.get_state() == State.DESTROY):
                    return

        self.interfaces = interfaces if interfaces else firewall.get_interfaces()

        # Clear the cached CLI prefix pattern and regular expression since the
        # interfaces will be different for each call.
        self.cli_prefix_pattern = None
        self.cli_prefix_re = None

        old_config = []
        new_config = []
        if ((nat_policy_state and nat_policy_state == State.CREATE) or
                self.get_state() == State.CREATE):
            new_config.extend(list(self.children.itervalues()))
        elif ((nat_policy_state and nat_policy_state == State.DESTROY) or
                self.get_state() == State.DESTROY):
            old_config.extend(list(self.children.itervalues()))
        else:
            if filter_first(lambda child: child.get_state() == State.MODIFY,
                            self):
                # Remove any entries to be deleted
                for key, nat in self.children.items():
                    if nat.get_state() == State.DESTROY:
                        del(self.children[key])
                # Change the remaining entries to CREATE
                for nat in self.children.itervalues():
                    nat.set_state(State.CREATE)

                self.mini_audit_command = ('show running-config nat | grep ^' +
                                           self.get_cli_prefix_re())
                self.mini_audit()

            # Optimize generated CLI by looking for NAT rules that have changed
            for nat in self.children.itervalues():
                state = nat.get_state()
                if state == State.NOCHANGE:
                    old_config.append(nat)
                    new_config.append(nat)
                elif state == State.CREATE:
                    new_config.append(nat)
                elif state == State.DESTROY:
                    old_config.append(nat)

        # Sort the old and new config
        self.sort_config(old_config)
        self.sort_config(new_config)

        last_equal = -1
        for i, (old_nat, new_nat) in enumerate(zip(old_config, new_config)):
            if new_nat.equals_no_order(old_nat):
                last_equal = i
            else:
                break

        # Remove the old NAT rules which are different
        for nat in old_config[last_equal + 1:]:
            nat.generate_delete(no_asa_cfg_stack)

        # Add the new NAT rules which are different
        for nat in new_config[last_equal + 1:]:
            nat.generate_command(asa_cfg_list)

        if (self.clear_translation_state in (State.CREATE, State.MODIFY, State.NOCHANGE)
                and self.nat_changed):
            # If translations are already cleared for ACLs, then not needed for NAT
            access_list_list = self.get_top().get_child('AccessList')
            if not access_list_list or not access_list_list.clear_xlate:
                self.generate_cli(asa_cfg_list, 'clear xlate interface ' +
                                  self.get_interface_name('external'))

    def normalize_order(self):
        # Normalize the order values so they start with '1' and increase by '1'.
        # This is needed to correctly match the CLI during an audit.
        rules = self.children.values()
        self.sort_config(rules)
        for i, nat in enumerate(rules, 1):
            nat.values['order'] = str(i)

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        self.delta_ifc_key = delta_ifc_key
        self.delta_ifc_cfg_value = delta_ifc_cfg_value
        for key, value in delta_ifc_cfg_value['value'].iteritems():
            if key[1] == 'NATRule':
                super(NATList, self).populate_model(key, value)
            elif key[1] == 'clear_translation':
                self.clear_translation_state = value['state']
        self.normalize_order()

    def parse_cli(self, cli, order):
        result = self.parse_static_cli(cli)
        if not result:
            result = self.parse_dynamic_cli(cli)
        if result:
            result['order'] = str(order)
            return result

    @classmethod
    def get_dynamic_cli_pattern(cls):
        # Dynamic NAT:
        # nat (<origin-ifc>,<translated-ifc>)
        #     source dynamic {<real-obj> | any}
        #         {[<mapped-obj>] [pat-pool [<mapped-obj>] [interface [ipv6]]
        #         [extended] [flat [include-reserve]] [round-robin]] |
        #         [interface [ipv6]]}
        #     [destination static {<mapped-obj> | {interface [ipv6]}} {<real-obj> |
        #         any}]
        #     [service <origin-svc-obj> <translated-svc-obj>]
        #     [dns] [unidirectional]
        if not cls.DYNAMIC_CLI_PATTERN:
            cls.DYNAMIC_CLI_PATTERN = re.compile(
                'nat \((.+),(.+)\)' +                                         # Interfaces, groups 1,2
                ' source dynamic (?:any|(\S+))' +                             # Source address, groups 3,4,5,6,7,8,9,10,11,12
                    '(?: (?!interface)(?!pat-pool)(\S+))?' +
                    '(?: (pat-pool)(?: (?!interface)(\S+))?)?' +
                    '(?: (interface)(?: (ipv6))?)?' +
                    '( extended)?(?: (flat)(?: (include-reserve))?)?( round-robin)?' +
                '(?: destination static (?:(interface)(?: (ipv6))?|(\S+))' +  # Destination address, groups 13,14,15,16
                    ' (?:any|(\S+)))?' +
                '(?: service (?:any|(\S+)) (\S+))?' +                         # Service, groups 17,18
                '( net-to-net)?( dns)?( unidirectional)?$')                   # Options, groups 19,20,21
        return cls.DYNAMIC_CLI_PATTERN

    def parse_dynamic_cli(self, cli):
        m = self.get_dynamic_cli_pattern().match(cli)
        if m:
            result = {}
            result['source_interface'] = self.get_connector_name(m.group(1),
                NATRule.DEFAULTS['source_interface'])
            result['destination_interface'] = self.get_connector_name(m.group(2),
                NATRule.DEFAULTS['destination_interface'])
            self.parse_dynamic_source_address(result, *m.group(3, 4, 5, 6, 7, 8, 9, 10, 11, 12))
            self.parse_destination_address(result, *m.group(13, 14, 15, 16))
            self.parse_service(result, *m.group(17, 18))
            self.parse_options(result, *m.group(19, 20, 21))
            return result

    @classmethod
    def get_static_cli_pattern(cls):
        # Static NAT:
        # nat (<origin-ifc>,<translated-ifc>)
        #     source static {<real-obj> | any} {<mapped-obj> | {interface [ipv6]} |
        #         any}
        #     [destination static {<mapped-obj> | {interface [ipv6]}} {<real-obj> |
        #         any}]
        #     [service {<origin-svc-obj> | any} <translated-svc-obj>]
        #     [net-to-net] [dns] [unidirectional] [no-proxy-arp]
        if not cls.STATIC_CLI_PATTERN:
            cls.STATIC_CLI_PATTERN = re.compile(
                'nat \((.+),(.+)\)' +                                        # Interfaces, groups 1,2
                ' source static (?:any|(\S+))' +                             # Source address, groups 3,4,5,6
                    ' (?:any|(interface)(?: (ipv6))?|(\S+))' +
                '(?: destination static (?:(interface)(?: (ipv6))?|(\S+))' + # Destination address, groups 7,8,9,10
                    ' (?:any|(\S+)))?' +
                '(?: service (?:any|(\S+)) (\S+))?' +                        # Service, groups 11,12
                '( net-to-net)?( dns)?( unidirectional)?( no-proxy-arp)?$')  # Options, groups 13,14,15,16
        return cls.STATIC_CLI_PATTERN

    def parse_static_cli(self, cli):
        m = self.get_static_cli_pattern().match(cli)
        if m:
            result = {}
            result['source_interface'] = self.get_connector_name(m.group(1),
                NATRule.DEFAULTS['source_interface'])
            result['destination_interface'] = self.get_connector_name(m.group(2),
                NATRule.DEFAULTS['destination_interface'])
            self.parse_static_source_address(result, *m.group(3, 4, 5, 6))
            self.parse_destination_address(result, *m.group(7, 8, 9, 10))
            self.parse_service(result, *m.group(11, 12))
            self.parse_options(result, *m.group(13, 14, 15, 16))
            return result

    def is_network_object_group(self, name):
        return name in self.get_top().get_child('NetworkObjectGroup').children

    def parse_destination_address(self, result, interface_token, ipv6_token,
                                  mapped_object_name, real_object_name):
        if interface_token or mapped_object_name:
            address = {}
            if interface_token:
                interface = {'ip_version': 'ipv4'}
                if ipv6_token:
                    interface['ip_version'] = ipv6_token
                address['mapped_object'] = {'interface': interface}
            elif mapped_object_name:
                address['mapped_object'] = self.parse_address_object(mapped_object_name)
            if real_object_name:
                address['real_object'] = self.parse_address_object(real_object_name)
            result['destination_translation'] = address

    def parse_address_object(self, object_name):
        address_object = {}
        if self.is_network_object_group(object_name):
            address_object['object_group_name'] = object_name
        else:
            # Default to network object whether found or not
            address_object['object_name'] = object_name
        return address_object

    def parse_dynamic_source_address(
            self, result,
            real_object_name, mapped_object_name,
            pat_pool_token, pat_pool_name,
            interface_token, ipv6_token,
            extended_token, flat_token, include_reserve_token, round_robin_token):
        address = {'nat_type': 'dynamic'}
        if real_object_name:
            address['real_object'] = self.parse_address_object(real_object_name)
        mapped_object = {}
        if mapped_object_name:
            mapped_object.update(self.parse_address_object(mapped_object_name))
        if pat_pool_token:
            pat_pool = {}
            if pat_pool_name:
                pat_pool.update(self.parse_address_object(pat_pool_name))
            if extended_token:
                pat_pool['extended'] = None
            if flat_token:
                flat = {}
                if include_reserve_token:
                    flat['include_reserve'] = None
                pat_pool['flat'] = flat
            if round_robin_token:
                pat_pool['round_robin'] = None
            mapped_object['pat_pool'] = pat_pool
        if interface_token:
            interface = {'ip_version': 'ipv4'}
            if ipv6_token:
                interface['ip_version'] = ipv6_token
            mapped_object['interface'] = interface
        address['mapped_object'] = mapped_object
        result['source_translation'] = address

    @staticmethod
    def parse_options(result, net_to_net, dns, unidirectional,
                      no_proxy_arp=None):
        if net_to_net:
            result['net_to_net'] = None
        if dns:
            result['dns'] = None
        if unidirectional:
            result['unidirectional'] = None
        if no_proxy_arp:
            result['no_proxy_arp'] = None

    @staticmethod
    def parse_service(result, real_object_name, mapped_object_name):
        if mapped_object_name:
            service = {'mapped_object_name': mapped_object_name}
            if real_object_name:
                service['real_object_name'] = real_object_name
            result['service_translation'] = service

    def parse_static_source_address(self, result, real_object_name,
                                    interface_token, ipv6_token,
                                    mapped_object_name):
        address = {'nat_type': 'static'}
        if real_object_name:
            address['real_object'] = self.parse_address_object(real_object_name)
        if interface_token:
            interface = {'ip_version': 'ipv4'}
            if ipv6_token:
                interface['ip_version'] = ipv6_token
            address['mapped_object'] = {'interface': interface}
        elif mapped_object_name:
            address['mapped_object'] = self.parse_address_object(mapped_object_name)
        result['source_translation'] = address

    @staticmethod
    def sort_config(config):
        '''
        Sort by key and then sort by order.  This will ensure that the NAT
        order is always the same for NATList that have the same order, even
        though a dictionary does not have an order.
        '''

        config.sort(key=lambda nat: nat.delta_ifc_key)
        config.sort(key=lambda nat: int(nat.values['order']))

    def set_nat_changed(self):
        self.nat_changed = True

    def validate_configuration(self):
        if self.delta_ifc_cfg_value.get('template_ifc_key'):
            'Only need to validate the template. The fault from the instantiated one would be the same.'
            return
        faults = []
        self.validate_duplicates(faults)
        for child in self.children.values():
            self.report_fault(child.validate_configuration(), faults)
        return faults

    def validate_duplicates(self, faults):
        config = []
        for nat in self.children.itervalues():
            # Check the final configuration only
            if nat.get_state() in (State.NOCHANGE, State.CREATE):
                config.append(nat)
    
        if len(config) > 1:
            # Sort into order
            self.sort_config(config)
    
            # Check for duplicates
            for i, this_nat in enumerate(config[:-1]):
                for that_nat in config[i + 1:]:
                    if this_nat.equals_no_order(that_nat):
                        faults.append(that_nat.generate_fault(
                            'Duplicate NAT rule.'))

class NATListList(DMList):
    'A list of NAT lists'

    INTERFACES_PATTERN = None

    def __init__(self):
        super(NATListList, self).__init__('NATList', NATList, 'nat')

    def find_nat_list(self, interfaces):
        '''
        Find the NAT list using these interfaces.
            1) Find the group using these interfaces
            2) If found, get the NAT policy.  Return a tuple with the firewall
               and the name that the NAT policy references (or None)
        '''

        for group in self.get_top().iter_groups():
            for firewall in group.iter_firewalls():
                firewall_interfaces = firewall.get_interfaces()
                if not firewall_interfaces:
                    continue
                if all(map(lambda x: x in firewall_interfaces, interfaces)):
                    nat_list_name = None
                    nat_policy = firewall.get_child('NATPolicy')
                    if nat_policy.has_ifc_delta_cfg():
                        nat_list_name = nat_policy.get_value()['nat_list_name']
                    return nat_policy, nat_list_name, firewall_interfaces
        return None, None, None

    def get_translator(self, cli):
        if isinstance(cli, basestring):
            interfaces = self.parse_interfaces(cli)
            if interfaces:
                nat_policy, nat_list_name, firewall_interfaces = self.find_nat_list(interfaces)
                if nat_list_name:
                    # There is a firewall using this NAT command, so the command
                    # needs to be handled
                    nat_list = self.get_child(nat_list_name)
                    if nat_list:
                        nat_policy.set_state(State.NOCHANGE)
                    else:
                        nat_list = self.child_class(nat_list_name)
                        self.register_child(nat_list)
                        nat_list.create_missing_ifc_delta_cfg()
                    if not nat_list.interfaces:
                        nat_list.interfaces = firewall_interfaces
                    return nat_list
                if firewall_interfaces:
                    # There is a firewall using the interfaces of this NAT
                    # command, but not using the command.  So it will need to be
                    # destroyed.
                    key = self.make_nat_list_name(str(firewall_interfaces))
                    nat_list = self.children.get(key)
                    if not nat_list:
                        nat_list = self.child_class(key)
                        self.register_child(nat_list)
                        nat_list.create_missing_ifc_delta_cfg()
                        nat_list.interfaces = firewall_interfaces
                        # Save the interfaces in the value dictionary
                        nat_list.delta_ifc_cfg_value['interfaces'] = firewall_interfaces
                    return nat_list

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        # A NATList can be used by multiple NAT policies, so call each policy
        # to generate the NAT rules for that policy's pair of interfaces.
        for group in self.get_top().iter_groups():
            for firewall in group.iter_firewalls():
                firewall.get_child('NATPolicy').deferred_ifc2asa(no_asa_cfg_stack,
                                                                 asa_cfg_list)

        # Remove the NAT rules that are not used by any NAT policies
        for child in self:
            if (child.has_ifc_delta_cfg() and
                    'interfaces' in child.delta_ifc_cfg_value):
                child.ifc2asa(no_asa_cfg_stack, asa_cfg_list,
                              interfaces=child.delta_ifc_cfg_value['interfaces'])

    def make_copy(self, name, copy_name):
        '''
        Make a copy of a NATList
        @param name: str
        @param copy_name: set of nameifs
        @return NATList of the copy
        '''
        result = self.get_child(copy_name)
        if result:
            return result
        nat_list = self.get_child(name)
        if not nat_list:
            return
        result = NATList(copy_name)
        self.register_child(result)
        kind, name, instance = nat_list.delta_ifc_key
        new_key = (kind, name, copy_name)
        result.populate_model(new_key, copy.deepcopy(nat_list.delta_ifc_cfg_value))
        self.parent.delta_ifc_cfg_value['value'][new_key] = result.delta_ifc_cfg_value
        'save the ifc_key of the source as template_ifc_key, use as config_path when generating fault, see CSCvb67004'
        result.delta_ifc_cfg_value['template_ifc_key'] = nat_list.delta_ifc_key
        return result

    @classmethod
    def get_interfaces_pattern(cls):
        if not cls.INTERFACES_PATTERN:
            cls.INTERFACES_PATTERN = re.compile('nat \((.+),(.+)\)')
        return cls.INTERFACES_PATTERN

    @classmethod
    def parse_interfaces(cls, cli):
        '''
        Returns a tuple of a tuple of the external,internal interfaces and a key
        that can be used for a NATList of an unknown NAT rule

        For example, a CLI of:
            'nat (inside,outside) source static any any'
        will return:
            ((inside, outside),'nat (inside,outside)')
        '''
        m = cls.get_interfaces_pattern().match(cli)
        if m:
            return (m.group(1),m.group(2))

    @classmethod
    def make_nat_list_name(self, seed):
        return 'nat ' + seed

class NATPolicy(DMObject):
    'A NAT policy associates a NAT list with a service graph'

    def __init__(self):
        super(NATPolicy, self).__init__(NATPolicy.__name__)

    def ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        # At the time this function is called, the interface CLI has not been
        # generated. Defer the CLI generation until NATListList.ifc2asa() is
        # called, since it occurs after the interfaces are generated.
        pass
 
    def deferred_ifc2asa(self, no_asa_cfg_stack, asa_cfg_list):
        # This function is called by NATListList.ifc2asa()
        if self.has_ifc_delta_cfg():
            nat_list_name = self.get_value()['nat_list_name']
            nat_list = self.get_top().get_child('NATList').get_child(nat_list_name)
            if nat_list:
                nat_list.ifc2asa(no_asa_cfg_stack, asa_cfg_list, self.parent,
                                 self.get_state())

    def update_references(self):
        '''
        Override the default to use a copy of the referenced NATList.
        This overcomes the problem with serviceAudit when we have multiple-graphs
        sharing the same NAT. CSCuz56618
        '''
        if not self.has_ifc_delta_cfg():
            return
        def get_nat_list_name ():
            config = self.delta_ifc_cfg_value
            result = config['value'].values()[0]['target']
            return result
        def set_nat_list_name (name):
            config = self.delta_ifc_cfg_value
            config['value'].values()[0]['target'] = name
        nat_list_name = get_nat_list_name()
        nat_list_copy_name = NATListList.make_nat_list_name(str(self.parent.get_interfaces()))
        self.get_top().get_child('NATList').make_copy(nat_list_name, nat_list_copy_name)
        set_nat_list_name(nat_list_copy_name)

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        '''
        To take care of graphs sharing the same NATList, do not remove it if its parent is removed.
        It will be automatically removed once the interfaces using it is removed. i.e. the last graph
        using it is removed.
        See CSCvb34578.
        '''
        if self.parent.get_state() == State.DESTROY:
            set_cfg_state(delta_ifc_cfg_value, State.NOCHANGE)
        return DMObject.populate_model(self, delta_ifc_key, delta_ifc_cfg_value)

class NATRule(DMObject):
    '''
    A single NAT rule

    For the dynamic NAT source address a mapped object, PAT pool mapped object,
    and an interface can all be specified at the same time.  The correct CLI
    will be generated, but the ASA will not currently accept it due to the ASA
    defect CSCtq20634.
    '''

    DEFAULTS = {
        'destination_interface': 'internal',
        'source_translation': {'nat_type': 'static'},
        'source_interface': 'external',
    }

    def __init__(self, instance):
        super(NATRule, self).__init__(instance)

    def diff_ifc_asa(self, cli):
        if not self.has_ifc_delta_cfg():
            self.delta_ifc_key = (Type.FOLDER, NATRule.__name__, cli)
            self.delta_ifc_cfg_value = {'state': State.DESTROY, 'value': ifcize_param_dict(self.values)}
            ancestor = self.get_ifc_delta_cfg_ancestor()
            if ancestor:
                ancestor.delta_ifc_cfg_value['value'][self.delta_ifc_key] =  self.delta_ifc_cfg_value

    def equals_no_order(self, nat):
        nat_values = nat.values.copy()
        if 'order' in nat_values:
            del nat_values['order']
        self_values = self.values.copy()
        if 'order' in self_values:
            del self_values['order']

        return self_values == nat_values

    def add_xlate(self, nat):
        firewall = self.get_ancestor_by_class(translator.firewall.Firewall)
        if firewall:
            nat_rule_deployment = firewall.get_child('NATRuleDeployment')
            if nat_rule_deployment:
                nat_rule_deployment.add_xlate(nat)

    def generate_command(self, asa_cfg_list):
        self.generate_cli(asa_cfg_list,
                          self.get_cli(),
                          response_parser=ignore_warning_response_parser)

    def generate_delete(self, asa_cfg_list):
        self.generate_cli(asa_cfg_list,
                          'no ' + self.get_cli(),
                          response_parser=ignore_response_parser)
        self.set_nat_changed()

    def get_cli(self):
        cli = []
        cli.append('nat (')
        cli.append(self.parent.get_interface_name(self.values['source_interface']))
        cli.append(',')
        cli.append(self.parent.get_interface_name(self.values['destination_interface']))
        cli.append(')')
        self.append_source_address(cli, self.values.get('source_translation'))
        self.append_destination_address(cli, self.values.get('destination_translation'))
        self.append_service(cli, self.values.get('service_translation'))
        if 'net_to_net' in self.values:
            cli.append(' net-to-net')
        if 'dns' in self.values:
            cli.append(' dns')
        if 'unidirectional' in self.values:
            cli.append(' unidirectional')
        if 'no_proxy_arp' in self.values:
            cli.append(' no-proxy-arp')

        return ''.join(cli)

    @classmethod
    def append_destination_address(cls, cli, address):
        if address:
            cli.append(' destination static')
            cls.append_object(cli, address.get('mapped_object'), '')
            cls.append_object(cli, address.get('real_object'), 'any')

    @staticmethod
    def append_dynamic_mapped_object(cli, object_dict):
        '''
        CSCvb77538: When APIC send an empty configuration to DP, in this case an empty 'real_object' under NATRule/source_translation, DP errors out.
        As a result, no CLI is generated. Solution here is to skip the empty value and continue to generate the rest of the CLI. For example,
        if we see configuration "A B C D" and 'C' is missing, we just skip 'C' and continue to generate CLI for "A B D".
        '''
        if object_dict == None:
            return
        if 'object_name' in object_dict:
            cli.append(' ' + object_dict['object_name'])
        elif 'object_group_name' in object_dict:
            cli.append(' ' + object_dict['object_group_name'])
        pat_pool = object_dict.get('pat_pool')
        if pat_pool != None:
            cli.append(' pat-pool')
            if 'object_name' in pat_pool:
                cli.append(' ' + pat_pool['object_name'])
            elif 'object_group_name' in pat_pool:
                cli.append(' ' + pat_pool['object_group_name'])

        interface = object_dict.get('interface')
        if interface != None:
            cli.append(' interface')
            if interface.get('ip_version') == 'ipv6':
                cli.append(' ipv6')

        if pat_pool != None:
            if 'extended' in pat_pool:
                cli.append(' extended')
            if 'flat' in pat_pool:
                cli.append(' flat')
                if 'include_reserve' in pat_pool['flat']:
                    cli.append(' include-reserve')
            if 'round_robin' in pat_pool:
                cli.append(' round-robin')
                

    @staticmethod
    def append_object(cli, object_dict, default):
        if object_dict:
            if 'object_name' in object_dict:
                cli.append(' ' + object_dict['object_name'])
                return
            elif 'object_group_name' in object_dict:
                cli.append(' ' + object_dict['object_group_name'])
                return
            elif 'interface' in object_dict:
                cli.append(' interface')
                if object_dict['interface'].get('ip_version') == 'ipv6':
                    cli.append(' ipv6')
                return
        cli.append(' ' + default)

    @staticmethod
    def append_service(cli, service):
        if service:
            cli.append(' service')
            cli.append(' ' + service.get('real_object_name', 'any'))
            cli.append(' ' + service.get('mapped_object_name', ''))

    @classmethod
    def append_source_address(cls, cli, address):
        nat_type = address['nat_type']
        cli.append(' source')
        cli.append(' ' + nat_type)
        cls.append_object(cli, address.get('real_object'), 'any')
        if nat_type == 'dynamic':
            cls.append_dynamic_mapped_object(cli, address.get('mapped_object'))
        else:
            cls.append_object(cli, address.get('mapped_object'), 'any')

    @classmethod
    def populate_defaults(cls, values, defaults):
        for name, default in defaults.iteritems():
            if name in values:
                if isinstance(values[name], dict) and isinstance(default, dict):
                    cls.populate_defaults(values[name], default)
            else:
                values[name] = default

    def populate_model(self, delta_ifc_key, delta_ifc_cfg_value):
        super(NATRule, self).populate_model(delta_ifc_key, delta_ifc_cfg_value)
        self.values = self.get_value()
        self.populate_defaults(self.values, self.DEFAULTS)

        # Normalize parameters whose values are ignored
        for param in ('dns', 'net_to_net', 'no_proxy_arp', 'unidirectional'):
            if param in self.values:
                self.values[param] = None
        source_translation = self.values['source_translation']
        if (source_translation['nat_type'] == 'dynamic' and
                'mapped_object' in source_translation and
                'pat_pool' in source_translation['mapped_object']):
            pat_pool = source_translation['mapped_object']['pat_pool']
            for param in ('extended', 'flat', 'round_robin'):
                if param in pat_pool:
                    if param == 'flat':
                        param = 'include_reserve'
                        flat = pat_pool['flat']
                        if param in flat:
                            flat[param] = None
                    else:
                        pat_pool[param] = None

    def set_nat_changed(self):
        self.parent.set_nat_changed()

    def validate_configuration(self):
        faults = []
        for folder in ('source_translation', 'destination_translation'):
            self.validate_address(faults, folder)
        self.validate_dns(faults)
        self.validate_net_to_net(faults)
        self.validate_no_proxy_arp(faults)
        self.validate_pat_pool(faults)
        self.validate_unidirectional(faults)
        return faults

    def validate_address(self, faults, address_folder):
        if address_folder in self.values:
            for folder in ('mapped_object', 'real_object'):
                self.validate_object(faults, address_folder, folder)

    def validate_dns(self, faults):
        if 'dns' in self.values:
            if 'destination_translation' in self.values:
                faults.append(self.generate_fault(
                    'dns cannot be configured if destination address translation is configured.',
                    'dns'))
            if 'service_translation' in self.values:
                faults.append(self.generate_fault(
                    'dns cannot be configured if service translation is configured.',
                    'dns'))

    def validate_net_to_net(self, faults):
        if ('net_to_net' in self.values and
                'destination_translation' not in self.values):
            faults.append(self.generate_fault(
                'net_to_net is only supported if destination address translation is configured.',
                'net_to_net'))

    def validate_no_proxy_arp(self, faults):
        if ('no_proxy_arp' in self.values and
                self.values['source_translation']['nat_type'] != 'static'):
            faults.append(self.generate_fault(
                'no_proxy_arp is only supported for static NAT.',
                'no_proxy_arp'))

    def validate_object(self, faults, address_folder, object_folder):
        address = self.values[address_folder]
        if object_folder in address:
            address_object = address[object_folder]
            keys = set(address_object.keys())
            if (object_folder == 'mapped_object' and
                    address.get('nat_type') != 'dynamic' and len(keys) > 1):
                faults.append(self.generate_fault(
                    object_folder + ' can only contain one of interface, object_name, or object_group_name.',
                    [address_folder, object_folder]))
            elif set(('object_name', 'object_group_name')).issubset(keys):
                faults.append(self.generate_fault(
                    object_folder + ' can only contain one of object_name or object_group_name.',
                    [address_folder, object_folder]))

    def validate_pat_pool(self, faults):
        address_folder = self.values['source_translation']
        if (address_folder['nat_type'] == 'dynamic' and
                'mapped_object' in address_folder and
                'pat_pool' in address_folder['mapped_object']):
            mapped_object = address_folder['mapped_object']
            pat_pool = mapped_object['pat_pool']
            if 'object_name' in pat_pool and 'object_group_name' in pat_pool:
                faults.append(self.generate_fault(
                    'pat_pool can only contain one of object_name or object_group_name.',
                    ('source_translation', 'mapped_object', 'pat_pool')))
            if 'extended' in pat_pool and 'interface' in mapped_object:
                faults.append(self.generate_fault(
                    'extended cannot be configured if interface is configured.',
                    ('source_translation', 'mapped_object', 'pat_pool', 'extended')))

    def validate_unidirectional(self, faults):
        if ('unidirectional' in self.values and
                self.values['source_translation']['nat_type'] != 'static'):
            faults.append(self.generate_fault(
                'unidirectional is only supported for static NAT.',
                'unidirectional'))

    def __str__(self):
        return self.get_cli()
