'''
Created on Jul 25, 2013

@author: jeffryp

Copyright (c) 2013 by Cisco Systems, Inc.
All rights reserved.

Common utility functions
'''

from collections import OrderedDict
from requests.exceptions import RequestException
from itertools import ifilter
import json
import re
import copy
import pprint
import bisect

from asaio.dispatch import HttpDispatch
from asaio.cli_interaction import CLIInteraction, format_cli
import translator
from translator.state_type import State, Type
from translator.type2key import type2key
from translator.structured_cli import convert_to_structured_commands, StructuredCommand
from utils.errors import ConnectionError, ASACommandError, ASABusyError, DeviceParameterError
import utils.env as env
from utils.errors import FaultCode
from utils.ipaddress import ip_address, ip_network
from utils.ipaddress import IPV4LENGTH, IPv4Address
from utils.ipaddress import IPv6Address, IPv6Interface, IPv6Network

def connection_exception_handler(f):
    ''' Decorator used to handle connection exceptions, only used for read_clis and delivelr_clis APIs for now
    '''
    def handler(*argv, **kwargs):
        try:
            return f(*argv, **kwargs)
        except (IOError, RequestException) as e:
            raise ConnectionError(str(e))
        except:
            raise
    return handler

def key_as_string(obj):
    ''' This function makes the key into string in a dictionary.
    '''

    if isinstance(obj, dict):
        result = OrderedDict()
        for key, value in obj.iteritems():
            if isinstance(value, dict):
                result[repr(key)] = key_as_string(value)
            else:
                result[repr(key)] = value
        return result
    else:
        return obj

def pretty_dict(obj, indent=3):
    ''' This function prints a dictionary in a nicer indented form.
    '''
    try:
        return json.dumps(key_as_string(obj), indent=indent)
    except Exception:
        # If not JSON serializable
        return str(obj)

def normalize_param_dict(param_dict, remove_destroyed_elements = False):
    '''
    Normalize a param dictionary by extracting the 'key' and 'value' into a new
    dictionary, recursively.

    @param param_dict: scalar value or param dictionary
        if it's a param dictionary, then each entry is of the format:
            (type, key, instance): {'state': state,
                                    'device': device,
                                    'connector': connector,
                                    'value':  scalar value or param dictionary}
    @param remove_destroyed_elements: boolean
        Take out destroyed elements in the dictionary
    @return: scalar value or param dictionary
        if param_dict is a param dictionary, then return the a new dictionary
            with just the 'key' and 'value'.
        otherwise param_dict is returned
    '''

    if not isinstance(param_dict, dict):
        return param_dict

    result = {}
    for (kind, key, instance), cfg_value in param_dict.iteritems():
        'Will deprecate the check for target in the future, it should be handled at the program initialized'
        if remove_destroyed_elements and cfg_value.get('state') == State.DESTROY:
            continue
        if 'target' in cfg_value:
            value = cfg_value['target']
        elif 'value' in cfg_value:
            value = cfg_value['value']
        else:
            value = None
        if isinstance(value, dict):
            value = normalize_param_dict(value, remove_destroyed_elements)
        """
        For certain types of entries in the configuration parameter, the key
        may not corresponding to our internal key.
        Example:
            (Type.ENCAP, '', '3')
        we look up the internal key for it here.
        """
        key = type2key.get(kind, key)
        result[key] = value
    return result

def ifcize_param_dict(param_dict):
    '''Reverse the operation of util.normalize_param_dict
    @param param_dict: dict
        A configuration diction in the normal format, e.g:
         {'Hostname': 'my-asa'}
    @return dict in the IFC format,
        For the above example, the return value is:
         {(Type.PARAM, 'Hostname', ''), {'state': State.NOCHANGE, 'value': 'my-asa'}}
    @todo: move it into util. But encountered cyclic-dependency problem.

    '''
    if not isinstance(param_dict, dict):
        return param_dict

    result = {}
    for key, value in param_dict.iteritems():
        kind = Type.FOLDER if isinstance(value, dict) else Type.PARAM
        if isinstance(value, dict):
            value = ifcize_param_dict(value)
        result[(kind, key, '')] = {'state': State.NOCHANGE, 'value': value}
    return result


def filter_out_sacred_commands(output_clis):
    '''Remove commands that may cause disruption of communication between IFC and the ASA device.
    @param output_clis: list of CLIInteraction's
    @return output_clis
    '''
    mgmt_intf = 'management'
    sacred_commands  = ['^no route ' + mgmt_intf,
                        '^no http enable',
                        '^no http .+ ' + mgmt_intf,
                        '^clear config interface Management0/0$',
                        '^clear config interface Management0/1$']

    for sacred in sacred_commands:
        cmds = filter(lambda text: re.compile(sacred).match(str(text).strip()), output_clis)
        for cmd in cmds:
            output_clis.remove(cmd)
    return output_clis

def get_write_mem_clis(device, clis):
    """
    @return list of CLIInteraction's
        For single mode ASA, just a single CLI 'write mem'.
        For multi mode ASA, it can be  combination of ['changeto system', 'write mem', 'changeto context <name>', 'write mem']
    """
    response_parser = lambda response: None if '[OK]' in response else response
    result = []
    asa_context = device.get('asa_context')
    if asa_context and asa_context.get('system_context_accessible'):
        system_clis = filter(lambda cli: cli.is_system_context, clis)
        context_clis = filter(lambda cli: not cli.is_system_context, clis)
        if system_clis:
            result.extend(['changeto system', 'write mem'])
        if context_clis:
            result.extend(['changeto context ' + asa_context.get('context_name'), 'write mem'])
    else:
        result.append('write mem')
    return map(lambda cli: CLIInteraction(cli, response_parser=response_parser), result)

def move_copy_cmd_to_the_end(output_clis):
    '''
    Move 'copy /noconfirm .. running-config' to the end of the list.
    This is required to restore output band configuration for ASA-FI-DP.
    See CSCvb90258.
    '''
    copy_cmd = filter_first(lambda cmd: str(cmd).startswith('copy '), output_clis)
    if copy_cmd:
        output_clis.remove(copy_cmd)
        output_clis.append(copy_cmd)
    return output_clis
    
@connection_exception_handler
def deliver_clis(device, clis, transformers=[filter_out_sacred_commands, move_copy_cmd_to_the_end], save_config = True):
    '''Deliver a list of CLI's to an ASA device
    @param device: dict
        a device dictionary
    @param clis: list of CLIIneraction's
    @param transformers: list of function that takes one argument and return an object
        the purpose of a transformer is to transform ASA configuration to a desired format
        before actually sending them down to the ASA device.
        The order of the application of the transformer is the reverse given in the parameter,
        i.e. if the transformers is [a, b], the result will be a(b(config)).
        a list of CLI objects.
    @param save_config: boolean
        indicate if the running-config should be saved to startup-config if the configuration
        is delivered successfully.
    @return: True if successful in delivery, or ASACommandError or ConnectionError exception will be raised.
    '''
    if not deliver_clis.enabled:
        env.debug("[CLIs would be delivered]\n%s\n" % '\n'.join([str(cli) for cli in clis]))
        return True
    if not clis:
        return True;

    if transformers:
        for transformer in reversed(transformers):
            clis = transformer(clis)
    dispatcher = HttpDispatch(device)
    def dispatch(clis):
        'deliver a list of CLIInteraction, and return list errors if any'
        messenger = dispatcher.make_command_messenger(clis)
        results = messenger.get_results()
        errs = filter(lambda x: x != None and x.err_msg != None and len(x.err_msg.strip()) > 0, results)
        return errs
    errs = dispatch(clis)
    if not errs:
        if save_config and 'failover' not in map(str, clis):
            # 'wr mem' will fail during failover setup so bypass now. Defer till it is stable.
            errs = dispatch(get_write_mem_clis(device, clis))
            if not errs:
                return True
        else:
            return True

    faults = []
    for err in errs:
        # Check if the ASA is busy with a previous configuration and only report one error
        if err.err_msg.startswith('Command Ignored, Configuration in progress...'):
            raise ASABusyError(err.err_msg)
        faults.append((err.model_key, FaultCode.CONFIGURATION_ERROR, err.err_msg))
    raise ASACommandError(faults)

''' One can turn deliver_clis.enabled attribute on/off during testing.
'''
deliver_clis.enabled = True

def command_filter(clis):
    'Ignore those commands that we dont care'
    def allowed(cli):
        return cli and not cli.startswith('!')
    return filter(allowed, clis)

def filter_n(count, predicate, iterable):
    'Same as filter but only return the first count number of elements'
    generator = ifilter(predicate, iterable)
    n = 0
    result = []
    while n < count:
        try:
            result.append(generator.next())
        except StopIteration as e:
            break
        n += 1
    return result

def filter_first(predicate, iterable):
    'return the first match'
    hit = filter_n(1, predicate, iterable)
    return hit[0] if hit else None

def consolidate_and_filter_interface_commands(commands):
    """
    Consolidate multiple instances of instances of interface commands for the same interface.
    This can happen when we are running in multi-context mode, where the configuration of the
    same interface are spread between the system context and the user context.
    Also filter out interface commands where the interfaces do not belong to the context.

    For example:
    In the system context we have:
            'interface GigabitEthernet0/4.501',
            ' vlan 501',
            'interface GigabitEthernet0/3.201',
            ' vlan 201',
    and in the user conrtext, we have
            'interface GigabitEthernet0/3.201',
            ' nameif Address-1',
            ' security-level 100',
            ' ip address 20.0.0.1 255.255.255.192',
    The function will combine the two into one, make it the same as for single context and remove
    GigabitEthernet0/4.501 because it is not used by this context.:
            'interface GigabitEthernet0/3.201',
            ' vlan 201',
            ' nameif Address-1',
            ' security-level 100',
            ' ip address 20.0.0.1 255.255.255.192',
    """
    get_main_cmd = lambda cmd: cmd.command if isinstance(cmd, StructuredCommand) else cmd
    is_interface_command = lambda cmd: get_main_cmd(cmd).startswith('interface ')
    interface_commands =  filter(is_interface_command, commands)
    commands_to_remove = []
    commands_to_keep = []
    for intf_cmd in interface_commands:
        if intf_cmd in commands_to_keep:
            'we have finished with interface commands in the system context.'
            break
        same_intf_cmds = filter_n(2, lambda cmd: get_main_cmd(cmd) == get_main_cmd(intf_cmd),
                                  interface_commands)
        if len(same_intf_cmds) == 2:
            'combine interface commands for the same interface'
            if isinstance(intf_cmd, StructuredCommand):
                same_intf_cmds[1].sub_commands.extend(intf_cmd.sub_commands)
            commands_to_keep.append(same_intf_cmds[1])
        commands_to_remove.append(intf_cmd)
    'Keep the main interface command whose sub-interface is used by the context'
    is_our_main_interaface_cmd = lambda cmd:\
        get_main_cmd(cmd).split()[1].startswith('BVI') or\
        any(map(lambda cmd2: get_main_cmd(cmd2).startswith(get_main_cmd(cmd) + '.'),
                commands_to_keep))
    for cmd in commands_to_remove:
        'only remove if it is not used at all'
        if not is_our_main_interaface_cmd(cmd):
            commands.remove(cmd)
    return commands

def associate_ace_remark_to_ace(clis):
    '''
    It will convert remarks to be the sub-commands of the next true ACE.
      e.g.
      Input:
           'access-list foo remark this is remarkable'
           'access-list foo extend permit ip any any'
      Output:
           'access-list foo extend permit ip any any'
           ' this is remarkable'
    '''
    result = []
    remarks = None # a continuous array of remarks for the same access-list
    acl_name = None # the name of the access-list
    for cli in clis:
        if cli.startswith('access-list '):
            m = re.match('^access-list (\S+) (\S+) (.+)$', cli)
            if m.group(2) == 'remark':# the CLI is a remark
                if acl_name == m.group(1): # the same remark set
                    remarks.append(m.group(3))
                else:#first of a new set of remarks
                    acl_name = m.group(1)
                    remarks = [m.group(3)]
            else: # a pure ACE
                result.append(cli)
                if m.group(1) == acl_name:# make the remarks sub-commands of the ACE
                    result.extend(map(lambda r: ' ' + r, remarks))
                acl_name = None
                remarks = None
        else:
            acl_name = None
            remarks = None
            result.append(cli)
    return result

@connection_exception_handler
def read_clis(device,
              transformers=[consolidate_and_filter_interface_commands,
                            convert_to_structured_commands,
                            associate_ace_remark_to_ace,
                            command_filter,
                            lambda x: x.split('\n')],
              read_system_context = False):
    '''Read running-configuration from ASA device
    @param device: dict or string or a list of strings
        an ASA device dictionary, or the name of the file containing ASA running-config, or a list of CLIs
    @param transformers: list of function that takes one argument and return an object
        the purpose of a transformer is to transform ASA configuration to a desired format.
        The order of the application of the transformer is the reverse given in the parameter,
        i.e. if the transformers is [a, b], the result will be a(b(config)).
    @param read_system_contex: boolean. Used for unit tests to indicate
        if the system context configuration is read.
    @return: list of CLI's
    '''
    if isinstance(device, dict):
        dispatcher = HttpDispatch(device)
        if device.get('asa_context') and device.get('asa_context').get('system_context_accessible'):
            messenger = dispatcher.make_read_config_messenger('system')
            result = messenger.read()
            context_name = device.get('asa_context').get('context_name')
            messenger = dispatcher.make_read_config_messenger(context_name)
            result += messenger.read()
            read_system_context = True
        else:
            messenger = dispatcher.make_read_config_messenger()
            result = messenger.read()
    elif isinstance(device, str):
        with open(device) as f:
            result = f.read()
    elif isinstance(device, list):
        result = '\n'.join(device)
    else:
        raise Exception("Unsupported device parameter")

    if result and transformers:
        for transformer in reversed(transformers):
            if (transformer == consolidate_and_filter_interface_commands and
               not read_system_context):
                continue
            result = transformer(result)
    return result

def deliver_sts_table(device, sts, is_audit):
    '''
    Send STS table to ASA device
    @param device: dict
        a device dictionary
    @param table; dict
        an STS dictionary
    @param is_audit: boolean
        True if this is an audit operation else False
    '''
    table = sts.sts_table
    if not deliver_clis.enabled:
        env.debug("[STS table would be delivered]\n\n")
        return True
    if not table:
        return True
    dispatcher = HttpDispatch(device)
    if is_audit:
        messenger = dispatcher.make_sts_audit_write_messenger(sts)
    else:
        messenger = dispatcher.make_sts_write_messenger(sts)
    results = messenger.get_results()
    if results:
        faults = []
        faults.append(('STS', FaultCode.CONFIGURATION_ERROR, results))
        raise ASACommandError(faults)

def query_sts_audit_table(device, sts):
    '''
    Query STS table from ASA device
    @param device: dict
        a device dictionary
    @param table; dict
        an STS dictionary
    '''
    if not device:
        error = "query_sts_table: fails to read response for quering sts table"
        env.debug(error)
        return (None, error)
    dispatcher = HttpDispatch(device)
    messenger = dispatcher.make_sts_audit_read_messenger(sts)
    results = messenger.get_results()
    if results:
        faults = []
        faults.append(('STS', FaultCode.CONFIGURATION_ERROR, results))
        raise ASACommandError(faults)

@connection_exception_handler
def query_asa(device, query_cmd, context = None, hide_exception = True):
    '''Read information back from the given device
    @param device: dict
        a device dictionary
    @param query: str, a show command cli, like "show run access-list bla"
    @param context: str, the name of a context to run, can be None.
    @param hide_exception: boolean. Indicate if the communication exception should be hidden
    @return tuple with:
        1) response string from ASA; or None if cannot connect to the device
        2) any error or exception string, otherwise None
    @attention:  PLEASE do not use this API to make configuration change on the device
    '''
    if not device:
        error = "query_asa: fails to read response for command '%s'. Error: %s" % (query_cmd, 'empty device dictionary')
        env.debug(error)
        return (None, error)
    if context:
        change_context_cmd = "changeto system" if "system"  == context  else "changeto context " + context
        cmds = [change_context_cmd, query_cmd]
    else:
        cmds = [query_cmd]
    if not query_cmd.strip().startswith('show'):
        error = "query_asa: '%s' is not a show command, discarded" % query_cmd
        env.debug(error)
        return (None, error)
    try:
        dispatcher = HttpDispatch(device)
        messenger = dispatcher.make_shows_messenger(cmds)
        result =  messenger.read()
        if result == 'Command failed\n':
            return None, None
        if '\nERROR: %' in result:
            return None,None
        return str(result), None
    except Exception as e:
        if hide_exception:
            env.debug("query_asa: fails to read response for command '%s'. Error: %s" % (query_cmd, e))
            return (None, str(e))
        else:
            raise(e)

@connection_exception_handler
def query_asa_n(device, query_cmds, hide_exception = True):
    '''Read information back from the given device
    @param device: dict
        a device dictionary
    @param query_cmds: list of CLIInteraction with show command cli, like "show run access-list bla"
    @param hide_exception: boolean. Indicate if the communication exception should be hidden
    @return tuple with:
        1) list of CLIResult from ASA; or None if cannot connect to the device
        2) any error or exception string, otherwise None
    @attention:  PLEASE do not use this API to make configuration change on the device
    '''
    if not device:
        error = "query_asa_n: empty device dictionary"
        env.debug(error)
        return (None, error)
    for ci in query_cmds:
        cmd = ci.command.strip()
        if not cmd.startswith('show'):
            error = "query_asa_n: '%s' is not a show command, discarded" % cmd
            env.debug(error)
            return (None, error)
    format_cli(query_cmds)
    try:
        dispatcher = HttpDispatch(device)
        messenger = dispatcher.make_command_messenger(query_cmds)
        return messenger.get_results(), None
    except Exception as e:
        if hide_exception:
            env.debug("query_asa_n: fails to read response. Error: %s" % e)
            return (None, str(e))
        else:
            raise(e)

def read_asa_version(device):
    '''Read the version information from the given device
    @param device: dict
        a device dictionary
    @return tuple with:
        1) version string, such as '9.0(3)'; or None if cannot connect to the device
        2) any error
    '''
    version_line = 'Cisco Adaptive Security Appliance Software Version'
    pattern = r'^'+ version_line + ' (\S+)'
    result, error =  query_asa(device, 'show version | grep ' + version_line)
    if not result:
        return (None, error)
    for line in result.split('\n'):
        m = re.match(pattern, line)
        if m:
            return (m.group(1), None)
    return (None, None)

def set_cfg_state(ifc_cfg_dict, state):
    '''Set the state of each entry in the a given IFC configuration
     to a given state.
    @param ifc_cfg_dict: dict
        IFC configuration dictionary
    @param state: State
    '''

    if 'state' in  ifc_cfg_dict:
        ifc_cfg_dict['state'] = state

    for value in ifc_cfg_dict.values():
        if not isinstance(value, dict):
            continue
        if not 'state' in  value:
            set_cfg_state(value, state)
            continue
        value['state'] = state

        if 'cifs' in value:
            key_of_value = 'cifs'
        elif 'target' in value:
            key_of_value = 'target'
        else:
            key_of_value = 'value'

        ifc_cfg_dict = value.get(key_of_value)
        if isinstance(ifc_cfg_dict, dict):
            set_cfg_state(ifc_cfg_dict, state)


def normalize_faults(faults, config):
    """
    Remove (0, '', '') from the fault path if it is not in the config paramter
    @param faults: list of fault
        Each fault is a triplet: (path, error_number, error_string),
        where path is list of triplets of the form (kind, key, instance). where the key might be modified by
        fill_key_string method during command generation.
    @param config: dict
        the original configuration parameter
    @return the same faults using the original dictionary key in the original configuration dictionary
    """

    def normalize_path(path, config):
        result = []
        if not path:
            return result
        for kind, key, instance in path:
            if kind == Type.DEV:
                """
                 The top level config might not be (Type.DEV, key, name)
                 this is the case of clusterXXX and deviceXXX APIs
                 Skip it in this case.
                """
                keys = config.keys()
                if keys and keys[0][0] != Type.DEV:
                    continue
            result.append((kind, key, instance))
        return result

    result = []
    for path, err_num, err_str in faults:
        path = normalize_path(path, config)
        result.append((path, err_num, err_str))
    return result


def get_config_root_key(configuration):
    'Return the key to the root entry of an IFC configuration parameter'
    return [configuration.keys()[0]] if configuration else []


def get_config_firewall_keys(configuration):
    'Return the list of keys to the Firewall entries in an IFC configuration parameter'
    result = []
    if not configuration:
        return result
    kind, root_key, instance = configuration.keys()[0]
    if kind != Type.DEV:
        return result
    grp_keys = filter(lambda (kind, key, instance):  kind == Type.GRP, configuration.values()[0]['value'].keys())
    if not grp_keys:
        return result
    root_key = configuration.keys()[0]
    for grp_key in grp_keys:
        grp_config = configuration.values()[0]['value'][grp_key]
        hits = filter(lambda (kind, key, instance):  kind == Type.FUNC, grp_config['value'].keys())
        for key in hits:
            result.append([root_key, grp_key, key])
    return result


def massage_param_dict(asa, ifc_delta_cfg_dict, transformers = None):
    '''Adjust the configuration dictionary parameter passed to us by IFC to a format suited to us.
    @param asa:  DeviceModel
    @param ifc_delta_cfg_dict: IFC configuration parameter dictionary
    @param transformers: list of function that takes one argument and return an object
        the purpose of a transformer is to transform IFC configuration dictionary to a desired format.
        The order of the application of the transformer is the reverse given in the parameter,
        i.e. if the transformers is [a, b], the result will be a(b(ifc_delta_cfg_dict)).
    @return dict
        if_cfg_dict is a equivalent of ifc_delta_cfg_dict but massaged for ifc2asa or
        generate_ifc_delta_cfg methods
    '''
    if not ifc_delta_cfg_dict:
        result = {(Type.DEV, asa.ifc_key, ''):  {'state': State.NOCHANGE, 'value': ifc_delta_cfg_dict}}
    else:
        kind, key, instance = ifc_delta_cfg_dict.keys()[0]
        if kind == Type.DEV: #for the complete configurations: device, group, function
            result = ifc_delta_cfg_dict
        else:#device configuration
            result = {(Type.DEV, asa.ifc_key, ''):  {'state': State.NOCHANGE, 'value': ifc_delta_cfg_dict}}
    'Do not modify the parameter passed to us by IFC.'
    result = copy.deepcopy(result)

    'Apply the transformations required'
    if not transformers:
        transformers = []
    for transformer in reversed(transformers):
        result = transformer(result)
    return result

def get_all_connectors(configuration):
    'Returns a tuple of firewalls and connectors'
    asa = translator.devicemodel.DeviceModel()
    ifc_cfg = massage_param_dict(asa, configuration)
    asa.populate_model(ifc_cfg.keys()[0], ifc_cfg.values()[0])

    connectors = []
    firewalls = []
    for group in asa.iter_groups():
        for firewall in group.iter_firewalls():
            firewalls.append(firewall)
            for connector in firewall.iter_connectors():
                connectors.append(connector)
    return (firewalls, connectors)

def normalize_ip_address(addr):
    '''
    Normalize an IPv4 or IPv6 host address string.

    Examples:
    '100.000.000.000' -> '100.0.0.0'
    'F800::1'         -> 'f800::1'
    '''

    # The backport of ipaddress.py requires a unicode address
    try:
        return str(ip_address(unicode(addr)))
    except: #due to illegal addr
        return addr

def normalize_ipv6_address(addr, interface_addr=False):
    '''
    Normalizes ipv6 address, including host address and network address.
    The following is a few example of normalized result: case converted to lower case,
    leading 0 suppressed, host part masked according to the prefix length in the case
    of network address. Example:
    'F800::1' -> 'f800::1'
    '2002:A00:55:6::0:0000:6' -> '2002:a00:55:6::6'
    '2002:A00:60:6::2/29' -> '2002:a00::/29'

    If interface_addr is True, don't mask the host part according to the prefix
    length.  Example:
    '2002:A00:60:6::2/29' -> '2002:a00:60:6::2/29'
    '''

    # The backport of ipaddress.py requires a unicode address
    try:
        addr = unicode(addr)
        if '/' in addr:
            # e.g: '2002:a00:60:6::2/29'
            if interface_addr:
                return str(IPv6Interface(addr))
            else:
                return str(IPv6Network(addr, strict=False)) # Don't check for host bits
        else: # host address
            return str(IPv6Address(addr))
    except:
        return addr

def normalize_network_address(addr, mask):
    '''
    Normalizes a network address to match its mask.

    Mask can be either an IP address (255.255.255.0) or a prefix length (24).
    
    Example:
    2.2.2.10, 255.255.255.0 -> 2.2.2.0
    2002:3::10, 64          -> 2002:3::
    '''

    # The backport of ipaddress.py requires a unicode address
    addr = unicode(addr)
    mask = unicode(mask)
    network = ip_network(addr + '/' + mask, strict=False)
    return str(network.network_address)

def normalize_ipv4_address_and_mask_4_asa(value):
    """
    Normalize the value of format 'address/mask' to 'address mask' for ASA.
    Mask could be prefix length.
    """
    address, mask  = value.split('/')
    if '.' not in mask:#prefix length
        mask = netmask_from_prefix_length(mask)
    return ' '.join((address, mask))

def normalize_interface_name(intf):
    '''
    Normalizes interface, sub-interface, and port-channel name. For example:
        'g0/0'  -> 'GigabitEthernet0/0'
        'Gig0/1'-> 'GigabitEthernet0/1'
        'm0/0'  -> 'Management0/0'
        'Te0/8' -> 'TenGigabitEthernet0/8'
        'Eth1/1.20' -> 'Ethernet1/1.20'
        'Po9'   -> 'Port-channel9'
    '''

    if not intf:
        return
    intf = intf.strip()
    intf = pattern_replace(r'([gG][\D]*)[\d]+/[\d]+.*$', 'GigabitEthernet', intf)
    intf = pattern_replace(r'([mM][\D]*)[\d]+/[\d]+.*$', 'Management', intf)
    intf = pattern_replace(r'([tT][\D]*)[\d]+/[\d]+.*$', 'TenGigabitEthernet', intf)
    intf = pattern_replace(r'([eE][\D]*)[\d]+/[\d]+.*$', 'Ethernet', intf)
    intf = pattern_replace(r'([pP][\D]*)[\d]+.*$', 'Port-channel', intf)
    return intf

def pattern_replace(pattern, replace, source):
    '''
    Find 'pattern' in 'source' and replace the first group with 'replace'.
        'pattern' example: '([gG][\D]*)[\d]+/[\d]+.*$'
        'replace' example: 'GigabitEthernet'
        'source' example: 'g0/0'
        return result: 'GigabitEthernet0/0'
    For interface replacement, it happens if and only if the interface will be accepted
    by the device, in another word, it must be the prefix of the full interface name.
    '''
    p = re.compile(pattern)
    m = p.match(source)
    if m:
        prefix = m.group(1)
        if replace.lower().startswith(prefix.lower()):
            return re.sub(prefix, replace, source)
    return source

def hide_passwords(device):
    '''
    Hide the passwords in the device dictonary by replacing them with '<hidden>.
    This should be done on a deep copy, since the dictionary will be modified.
    '''

    if 'devs' in device:
        for value in device['devs'].itervalues():
            hide_passwords(value)
    if 'creds' in device and 'password' in device['creds']:
        device['creds']['password'] = '<hidden>'

    return device

def get_asa_context_information(device):
    '''
    Get the name of the user-context if the device is an ASA in multi-context mode.
    This user-context is the target CDev for applying changes.
    The result context information is saved in the device dictionary provided.
    If the ASA is in multi-context mode, the name of context is stored in the device dictionary.
    e.g.
     device['asa_context'] = {'context_name': 'admin', 'system_context_accessible': True }

    @param device: device dictionary
    @return error information
    TODO error handling.
    '''
    result, error = query_asa(device, "show mode", hide_exception=False)
    if error:
        return
    if not result:
        return
    if not result.strip().endswith("multiple"):
        "single mode"
        return

    def get_ip2context_name_map(device):
        '@return the mapping of management IP address to context name'

        def get_asa_context_names(device):
            '@return the names of all the context in the ASA device'
            result, error = query_asa(device, "show run context | grep ^context", "system", hide_exception=False)
            return map(lambda line: line.split()[1], result.strip().split('\n'))

        def get_man_ip_address(device, context_name):
            '@return the ip address of the management interface in the given context'
            config, error = query_asa(device, "show ip address management | begin Current IP Address:",
                                      context_name, hide_exception=False)
            """
            The format of config looks like forASA which is not a cluster master:
                Current IP Address:
                Interface                Name                   IP address      Subnet mask     Method 
                Management0/0            management             172.23.204.243  255.255.255.0   CONFIG

            For cluster master, there is an addtional entry for cluster address.

            For ASA 9.1(3), the format of config for a cluster master looks like :
                Current IP Address:
                Interface                Name                   IP address      Subnet mask     Method 
                Management0/0            management             172.23.204.243  255.255.255.0   CONFIG
                                                                172.23.204.241  255.255.255.0 
             For ASA 9.5(1), the format of config for a cluster master looks like:
                Current IP Address:
                Interface                Name                   IP address      Subnet mask     Method 
                Management0/0            management             172.23.204.243  255.255.255.0   IP-POOL
                                         management             172.23.204.241  255.255.255.0   VIRTUAL
            """
            if not config:
                return
            config =  config.strip().split('\n')
            if len(config) < 3:
                return
            result = [config[2].split()[2]] # the device address
            if len(config) >= 4:
                #take care of the cluster address
                cluster_ip = config[3].strip().split()
                if len(cluster_ip) == 2:#old format
                    result.append(cluster_ip[0])
                elif len(cluster_ip) == 4:#new format
                    result.append(cluster_ip[1])
            return result

        result = {}
        context_names = get_asa_context_names(device)
        for name in context_names:
            man_ip = get_man_ip_address(device, name)
            if man_ip:
                for ip in man_ip:
                    result[ip] = name
        return result

    def get_context_name(device):
        '@return the name of the context whose management address is given as LDevVIP'
        result, error = query_asa(device, "show context", hide_exception=False)
        lines = result.split('\n')
        name = lines[1].strip().split()[0]
        return name

    def get_admin_context_name(device):
        result, error = query_asa(device, 'show run context | grep ^admin-context', 'system', hide_exception=False)
        name = result.split()[1]
        return name

    "get the list of IP addresses of devs, one of them is the targeted context for the call-out"
    context_name = get_context_name(device)
    is_admin_context = context_name[0] == '*'
    if is_admin_context:
        context_name = context_name[1:]
    devs = device.get('devs')
    if not devs: #due to deviceModify or deviceAudit call-out
        asa_context = {'context_name': context_name,
                       'is_admin_context': is_admin_context,
                       'system_context_accessible': is_admin_context}
        device['asa_context'] = asa_context
        device['asa_all_context_names'] = [context_name]
        return

    if not is_admin_context:
        raise DeviceParameterError('The context specified by cluster IP address %s '
                                   'is not admin context. It must be admin context.' % device.get('host'))

    ip_addresses = map(lambda dev: dev['host'], devs.values())

    'get the context name'
    man_ip2context_name = get_ip2context_name_map(device)
    hits = filter(lambda ip: ip in ip_addresses, man_ip2context_name.keys())
    if len(hits) == 0:
        env.debug("asa_ip_2_context_map:\n%s" % pprint.pformat(man_ip2context_name))
        raise DeviceParameterError('No user context from the ASA device addressed by %s is '
                                   'found in the cluster member devices. There must be one.' % device.get('host'))
    if len(hits) != 1:
        env.debug("asa_ip_2_context_map:\n%s" % pprint.pformat(man_ip2context_name))
        raise DeviceParameterError('More than one user contexts from the the ASA device addressed by %s '
                                   'are found among the cluster member devices. There can be only one.' % device.get('host'))
    context_name = man_ip2context_name[hits[0]]
    admin_context_name = get_admin_context_name(device)

    asa_context = {'context_name': context_name,
                   'system_context_accessible': True,
                   'is_admin_context': admin_context_name == context_name}
    device['asa_context'] = asa_context
    device['asa_all_context_names'] = man_ip2context_name.values()
    device['asa_ip_2_context_map'] = man_ip2context_name

def populate_asa_version(device):
    '''
    Populate the device dictionary with the ASA version information

    Example for a version of '9.3(2)1':
        'device': {
            'asa_version': (9.3, 2, 1),
        }

    For the device* API calls, the device dictionary has only one device and one
    version.  For example:
        'device': {
            'version': '9.3(2)1',
        }

    For the cluster* and service* API calls, the device dictionary has the lDev
    device and all of the CDev devices.  There is a version for each CDev, but
    none for the LDev.  For example:
        'device': {
            'devs': {
                'ASA-205': {
                    'version': '9.3(2)1',
                }
            }
        } 

    @param device: device dictionary
    '''

    def get_version_str():
        if 'devs' in device:
            dev = filter_first(lambda dev: 'version' in dev,
                               device['devs'].itervalues())
            if dev:
                return dev['version']
        if 'version' in device:
            return device['version']
        return read_asa_version(device)[0]

    def get_version():
        version_str = get_version_str()
        if version_str:
            m = re.match(r'(\d+\.\d+)(?:\((\d+)\)(\d+)?)?', version_str)
            if m:
                return (float(m.group(1)),
                        int(m.group(2)) if m.group(2) else 0,
                        int(m.group(3)) if m.group(3) else 0)
        return (0.0, 0, 0)

    device['asa_version'] = get_version()

def netmask_from_prefix_length(prefix_length):
    '''
    Convert a prefix length to a netmask.
         0 -> 0.0.0.0
        24 -> 255.255.255.0
        32 -> 255.255.255.255 
    '''

    netmask = (0xffffffff00000000 >> int(prefix_length)) & 0xffffffff
    return str(IPv4Address(netmask)) 

def prefix_length_from_netmask(netmask):
    '''
    Convert a netmask to a prefix length
        0.0.0.0         -> 0
        255.255.255.0   -> 24
        255.255.255.255 -> 32
    '''

    number = long(int(IPv4Address(unicode(netmask))))
    # Count number of leading 1s
    for n in xrange(IPV4LENGTH):
        if not (number & (0x80000000 >> n)):
            return n
    return IPV4LENGTH

def binary_search(cli2obj_dict, sorted_clis, obj):
    '''
    binary search for an object based on sorted CLIs.
    @param cli2obj_dict: input dict
        mapping from the CLI of an object to the object.
    @param sorted_clis: input list of strings
        the list of CLIs sorted. Each CLI is the key to map an object in cli2obj_dict
    @param obj: input CLI string, or DMObject
        The object to be searched from cli2obj_dict
    @return None if cannot be found, otherwise the Object in cli2obj_dict that has the same CLI as obj.
    '''
    if isinstance(obj, basestring):
        cli = str(obj)
    else:
        try:
            cli = obj.get_cli()
        except Exception:
            return
    n = bisect.bisect_left(sorted_clis, cli)
    if n == len(sorted_clis):
        return
    return cli2obj_dict.get(cli)
