/*
 * Copyright (c) 2015 Cisco and/or its affiliates.  All rights reserved.
 */

package com.jabberguest.sample.custom;

import java.util.List;

import android.net.Uri;
import android.os.Bundle;
import android.app.Activity;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.util.TypedValue;
import android.view.View;
import android.view.View.OnClickListener;
import android.widget.LinearLayout;
import android.widget.ProgressBar;
import android.widget.RelativeLayout;

import com.cisco.jabber.guest.JabberGuestCall;
import com.cisco.jabber.guest.SelfView;
import com.cisco.jabber.guest.JabberGuestCall.JabberGuestInvalidCertificateCallback;
import com.cisco.jabber.guest.JabberGuestCall.State;
import com.jabberguest.sample.custom.R;

public class MainActivity extends Activity {

    // !!! Specify your own Server and Address !!!     
    public static final String DEFAULT_SERVER = "jabberguestsandbox.cisco.com";
    public static final String DEFAULT_ADDRESS = "5555";

    private Uri mCallUri;
    private JabberGuestCall mInstance;
    private ProgressBar mProgressBar;
    private LinearLayout mCallPlaceholder;
    private SelfView mSelfView;

    /**
     * Class to handle invalid certificate notifications
     */
    private class JabberGuestCertificateHandler implements JabberGuestInvalidCertificateCallback {

        @Override
        public void onInvalidCertificate(String certFingerprint,
                String identifierToDisplay,
                String certSubjectCN, String referenceID,
                List<String> invalidReason, String subjectCertificateData,
                List<String> intermediateCACertificateData,
                boolean allowUserToAccept) {

            // For this sample we are accepting all certificates
            JabberGuestCall.getInstance().acceptInvalidCertificate(referenceID, subjectCertificateData);
        }
    } 
    JabberGuestCertificateHandler mCertificateHandler = new JabberGuestCertificateHandler();

    /**
     * Receiver to handle JabberGuestCall State changes
     * This receiver is used to update the UI based on what state the call is in.
     */
    private class JabberGuestStateBroadcastReceiver extends BroadcastReceiver {

        @Override
        public void onReceive(Context context, Intent intent) {

            String action = intent.getAction();
            if (JabberGuestCall.ACTION_CALL_STATE_CHANGED.equals(action)) {

                State state = (JabberGuestCall.State) intent.getSerializableExtra(JabberGuestCall.ARG_CALL_STATE_VALUE);

                if (state == State.GuestCallStateDisconnected) {
                    // Show the SelfView
                    adjustSelfView(false);
                    //create a new instance for the next call
                    mInstance = JabberGuestCall.createInstance(context, mCallUri);
                    hideCall();
                }

                if (state == State.GuestCallStateConnected) {
                    // Hide the SelfView
                    adjustSelfView(true);
                    showCall();
                }
            }
        }
    }
    JabberGuestStateBroadcastReceiver mStateReceiver = new JabberGuestStateBroadcastReceiver();

    /*
     * (non-Javadoc)
     * @see android.app.Activity#onCreate(android.os.Bundle)
     */
    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_main);

//      The Jabber Guest SDK for Android uses H.264 AVC video, which requires activation of license agreement
//      by each user. If your application uses any of our included SDK components that use this functionality
//      (SelfView, PreviewView, PreviewFragment, or JabberGuestCallActivity), the user will be prompted to 
//      activate the video license with a DialogFragment. Once the user activates the video license, they will
//      not be prompted to do so again for subsequent calls. If the app is uninstalled or reinstalled, the video 
//      license will need to be reactivated.
//
//      If your application does not use any of those SDK components, you will have to provide the JabberGuestCall
//      instance with a valid activity context where this dialog can be displayed prior to calling start() or 
//      setSelfTextureView(TextureView textureView) on your JabberGuestCall instance; see the documentation for
//      the registerContext(Context context) method of the JabberGuestCall class for more details. Failure to
//      register a context with your JabberGuestCall instance prior to calling start() or 
//      setSelfTextureView(TextureView textureView) will result in an error being logged, and your calls will
//      not be allowed to proceed.
//        
//      If your company has it’s own license agreement for H.264 AVC video from MPEG LA, and you wish to disable
//      the display of the video license DialogFragment, place the following code in your app before calls are 
//      made to start() or setSelfTextureView(TextureView textureView) on your JabberGuestCall instance:
//        
//      JabberGuestCall.disableVideoLicenseActivation(getApplicationContext()); 

        mCallPlaceholder = (LinearLayout) findViewById(R.id.callPlaceholder);
        mProgressBar = (ProgressBar) this.findViewById(R.id.progressBar);
        mSelfView = (SelfView) this.findViewById(R.id.selfView);

        // Compose a URI to initialize the JabberGuestCall
        mCallUri = JabberGuestCall.createUri(DEFAULT_SERVER, DEFAULT_ADDRESS, null);

        // Initialize the JabberGuestCall instance for a new call
        mInstance = JabberGuestCall.createInstance(this, mCallUri);

        // Register the JabberGuestInvalidCertificateCallback
        JabberGuestCall.registerInvalidCertificateHandler(mCertificateHandler);

        
        mInstance.registerContext(this);
        
        // Since we are inflating the self view from XML, set the texture view to the instance
        // NOTE This is a current limitation of the SelfView implementation, PreviewView does
        //    have this limitation.
        mInstance.setSelfTextureView(mSelfView.getTextureView());

        // Register the JabberGuestCall State receiver
        JabberGuestCall.registerReceiver(this, mStateReceiver);

        // Handle the user clicking the button
        findViewById(R.id.callButton).setOnClickListener(new OnClickListener() {

            @Override
            public void onClick(View v) {

                // Actually start the JabberGuestCall instance
                if (mInstance != null) {
                    mInstance.start();
                }

                // Show the progress indicator as we attempt to connect
                mProgressBar.setVisibility(View.VISIBLE);
            }

        });

    }

    /**
     * Hide the Jabber Guest Call Fragment.
     */
    private void hideCall() {

        // Hide the placeholder and progress bar
        mCallPlaceholder.setVisibility(View.INVISIBLE);
        mProgressBar.setVisibility(View.INVISIBLE);
        if (mInstance != null) {
            // Reattach the SelfView to JabberGuestCall object and restart 
            // the selfview video capture.
            mInstance.setSelfTextureView(mSelfView.getTextureView());
        }
    }

    /**
     * Show the Jabber Guest Call Fragment.
     */
    private void showCall() {

        CustomCallFragment callFragment = (CustomCallFragment) getFragmentManager().findFragmentByTag("CallFragment");

        if (callFragment == null)
            callFragment = new CustomCallFragment();

        getFragmentManager().beginTransaction().replace(R.id.callPlaceholder, callFragment, "CallFragment").commit();

        mCallPlaceholder.setVisibility(View.VISIBLE);
        mProgressBar.setVisibility(View.INVISIBLE);
    }

    /*
     * (non-Javadoc)
     * @see android.app.Activity#onPause()
     */
    @Override
    public void onPause() {
        super.onPause();

        if (mInstance != null) {

            // For this sample, just end any active call when the User exits the application
            JabberGuestCall.State state = mInstance.getState();
            if (state == JabberGuestCall.State.GuestCallStateConnecting || state == JabberGuestCall.State.GuestCallStateConnected)
                mInstance.end();
        }
    }

    /**
     * Minimize or restore the SelfView to/from the bottom left corner and adjust the size
     * @param minimize
     */
    private void adjustSelfView(boolean minimize) {

        RelativeLayout.LayoutParams lp = new RelativeLayout.LayoutParams(mSelfView.getLayoutParams());
        if (minimize) {
            lp.width = (int) TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP, 120, this.getResources().getDisplayMetrics());
            lp.height = (int) TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP, 180, this.getResources().getDisplayMetrics());
            lp.addRule(RelativeLayout.ALIGN_PARENT_BOTTOM);
            lp.addRule(RelativeLayout.ALIGN_PARENT_LEFT);
        }
        else {	
            lp.width = (int) TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP, 240, this.getResources().getDisplayMetrics());
            lp.height = (int) TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP, 360, this.getResources().getDisplayMetrics());
            lp.addRule(RelativeLayout.ALIGN_PARENT_BOTTOM);
            lp.addRule(RelativeLayout.CENTER_HORIZONTAL);
        }

        mSelfView.setLayoutParams(lp);
    }

}
