/*
 File: LiveSupportView.m
 Abstract: The live support view that wraps up the CJGuestCall views.
 
 Copyright (c) 2013 Cisco Systems, Inc. All Rights Reserved.
 
 */

#import "LiveSupportView.h"

#import <JabberGuest/JabberGuest.h>


@interface LiveSupportView () {
    UIImageView * remoteView;
    UIImageView * remotePresoView;
    UIActivityIndicatorView * aiView;
    CJGuestCallBarView * callBar;
    BOOL callTriedToConnect;
}
@end

@implementation LiveSupportView

- (id)initWithFrame:(CGRect)frame
{
    self = [super initWithFrame:frame];
    
    if (self) {
        remoteView = [[UIImageView alloc] initWithFrame:CGRectMake(0, 0, frame.size.width, frame.size.height)];
        [remoteView setBackgroundColor:[UIColor blackColor]];
        [remoteView setContentMode:UIViewContentModeScaleAspectFill];
        [remoteView setClipsToBounds:YES];
        [self addSubview:remoteView];
        
        //UImageview for remote share
        remotePresoView = [[UIImageView alloc] initWithFrame:CGRectMake(0, 0, frame.size.width, frame.size.height)];
        [remoteView setBackgroundColor:[UIColor blackColor]];
        [remoteView setContentMode:UIViewContentModeScaleAspectFill];
        [remoteView setClipsToBounds:YES];
        [self addSubview:remotePresoView];
        
        aiView = [[UIActivityIndicatorView alloc] initWithActivityIndicatorStyle:UIActivityIndicatorViewStyleWhiteLarge];
        
        CGRect aiFrame = aiView.frame;
        aiFrame.origin.x = remoteView.frame.origin.x + ((remoteView.frame.size.width - aiView.frame.size.width) / 2.0);
        aiFrame.origin.y = remoteView.frame.origin.y + ((remoteView.frame.size.width - aiView.frame.size.width) / 2.0);
        aiView.frame = aiFrame;
        
        [self addSubview:aiView];
        [aiView startAnimating];
        
        callTriedToConnect = FALSE;
        
        [[NSNotificationCenter defaultCenter] addObserver:self
                                                 selector:@selector(handleCallStateChange:)
                                                     name:CJGuestCallStateChangeNotification
                                                   object:nil];
        
        [[NSNotificationCenter defaultCenter] addObserver:self
                                                 selector:@selector(handleShareActiveStateChange:)
                                                     name:CJGuestCallRemotePresoVideoActiveNotification
                                                   object:nil];
        
        // The Jabber Guest SDK for iOS uses H.264 AVC video, which requires activation of a license agreement
        // by each user. The user will be prompted to activate the video license with an AlertView.
        //
        // If your company has it’s own license agreement for H.264 AVC video from MPEG LA, and you wish to
        // disable the display of the video license AlertView, you may call the disableActivation API (see below)
        //
        // If you disable the built-in activation capability of the Jabber SDK for iOS, you do so at your own risk
        // and are responsible for any royalties or other fees associated with your use of the H.264 AVC video codec.
        //
        // [[CJGuestVideoLicense sharedInstance] disableActivation];
        
        CJGuestCall * call = [CJGuestCall sharedInstance];
        [call setServerName:@"jabberguestsandbox.cisco.com"];
        [call setToURI:@"5555"];
        
        call.remoteView = remoteView;
        call.remotePresoView = remotePresoView;
        
        [call startCall: ^(BOOL isActivated) {
            if (!isActivated) {
                [_delegate supportCallIsFinished];
            }
        }];
    }
    
    return self;
}

- (void)layoutCJGuestCallViews
{
    [[CJGuestCall sharedInstance] setRemoteView:remoteView];
    
    CGPoint callBarOrigin;
    callBarOrigin.x = remoteView.frame.origin.x + ((remoteView.frame.size.width - ([CJGuestCallBarView callBarButtonSize].width * 2)) / 2.0);
    callBarOrigin.y = remoteView.frame.origin.y + remoteView.frame.size.height - [CJGuestCallBarView callBarButtonSize].height;
    callBar = [[CJGuestCallBarView alloc] initWithOrigin:callBarOrigin
                                                    mask:GuestCallBarButtonOptionAudioMute | GuestCallBarButtonOptionHook];
    callBar.delegate = self;
    [self addSubview:callBar];
}

- (void)handleCallStateChange:(NSNotification *)notification
{
    CJGuestCall * call = [CJGuestCall sharedInstance];
    
    switch (call.callState) {
        case GuestCallStateConnecting:
            callTriedToConnect = TRUE;
            break;
        case GuestCallStateConnected:
            [aiView stopAnimating];
            [self layoutCJGuestCallViews];
            break;
        case GuestCallStateDisconnecting:
        case GuestCallStateDisconnected:
            if (callTriedToConnect)
                [_delegate supportCallIsFinished];
            break;
        default:
            break;
    }
}

- (void)handleShareActiveStateChange:(NSNotification *)notification
{
    // CJGuestCall * call = [CJGuestCall sharedInstance];
    CJGuestCall *call = [ notification object];
    
    if(call.remotePresoActive) {
        remotePresoView.hidden = NO;
    } else {
        remotePresoView.hidden = YES;
    }
    
}

- (void)dealloc
{
    [[NSNotificationCenter defaultCenter] removeObserver:self
                                                    name:CJGuestCallStateChangeNotification
                                                  object:nil];
    [[NSNotificationCenter defaultCenter] removeObserver:self
                                                    name:CJGuestCallRemotePresoVideoActiveNotification
                                                  object:nil];
    
}

@end
