/**************************************************************************
*       Copyright (c) 2008, Cisco Systems, All Rights Reserved
***************************************************************************
*
*  File:    VpnStatsBase.h
*  Date:    01/2008
*
***************************************************************************
*
*   VPN Statistics base class implementation for the Client API.
*
***************************************************************************/
#ifndef _VPNSTATSBASE_
#define _VPNSTATSBASE_

#include <list>
#if defined(_WIN32) && (_MSC_VER < 1800)
typedef unsigned long   uint32_t;
#else
#include <stdint.h>
#endif

#include "ProtocolInfo.h"
#include "RouteInfo.h"
#include "api.h"


// This is the base class for VPNStats.  It provides underlying functionality
// and data storage.  The intent is to expose only what is needed by a user
// of the API in the VPNStats class.
//

class CStatisticsTlv;
class CStateTlv;
class CSessionInfoTlv;
class CManualLock;
class FirewallInfo;
class CExtensibleStats;

class VPN_VPNAPI VPNStatsBase
{
    public:

        // various ctor
        //
        VPNStatsBase();
        VPNStatsBase(const VPNStatsBase &existing);
        VPNStatsBase & operator= (const VPNStatsBase &);

        virtual ~VPNStatsBase();

        // get a list of Secure Routes.
        // Method returns a list of RouteInfo pointers
        //
        const std::list<RouteInfo *> &getSecureRoutes();

        // Returns the number of applied secure routes for the specified IP protocol.
        // This may be different from the number of routes
        // pushed to UI, which cannot exceed 200.
        //
        uint16_t getAppliedSecureRouteCount(bool bIPv6)
        {
            return (bIPv6 ? m_uiAppliedSecureRouteCountV6 :
                            m_uiAppliedSecureRouteCountV4);
        }

        // get a list of Nonsecure Routes.
        // Method returns a list of RouteInfo pointers
        //
        const std::list<RouteInfo *> &getNonsecureRoutes();

        // Returns the number of applied non-secure routes for the specified IP protocol.
        // This may be different from the number of routes
        // pushed to UI, which cannot exceed 200.
        uint16_t getAppliedNonsecureRouteCount(bool bIPv6)
        {
            return (bIPv6 ? m_uiAppliedNonsecureRouteCountV6 :
                            m_uiAppliedNonsecureRouteCountV4);
        }

        // get a list of protocol Info objects.
        // Method returns a list of ProtocolInfo pointers
        //
        const std::list<ProtocolInfo *> &getProtocolInfo();

        // get a list of firewall Info objects.
        // Method returns a list of FirewallInfo pointers
        //
        const std::list<FirewallInfo *> &getFirewallInfo();

        // method used to retireve statistices by name.
        //
        const tstring &getStatValue(tstring &label);

        void setConnectionStatistics(CStatisticsTlv *connectionStats);

        bool ClearConnectionStatistics();

        bool ClearVPNSessionStats();

        // these two methods are helpers used to make sure SessionInfo
        // is available.
        //
        bool hasSessionInfo();

        bool IsFirewallInfoEqual (VPNStatsBase &stats); 
        
        void setDisconnected();

        // allow for updating state data.  Returns the state that should
        // be displayed to the user.
        //
        // TODO: Paul agreed to removing dependency on Gendefs
        // he'll perform the actual commit for this
        bool setStateInfo(CStateTlv* stateInfo,
                          STATE& rSessionState,
                          VPNCON_SUBSTATE& rSessionSubState,
                          STATE& rMgmtTunSessionState,
                          VPNCON_SUBSTATE& rMgmtTunSessionSubState,
                          NETENV_STATE& rNetEnvState,
                          NETCTRL_STATE& rNetCtrlState,
                          NETWORK_TYPE& rNetType);

        // allow for updating sessionInfo data 
        //
        void setSessionInfo(CSessionInfoTlv *sessionInfo);

        // allow for updating strictly TND related data
        //
        void setTNDInfo(CSessionInfoTlv& sessionInfo);

        // set the client operating Mode
        //
        void setClientOperatingMode(uint32_t operatingMode);

        static const tstring::value_type* const* getVpnStatsStringTable();

        void StreamStats(tostream &outStream);

protected:
        static const tstring::value_type* const sm_pVpnStatsStringTable[];

private:
        ApiStringMap mo_StatNameValue;
        ApiStringMap mo_SessionStats;

        tstring ms_CurrentState;
        tstring ms_CurrentMgmtTunState;

        bool mb_GotSessionInfo;

        std::list<RouteInfo *> mlop_SecureRoutes;
        uint16_t m_uiAppliedSecureRouteCountV4;
        uint16_t m_uiAppliedSecureRouteCountV6;

        std::list<RouteInfo *> mlop_NonsecureRoutes;
        uint16_t m_uiAppliedNonsecureRouteCountV4;
        uint16_t m_uiAppliedNonsecureRouteCountV6;

        std::list<ProtocolInfo *> mlop_ProtocolInfo;
    
        std::list<FirewallInfo *> mlop_FirewallInfo;
    
        static CManualLock sm_statsLock;

        static tstring EmptyString;
        static tstring NotAvailable;

        ApiStringMap &getStatsMap();
        ApiStringMap &getSessionStatsMap();

        void setStatValue(tstring &label, tstring value);
        void setStatValue(ApiStringMap &map, tstring &label, tstring value);

        void synchronizeLists(const VPNStatsBase &existing);

        void clearProtocolInfo();
        void clearSecureRoutes();
        void clearNonSecureRoutes();
        void clearFirewallInfo();

        void init();

        void setIndividualStats(CExtensibleStats *stats);

        friend VPN_VPNAPI tostream& operator<<(tostream &outStream,
                                               VPNStatsBase &stats);

};


#endif // _VPNSTATSBASE_
