/********************************************************************************\
*  This sample is supplied as is with no implied warranty.  
*  It is designed to assist you in using the Cisco AnyConnect VPN API. 
*  It is assumed that you will build a production application and 
*  refer to this sample as a reference only.
\********************************************************************************/
#include <map>
#include "stdafx.h"
#include "CppComSample.h"
#include "CppComSampleDlg.h"

#include "EventCallback.h"

#include "resource.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#endif

#define IDC_COMBO_TUNNELGROUP 9000

//maximum time to wait for the vpndownloader to perform an upgrade of the vpn software.
//
static const unsigned int sMaxTimeToWaitForUpdate = 120000; //2-minutes

//|||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
//
//Uncomment the next line to run the VPN API COM Server INPROC
//#define COM_INPROC_EXAMPLE
//
//|||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
    //NOTE: In this example, the COM object is hosted out of process by design.
    //      Additionally, the threading for COM is set for STA in this sample.
    //      As a result of both of these items, COM notifications will be
    //      serialized to your UI thread.
    //
    //      This removes the need for you to worry about threading and also 
    //      provides a layer of fault isolation between your application and
    //      the VPN API.  For example, you might be writing a module whose
    //      primary role is not VPN related, and want to ensure the component
    //      is isolated from VPN components (and vice-versa).
    //
    //      If you choose to explicitly host the API COM server INPROC, using
    //      an STA threading model, then uncomment the #define COM_INPROC_EXAMPLE
    //      at the top of this file.  This allows you to use the COM API
    //      in a manner that processes events on your UI thread.  This is accomplished
    //      by setting the EnableConsumerDrivenEventModel = VARIANT_TRUE
    //      Notice that the EnableConsumerDrivenEventModel should always been set to TRUE
    //      as the old event model is deprecated.
    //
    //      THIS EXAMPLE DEMONSTRATES STA / OUT-OF-PROC COM BY DEFAULT.
    //


/***************************************************************************************/
// CAboutDlg dialog used for App About
class CAboutDlg : public CDialog
{
public:
    CAboutDlg();

    enum { IDD = IDD_ABOUTBOX };

    protected:
    virtual void DoDataExchange(CDataExchange* pDX);    // DDX/DDV support

// Implementation
protected:
    DECLARE_MESSAGE_MAP()
};

CAboutDlg::CAboutDlg() : CDialog(CAboutDlg::IDD)
{
}

void CAboutDlg::DoDataExchange(CDataExchange* pDX)
{
    CDialog::DoDataExchange(pDX);
}

BEGIN_MESSAGE_MAP(CAboutDlg, CDialog)
END_MESSAGE_MAP()

/***************************************************************************************/
// CCppComSampleDlg dialog
CCppComSampleDlg::CCppComSampleDlg(CWnd* pParent /*=NULL*/)
    : CDialog(CCppComSampleDlg::IDD, pParent),
      m_currentVpnState(UNKNOWN),
      m_hWatchHandle(NULL),
      m_dwPreviousUpgradeState(RegValueUpgradeNotInProgress)
{
    m_hIcon = AfxGetApp()->LoadIcon(IDR_MAINFRAME);
    m_hOptionsIcon = AfxGetApp()->LoadIconW(IDI_OPTIONS);
    m_IDArray[0] = IDS_NOTIFICATION;

    m_stateStr.LoadString(IDS_UNKNOWN);
    m_serviceStr.LoadString(IDS_UNAVAILABLE);
}

//WaitOrTimer callback function used to monitor the upgrade registry key
/*static*/
void CALLBACK CCppComSampleDlg::RegistryWaitCallback(void * pThis, BOOLEAN bTimedOut)
{
    //Since the registry notification cannot be cancelled from the callback, serialize
    //the message to the main UI thread as a notification.
    if (NULL != pThis)
    {
        reinterpret_cast<CCppComSampleDlg*>(pThis)->PostMessage(WM_VPN_REGISTRY_NOTIFICATION, bTimedOut);
    }
}

//Called as a result of a value in the vpn update registry key being changed.
LRESULT CCppComSampleDlg::OnVpnRegistryNotification(WPARAM bTimedOut, LPARAM /*not used*/)
{
    if (bTimedOut)
    {
        //Timedout waiting for the upgrade to complete.
        //
        if (NULL != m_hWatchHandle)
        {
            ::UnregisterWait(m_hWatchHandle);
            m_hWatchHandle = NULL;
        }

        m_dwPreviousUpgradeState = RegValueUpgradeError;

        MessageBox(_T("The VPN Software failed during upgrade, contact your system administrator."), _T("ERROR"), MB_OK | MB_ICONERROR);
        CString strPaneText;
        strPaneText.LoadString(IDS_NOTIFICATION);
        strPaneText += _T(" The VPN Software failed during upgrade, contact your system administrator.");
        m_StatusBar.SetPaneText(0, strPaneText);
    }
    else
    {
        //Did the upgrade complete?
        //
        DWORD dwCurrentUpdateValue = RegValueUpgradeNotInProgress;
        m_keyUpgrade.QueryDWORDValue(m_strUpgradeValueName, dwCurrentUpdateValue);

        if (RegValueUpgradeInProgress != dwCurrentUpdateValue 
                && RegValueUpgradeInProgress == m_dwPreviousUpgradeState)
        {
            //Release the registry watch resource if active
            //
            if (NULL != m_hWatchHandle)
            {
                ::UnregisterWait(m_hWatchHandle);
                m_hWatchHandle = NULL;
            }

            //If there is not an active upgrade in progress,
            //try to re-create the VPN API.  If the upgrade failed,
            //then the attempt to create the COM API will catch that case.
            //
            StartupCOMVpnApi();
        }
        else
        {
            //Re-register the key watch.
            LONG lRet = m_keyUpgrade.NotifyChangeKeyValue(true, REG_NOTIFY_CHANGE_LAST_SET, 
                                                                  (HANDLE)m_eventUpgradeComplete);
            if (lRet != ERROR_SUCCESS)
            {
                MessageBox(_T("Failed to register the notification for the registry value."), _T("ERROR"), MB_OK | MB_ICONERROR);
            }
        }

        m_dwPreviousUpgradeState = dwCurrentUpdateValue;
    }

    return 0;
}


//Called as a result of a banner arrival
LRESULT CCppComSampleDlg::OnBannerAvailableNotification(WPARAM /*not used*/, LPARAM /*not used*/)
{
    if (!m_spVpnApi)
    {
        return 0; //nothing to do here as the API is not running anymore.
    }

    CBannerDlg dlg(m_strBannerMsg);
    INT_PTR nReturn = dlg.DoModal();
    
    try
    {
        m_spVpnApi->BannerResponse = ((nReturn == IDOK) ? VARIANT_TRUE : VARIANT_FALSE);
    }
    catch (_com_error err)
    {
        CString strErr; 
        strErr.Format(_T("Failed to set the banner response, Error = 0x%.8X, [%s] description: %s"),
            err.Error(), (LPCTSTR)err.Source(), (LPCTSTR)err.Description());                              
        MessageBox(strErr, _T("ERROR"), MB_OK | MB_ICONERROR);
    }

    return 0;
}

//Called as a result of a pre-conect reminder message arrival
LRESULT CCppComSampleDlg::OnPreConnectReminderAvailableNotification(WPARAM /*not used*/, LPARAM /*not used*/)
{
    if (!m_spVpnApi)
    {
        return 0; //nothing to do here as the API is not running anymore.
    }

    INT_PTR nReturn = MessageBox(m_strPreConnectReminderMsg, _T("Cisco AnyConnect"), MB_OK);
    try
    {
        m_spVpnApi->PreConnectReminderResponse = ((nReturn == IDOK) ? VARIANT_TRUE : VARIANT_FALSE);
    }
    catch (_com_error err)
    {
        CString strErr; 
        strErr.Format(_T("Failed to set the pre-connect reminder response, Error = 0x%.8X, [%s] description: %s"),
            err.Error(), (LPCTSTR)err.Source(), (LPCTSTR)err.Description());                              
        MessageBox(strErr, _T("ERROR"), MB_OK | MB_ICONERROR);
    }

    return 0;
}


//Called as a result untrusted server cert being blocked
LRESULT CCppComSampleDlg::OnCertBlockedNotification(WPARAM /*not used*/, LPARAM /*not used*/)
{
    if (!m_spVpnApi)
    {
        return 0; //nothing to do here as the API is not running anymore.
    }

    CCertBlockedDlg dlg(m_strUntrustedServer);
    INT_PTR nReturn = dlg.DoModal();
    
    try
    {
        m_spVpnApi->CertBlockedResponse = ((nReturn == IDC_CHANGE_SETTING) ? VARIANT_TRUE : VARIANT_FALSE);
        // enable button so that user can attempt connect again
        m_ConnectButton.EnableWindow(TRUE);
    }
    catch (_com_error err)
    {
        CString strErr; 
        strErr.Format(_T("Failed to show cert blocked error, Error = 0x%.8X, [%s] description: %s"),
            err.Error(), (LPCTSTR)err.Source(), (LPCTSTR)err.Description());                              
        MessageBox(strErr, _T("ERROR"), MB_OK | MB_ICONERROR);
    }

    return 0;
}

LRESULT CCppComSampleDlg::OnCertWarningNotification(WPARAM /*not used*/, LPARAM /*not used*/)
{
    if (!m_spVpnApi)
    {
        return 0; //nothing to do here as the API is not running anymore.
    }

    CCertWarningDlg dlg(m_strUntrustedServer, m_lCertErrors, m_bImportAllowed);
    INT_PTR nReturn = dlg.DoModal();
    
    try
    {
        if (IDC_CONNECT_ANYWAY == nReturn)
        {
            m_spVpnApi->PutCertWarningResponse(VARIANT_TRUE, VARIANT_FALSE);
        }
        else if (IDC_ALWAYS_CONNECT == nReturn)
        {
            m_spVpnApi->PutCertWarningResponse(VARIANT_TRUE, VARIANT_TRUE);
        }
        else
        {
            m_spVpnApi->PutCertWarningResponse(VARIANT_FALSE, VARIANT_FALSE);
            m_ConnectButton.EnableWindow(TRUE);
        }
    }
    catch (_com_error err)
    {
        CString strErr; 
        strErr.Format(_T("Failed to show cert warning, Error = 0x%.8X, [%s] description: %s"),
            err.Error(), (LPCTSTR)err.Source(), (LPCTSTR)err.Description());                              
        MessageBox(strErr, _T("ERROR"), MB_OK | MB_ICONERROR);
    }

    return 0;
}

void CCppComSampleDlg::DoDataExchange(CDataExchange* pDX)
{
    CDialog::DoDataExchange(pDX);
    DDX_Text(pDX, IDC_STATESTRING, m_stateStr);
    DDX_Text(pDX, IDC_SERVICESTRING, m_serviceStr);
    DDX_Control(pDX, IDC_COMBO_HOSTS, m_HostSelectionComboBox);
    DDX_Control(pDX, IDC_BUTTON_OPTIONS, m_OptionsButton);
    DDX_Control(pDX, IDC_BUTTON_SUBMIT, m_SubmitButton);
    DDX_Control(pDX, IDC_BUTTON_CONNECT, m_ConnectButton);
}

BEGIN_MESSAGE_MAP(CCppComSampleDlg, CDialog)
    ON_WM_SYSCOMMAND()
    ON_WM_PAINT()
    ON_WM_QUERYDRAGICON()
    ON_WM_DESTROY()
    //}}AFX_MSG_MAP
    ON_BN_CLICKED(IDC_BUTTON_SHOW_VPN_STATS, &CCppComSampleDlg::OnBnClickedButtonShowVpnStats)
    ON_BN_CLICKED(IDC_BUTTON_SHOW_ROUTES, &CCppComSampleDlg::OnBnClickedShowRouteDetail)
    ON_BN_CLICKED(IDC_BUTTON_OPTIONS, &CCppComSampleDlg::OnBnClickedOptions)
    ON_CBN_SELCHANGE(IDC_COMBO_HOSTS, &CCppComSampleDlg::OnHostsComboBoxSelectionChanged)
    ON_BN_CLICKED(IDC_FIREWALLRULES, &CCppComSampleDlg::OnBnClickedFirewallrules)
    ON_BN_CLICKED(IDC_BUTTON_CONNECT, &CCppComSampleDlg::OnBnClickedButtonConnect)

    //This is called when an upgrade-complete registry key is changed.
    //
    ON_MESSAGE(WM_VPN_REGISTRY_NOTIFICATION, &CCppComSampleDlg::OnVpnRegistryNotification)

    //This is used if VpnApi::put_EnableConsumerDrivenEventModel is set to true.
    //
    ON_MESSAGE(WM_VPN_EVENT_AVAILABLE, &CCppComSampleDlg::OnVpnEventAvailable)

    ON_BN_CLICKED(IDC_BUTTON_SUBMIT, &CCppComSampleDlg::OnBnClickedButtonSubmit)
    ON_BN_CLICKED(IDC_BUTTON_SHOW_PROTOCOL, &CCppComSampleDlg::OnBnClickedButtonShowProtocol)

    //This is called when the group combobox selection changes.
    //
    ON_CBN_SELCHANGE(IDC_COMBO_TUNNELGROUP, &CCppComSampleDlg::OnPromptComboBoxSelectionChanged)
    ON_MESSAGE(WM_VPN_GROUP_CHANGED, &CCppComSampleDlg::OnGroupChanged)

    //This is called when a banner needs to be displayed
    //
    ON_MESSAGE(WM_VPN_BANNER_AVAILABLE, &CCppComSampleDlg::OnBannerAvailableNotification)

    //This is called when a pre-connect reminder needs to be displayed
    //
    ON_MESSAGE(WM_VPN_PRECONNECT_REMINDER_AVAILABLE, &CCppComSampleDlg::OnPreConnectReminderAvailableNotification)

    ON_MESSAGE(WM_VPN_CERT_BLOCKED, &CCppComSampleDlg::OnCertBlockedNotification)

    ON_MESSAGE(WM_VPN_CERT_WARNING, &CCppComSampleDlg::OnCertWarningNotification)
    
END_MESSAGE_MAP()


// CCppComSampleDlg message handlers

BOOL CCppComSampleDlg::OnInitDialog()
{
    CDialog::OnInitDialog();

    // IDM_ABOUTBOX must be in the system command range.
    ASSERT((IDM_ABOUTBOX & 0xFFF0) == IDM_ABOUTBOX);
    ASSERT(IDM_ABOUTBOX < 0xF000);

    CMenu* pSysMenu = GetSystemMenu(FALSE);
    if (pSysMenu != NULL)
    {
        CString strAboutMenu;
        strAboutMenu.LoadString(IDS_ABOUTBOX);
        if (!strAboutMenu.IsEmpty())
        {
            pSysMenu->AppendMenu(MF_SEPARATOR);
            pSysMenu->AppendMenu(MF_STRING, IDM_ABOUTBOX, strAboutMenu);
        }
    }

    // Set the icon for this dialog.  The framework does this automatically
    //  when the application's main window is not a dialog.
    SetIcon(m_hIcon, TRUE);			// Set big icon.
    SetIcon(m_hIcon, FALSE);		// Set small icon.

    //Load the Icon for the Options button.
    m_OptionsButton.SetIcon(m_hOptionsIcon);

    // Create the VPN Stats / Routes / Protocol Dlg
    m_StatsDlg.Create(m_StatsDlg.IDD, this);
    m_RoutesDlg.Create(m_RoutesDlg.IDD, this);
    m_ProtocolDlg.Create(m_ProtocolDlg.IDD, this);
    m_FirewallDlg.Create(m_FirewallDlg.IDD, this);

    // Create status bar at the bottom of the dialog window.
    if (m_StatusBar.Create(this))
    {
        m_StatusBar.SetIndicators(m_IDArray, ARRAYSIZE(m_IDArray));

        // Make a sunken or recessed border around the first pane.
        m_StatusBar.SetPaneInfo(0, 
                                m_StatusBar.GetItemID(0),
                                SBPS_STRETCH, 0 );
    }

    // Resize the dialog to make room for control bar.
    // First, figure out how big the control bar is.
    CRect rectOriginal;
    CRect rectFinal;
    GetClientRect(rectOriginal);
    RepositionBars(AFX_IDW_CONTROLBAR_FIRST, 
                   AFX_IDW_CONTROLBAR_LAST,
                   0, reposQuery, rectFinal);

    // Now move all the controls so they are in the same relative
    // position within the remaining client area as they would be
    // with no control bars.
    CPoint ptOffset(rectFinal.left - rectOriginal.left,
                    rectFinal.top - rectOriginal.top);

    CRect  rcChild;
    CWnd* pwndChild = GetWindow(GW_CHILD);
    while (pwndChild)
    {
        pwndChild->GetWindowRect(rcChild);
        ScreenToClient(rcChild);
        rcChild.OffsetRect(ptOffset);
        pwndChild->MoveWindow(rcChild, FALSE);
        pwndChild = pwndChild->GetNextWindow();
    }

    // Adjust the dialog window dimensions.
    CRect rcWindow;
    GetWindowRect(rcWindow);
    rcWindow.right += rectOriginal.Width() - rectFinal.Width();
    rcWindow.bottom += rectOriginal.Height() - rectFinal.Height();
    MoveWindow(rcWindow, FALSE);

    // And position the control bars.
    RepositionBars(AFX_IDW_CONTROLBAR_FIRST, AFX_IDW_CONTROLBAR_LAST, 0);


    ///Initialize our COM VPN API.
    StartupCOMVpnApi();

    return TRUE;
}

void CCppComSampleDlg::OnSysCommand(UINT nID, LPARAM lParam)
{
    if ((nID & 0xFFF0) == IDM_ABOUTBOX)
    {
        CAboutDlg dlgAbout;
        dlgAbout.DoModal();
    }
    else
    {
        CDialog::OnSysCommand(nID, lParam);
    }
}

void CCppComSampleDlg::OnPaint()
{
    if (IsIconic())
    {
        CPaintDC dc(this); // Device context for painting

        SendMessage(WM_ICONERASEBKGND, reinterpret_cast<WPARAM>(dc.GetSafeHdc()), 0);

        // Center icon in client rectangle
        int cxIcon = GetSystemMetrics(SM_CXICON);
        int cyIcon = GetSystemMetrics(SM_CYICON);
        CRect rect;
        GetClientRect(&rect);
        int x = (rect.Width() - cxIcon + 1) / 2;
        int y = (rect.Height() - cyIcon + 1) / 2;

        // Draw the icon
        dc.DrawIcon(x, y, m_hIcon);
    }
    else
    {
        CDialog::OnPaint();
    }
}

HCURSOR CCppComSampleDlg::OnQueryDragIcon()
{
    return static_cast<HCURSOR>(m_hIcon);
}

void CCppComSampleDlg::OnDestroy()
{
    ResetCredentialsUI();

    ///Shutdown our COM VPN API.
    ShutdownCOMVpnApi();

    //Release the registry watch resource if active.
    //
    if (NULL != m_hWatchHandle)
    {
        ::UnregisterWait(m_hWatchHandle);
        m_hWatchHandle = NULL;
    }
}

void CCppComSampleDlg::ResetStrings(void)
{
    CString strPaneText;
    strPaneText.LoadString(IDS_NOTIFICATION);
    m_stateStr.LoadString(IDS_UNKNOWN);
    m_serviceStr.LoadString(IDS_UNAVAILABLE);
    UpdateData(FALSE);
}


//INITIALIZES THE COM VPN API.
//
HRESULT CCppComSampleDlg::StartupCOMVpnApi(void)
{
    HRESULT hr = S_OK;
    
    CString strPaneText;

    //Create the VPNAPI COM Component.
    if (!m_spVpnApi)
    {
        strPaneText.LoadString(IDS_NOTIFICATION);
        CString strTemp;
        strTemp.LoadString(IDS_API_INITIALIZING);
        strPaneText += _T(" ") + strTemp;
        m_StatusBar.SetPaneText(0, strPaneText);

#if defined(COM_INPROC_EXAMPLE)
        hr = m_spVpnApi.CoCreateInstance(__uuidof(VpnApi)); //run INPROC
#else
        hr = m_spVpnApi.CoCreateInstance(__uuidof(VpnApi), 0, CLSCTX_LOCAL_SERVER); //run OUTOFPROC
#endif
        if (FAILED(hr))
        {
            MessageBox(_T("Failed to create the VpnApi COM Component"), _T("ERROR"), MB_OK | MB_ICONERROR);
            strPaneText.LoadString(IDS_NOTIFICATION);
            strPaneText += _T(" Failed to initialize the VPN API COM Server.");
            m_StatusBar.SetPaneText(0, strPaneText);
            return hr;
        }
    }
    else
    {
        MessageBox(_T("The VPN COM API is already initialized"), _T("INFORMATION"), MB_OK | MB_ICONINFORMATION);
        return S_FALSE; //Already running, so return a non-failure HRESULT.
    }

    //Now obtain the COM interface of this object to pass to the COM VPN API Server.
    CComPtr<IVpnApiEvents> spVpnApiEvents = NULL;
    hr = QueryInterface(__uuidof(IVpnApiEvents), 
                        reinterpret_cast<void**>(&spVpnApiEvents));
    if (FAILED(hr))
    {
        MessageBox(_T("CCppComSampleDlg::QueryInterface - failed"), _T("ERROR"), MB_OK | MB_ICONERROR);
    }
    else
    {
        //Attach the COM VPN API interface to the VPN Stats dialog.
        m_StatsDlg.AttachVpnApi(m_spVpnApi);

        try
        {
           /* NOTE: USE THE RAW INTERFACES IF YOU DON'T WANT EXCEPTIONS.
              For example, m_spVpnApi->raw_RegisterAndAttach()
           */
            //Enable the consumer driven event model.
            //It's required to enable this as the old event model is deprecated.
            m_spVpnApi->EnableConsumerDrivenEventModel = VARIANT_TRUE;

            m_spVpnApi->RegisterAndAttach(spVpnApiEvents, 
                                          ClientType_GUI,
                                          VARIANT_TRUE,   /*request master*/
                                          VARIANT_TRUE);  /*don't perform auto connect to last host*/
        }
        catch(_com_error err)
        {
            hr = err.Error();
            strPaneText.LoadString(IDS_NOTIFICATION);
            strPaneText += _T(" Failed to Register and Attach to the VPN API COM Server.");
            m_StatusBar.SetPaneText(0, strPaneText);
            CString strErr; 
            strErr.Format(_T("Failed to initialize the VPN API engine, error = 0x%.8X, [%s] description: %s"),
                          hr, (LPCTSTR)err.Source(), (LPCTSTR)err.Description());                              
            MessageBox(strErr, _T("ERROR"), MB_OK | MB_ICONERROR);
        }
    }

    return hr;
}

//SHUTSDOWN THE COM VPN API.
//
HRESULT CCppComSampleDlg::ShutdownCOMVpnApi(void)
{
    if (!m_spVpnApi)
    {
        return S_FALSE; //nothing to do here as the API is not running.
    }

    HRESULT hr = S_OK;

    try
    {
        /* NOTE: USE THE RAW INTERFACES IF YOU DON'T WANT EXCEPTIONS.
           For example, m_spVpnApi->raw_UnregisterAndDetach()
       */
        m_spVpnApi->UnregisterAndDetach();

        //explicitly release the API.
        m_spVpnApi.Release();

        //tell COM to immediately unload the VPN API COM binaries.
        ::CoFreeUnusedLibrariesEx(0,0); //only supported on XP and later.  
                                        //Win2k is not as easily controlled,
                                        //and left as an exercise to the user.
    }
    catch(_com_error err)
    {
        hr = err.Error();
        CString strPaneText;
        strPaneText.LoadString(IDS_NOTIFICATION);
        strPaneText += _T(" Failed to shutdown the VPN API COM Server.");
        m_StatusBar.SetPaneText(0, strPaneText);
        CString strErr; 
        strErr.Format(_T("Failed to shutdown the VPN API engine, error = 0x%.8X, [%s] description: %s"),
                      hr, (LPCTSTR)err.Source(), (LPCTSTR)err.Description());            
        MessageBox(strErr, _T("ERROR"), MB_OK | MB_ICONERROR);
    }

    ResetStrings();

    return hr;
}


void CCppComSampleDlg::OnBnClickedButtonShowVpnStats()
{
    m_StatsDlg.ShowWindow(SW_SHOW);
}

void CCppComSampleDlg::OnBnClickedShowRouteDetail()
{
    m_RoutesDlg.ShowWindow(SW_SHOW);
}

void CCppComSampleDlg::OnBnClickedButtonShowProtocol()
{
    m_ProtocolDlg.ShowWindow(SW_SHOW);
}

void CCppComSampleDlg::OnBnClickedFirewallrules()
{
   m_FirewallDlg.ShowWindow(SW_SHOW);
}




void CCppComSampleDlg::OnBnClickedOptions()
{
    // TODO: Add your control notification handler code here.
}

//The 'submit' button is used for both connect and disconnect.
//
void CCppComSampleDlg::OnBnClickedButtonSubmit()
{
    //disable the credentials UI.
    DisableCredentialsUI();

    //If the COM API is not initialized, just return. Nothing to do here.
    //
    if (!m_spVpnApi)
    {
        return;
    }

    //Determine whether this is a connect or disconnect.
    //If the IConnectPromptInfo object is valid, then this is a connect attempt.
    //
    if (m_spConnectPromptInfo)
    {
        try
        {
            //Copy values from credential controls to the API object.
            //
            StoreCredentialsUI(m_spConnectPromptInfo);

            //Release the IConnectPromptInfo object, then call VpnApi::UserSubmit.
            //
            m_spConnectPromptInfo.Release();
            m_spVpnApi->UserSubmit();
        }
        catch(_com_error err)
        {
            HRESULT hr = err.Error();
            CString strPaneText;
            strPaneText.LoadString(IDS_NOTIFICATION);
            strPaneText += _T(" Failed Submitting the credentials.");
            m_StatusBar.SetPaneText(0, strPaneText);
            CString strErr; 
            strErr.Format(_T("Failed Submitting the credentials., error = 0x%.8X, [%s] description: %s"),
                          hr, (LPCTSTR)err.Source(), (LPCTSTR)err.Description());                              
            MessageBox(strErr, _T("ERROR"), MB_OK | MB_ICONERROR);

            //Reset the controls.
            ResetCredentialsUI();
        }
    }
    else
    {
        //Check for a connected status.
        try
        {
            if (m_spVpnApi->IsConnected)
            {
                m_spVpnApi->DisconnectVpn();
            }

            //Reset the controls.
            ResetCredentialsUI();
        }
        catch(_com_error err)
        {
            HRESULT hr = err.Error();
            CString strPaneText;
            strPaneText.LoadString(IDS_NOTIFICATION);
            strPaneText += _T(" Failed to disconnect the vpn session.");
            m_StatusBar.SetPaneText(0, strPaneText);
            CString strErr; 
            strErr.Format(_T("Failed to disconnect the vpn session., error = 0x%.8X, [%s] description: %s"),
                          hr, (LPCTSTR)err.Source(), (LPCTSTR)err.Description());                              
            MessageBox(strErr, _T("ERROR"), MB_OK | MB_ICONERROR);
        }
    }
}

void CCppComSampleDlg::OnHostsComboBoxSelectionChanged()
{
    //When the user changes the selection, clear the credential related UI fields
    // and perform a connection operation (IVpnApi::Connect).
    //
    ResetCredentialsUI();
    
    int nItem = m_HostSelectionComboBox.GetCurSel();
    if (CB_ERR != nItem)
    {
        CString strHostSelected;
        m_HostSelectionComboBox.GetLBText(nItem, strHostSelected);
        
        _bstr_t bstrHostSelected((LPCTSTR)strHostSelected); 
        try
        {
            m_spVpnApi->ConnectVpn(bstrHostSelected);
        }
        catch(_com_error err)
        {
            HRESULT hr = err.Error();
            CString strPaneText;
            strPaneText.LoadString(IDS_NOTIFICATION);
            strPaneText += _T(" Failed to Connect to the VPN server selected.");
            m_StatusBar.SetPaneText(0, strPaneText);
            CString strErr; 
            strErr.Format(_T("Failed to Connect to the VPN server selected, error = 0x%.8X, [%s] description: %s"),
                          hr, (LPCTSTR)err.Source(), (LPCTSTR)err.Description());                              
            MessageBox(strErr, _T("ERROR"), MB_OK | MB_ICONERROR);
        }
    }
}

//Helper method to populate the CComboBox using the VPN API interfaces and methods.
//
HRESULT CCppComSampleDlg::BuildHostComboBoxEntries()
{
    m_HostSelectionComboBox.ResetContent();
    HRESULT hr = S_OK;
    if (!m_spVpnApi)
    {
        hr = S_FALSE; //Nothing to do here, as the API is not running.
    }
    else
    {
        try
        {
            //Determine the default host.
            _bstr_t bstrDefaultHost = m_spVpnApi->DefaultHostName;

             //Obtain the hosts collection.
            CComPtr<IStringCollection> spStringCollection = m_spVpnApi->HostNames;
            long lTotal = spStringCollection->Count;

            //Populate the combobox with host entries.
            //NOTE: COM Collection Indexes are 1-based as the starting point.
            //
            for (long lCount = 1; lCount <= lTotal; lCount++)
            {
                _bstr_t bstrHost = spStringCollection->Item[lCount];
                m_HostSelectionComboBox.InsertString(lCount - 1, (LPCTSTR)bstrHost);

                //If the current value is the default host, set it as selected.
                if (bstrDefaultHost == bstrHost)
                {
                    m_HostSelectionComboBox.SetCurSel(lCount - 1);
                }
            }


            //Lastly, disable the control if the VPN Session is connected.
            if (VARIANT_TRUE == m_spVpnApi->IsConnected)
            {
                m_HostSelectionComboBox.EnableWindow(FALSE);
            }
        }
        catch(_com_error err)
        {
            CString strErr; 
            hr = err.Error();
            strErr.Format(_T("Failed to obtain the the Host Names collection and / or the Default Host, error = 0x%.8X, [%s] description: %s"),
                          hr, (LPCTSTR)err.Source(), (LPCTSTR)err.Description());                              
            MessageBox(strErr, _T("ERROR"), MB_OK | MB_ICONERROR);
        }
    }

    return hr;
}

//Helper method to delete credential controls.
//
void CCppComSampleDlg::ResetCredentialsUI(void)
{
    m_SubmitButton.ShowWindow(SW_HIDE);
    m_ConnectButton.EnableWindow(TRUE);
    m_HostSelectionComboBox.EnableWindow(TRUE);

    for (ControlMap::iterator it = m_promptLabels.begin();
         it != m_promptLabels.end();
         it++)
    {
        CWnd *pWindow = it->second;
        if (NULL != pWindow)
        {
            pWindow->DestroyWindow();
            delete pWindow;
        }
    }
    m_promptLabels.clear();

    for (ControlMap::iterator it = m_promptInputControls.begin();
         it != m_promptInputControls.end();
         it++)
    {
        CWnd *pWindow = it->second;
        if (NULL != pWindow)
        {
            pWindow->DestroyWindow();
            delete pWindow;
        }
    }
    m_promptInputControls.clear();
}

//Helper method to disable credential controls.
//
void CCppComSampleDlg::DisableCredentialsUI(void)
{
    m_ConnectButton.EnableWindow(FALSE);
    m_SubmitButton.EnableWindow(FALSE);
    m_HostSelectionComboBox.EnableWindow(FALSE);

    for (ControlMap::iterator it = m_promptLabels.begin();
         it != m_promptLabels.end();
         it++)
    {
        CWnd *pWindow = it->second;
        if (NULL != pWindow)
        {
            pWindow->EnableWindow(FALSE);
        }
    }

    for (ControlMap::iterator it = m_promptInputControls.begin();
         it != m_promptInputControls.end();
         it++)
    {
        CWnd *pWindow = it->second;
        if (NULL != pWindow)
        {
            pWindow->EnableWindow(FALSE);
        }
    }
}

//Helper method to store credential fields to API object.
// This function does not explicitly catch COM exceptions, and
// it relies on the caller to provide a try/catch block.
//
HRESULT CCppComSampleDlg::StoreCredentialsUI(CComPtr<IConnectPromptInfo> & spConnectPromptInfo)
{
    //Get the collection of prompt entries.
    //
    CComPtr<IInterfaceCollection> spCollection = spConnectPromptInfo->PromptEntries;
    
    //NOTE: COM Collections are 1's-based indexed.
    for (long nCount = 1; nCount <= spCollection->Count; nCount++)
    {
        //Obtain the text, and then insert it in the appropriate column.
        //
        CComQIPtr<IPromptEntry> spPromptEntry(spCollection->Item[nCount]);
        if (spPromptEntry) //NOTE: This is not checking the smart pointer's address on the stack.
                           //      This actually calls implicit cast, CComQIPtr::operator T*(), which is what we want.
                           //      This code checks that the interface being queried is valid.
        {
            //Obtain the type of prompt and set the appropriate item.
            //
            if (spPromptEntry->IsVisible)
            {
                _bstr_t bstrPromptName = spPromptEntry->PromptName;

                //Because the return enumerator name matches the name of the property, 
                //the typelibrary importer will prepend an '_' to the property name.
                //A developer can use GetPromptType() or the raw, get_PromptType() as well.
                //
                PromptType ePromptType = spPromptEntry->_PromptType; 

                switch (ePromptType)
                {
                case Prompt_Input:
                case Prompt_Password:
                case Prompt_Combo:
                {
                    ControlMap::iterator it = m_promptInputControls.find(bstrPromptName);
                    if (it == m_promptInputControls.end())
                    {
                        return Error(L"Could not find input control for prompt entry",
                                     __uuidof(IPromptEntry), E_UNEXPECTED);
                    }
                    CWnd *pInput = it->second;

                    CString strInputValue;
                    pInput->GetWindowText(strInputValue);
                    _bstr_t bstrInputValue((LPCTSTR)strInputValue);
                    spPromptEntry->Value = bstrInputValue;

                    if (ePromptType == Prompt_Password)
                    {
                        //Clear editbox of pwd
                        pInput->SetWindowText(_T("")); 

                        //Reduce the window of time the password is in memory.
                        //BSTRs are a garbage collected type, so the time the data
                        //is in memory is indeterminite.  Securely clear the data.
                        //
                        SecureZeroMemory(strInputValue.GetBuffer(), strInputValue.GetLength()*sizeof(TCHAR));
                        strInputValue.ReleaseBuffer();
                        BSTR rawBSTR = bstrInputValue.GetBSTR();
                        SecureZeroMemory(rawBSTR, ::SysStringByteLen(rawBSTR));
                    }
                }
                break;
                case Prompt_Banner:
                    break;
                case Prompt_Header:
                    break;
                case Prompt_Hidden:
                    break;
                case Prompt_CheckBox:
                    break;
                default:
                Error(L"An unexpected value for PromptType was returned",
                      __uuidof(IPromptEntry), E_UNEXPECTED);
                }
            }
        }
    }
    return S_OK;
}

//Helper method to populate the credential fields using the VPN API interfaces and methods.
//
HRESULT CCppComSampleDlg::BuildCredentialsUI(CComPtr<IConnectPromptInfo> & spConnectPromptInfo)
{
    HRESULT hr = S_OK;
    try
    {
        ResetCredentialsUI();

        //Obtain the rectangle used for placing the dynamic controls.
        //
        CRect rectLocation;
        CWnd* tmpWnd = this->GetDlgItem(IDC_PLACEHOLDER);
        tmpWnd->GetWindowRect(&rectLocation);
        this->ScreenToClient(&rectLocation);

        //Get the collection of prompt entries.
        //
        CComPtr<IInterfaceCollection> spCollection = spConnectPromptInfo->PromptEntries;
        
        //NOTE: COM Collections are 1's-based indexed.
        for (long nCount = 1; nCount <= spCollection->Count; nCount++)
        {
            //Obtain the IPromptEntry interface.
            //
            CComQIPtr<IPromptEntry> spPromptEntry(spCollection->Item[nCount]);
            if (spPromptEntry) //NOTE: This is not checking the smart pointer's address on the stack.
                               //      This actually calls implicit cast, CComQIPtr::operator T*(), which is what we want.
                               //      This code checks that the interface being queried is valid.
            {
                //Obtain the type of prompt and set the appropriate UI item.
                //
                if (spPromptEntry->IsVisible)
                {
                    //Because the return enumerator name matches the name of the property, 
                    //the typelibrary importer will prepend an '_' to the property name.
                    //A developer can use GetPromptType() or the raw, get_PromptType() as well.
                    //
                    PromptType ePromptType = spPromptEntry->_PromptType;
                    switch (ePromptType)
                    {
                    case Prompt_Input:
                        CreateLabel(spPromptEntry, rectLocation);
                        CreateEditPrompt(spPromptEntry, rectLocation);
                        break;
                    case Prompt_Password:
                        CreateLabel(spPromptEntry, rectLocation);
                        CreateEditPrompt(spPromptEntry, rectLocation);
                        break;
                    case Prompt_Banner:
                        break;
                    case Prompt_Combo:
                        CreateLabel(spPromptEntry, rectLocation);
                        CreateComboPrompt(spPromptEntry, rectLocation);
                        break;
                    case Prompt_Header:
                        break;
                    case Prompt_Hidden:
                        break;
                    case Prompt_CheckBox:
                        break;
                    default:
                    return Error(L"An unexpected value for PromptType was returned",
                                 __uuidof(IPromptEntry), E_UNEXPECTED);
                    }
                }
            }
        }

        _bstr_t bstrText = spConnectPromptInfo->SubmitButtonName;
        m_SubmitButton.SetWindowText((LPCTSTR)bstrText);
        m_SubmitButton.EnableWindow(TRUE);
        m_SubmitButton.ShowWindow(SW_SHOW);
        m_ConnectButton.ShowWindow(SW_SHOW);
    }
    catch (_com_error err)
    {
        CString strErr; 
        hr = err.Error();
        strErr.Format(_T("Failed Building Prompt components, Error = 0x%.8X, [%s] description: %s"),
                      hr, (LPCTSTR)err.Source(), (LPCTSTR)err.Description());                              
        MessageBox(strErr, _T("ERROR"), MB_OK | MB_ICONERROR);
    }

    return hr;
}

//Helper method to create a label control for a prompt entry.
//This function does not explicitly catch COM exceptions, and
//it relies on the caller to provide a try/catch block.
//
void CCppComSampleDlg::CreateLabel(const CComQIPtr<IPromptEntry> & spPromptEntry, CRect & rectLocation)
{
    CStatic* pLabel = new CStatic();
    pLabel->CreateEx(WS_EX_NOPARENTNOTIFY,
                     _T("STATIC"),
                     _T(""),
                     WS_CHILD | WS_VISIBLE,
                     rectLocation,
                     this,
                     0);
    pLabel->SetFont(GetFont());
    pLabel->EnableWindow(spPromptEntry->IsEnabled);

    _bstr_t bstrLabel = spPromptEntry->PromptLabel;
    pLabel->SetWindowText((LPCTSTR)bstrLabel);

    _bstr_t bstrName = spPromptEntry->PromptName;
    m_promptLabels[bstrName] = pLabel;

    rectLocation.MoveToX(rectLocation.left + rectLocation.Width());
}

//Helper method to create an edit control from an input or password prompt entry.
//This function does not explicitly catch COM exceptions, and
//it relies on the caller to provide a try/catch block.
//
void CCppComSampleDlg::CreateEditPrompt(const CComQIPtr<IPromptEntry> & spPromptEntry, CRect & rectLocation)
{
    CEdit* pInput = new CEdit();
    pInput->CreateEx(WS_EX_CLIENTEDGE | WS_EX_NOPARENTNOTIFY,
                     _T("EDIT"),
                     _T(""),
                     WS_CHILD | WS_VISIBLE | WS_TABSTOP | ES_AUTOHSCROLL,
                     rectLocation,
                     this,
                     0);
    pInput->SetFont(GetFont());
    pInput->EnableWindow(spPromptEntry->IsEnabled);

    if (spPromptEntry->GetPromptType() == Prompt_Password)
    {
        //NOTE: The value contents of the password prompt type is an
        //      empty string.
        //
        pInput->SetPasswordChar(_T('*'));
    }
    else
    {
        _bstr_t bstrInitialValue = spPromptEntry->Value;
        pInput->SetWindowText((LPCTSTR)bstrInitialValue);
    }

    _bstr_t bstrName = spPromptEntry->PromptName;
    m_promptInputControls[bstrName] = pInput;

    rectLocation.MoveToX(rectLocation.left - rectLocation.Width());
    rectLocation.MoveToY(rectLocation.top + rectLocation.Height() + 5);
}

//Helper method to create a combobox control from a prompt entry.
//This function does not explicitly catch all COM exceptions, and
//it relies on the caller to provide a try/catch block.
//
void CCppComSampleDlg::CreateComboPrompt(const CComQIPtr<IPromptEntry> & spPromptEntry, CRect & rectLocation)
{
    //NOTE: If we are creating a group prompt entry, then create the combobox with a known
    //ID (IDC_COMBO_TUNNELGROUP) in order to handle group selection changed events.
    //There is at most one group selection prompt entry.
    //
    CComboBox* pInput = new CComboBox();
    pInput->CreateEx(WS_EX_NOPARENTNOTIFY,
                     _T("COMBOBOX"),
                     _T(""),
                     WS_CHILD | WS_VISIBLE  | WS_TABSTOP | CBS_DROPDOWNLIST | CBS_HASSTRINGS | CBS_AUTOHSCROLL,
                     rectLocation,
                     this,
                     spPromptEntry->IsEntryGroup ? IDC_COMBO_TUNNELGROUP : 0);
    pInput->SetFont(GetFont());
    pInput->EnableWindow(spPromptEntry->IsEnabled);

    try
    {
        CComPtr<IStringCollection> spCollection = spPromptEntry->ValueOptions;

        _bstr_t bstrInitialValue = spPromptEntry->Value;
        long nSelectedIndex = 0;
        
        //NOTE: COM Collections are 1's based indexed.
        for (long nCount = 1; nCount <= spCollection->Count; nCount++)
        {
            _bstr_t bstrValue = spCollection->Item[nCount];
            pInput->InsertString(nCount - 1, (LPCTSTR)bstrValue);
            if (bstrValue == bstrInitialValue)
            {
                nSelectedIndex = nCount - 1;
            }
        }

        pInput->SetCurSel(nSelectedIndex);
    }
    catch (_com_error err)
    {
        CString strErr; 
        HRESULT hr = err.Error();
        strErr.Format(_T("Failed Building combo-box entries, Error = 0x%.8X, [%s] description: %s"),
                      hr, (LPCTSTR)err.Source(), (LPCTSTR)err.Description());                              
        MessageBox(strErr, _T("ERROR"), MB_OK | MB_ICONERROR);
    }

    _bstr_t bstrName = spPromptEntry->PromptName;
    m_promptInputControls[bstrName] = pInput;

    rectLocation.MoveToX(rectLocation.left - rectLocation.Width());
    rectLocation.MoveToY(rectLocation.top + rectLocation.Height() + 5);
}

//Handler for the selection changed event on the group selection combobox.
//
void CCppComSampleDlg::OnPromptComboBoxSelectionChanged()
{
    CWnd* pGroupComboBox = this->GetDlgItem(IDC_COMBO_TUNNELGROUP);

    if (pGroupComboBox != NULL)
    {
        CString strSelectedGroup;
        pGroupComboBox->GetWindowText(strSelectedGroup);
        _bstr_t bstrSelectedGroup((LPCTSTR)strSelectedGroup);

        try
        {
            //Notify the AnyConnect API that the selected group has changed.
            //
            m_spConnectPromptInfo->TunnelGroup = bstrSelectedGroup;

            //Post a message to this dialog to recreate the dynamic controls,
            //since the prompt entry attributes may have changed as a result
            //of the group change.
            //
            PostMessage(WM_VPN_GROUP_CHANGED);
        }
        catch (_com_error err)
        {
            CString strErr; 
            HRESULT hr = err.Error();
            strErr.Format(_T("Failed to notify API of tunnel group change, Error = 0x%.8X, [%s] description: %s"),
                          hr, (LPCTSTR)err.Source(), (LPCTSTR)err.Description());                              
            MessageBox(strErr, _T("ERROR"), MB_OK | MB_ICONERROR);
        }
    }
}

//This is called whem WM_VPN_GROUP_CHANGED is posted to the message queue
//signalling the GUI to recreate the credential controls.
//
LRESULT CCppComSampleDlg::OnGroupChanged(WPARAM /*not used*/, LPARAM /*not used*/)
{
    if (m_spConnectPromptInfo)
    {
        BuildCredentialsUI(m_spConnectPromptInfo);
    }
    return 0;
}

//This is called when WM_VPN_EVENT_AVAILABLE is posted to the message queue
//when configured for consumer driven events.
//
LRESULT CCppComSampleDlg::OnVpnEventAvailable(WPARAM /*not used*/, LPARAM /*not used*/)
{
    if (!m_spVpnApi)
    {
        return 0; //nothing to do here as the API is not running anymore.
    }

    //If this method is called, it is because you enabled the consumer-driven event model.
    //Call the ProcessEvents method to consume an event.
    //
    try
    {
        m_spVpnApi->ProcessEvents();
    }
    catch(_com_error err)
    {
        CString strErr; 
        HRESULT hr = err.Error();
        strErr.Format(_T("Failed to process events queued in the VPN API engine, error = 0x%.8X, [%s] description: %s"),
                      err.Error(), (LPCTSTR)err.Source(), (LPCTSTR)err.Description());      
        MessageBox(strErr, _T("ERROR"), MB_OK | MB_ICONERROR);
    }

    return 0;
}


/////////////////////////////// IVpnApiEvents Implementation //////////////////////////////
//This is the implemetation of the methods for the event interface 
//that is registered with the COM server.
///////////////////////////////////////////////////////////////////////////////////////////
STDMETHODIMP CCppComSampleDlg::raw_VpnStatsNotification(IN IVpnStats * pVpnStats)
{
    CComPtr<IVpnStats> spVpnStats = pVpnStats;
    if (!spVpnStats)
    {
        MessageBox(_T("Failed to get IVpnStats object"), _T("ERROR"), MB_OK | MB_ICONERROR);

        //Error() is used to return a detailed error message to the caller of this method.
        //It is built into the ATL COM infrastructure base classes.
        //
        return Error(L"An unexpected NULL pointer was passed to the VpnStatsNotification method",
                     __uuidof(IVpnApiEvents), E_POINTER);
    }


    //update the route info and stats.
    //
    m_RoutesDlg.UpdateStats(spVpnStats);
    m_StatsDlg.UpdateStats(spVpnStats);
    m_FirewallDlg.UpdateStats(spVpnStats);
    m_ProtocolDlg.UpdateStats(spVpnStats);


    return S_OK;
}

STDMETHODIMP CCppComSampleDlg::raw_VpnStateNotification(IN enum VPNState eState,
                                                        IN enum VPNSubState eSubState,
                                                        IN BSTR rawbstrState )
{
    if (NULL == rawbstrState)
    {
        MessageBox(_T("The bstrState BSTR is NULL"), _T("ERROR"), MB_OK | MB_ICONERROR);
        return Error(L"An unexpected NULL pointer was passed to the VpnStateNotification method",
                     __uuidof(IVpnApiEvents), E_POINTER);

    }

    m_currentVpnState = static_cast<VPNState>(eState);

    //You could work with the raw BSTR type and not incur the extra allocation
    //in _bstr_t.  Given this is not a high performance server application
    //the extra allocations are insignificant.  Additionally, this sample demonstrates
    //the convenience of _bstr_t and other COM helper classes for those not familiar with them.
    //
    _bstr_t bstrState = rawbstrState;

    //Technically, a BSTR does not have to be NUL terminated, however our use of the 
    //type in the COM Server is ALWAYS NUL terminated.  The _bstr_t class deals with 
    //BSTRs in a safe manner
    //
    m_stateStr = (LPCTSTR)bstrState;

    HRESULT hr = S_OK;

    //Lastly, change the combobox to enabled and repopulate the values
    //if the state is 'disconnected'.
    //
    if (DISCONNECTED == m_currentVpnState)
    {
        hr = BuildHostComboBoxEntries();
        m_HostSelectionComboBox.EnableWindow(TRUE);
    }
    else if (CONNECTED == m_currentVpnState)
    {
        CString strDisconnect;
        strDisconnect.LoadString(IDS_DISCONNECT);
        m_SubmitButton.SetWindowText(strDisconnect);
        m_SubmitButton.EnableWindow(TRUE);
        m_SubmitButton.ShowWindow(SW_SHOW);
    }

    //There are many ways to update the UI, this is one way.
    UpdateData(FALSE);

    return hr;
}

STDMETHODIMP CCppComSampleDlg::raw_VpnBannerNotification(IN BSTR rawbstrBannerMessage)
{
    if (NULL == rawbstrBannerMessage)
    {
        MessageBox(_T("The bstrBannerMessage BSTR"), _T("ERROR"), MB_OK | MB_ICONERROR);
        return Error(L"An unexpected NULL pointer was passed to the VpnBannerNotification method",
                     __uuidof(IVpnApiEvents), E_POINTER);

    }

    _bstr_t bstrBannerMsg = rawbstrBannerMessage;
    m_strBannerMsg = (LPCTSTR)bstrBannerMsg;

    PostMessage(WM_VPN_BANNER_AVAILABLE);

    return S_OK;
}

STDMETHODIMP CCppComSampleDlg::raw_VpnPreConnectReminderNotification(IN BSTR rawbstrPreConnectReminderMessage)
{
    if (NULL == rawbstrPreConnectReminderMessage)
    {
        MessageBox(_T("The bstrPreConnectReminderMessage BSTR"), _T("ERROR"), MB_OK | MB_ICONERROR);
        return Error(L"An unexpected NULL pointer was passed to the VpnPreConnectReminderNotification method",
                     __uuidof(IVpnApiEvents), E_POINTER);

    }

    _bstr_t bstrPreConnectReminderMsg = rawbstrPreConnectReminderMessage;
    m_strPreConnectReminderMsg = (LPCTSTR)bstrPreConnectReminderMsg;

    PostMessage(WM_VPN_PRECONNECT_REMINDER_AVAILABLE);

    return S_OK;
}

STDMETHODIMP CCppComSampleDlg::raw_VpnCertBlockedNotification(BSTR rawbstrUntrustedServer)
{
    if (NULL == rawbstrUntrustedServer)
    {
        MessageBox(_T("The strUnTrustedServer BSTR"), _T("ERROR"), MB_OK | MB_ICONERROR);
        return Error(L"An unexpected NULL pointer was passed to the VpnCertBlockedNotification method",
                     __uuidof(IVpnApiEvents), E_POINTER);

    }

    _bstr_t bstrUntrustedServer = rawbstrUntrustedServer;
    m_strUntrustedServer = (LPCTSTR)bstrUntrustedServer;

    PostMessage(WM_VPN_CERT_BLOCKED);

    return S_OK;
}

STDMETHODIMP CCppComSampleDlg::raw_VpnCertWarningNotification(BSTR rawbstrUntrustedServer,
                                                              IStringCollection *pCertErrors,
                                                              VARIANT_BOOL bImportAllowed)
{
    if (NULL == rawbstrUntrustedServer ||
        NULL == pCertErrors)
    {
        MessageBox(_T("Unexpected NULL pointer"), _T("ERROR"), MB_OK | MB_ICONERROR);
        return Error(L"An unexpected NULL pointer was passed to the VpnCertWarningNotification method",
                     __uuidof(IVpnApiEvents), E_POINTER);

    }

    _bstr_t bstrUntrustedServer = rawbstrUntrustedServer;
    m_strUntrustedServer = (LPCTSTR)bstrUntrustedServer;

    m_lCertErrors.clear();

    CComPtr<IStringCollection> spStringCollection = pCertErrors;
    long lTotal = spStringCollection->Count;

    //NOTE: COM Collection Indexes are 1-based as the starting point.
    //
    for (long lCount = 1; lCount <= lTotal; lCount++)
    {
        _bstr_t bstrError = spStringCollection->Item[lCount];
        m_lCertErrors.push_back((LPCTSTR)bstrError);
    }

    m_bImportAllowed = (bImportAllowed == VARIANT_TRUE);

    PostMessage(WM_VPN_CERT_WARNING);

    return S_OK;
}

STDMETHODIMP CCppComSampleDlg::raw_VpnNoticeNotification(IN BSTR rawbstrNoticeMessage, 
                                                      IN enum MessageType eMessageType)
{
    if (NULL == rawbstrNoticeMessage)
    {
        MessageBox(_T("The bstrNoticeMessage BSTR is NULL"), _T("ERROR"), MB_OK | MB_ICONERROR);
        return Error(L"An unexpected NULL pointer was passed to the VpnNoticeNotification method",
                     __uuidof(IVpnApiEvents), E_POINTER);

    }

    _bstr_t bstrNoticeMessage = rawbstrNoticeMessage;

    CString strPaneText;
    strPaneText.LoadString(IDS_NOTIFICATION);

    strPaneText += _T(" ");
    //Technically, a BSTR does not have to be NUL terminated, however our use of the 
    //type in the COM Server is ALWAYS NUL terminated.  The _bstr_t class deals with 
    //BSTRs in a safe manner
    //
    strPaneText += (LPCTSTR)bstrNoticeMessage;

    m_StatusBar.SetPaneText(0, strPaneText);

    return S_OK;
}

STDMETHODIMP CCppComSampleDlg::raw_VpnExitNotification(IN BSTR rawbstrExitMessage,
                                                       IN long exitCode)
{
    if (NULL == rawbstrExitMessage)
    {
        MessageBox(_T("The bstrExitMessage BSTR is NULL"), _T("ERROR"), MB_OK | MB_ICONERROR);
        return Error(L"An unexpected NULL pointer was passed to the VpnExitNotification method",
                     __uuidof(IVpnApiEvents), E_POINTER);
    }

    _bstr_t bstrExitMessage = rawbstrExitMessage;

    CString strPaneText;
    strPaneText.LoadString(IDS_NOTIFICATION);

    strPaneText += _T(" ");
    //Technically, a BSTR does not have to be NUL terminated, however our use of the 
    //type in the COM Server is ALWAYS NUL terminated.  The _bstr_t class deals with 
    //BSTRs in a safe manner
    //
    strPaneText += (LPCTSTR)bstrExitMessage;

    CString strFormatter;
    strFormatter.Format(_T(" exit code: %d"), exitCode);

    m_StatusBar.SetPaneText(0, strPaneText);

    HRESULT hr = S_OK;

    //An exit notification must be obeyed.  The COM API *MUST* be unloaded!
    //The most common reason for this notification is that an upgrade of software is being performed.
    //Failure to heed this will result in vpndownloader.exe terminating the process hosting the
    //COM server.  In the default case of this example, that would be DllHost.exe.  If you
    //defined COM_INPROC_EXAMPLE, then this application would be killed for not
    //heeding the notice.
    //
    //NOTE: Just prior to shutting down, obtain the name of the key to 
    //use to monitor for the completion of the upgrade and the value name to query
    //for the upgrade result.
    //
    try
    {
        //Get the name of the registry value to query for update status when changed.
        //
        _bstr_t bstrUpgradeRegValueName = m_spVpnApi->UpgradeRegistryValueName;
        m_strUpgradeValueName = (LPCTSTR) bstrUpgradeRegValueName;

        //Get the name of the registry key to monitor for update status changes.
        //
        _bstr_t bstrUpgradeRegKeyName = m_spVpnApi->UpgradeRegistryKeyName;
        long lRet = m_keyUpgrade.Open(HKEY_LOCAL_MACHINE, 
                                     (LPCTSTR) bstrUpgradeRegKeyName, 
                                     KEY_READ);
        if (ERROR_SUCCESS == lRet)
        {
            lRet = m_keyUpgrade.NotifyChangeKeyValue(true, REG_NOTIFY_CHANGE_LAST_SET, (HANDLE)m_eventUpgradeComplete);
            if (ERROR_SUCCESS == lRet)
            {
                //this watch will allow for checking when the upgrade is completed by the vpndownloader software.
                //
                BOOL bReturn = ::RegisterWaitForSingleObject(&m_hWatchHandle, 
                                                             (HANDLE)m_eventUpgradeComplete,
                                                             CCppComSampleDlg::RegistryWaitCallback,
                                                             this,
                                                             sMaxTimeToWaitForUpdate,
                                                             WT_EXECUTEINWAITTHREAD);
                if (!bReturn)
                {
                    _com_error err(HRESULT_FROM_WIN32(::GetLastError()));
                    throw err;
                }
            }
            else
            {
                _com_error err(HRESULT_FROM_WIN32(::GetLastError()));
                throw err;
            }
        }
        else
        {
            _com_error err(HRESULT_FROM_WIN32(::GetLastError()));
            throw err;
        }
    }
    catch (_com_error err)
    {
        CString strErr; 
        hr = err.Error();
        strErr.Format(_T("Failed to obtain the upgrade registry key name/value, Error = 0x%.8X, [%s] description: %s"),
                      hr, (LPCTSTR)err.Source(), (LPCTSTR)err.Description());                              
        MessageBox(strErr, _T("ERROR"), MB_OK | MB_ICONERROR);
    }

    hr = ShutdownCOMVpnApi();


    return hr;
}
    
STDMETHODIMP CCppComSampleDlg::raw_VpnServiceReadyNotification(void)
{
    //Indicates that the VPN Service is running.
    //

    //If there was a previous Upgrade registry watch operation, cancel it here.
    if (NULL != m_hWatchHandle)
    {
        ::UnregisterWait(m_hWatchHandle);
        m_hWatchHandle = NULL;
    }

    CString strPaneText;
    strPaneText.LoadString(IDS_NOTIFICATION);
    CString strTemp;
    strTemp.LoadString(IDS_AVAILABLE);
    strPaneText += _T(" ") + strTemp;
    m_StatusBar.SetPaneText(0, strPaneText);

    m_serviceStr = strTemp;

    //This call could be done from a variety of places including right after 
    //performing the IVpnApi::RegisterAndAttach.
    //Since this notification comes as a result of determining that the 
    //vpn service is online, it is also a logical place to perform the action.
    //Obtain a list of host names to populate the list box.
    //Compare the default host to the list of hosts and set that as selected.
    //
    HRESULT hr = BuildHostComboBoxEntries();

    //There are many ways to update the UI, this is one way.
    UpdateData(FALSE);

    return hr;
}

STDMETHODIMP CCppComSampleDlg::raw_VpnUserPromptNotification(IN IConnectPromptInfo * pConnectPromptInfo)
{
    m_spConnectPromptInfo = pConnectPromptInfo;
    if (!m_spConnectPromptInfo)
    {
        MessageBox(_T("Failed to get IConnectPromptInfo object"), _T("ERROR"), MB_OK | MB_ICONERROR);
        return Error(L"An unexpected NULL pointer was passed to the ConnectPromptInfoNotification method",
                     __uuidof(IVpnApiEvents), E_POINTER);
    }
 
    HRESULT hr = S_OK;

    //Consume the prompt info that was delivered.
    //
    try
    {
        ConnectPromptType ePromptType = m_spConnectPromptInfo->GetConnectPromptType();
        switch(ePromptType)
        {
        case CREDENTIALS:
            //This is a credential prompt. Configure the various 
            //fields for display.
            //
            hr = BuildCredentialsUI(m_spConnectPromptInfo); //helper method to build the creds on the UI
            break;
        case PROXY:
            //This is a proxy auth credential prompt. Collect the 
            //proxy information.
            //
            break;
        case CERTIFICATE:
            //This is a certificate prompt. Collect cert info.
            //
            break;
        default:
            hr = Error(L"An unexpected value for ConnectPromptType was returned",
                       __uuidof(IConnectPromptInfo), E_UNEXPECTED);
        }
    }
    catch(_com_error err)
    {
        CString strErr; 
        hr = err.Error();
        strErr.Format(_T("Failed processing ConnectPromptInfo, Error = 0x%.8X, [%s] description: %s"),
                      hr, (LPCTSTR)err.Source(), (LPCTSTR)err.Description());                              
        MessageBox(strErr, _T("ERROR"), MB_OK | MB_ICONERROR);
    }

    return hr;
}

STDMETHODIMP CCppComSampleDlg::raw_VpnWMHintNotification(IN enum WMHint eHint,
                                                      IN enum WMHintReason eReason)
{
    return S_OK;
} 

STDMETHODIMP CCppComSampleDlg::raw_VpnWebLaunchHostNotification(IN BSTR rawbstrActiveHost)
{
    if (NULL == rawbstrActiveHost)
    {
        MessageBox(_T("The bstrExitMessage BSTR is NULL"), _T("ERROR"), MB_OK | MB_ICONERROR);
        return Error(L"An unexpected NULL pointer was passed to the VpnExitNotification method",
                     __uuidof(IVpnApiEvents), E_POINTER);
    }

    return E_NOTIMPL;
}

//This is a mandatory interface method, it is called when EnableConsumerDrivenEventModel = VARIANT_TRUE.  
//
STDMETHODIMP CCppComSampleDlg::raw_VpnEventAvailableNotification(void)
{
    HRESULT hr = S_OK;

    //Post to the UI thread for processing.
    if (!PostMessage(WM_VPN_EVENT_AVAILABLE))
    {
        CString strErr = _T("Failed to post a message to the message queue to indicate that VPN API Event is available");
        hr = Error((LPCTSTR)strErr, __uuidof(IVpnApiEvents), HRESULT_FROM_WIN32(::GetLastError()));
    }

    return hr;
}










/*************************************************************************************************/
#ifdef COM_INPROC_EXAMPLE
#pragma message(" ******** The VPN API COM Server will be hosted INPROC ********")
#else
    #pragma message("******** The VPN API COM Server will be hosted high isolation (OUT_OF_PROC) ********")
#endif



void CCppComSampleDlg::OnBnClickedButtonConnect()
{
    ResetCredentialsUI();
    m_ConnectButton.EnableWindow(FALSE);

    CString strHost;
    m_HostSelectionComboBox.GetWindowTextW(strHost);
    
    if (strHost.IsEmpty())
    {
        m_StatusBar.SetPaneText(0, _T("Hostname cannot be empty!"));
        m_ConnectButton.EnableWindow(TRUE);
        return;
    }
        
    _bstr_t bstrHostSelected((LPCTSTR)strHost); 

    try
    {
        m_StatusBar.SetPaneText(0, _T("Connecting to ") + strHost);
        m_spVpnApi->ConnectVpn(bstrHostSelected);
    }
    catch(_com_error err)
    {
        HRESULT hr = err.Error();
        CString strPaneText;
        strPaneText.LoadString(IDS_NOTIFICATION);
        strPaneText += _T(" Failed to Connect to the VPN server selected.");
        m_StatusBar.SetPaneText(0, strPaneText);
        CString strErr; 
        strErr.Format(_T("Failed to Connect to the VPN server selected, error = 0x%.8X, [%s] description: %s"),
                      hr, (LPCTSTR)err.Source(), (LPCTSTR)err.Description());                              
        MessageBox(strErr, _T("ERROR"), MB_OK | MB_ICONERROR);
        m_ConnectButton.EnableWindow(TRUE);
    }

}
