'''
Created on Apr 12, 2015

@author: Puneet Garg

Copyright (c) 2015 by Cisco Systems, Inc.
All rights reserved.

Classes used for Device Model Service.
'''
from devpkg.utils.errors import IFCConfigError
from devpkg.utils.state_type import State
from devpkg.utils.util import set_cfg_state, massage_param_dict
from devpkg.devicemodel import Features


class DeviceModelService(object):
    def __init__(self, device):
        self.device = device

    def ifc2dm(self, ifc_delta_cfg_dict, device):
        '''Generate Device delta configuration from configuration in IFC delta dictionary form.
    
        @param ifc_delta_cfg_dict: dict
        @param device: device
            Device Model Object
        @return list of Command's
        '''
        'ifc_cfg is a copy of ifc_delta_cfg_dict suited for us.'
        ifc_cfg = massage_param_dict(device, ifc_delta_cfg_dict)
    
        '''Since the input dictionary is in arbitrary order, we use two-pass approach to make sure resulting
        command list is in the right order given in the device model.
        '''
    
        #1st pass
        device.populate_model(ifc_cfg.keys()[0], ifc_cfg.values()[0])
        faults = device.validate_configuration()
        if faults:
            raise IFCConfigError(faults)
    
        #2nd pass
        command_list = []
        no_command_stack = []
        device.ifc2dm(no_command_stack, command_list)
    
        #gather result from positive and negative forms for the commands
        result = no_command_stack
        result.reverse() #last in first out for the 'no' commands
        result.extend(command_list)
       
        return result
    
    def generate_ifc_delta_cfg(self, ifc_cfg_dict, asa_clis, device = None, interfaces = None,
                               features = Features.vnsAll):
        '''Compare ifc configuration with asa configuration , and produce
        ifc configuration delta.
    
        @param asa_clis: list
            input/out list of CLI's read from ASA device
        @param ifc_cfg_dict: dict
            the configuration of ASA in IFC dictionary format. It will be modified
            after the comparison.
        @param device: dict
            to identify the ASA device, passed in from device_script APIs
        @param interfaces: list
            physical interfaces names passed in from device_script APIs
        @param features: bit-mask from Features
            to indicate what part of configurations to generate
        @return dict
            a copy of ifc_cfg_dct but with the state of each entry set to indicate
            if a modification operation or create operation is required.
        '''
        asa = self.device
    
        def create_missing_ifc_delta(ifc_cfg):
            #This is not needed anymore we will just return. creates random configs
            return ifc_cfg
    
        def set_cfg_state_2_CREATE(ifc_cfg):
            'Transformer to set the State of each entry to CREATE, the default state of all entries'
            set_cfg_state(ifc_cfg, State.CREATE)
            return ifc_cfg
    
        'ifc_cfg is a copy of ifc_cfg_dict suited for us.'
        ifc_cfg = massage_param_dict(asa, ifc_cfg_dict,
                                     transformers=[set_cfg_state_2_CREATE,
                                                   create_missing_ifc_delta])
    
        '''Since the input dictionary is in arbitrary order, we use two-pass approach to make sure resulting
        CLI list is in the right order given in the device model.
        '''
        if asa_clis is not None:
            for cli in asa_clis:
                translator = asa.get_translator(cli)
                if translator:
                    translator.diff_ifc_asa(cli)
        return ifc_cfg
    
    def generate_dm_delta_cfg(self, ifc_cfg, asa_cfg, device = None, interfaces = None, sts = None,
                               features = Features.vnsAll):
        '''Generate CLIs that represent the delta between given IFC configuration and ASA configuration
        @param ifc_cfg: dict
        @param asa_cfg: list of CLIs
        @param device: dict
            to identify the ASA device, passed in from device_script APIs
        @param interfaces: list
            physical interfaces names passed in from device_script APIs
        @param sts: dict
            to identify the STS external and internal tag
        @param features: bit-mask from Features
            to indicate what part of configurations to generate
        @return: list of CLIs
        '''
        ifc_delta_cfg = self.generate_ifc_delta_cfg(ifc_cfg, asa_cfg, device, interfaces, features)
        return self.ifc2dm(ifc_delta_cfg, self.device)
    
    'ToDo: CSCvd20931 - Comment out the code block for now to improve the coder coverage. The code is not executed for current implementation.'
#     def generate_clear_device_cfg_commands(self, clis):
#         '''Generate commands that FSMC configuration
#         @param self
#             devicemodelservice object.
#         @param clis
#             generated commands to configure on FSMC against that clear configuration commands generates.
#         '''
#         return
