"""
Copyright (c) 2015 by Cisco Systems, Inc.
All rights reserved.
"""


import urllib, hashlib, time
import devpkg.utils.util
from devpkg.utils.errors import DeviceConfigError, FaultCode
import devpkg.utils.env as env
import devpkg.base.dmobject
import fmc.config_keeper


API_PATH = 'api/'
TOKEN = []
# FMC REST degraded rapidly, see CSCvf53159. Adjust timeout for now till CSCvf53159 is fixed.
#DEFAULT_TIMEOUT = 30
DEFAULT_TIMEOUT = 90
MOCK = False
class CommandDispatch(object):
    '''
    Dispatch communications via HTTPS to the device.
    
    '''

    def __init__(self, device, executer = None):
        '''
        @param device: format example {'host': '172.23.204.63',
                                       'port': 443,
                                       'creds': {
                                           'username': 'cisco',
                                           'password': 'cisco'
                                       }
                                      }
        '''
        self.is_last = None
        self.auth_token = None
        self.device_token = None
        self.device_uuid = None
        self.domain_uuid = None
        self.baseIp = None
        'Flag keeps the info to trigger apply changes API of FSMC. It is ignored and set to False in case of Access control policy.'
        self.is_apply_changes = True
        self.device = device
        self.executor = executer
        self.all_devices = []
        self.active_device_index = 0
        # Checks device configuration.
        self.save_all_devices(device)
        devConf = self.get_next_device()
        if devConf == None:
            raise DeviceConfigError([([(0, '', None)], FaultCode.CONFIGURATION_ERROR, "No valid device config found.")])
        self.top_uuid = devpkg.base.dmobject.DMObject.get_unique_graph_string_from_device_dn(device['dn']) if device.has_key('dn') else ''
        self.init_device(devConf)

    def save_all_devices(self, device):
        self.all_devices = []
        self.active_device_index = 0
        try:
            for dev in device['devs'].itervalues():
                self.all_devices.append(dev)
        except:
            self.all_devices.append(device)

    def get_next_device(self):
        if self.active_device_index >= len(self.all_devices):
            return None
        active_device = self.all_devices[self.active_device_index]
        self.active_device_index += 1
        return active_device

    def init_device(self, devConf):
        if devConf.has_key('host'):
            self.device_ip = devConf['host']
        else:
            raise DeviceConfigError([([(0, '', None)], FaultCode.CONFIGURATION_ERROR, "Device IP or Port configuration missing.")])
        
        # Checks device configuration username and password property. 
        device = self.device
        executer = self.executor
        if devConf.has_key('manager') and devConf['manager']['creds'].has_key('username') and devConf['manager']['creds'].has_key('password'):
            self.auth = (devConf['manager']['creds']['username'], devConf['manager']['creds']['password'])
        elif device['creds'].has_key('username') and device['creds'].has_key('password'):
            self.auth = (device['creds']['username'], device['creds']['password'])
        else:
            raise DeviceConfigError([([(0, '', None)], FaultCode.CONFIGURATION_ERROR, "Device Username or Password configuration missing.")])
        
        # Checks device configuration host property.
        if devConf.has_key('manager') and devConf['manager'].has_key('hosts'):
            if len(devConf['manager']['hosts']) != 1: #if we have more than 1 managers or 0 managers
                raise DeviceConfigError([([(0, '', None)], FaultCode.CONFIGURATION_ERROR, "Device IP or Port configuration missing.")])
            self.baseUrl = ('https://' + devConf['manager']['hosts'].iterkeys().next() + ':' + str(devConf['manager']['hosts'].itervalues().next()['port']) + '/' + API_PATH)
            self.baseIp = devpkg.utils.util.asciistr(devConf['manager']['hosts'].iterkeys().next())
        elif device.has_key('host') and device.has_key('port'):
            self.baseUrl = ('https://' + device['host'] + ':' + str(device['port']) + '/' + API_PATH)
            self.baseIp = device['host']
        else:
            raise DeviceConfigError([([(0, '', None)], FaultCode.CONFIGURATION_ERROR, "Device IP or Port configuration missing.")])
            
        self.url = self.baseUrl
        self.executer = executer
        self.DEFAULT_TIMEOUT = DEFAULT_TIMEOUT
        self.is_device_virtual = device['virtual']

    @staticmethod
    def encode_url_path(show_commands):
        url = []
        for command in show_commands:
            if len(url) > 0:
                url.append('/')
            url.append(urllib.quote(command, safe=''))
        return ''.join(url)
    
    def execute(self):
        if self.executer:
            return self.executer(self)
        return None

class TokenObject(object):
    """
    The object that will store all of the hashed authentication storage for future usage
    """
    def __init__(self, dispatcher):
        hash_object = hashlib.sha256("%s%s%s" % (dispatcher.auth[0], dispatcher.auth[1], dispatcher.baseIp))
        self.hash = hash_object.hexdigest()
        self.token = dispatcher.auth_token
        self.domain_uuid = dispatcher.domain_uuid
        self.time_made = time.time()
    
    def is_equal(self, dispatcher):
        hash_object = hashlib.sha256("%s%s%s" % (dispatcher.auth[0], dispatcher.auth[1], dispatcher.baseIp))
        if hash_object.hexdigest() == self.hash:
            return True
        return False
    
    def __str__(self, *args, **kwargs):
        return "hash: %s \ntoken: %s \ndomain_uuid: %s \ntime_made: %s" % (str(self.hash), str(self.token),self.domain_uuid, str(self.time_made))

class TokenObjectHelper():
    def __init__(self):
        pass
    
    @staticmethod
    def get_token(dispatch):
        """
        Based on your dispatcher will get you the correct token
        """
        for token in TOKEN:
            if token.is_equal(dispatch):
                dispatch.auth_token = token.token
                dispatch.domain_uuid = token.domain_uuid
                return token
      
    @staticmethod   
    def add_new_token(dispatch):
        """
        Will add a new token to the global array
        """
        remove = [] #remove array
        for token in TOKEN:
            if token.is_equal(dispatch):
                remove.append(token) #we remove append token when we see it already existing since it is now old
            elif abs(time.time() - token.time_made) > 3600: #one hour has passed. lets remove it for memory managment
                remove.append(token)
        for rem in remove: #removal
            TOKEN.remove(rem)
        TOKEN.append(TokenObject(dispatch))
    
    @staticmethod
    def get_token_attribute(dispatch, attr):
        """
        Will get a new token attribute based on the dispatcher
        @param attr: the attribute that you want. Will return a 
        string with the attribute name if it could not find it for debugging purposes.
        """
        for token in TOKEN:
            if token.is_equal(dispatch):
                try:
                    return getattr(token, attr)
                except:
                    return "token did not have attribute %s" % devpkg.utils.util.asciistr(attr) # we should not get here in the final release. Only there for debug
    
    @staticmethod
    def delete_token_attribute(dispatch):
        """
        Will delete the token attribute with the same token_id as the dispatch if the dispatch has a token_id
        """
        if not dispatch.auth_token:
            return
        for token in TOKEN:
            if token.token == dispatch.auth_token:
                TOKEN.remove(token)
                return
    
    @staticmethod
    def print_all_tokens():
        """
        Prints all tokens
        """
        for token in TOKEN:
            env.debug(devpkg.utils.util.asciistr(token))
