'''
Created on August 6, 2018

Copyright (c) 2018 by Cisco Systems, Inc.
All rights reserved.

@author: feliu@cisco.com
'''

from devpkg.utils.util import asciistr
from devpkg.base.command_interaction import CommandInteraction
import devpkg.base.command_service
import re
from fmc.config_keeper import ConfigKeeper
import devpkg.utils.util

class SelectiveProbe(object):
    """
    Class that Probes the FMC to find all data that is in it.
    """

    def __init__(self, ldev_id, object_type):
        """
        creates all the clis requried to probe the fmc
        """
        self.probe = ConfigKeeper.get_global(ldev_id)
        self.data = self.probe.data
        self.has_run = False
        self.ldev_id = ldev_id
        self.object_type = object_type
        if self.object_type == 'securityzones':
            sz_probe = CommandInteraction("GetAllSZ", probe=self.probe)
            sz_probe.add_basic_interaction("fmc_config/v1/domain/<domainUUID>/object/securityzones", None, None, None)
            self.cliis = [sz_probe]
            self.securityzones = []
        elif self.object_type == 'networks':
            network_probe = CommandInteraction("GetAllNetworks", probe=self.probe)
            network_probe.add_basic_interaction("fmc_config/v1/domain/<domainUUID>/object/networks", None, None, None)
            self.cliis = [network_probe]
            self.networks = []
        elif self.object_type == 'hosts':
            host_probe = CommandInteraction("GetAllHosts", probe=self.probe)
            host_probe.add_basic_interaction("fmc_config/v1/domain/<domainUUID>/object/hosts", None, None, None)
            self.cliis = [host_probe]
            self.hosts = []
        elif self.object_type == 'physicalinterfaces':
            interface_probe = CommandInteraction("GetAllInterfaces")
            interface_probe.add_basic_interaction("fmc_config/v1/domain/<domainUUID>/devices/devicerecords/<deviceUUID>/physicalinterfaces", None, None, None)
            self.cliis = [interface_probe]
            self.physicalinterfaces = []
        elif self.object_type == 'accesspolicies':
            acpolicy_probe = CommandInteraction("GetACPolicy")
            #Set Up acpolicy_probe
            acpolicy_probe.add_basic_interaction('fmc_config/v1/domain/<domainUUID>/policy/accesspolicies', None, None, None)
            acpolicy_probe.add_param("regex", ".*:%s:" % ldev_id, find_ac_policy_by_regex, None)
            self.accesspolicies = []
            self.cliis = [acpolicy_probe]
        elif self.object_type == 'accessrules':
            acrule_probe = CommandInteraction("GetAllACRules")
            #Set Up acrule_probe. This is dependant on acpolicy_probe
            acrule_probe.add_basic_interaction('fmc_config/v1/domain/<domainUUID>/policy/accesspolicies/<acUUID>/accessrules', None, None, None)
            acrule_probe.add_data_param(acpolicy_probe.idParam, "acUUID")
            self.cliis = [acrule_probe]
            self.accessrules = []
        elif self.object_type == 'inlinesets':
            inline_set_probe = CommandInteraction("GetAllIS")
            inline_set_probe.add_basic_interaction("fmc_config/v1/domain/<domainUUID>/devices/devicerecords/<deviceUUID>/inlinesets", None, None, None)
            self.cliis = [inline_set_probe]
            self.inlinesets = []
        elif self.object_type == 'bridgegroupinterfaces':
            bgi_probe = CommandInteraction("GetAllBGI")
            bgi_probe.add_basic_interaction("fmc_config/v1/domain/<domainUUID>/devices/devicerecords/<deviceUUID>/bridgegroupinterfaces", None, None, None)
            self.cliis = [bgi_probe]
            self.bridgegroupinterfaces = []
        elif self.object_type == 'subinterfaces':
            subinterface_probe = CommandInteraction("GetAllSubInterfaces")
            subinterface_probe.add_basic_interaction("fmc_config/v1/domain/<domainUUID>/devices/devicerecords/<deviceUUID>/subinterfaces", None, None, None)
            self.cliis = [subinterface_probe]
            self.subinterfaces = []
        elif self.object_type == 'etherchannelinterfaces':
            etherchannel_probe = CommandInteraction("GetAllEtherChannelInterfaces")
            etherchannel_probe.add_basic_interaction("fmc_config/v1/domain/<domainUUID>/devices/devicerecords/<deviceUUID>/etherchannelinterfaces", None, None, None)
            self.cliis = [etherchannel_probe]
            self.etherchannelinterfaces = []
        elif self.object_type == 'ipv4staticroutes':
            ipv4staticroute_probe = CommandInteraction("GetAllIPv4StaticRoutes")
            ipv4staticroute_probe.add_basic_interaction("fmc_config/v1/domain/<domainUUID>/devices/devicerecords/<deviceUUID>/routing/ipv4staticroutes", None, None, None)
            self.cliis = [ipv4staticroute_probe]
            self.ipv4staticroutes = []
        elif self.object_type == 'networkgroups':
            nog_probe = CommandInteraction("GetAllNOG")
            nog_probe.add_basic_interaction("fmc_config/v1/domain/<domainUUID>/object/networkgroups", None, None, None)
            self.cliis = [nog_probe]
            self.networkgroups = []


    def run(self, device):
        """
        We run our clis and get the data that we need. Then we store it.
        @param device: the device config
        """

        # probe should be run as needed in the new scheme as the configuration of a device can be changed dynamically
        
        devpkg.base.command_service.CommandService(device, self.cliis).execute(False)

        for cli in self.cliis:
            if cli.does_id_exist():
                if cli.dataParam.param_uuid.has_key('items'):#multiple
                    for item in cli.dataParam.param_uuid['items']:
                        if not self.item_in_data(item, self.data):
                            self.data.append(item)
                        self.add_object(item)
                else:
                    if not self.item_in_data(cli.dataParam.param_uuid, self.data):
                        self.data.append(cli.dataParam.param_uuid)
                    self.add_object(cli.dataParam.param_uuid)
            else:
                # See if items are present in 'param_value'
                try:
                    if cli.dataParam and cli.dataParam.param_value and cli.dataParam.param_value.has_key('items'):
                        for item in cli.dataParam.param_value['items']:
                            if not self.item_in_data(item, self.data):
                                self.data.append(item)
                            self.add_object(item)
                except Exception:
                    pass
        ldev = devpkg.utils.util.get_top_uuid_from_device(device)
        self.save_data(self.probe)
        self.probe.data = self.data
        ConfigKeeper.set_global(ldev, self.probe)
        
    def item_in_data(self, candidate, data):
        for item in data:
            if item['id'] == candidate['id']:
                return True
        return False

    def save_data(self, probe):
        if self.object_type == 'securityzones':
            probe.config_keeper.set('securityzones', self.securityzones)
            probe.securityzones = self.securityzones
        elif self.object_type == 'networks':
            probe.config_keeper.set('networks', self.networks)
            probe.networks = self.networks
        elif self.object_type == 'hosts':
            probe.config_keeper.set('hosts', self.hosts)
            probe.hosts = self.hosts
        elif self.object_type == 'physicalinterfaces':
            probe.config_keeper.set('physicalinterfaces', self.physicalinterfaces)
            probe.physicalinterfaces = self.physicalinterfaces
        elif self.object_type == 'accesspolicies':
            probe.config_keeper.set('accesspolicies', self.accesspolicies)
            probe.accesspolicies = self.accesspolicies
        elif self.object_type == 'accessrules':
            probe.config_keeper.set('accessrules', self.accessrules)
            probe.accessrules = self.accessrules
        elif self.object_type == 'inlinesets':
            probe.config_keeper.set('inlinesets', self.inlinesets)
            probe.inlinesets = self.inlinesets
        elif self.object_type == 'bridgegroupinterfaces':
            probe.config_keeper.set('bridgegroupinterfaces', self.bridgegroupinterfaces)
            probe.bridgegroupinterfaces = self.bridgegroupinterfaces
        elif self.object_type == 'subinterfaces':
            probe.config_keeper.set('subinterfaces', self.subinterfaces)
            probe.subinterfaces = self.subinterfaces
        elif self.object_type == 'etherchannelinterfaces':
            probe.config_keeper.set('etherchannelinterfaces', self.etherchannelinterfaces)
            probe.etherchannelinterfaces = self.etherchannelinterfaces
        elif self.object_type == 'ipv4staticroutes':
            probe.config_keeper.set('ipv4staticroutes', self.ipv4staticroutes)
            probe.ipv4staticroutes = self.ipv4staticroutes
        elif self.object_type == 'interfacesecurityzones':
            probe.config_keeper.set('interfacesecurityzones', self.interfacesecurityzones)
            probe.interfacesecurityzones = self.interfacesecurityzones
        elif self.object_type == 'networkgroups':
            probe.config_keeper.set('networkgroups', self.networkgroups)
            probe.networkgroups = self.networkgroups

    
    def add_object(self, obj):
        obj_type = obj['type']
        if obj_type == 'SecurityZone':
            # Get unique interfaces
            if isinstance(obj, dict) and obj.has_key('interfaces'):
                unique_intf = [dict(x) for x in set(tuple(item.items()) for item in obj['interfaces'])]
                obj['interfaces'] = unique_intf
            self.securityzones.append(obj)
        elif obj_type == 'Network':
            self.networks.append(obj)
        elif obj_type == 'Host':
            self.hosts.append(obj)
        elif obj_type == 'InterfaceSecurityZone':
            self.interfacesecurityzones.append(obj)
        elif obj_type == 'InlineSet':
            self.inlinesets.append(obj)
        elif obj_type == 'PhysicalInterface':
            self.physicalinterfaces.append(obj)
        elif obj_type == 'SubInterface':
            self.subinterfaces.append(obj)
        elif obj_type == 'EtherChannelInterface':
            self.etherchannelinterfaces.append(obj)
        elif obj_type == 'IPv4StaticRoute':
            self.ipv4staticroutes.append(obj)
        elif obj_type == 'BridgeGroupInterface':
            self.bridgegroupinterfaces.append(obj)
        elif obj_type == 'NetworkGroup':
            self.networkgroups.append(obj)

    def find_all_with_param_regex(self, param, reg):
        """
        return a list of jsons based on the regex of the param
        @param param: paramater to look in based on the database>
        EX: "name", "id", "commentHistoryList"
        @param reg: a regex string
        """
        return_arr = []
        for dat in self.data:
            if isinstance(dat, dict) and dat.has_key(param):
                try:
                    if re.search(reg, asciistr(dat[param])):
                        return_arr.append(dat)
                except:
                    pass
        return return_arr
                
    
    def find_all_with_params_regex(self, **kwargs):
        """
        get a dictionary that contains a param : reg model. returns the intersection of all.
        """
        first = True
        return_arr = []
        for k, v in kwargs.iteritems():
            temp = self.find_all_with_param_regex(k, v)
            if first:
                if temp == None:
                    temp = []
                return_arr = temp
                first = False
            elif len(return_arr) > 0:
                return_arr = intersection_of_arrays_ftd(return_arr, temp)
            else:
                return []
        return return_arr
    
    def get_all_items_by_ldevid(self, ldevid):
        arr = []
        arr += self.find_all_with_param_regex("name", asciistr(ldevid))
        arr += self.find_all_with_param_regex("ifname", asciistr(ldevid))
        arr += self.find_all_with_param_regex("description", asciistr(ldevid))
        arr += self.find_all_with_param_regex("commentHistoryList", asciistr(ldevid))
        return arr

    def get_physicalinterfaces(self):
        return self.physicalinterfaces
    
def intersection_of_arrays_ftd(arr1, arr2):
    """
    returns the intersection of the two arrays
    """
    ret_arr = []
    for item1 in arr1:
        for item2 in arr2:
            if item1 == item2:
                ret_arr.append(item1)
    return ret_arr

def find_ac_policy_by_regex(command_executor):
    """
    Finds the Access Policy from a list of Access Policy based on the regex
    """
    try:
        ac_items = command_executor.command_holder.parent.dataParam.param_value['items']
    except:
        return [None]
    reg = command_executor.command_holder.param_value
    for ac in ac_items:
        try:
            if re.match(reg, asciistr(ac)) != None:
                command_executor.command_holder.parent.dataParam.param_uuid = ac
                command_executor.command_holder.parent.idParam.param_uuid = ac['id']
                return [None]
        except:
            pass
    command_executor.command_holder.parent.dataParam.param_uuid = None
    return [None]
