'''
Created on Jul 25, 2013

@author: jeffryp

Copyright (c) 2013 by Cisco Systems, Inc.
All rights reserved.

Common utility functions
'''

from collections import OrderedDict
import copy
import json
import re
import time

from requests.exceptions import RequestException

from devpkg.utils.errors import ConnectionError, DeviceConfigError
from devpkg.utils.state_type import State, Type
from devpkg.utils.type2key import type2key
from devpkg.utils.ipaddress import ip_network, IPv4Address
import devpkg.base.command_dispatch
import devpkg.base.dmobject


def sleep(stime):
    '''
    use specilized sleep so that we can turn it off when we do a mock call
    '''
    if not devpkg.base.command_dispatch.MOCK:
        time.sleep(stime)
    
def connection_exception_handler(f):
    ''' Decorator used to handle connection exceptions.
    '''
    def handler(*argv):
        try:
            return f(*argv)
        except DeviceConfigError as e:
            raise e
        except (IOError, RequestException) as e:
            raise ConnectionError(asciistr(e))
        except:
            raise
    return handler

def key_as_string(obj):
    ''' This function makes the key into string in a dictionary.
    '''

    if isinstance(obj, dict):
        result = OrderedDict()
        for key, value in obj.iteritems():
            if isinstance(value, dict):
                result[repr(key)] = key_as_string(value)
            else:
                result[repr(key)] = value
        return result
    else:
        return obj

def pretty_dict(obj, indent=3):
    ''' This function prints a dictionary in a nicer indented form.
    '''
    try:
        return json.dumps(key_as_string(obj), indent=indent)
    except Exception:
        # If not JSON serializable
        return asciistr(obj)

def normalize_param_dict(param_dict):
    '''
    Normalize a param dictionary by extracting the 'key' and 'value' into a new
    dictionary, recursively.

    @param param_dict: scalar value or param dictionary
        if it's a param dictionary, then each entry is of the format:
            (type, key, instance): {'state': state,
                                    'device': device,
                                    'connector': connector,
                                    'value':  scalar value or param dictionary}
    @return: scalar value or param dictionary
        if param_dict is a param dictionary, then return the a new dictionary
            with just the 'key' and 'value'.
        otherwise param_dict is returned
    '''

    if not isinstance(param_dict, dict):
        
        return param_dict
    
    result = {}
    for (kind, key, instance), cfg_value in param_dict.iteritems():
        'Will deprecate the check for target in the future, it should be handled at the program initialized'
        if 'target' in cfg_value:
            value = cfg_value['target']
        elif 'value' in cfg_value:
            value = cfg_value['value']
        else:
            value = None
        if isinstance(value, dict):
            value = normalize_param_dict(value)
        """
        For certain types of entries in the configuration parameter, the key
        may not corresponding to our internal key.
        Example:
            (Type.ENCAP, '', '3')
        we look up the internal key for it here.
        """
        
        
        key = type2key.get(kind, key)
        if result.has_key(key):#we dont want to override that information we will add it as an array
            arr = result[key]
            if not isinstance(arr, list): #therefore not an array
                result[key] = []
                result[key].append(arr)
                result[key].append(value)
            else:
                result[key].append(value)
        else: 
            result[key] = value
        
    return result

'ToDo: CSCvd20931 - Comment out the code block for now to improve the coder coverage. The code is not executed for current implementation.'
# def normalize_state_param_dict(param_dict):
#     '''
#     Normalize a param dictionary by extracting the 'key' and 'value' into a new
#     dictionary, recursively.
# 
#     @param param_dict: scalar value or param dictionary
#         if it's a param dictionary, then each entry is of the format:
#             (type, key, instance): {'state': state,
#                                     'device': device,
#                                     'connector': connector,
#                                     'value':  scalar value or param dictionary}
#     @return: scalar value or param dictionary
#         if param_dict is a param dictionary, then return the a new dictionary
#             with just the 'key' and 'value'.
#         otherwise param_dict is returned
#     '''
# 
#     if not isinstance(param_dict, dict):
#         return param_dict
# 
#     result = {}
#     for (kind, key, instance), cfg_value in param_dict.iteritems():
#         'Will deprecate the check for target in the future, it should be handled at the program initialized'
#         if 'target' in cfg_value:
#             value = cfg_value['target']
#         elif 'value' in cfg_value:
#             value = cfg_value['value']
#         if isinstance(value, dict):
#             value = normalize_state_param_dict(value)
#         result[key] = value
#         result[key + '_state'] = cfg_value['state']
#     return result
# 
# def ifcize_param_dict(param_dict):
#     '''Reverse the operation of util.normalize_param_dict
#     @param param_dict: dict
#         A configuration diction in the normal format, e.g:
#          {'Hostname': 'my-asa'}
#     @return dict in the IFC format,
#         For the above example, the return value is:
#          {(Type.PARAM, 'Hostname', ''), {'state': State.NOCHANGE, 'value': 'my-asa'}}
#     @todo: move it into util. But encountered cyclic-dependency problem.
# 
#     '''
#     if not isinstance(param_dict, dict):
#         return param_dict
# 
#     result = {}
#     for key, value in param_dict.iteritems():
#         kind = Type.FOLDER if isinstance(value, dict) else Type.PARAM
#         if isinstance(value, dict):
#             value = ifcize_param_dict(value)
#         result[(kind, key, '')] = {'state': State.NOCHANGE, 'value': value}
#     return result
# 
# 
# def filter_out_sacred_commands(output_clis):
#     '''Remove commands that may cause disruption of communication between IFC and the ASA device.
#     @param output_clis: list of RESTInteraction's
#     @return output_clis
#     '''
#     mgmt_intf = 'management'
#     sacred_commands  = ['^no route ' + mgmt_intf,
#                         '^no http enable',
#                         '^no http .+ ' + mgmt_intf,
#                         '^clear config interface Management0/0$',
#                         '^clear config interface Management0/1$']
# 
#     for sacred in sacred_commands:
#         cmds = filter(lambda text: re.compile(sacred).match(str(text).strip()), output_clis)
#         for cmd in cmds:
#             output_clis.remove(cmd)
#     return output_clis
# 
# def command_filter(clis):
#     'Ignore those commands that we dont care'
#     def allowed(cli):
#         return cli and not cli.startswith('!')
#     return filter(allowed, clis)

def set_cfg_state(ifc_cfg_dict, state):
    '''Set the state of each entry in the a given IFC configuration
     to a given state.
    @param ifc_cfg_dict: dict
        IFC configuration dictionary
    @param state: State
    '''

    if 'state' in  ifc_cfg_dict:
        ifc_cfg_dict['state'] = state

    for value in ifc_cfg_dict.values():
        if not isinstance(value, dict):
            continue
        if not 'state' in  value:
            set_cfg_state(value, state)
            continue
        value['state'] = state

        if 'cifs' in value:
            key_of_value = 'cifs'
        elif 'target' in value:
            key_of_value = 'target'
        else:
            key_of_value = 'value'

        ifc_cfg_dict = value.get(key_of_value)
        if isinstance(ifc_cfg_dict, dict):
            set_cfg_state(ifc_cfg_dict, state)


def normalize_faults(faults, config):
    """
    Remove (0, '', '') from the fault path if it is not in the config paramter
    @param faults: list of fault
        Each fault is a triplet: (path, error_number, error_string),
        where path is list of triplets of the form (kind, key, instance). where the key might be modified by
        fill_key_string method during command generation.
    @param config: dict
        the original configuration parameter
    @return the same faults using the original dictionary key in the original configuration dictionary
    """

    def normalize_path(path, config):
        result = []
        if path == None:
            path = get_config_root_key(config)
        if config == None:
            return None
        if len(config) == 0:
            return None
        try:
            for kind, key, instance in path:
                if kind == Type.DEV:
                    """
                     The top level config might not be (Type.DEV, key, name)
                     this is the case of clusterXXX and deviceXXX APIs
                     Skip it in this case.
                    """
                    keys = config.keys()
                    if keys and keys[0][0] != Type.DEV:
                        continue
                if instance is not None:    
                    result.append((kind, key, instance))
                else:
                    'This is to handle device missing configuration faults. Instance value should be graph instance id.'
                    keys = config.keys()
                    result.append((kind, key, keys[0][2]))
        except:
            return path
        return result

    result = []
    for path, err_num, err_str in faults:
        path = normalize_path(path, config)
        result.append((path, err_num, err_str))
    return result


def get_config_root_key(configuration):
    'Return the key to the root entry of an IFC configuration parameter'
    return [configuration.keys()[0]] if configuration else []

'ToDo: CSCvd20931 - Comment out the code block for now to improve the coder coverage. The code is not executed for current implementation.'
# def get_config_firewall_keys(configuration):
#     'Return the list of keys to the Firewall entries in an IFC configuration parameter'
#     result = []
#     if not configuration:
#         return result
#     kind, root_key, instance = configuration.keys()[0]
#     if kind != Type.DEV:
#         return result
#     grp_keys = filter(lambda (kind, key, instance):  kind == Type.GRP, configuration.values()[0]['value'].keys())
#     if not grp_keys:
#         return result
#     root_key = configuration.keys()[0]
#     for grp_key in grp_keys:
#         grp_config = configuration.values()[0]['value'][grp_key]
#         hits = filter(lambda (kind, key, instance):  kind == Type.FUNC, grp_config['value'].keys())
#         for key in hits:
#             result.append([root_key, grp_key, key])
#     return result


def massage_param_dict(asa, ifc_delta_cfg_dict, transformers = None):
    '''Adjust the configuration dictionary parameter passed to us by IFC to a format suited to us.
    @param asa:  DeviceModel
    @param ifc_delta_cfg_dict: IFC configuration parameter dictionary
    @param transformers: list of function that takes one argument and return an object
        the purpose of a transformer is to transform IFC configuration dictionary to a desired format.
        The order of the application of the transformer is the reverse given in the parameter,
        i.e. if the transformers is [a, b], the result will be a(b(ifc_delta_cfg_dict)).
    @return dict
        if_cfg_dict is a equivalent of ifc_delta_cfg_dict but massaged for ifc2asa or
        generate_ifc_delta_cfg methods
    '''
    if not ifc_delta_cfg_dict:
        result = {(Type.DEV, asa.ifc_key, ''):  {'state': State.NOCHANGE, 'value': ifc_delta_cfg_dict}}
    else:
        kind, key, instance = ifc_delta_cfg_dict.keys()[0]
        if kind == Type.DEV: #for the complete configurations: device, group, function
            result = ifc_delta_cfg_dict
        else:#device configuration
            result = {(Type.DEV, asa.ifc_key, ''):  {'state': State.NOCHANGE, 'value': ifc_delta_cfg_dict}}
        'Do not modify the parameter passed to us by IFC.'
        result = copy.deepcopy(result)

    'Apply the transformations required'
    if not transformers:
        transformers = []
    for transformer in reversed(transformers):
        result = transformer(result)
    return result

'ToDo: CSCvd20931 - Comment out the code block for now to improve the coder coverage. The code is not executed for current implementation.'
# def get_all_connectors(configuration, DeviceModelType):
#     'Returns a tuple of firewalls and connectors'
#     asa = DeviceModelType()
#     ifc_cfg = massage_param_dict(asa, configuration)
#     asa.populate_model(ifc_cfg.keys()[0], ifc_cfg.values()[0])
# 
#     connectors = []
#     firewalls = []
#     for group in asa.iter_groups():
#         firewall = group.get_child('NGIPS')
#         firewalls.append(firewall)
#         for connector in firewall.iter_connectors():
#             connectors.append(connector)
#     return (firewalls, connectors)

def normalize_interface_name(intf):
    '''
    Normalizes interface, sub-interface, and port-channel name. For example:
        'g0/0'  -> 'GigabitEthernet0/0'
        'Gig0/1'-> 'GigabitEthernet0/1'
        'm0/0'  -> 'Management0/0'
        'Te0/8' -> 'TenGigabitEthernet0/8'
        'Eth1/1.20' -> 'Ethernet1/1.20'
        'Po9'   -> 'Port-channel9'
    '''

    if not intf:
        return
    intf = intf.strip()
    intf = pattern_replace(r'([gG][\D]*)[\d]+/[\d]+.*$', 'GigabitEthernet', intf)
    intf = pattern_replace(r'([mM][\D]*)[\d]+/[\d]+.*$', 'Management', intf)
    intf = pattern_replace(r'([tT][\D]*)[\d]+/[\d]+.*$', 'TenGigabitEthernet', intf)
    intf = pattern_replace(r'([eE][\D]*)[\d]+/[\d]+.*$', 'Ethernet', intf)
    intf = pattern_replace(r'([pP][\D]*)[\d]+.*$', 'Port-channel', intf)
    return intf

def pattern_replace(pattern, replace, source):
    '''
    Find 'pattern' in 'source' and replace the first group with 'replace'.
        'pattern' example: '([gG][\D]*)[\d]+/[\d]+.*$'
        'replace' example: 'GigabitEthernet'
        'source' example: 'g0/0'
        return result: 'GigabitEthernet0/0'
    For interface replacement, it happens if and only if the interface will be accepted
    by the device, in another word, it must be the prefix of the full interface name.
    '''
    p = re.compile(pattern)
    m = p.match(source)
    if m:
        prefix = m.group(1)
        if replace.lower().startswith(prefix.lower()):
            return re.sub(prefix, replace, source)
    return source

def hide_passwords(device):
    '''
    Hide the passwords in the device dictonary by replacing them with '<hidden>.
    This should be done on a deep copy, since the dictionary will be modified.
    '''

    if 'devs' in device:
        for value in device['devs'].itervalues():
            hide_passwords(value)
    if 'creds' in device and 'password' in device['creds']:
        device['creds']['password'] = '<hidden>'

    return device

def netmask_from_prefix_length(prefix_length):
    '''
    Convert a prefix length to a netmask.
         0 -> 0.0.0.0
        24 -> 255.255.255.0
        32 -> 255.255.255.255 
    '''

    netmask = (0xffffffff00000000 >> int(prefix_length)) & 0xffffffff
    return str(IPv4Address(netmask)) 

def normalize_network_address(addr, mask):
    '''
    Normalizes a network address to match its mask.

    Mask can be either an IP address (255.255.255.0) or a prefix length (24).
    
    Example:
    2.2.2.10, 255.255.255.0 -> 2.2.2.0
    2002:3::10, 64          -> 2002:3::
    '''

    # The backport of ipaddress.py requires a unicode address
    addr = unicode(addr)
    mask = unicode(mask)
    network = ip_network(addr + '/' + mask, strict=False)
    return str(network.network_address)

def get_top_uuid_from_device(device):
    return asciistr(devpkg.base.dmobject.DMObject.get_unique_graph_string_from_device_dn(device['dn'])) if device.has_key('dn') else ''

def get_top_uuid_from_name(name):
    # name e.g.: ExtINT2_tenant1_dev1
    if name is not None:
        name_list = name.split('_')
        return '_'.join(name_list[1:]) if len(name_list) > 1 else ''
    return ''

def get_access_policy_from_config(config):
    if len(config.values()) == 0 or not config.values()[0].has_key('value'):
        return ''
    items = config.values()[0]['value'].keys()
    for item in items:
        (item_id, item_type, item_name) = item
        if item_type == 'AccessPolicy':
            return item_name
    return ''

def asciistr(instr):
    try:
        outstr = instr
        if isinstance(instr, unicode):
            outstr = instr.encode('ascii', 'ignore')
        else:
            outstr = str(instr)
    except:
        return ''
    return outstr
