'''
Created on January 25, 2018

@author: feliu@cisco.com, linlilia@cisco.com

Copyright (c) 2018 by Cisco Systems, Inc.
All rights reserved.

Classes used for Interface.
'''


from devpkg.utils.util import asciistr
from devpkg.base.dmboolean import DMBoolean
from devpkg.base.simpletype import SimpleType
from devpkg.interface import InterfaceConfig, InterfaceConfigs
from devpkg.base.command_interaction import CommandInteraction
from devpkg.utils.errors import DeviceConfigError, FaultCode
from devpkg.utils.error_messages import NO_VLAN_IN_INLINE_MODE_ERR
from devpkg.utils.state_type import State
from fmc.parsers import delete_interface_executor, delete_device_object_executor, parse_response_for_target_and_store_json, remove_interface_from_inline_set_executor
from parsers import physical_interface_enable_executor,interface_executor
from devpkg.base.specilized_param.command_param_value import CommandParamValue
from devpkg.base.specilized_param.command_param_append import CommandParamAppend
import inline_set
from static_route import StaticRoute
from fmc.config_keeper import ConfigKeeper

class NGIPSEtherChannelConfig(InterfaceConfig):
    '''
    Interface related configuration

    '''
    GENERAL_INTERFACE = """
    {
      "type": "SubInterface",
      "enabled": true,
      "MTU": 1500,
      "managementOnly": false,
      "ifname": "Outside",
      "enableDNSLookup": false,
      "enableAntiSpoofing": false,
      "name": "Port-channel2"
    }
    """
    MTU_DEFAULT_VALUE = '1500'
    ETHERCHANNEL_TYPE = 'EtherChannelInterface'
    
    
    def __init__(self, instance, probe):
        super(NGIPSEtherChannelConfig, self).__init__(instance, dm_key = 'SubInterface', probe=probe)
        self.dm_key_command = 'fmc_config/v1/domain/<domainUUID>/devices/devicerecords/<deviceUUID>/etherchannelinterfaces/<id>' 
        
        'Must be in the order in which to generate the command.'
        self.register_child(DMBoolean('enabled', 'enabled', 'enabled', defaults = 'true'))
        
        
        self.register_child(SimpleType('int_security_zone', 'szUUID', 'fmc_config/v1/domain/<domainUUID>/object/securityzones', defaults = {},  response_parser=parse_response_for_target_and_store_json, response_parser_arg={"target":"id"}, param_req_formater=CommandInteraction.DO_NOT_ADD_TO_PUT_POST_JSON, throw_error=True, probe=probe))
        self.register_child(SimpleType('mtu', 'MTU', defaults = NGIPSEtherChannelConfig.MTU_DEFAULT_VALUE))
        
        self.register_child(SimpleType('int_inline_set', 'Dummy',"fmc_config/v1/domain/<domainUUID>/devices/devicerecords/<deviceUUID>/inlinesets", defaults='None', delete_executor=remove_interface_from_inline_set_executor, param_req_formater=CommandInteraction.DO_NOT_ADD_TO_PUT_POST_JSON, typeToMake=CommandParamAppend, probe=probe))
        self.register_child(SimpleType('IPv4Config', 'ipv4', defaults={}, param_req_formater=CommandInteraction.DO_NOT_ADD_TO_PUT_POST_JSON, probe=probe))
        self.register_child(SimpleType('ifname', 'ifname', defaults=''))
        self.register_child(SimpleType('activeMACAddress', 'activeMACAddress', defaults='', probe=probe))
        self.register_child(StaticRoute(probe))
        self.dependent_on = []
        self.probe = probe
        
    def is_ENCAPASS_state_delete(self, intf):
        '''
        Algorithm:
        1. From the ifc_key, which should be the target name, we try to determine if this target is associated with ExIntConfigRelFolder or InIntConfigRelFolder.
        2. If It's associated with ExIntConfigRelFolder, then the connector type is 'external', otherwise it's 'internal'
        3. Then, based on the connector type, we determine if it's associated with 'consumer' child or 'provider' child of the connector, and get the corresponding instance name, which will be used to identify the ENCAPASS folder
        4. The ENCAPASS folder mentioned in step 3 above tracks the state of a connector interface, which we'll use to determine if an interface config should be modified or not. For example, if state is 0 (nochange), corresponding
        interface config should be kept even if the state is '3' (delete) in the ExIntConfigRelFolder or InIntConfigRelFolder folder itself.
        '''
        if len(self.parent.parent.get_child('SharedConfig').children.values()) > 0:
            # This is the case when the configuration is coming from APIC and we need to check the ENCAPASS state. 
            target = intf.ifc_key
            interface_type = 'external' if self.get_top().children['SharedConfig'].children.values()[0].children['FTD'].children['ExIntfConfigRelFolder'].children['ExIntfConfigRel'].delta_ifc_cfg_value['target'] == target else 'internal'
            encapass_name = self.parent.parent.get_child('SharedConfig').children.values()[0].get_child('FTD').get_child('CONN').children['consumer'].get_value()['ENCAPREL'] if \
                self.parent.parent.get_child('SharedConfig').children.values()[0].get_child('FTD').get_child('CONN').children['consumer'].conn_type == interface_type else \
                self.parent.parent.get_child('SharedConfig').children.values()[0].get_child('FTD').get_child('CONN').children['provider'].get_value()['ENCAPREL']
            return self.get_top().get_child('ENCAPASS').children[encapass_name].state == State.DESTROY
        else:
            # This is the case for configuration coming from FMC where ENCAPASS folder does not apply, so leave the logic as is. 
            return True
    
    def ifc2dm(self, no_dm_cfg_stack, dm_cfg_list):
        try:
            self.dependent_on.append('/SecurityZone/%s' % self.find('int_security_zone').get_value()['security_zone'])
            self.dependent_on.append('/BridgeGroupInterface/Interfaces') 
        except:
            pass
        'Populate current instance recursive state based on children and dependent objects.'
        self.recursive_state = self.get_state_recursive()
        if not self.has_ifc_delta_cfg():
            return

        is_vlan_used = self.get_top().is_vlan_supported()
        is_virtual = self.get_top().is_virtual()
        k, t, n = self.delta_ifc_key
        if not hasattr(self, 'vif'):
            # If there are no connectors using this interface, get the vif and
            # vlan from the value dictionary
            if self.has_ifc_delta_cfg() and 'vif' in self.delta_ifc_cfg_value:
                self.vif = self.delta_ifc_cfg_value['vif']
                if 'vlan' in self.delta_ifc_cfg_value:
                    self.vlan = self.delta_ifc_cfg_value['vlan']
            elif k != 13:
                # No interface information to generate commands
                return
            else:
                try:
                    self.vlan = self.delta_ifc_cfg_value['value'][(13, 'vlanId', 'vlanId')]['value']

                    # There is a case that VLAN is not supported for vFTD but there is subInterface 
                    # on the vFTD with VLAN - when the admin check the "Trunking Port" checkbox through
                    # GUI of L4-L7 Device on APIC after a graph is deployed.
                    if self.get_state() == State.DESTROY and is_virtual:
                        is_vlan_used = True

                except:
                    self.vlan = "1234"
                self.vif = self.ifc_key.split('_SPEC_ID')[0]

        local_dm_cfg_list = []
        inline_mode = False
        # Process for commands if state modified.
        if self.get_state() == State.CREATE or self.get_state() == State.MODIFY:
            super(NGIPSEtherChannelConfig, self).ifc2dm(no_dm_cfg_stack, local_dm_cfg_list)

            # raise fault if trying to turn on VLAN trunking in inline mode
            if is_virtual and self.get_top().is_vlan_supported() and local_dm_cfg_list[0].params.has_key('int_inline_set'):
                raise DeviceConfigError([([(0, '', None)], FaultCode.CONFIGURATION_ERROR, NO_VLAN_IN_INLINE_MODE_ERR)])
            # Add command for later execution.        
            for item in local_dm_cfg_list:
                if item.toDict().has_key('command') and item.command == 'IPv4StaticRoute':
                    if not item in (dm_cfg_list):
                        self.add_cli_to_array(item, dm_cfg_list)
                elif item.params.__len__() > 0:
                    item.add_data_param(self.vif, "name", paramType=CommandParamValue)
                    
                    if item.params.has_key('IPv4Config'):
                        if len(item.params['IPv4Config'].param_value) > 1:
                            raise Exception("Only one IPV4 Config should be set out of Static, PPPoE, and DHCP")
                        else:
                            ipv4 = {}
                            if item.params['IPv4Config'].param_value.has_key('static') and \
                            item.params['IPv4Config'].param_value['static'].has_key('address'): #we need to change static from what is there and what we have
                                addr = item.params['IPv4Config'].param_value['static']['address']
                                if '/' in addr: #we have xxx.xxx.xxx.xxx/xx
                                    addr_split = addr.split('/')
                                    ipv4 = {"static":{"address":addr_split[0], "netmask":addr_split[1]}}
                                else:
                                    ipv4 = item.params['IPv4Config'].param_value['static']
    
                            item.add_data_param(ipv4, 'ipv4')
    
                    if (is_virtual is False or (is_virtual and is_vlan_used)) and not item.params.has_key('int_inline_set'): # if vlan is supported(physical interface or virtual with vlan truncking) and not in inline mode. we make sub interfaces
                        item.add_basic_interaction("fmc_config/v1/domain/<domainUUID>/devices/devicerecords/<deviceUUID>/subinterfaces", self.vif, interface_executor, NGIPSEtherChannelConfig.GENERAL_INTERFACE)
                        item.name = self.vif+'.'+asciistr(self.vlan)
                        item.params['GET'].param_value['vlanId'] = self.vlan
    
                        item.add_data_param('SubInterface', 'type')
                        item.add_data_param(asciistr(self.vlan), 'vlanId')
                        item.add_data_param(asciistr(self.vlan), 'subIntfId')
                    else:
                        self.dm_key = NGIPSEtherChannelConfig.ETHERCHANNEL_TYPE
                        item.add_basic_interaction("fmc_config/v1/domain/<domainUUID>/devices/devicerecords/<deviceUUID>/etherchannelinterfaces",self.vif, interface_executor, NGIPSEtherChannelConfig.GENERAL_INTERFACE, True)
                        item.add_data_param("EtherChannelInterface", "type")
                    
                    item.add_data_param(self.dm_key, "type", paramType=CommandParamValue)
                    if item.params.has_key('ifname'):
                        item.params['ifname'].param_value += self.get_unique_graph_string_append() #add Tenant + Ldev to the name
                    if item.params.has_key('activeMACAddress'):
                        if self.children.has_key('activeMACAddress') and self.children['activeMACAddress'].delta_ifc_cfg_value and self.children['activeMACAddress'].delta_ifc_cfg_value.has_key('value') \
                                and self.children['activeMACAddress'].delta_ifc_cfg_value['state'] != State.DESTROY:
                            item.params['activeMACAddress'].param_value = self.children['activeMACAddress'].delta_ifc_cfg_value['value'].lower()
    
                    try:
                        #check for BGI
                        bgi_mode = False
                        bgi_interfaces = self.find("/BridgeGroupInterface/Interfaces") #find all bgi interfaces
                        if not bgi_interfaces is None:
                            bgi_interfaces = bgi_interfaces.get_value()
                            for interface in bgi_interfaces['Interface']:
                                if interface['InterfaceHolder'] == self.ifc_key: #check to see if current interface is in a BGI
                                    bgi_mode = True
                                    break
                    except:
                        pass
                    
                    if not (item.params.has_key('int_inline_set') or bgi_mode): #if not inline we add the Security Zone. If Inline we need to add is seperatly. Same for BGI
                        try:
                            if self.find('int_security_zone').get_state() != State.DESTROY:
                                sz_id = self.find("/SecurityZone/%s" % item.params['int_security_zone'].param_value['security_zone']).cli.idParam
                                item.add_data_param({"id" : sz_id, "type" : "SecurityZone"}, "securityZone")
                            else:
                                item.add_data_param({}, "securityZone")
                        except:
                            pass
                    elif not bgi_mode: # only for inline mode
                        try:
                            item.params['int_inline_set'].param_value['inline_set'] += self.get_unique_graph_string_append()#We add the Tenant + Ldev to the inline set
                        except:
                            pass
                        inline_mode = True
                        
                    # When IPv4StaticRoute is added, we need to add all of them into dm_cfg_list
                    # Interface may contain IPv4 cli, adding a cli to dm_cfg_list will overwrite the cli attribute associated to this interface
                    # so we do 2-pass addition here so that the cli associated to the NGIPSInterface base class will be kept as
                    # interface/sub-interface correctly rather than being overwritten with an IPv4StaticRoute cli.
                    for cli in range(0, len(local_dm_cfg_list)): 
                        if local_dm_cfg_list[cli].command == 'IPv4StaticRoute':
                           self.add_cli_to_array(local_dm_cfg_list[cli], dm_cfg_list)
                    # 2nd pass
                    for cli in range(0, len(local_dm_cfg_list)): 
                        if local_dm_cfg_list[cli].command != 'IPv4StaticRoute':
                           self.add_cli_to_array(local_dm_cfg_list[cli], dm_cfg_list)
                    try:
                        if item.params.has_key('int_inline_set') and self.children['int_inline_set'].get_state() != State.NOCHANGE and self.children['int_inline_set'].get_state() != State.DESTROY:
                            inline_set.InterfaceInlineSet().Interface(item.params['int_inline_set'].model_key, item.params['int_inline_set'].param_value['inline_set'] , dm_cfg_list, item)
                    except:
                        pass
    
                    if inline_mode: #system works for Inline
                        dm_cfg_list.append(SecurityZoneInterfaceInline(self, item, probe=self.probe))
    
                        'For Inline set mode, the EtherChannel interfaces MTU value need be the same as inline set MTU value'
                        'Updating inline set MTU will make related interface MTU value change as well'
                        'So no need to update interface MTU when inline set is being updated. '
                        'APIC allows user to set different MTU value on inline set member interface, which should throw fault as what FMC does.'
                        intf_mtu_to_be = item.params['mtu'].param_value if item.params.has_key('mtu') else inline_set.InlineSet.MTU_DEFAULT_VALUE
                        if not item.params.has_key('mtu') and isinstance(self.parent, NGIPSEtherChannelConfigs):
                            intf_mtu_to_be = self.parent.parent.delta_ifc_cfg_value['value'][(4,'InlineSet','IPSSet')]['value'][(5,'mtu','mtu')]['value']
                        intf = self.probe.config_keeper.get_interface_from_probe(item.params['name'].param_value, 'etherchannelinterfaces')
                        if intf is not None:
                            intf_mtu = asciistr(intf['MTU']) if intf.has_key('MTU') else inline_set.InlineSet.MTU_DEFAULT_VALUE
                            item.add_data_param(intf_mtu, 'MTU')
                        else:
                            item.add_data_param(intf_mtu_to_be, 'MTU')
    
                    elif self.get_state() == State.CREATE:
                        intf_mtu_to_be = item.params['mtu'].param_value if item.params.has_key('mtu') else NGIPSEtherChannelConfig.MTU_DEFAULT_VALUE
                        item.add_data_param(intf_mtu_to_be, 'MTU')
        elif self.get_state() == State.DESTROY:
            need_to_delete_intf = self.is_ENCAPASS_state_delete(self)
            super(InterfaceConfig, self).ifc2dm(no_dm_cfg_stack, local_dm_cfg_list)
            if local_dm_cfg_list.__len__() > 0 and local_dm_cfg_list[0].params.__len__() > 0:     
                for item in local_dm_cfg_list:
                    command = item.get_command()
                    if command in ('SubInterface', 'EtherChannelInterface'):
                        if (is_virtual and not is_vlan_used) or local_dm_cfg_list[0].params.has_key('int_inline_set'):
                            clii = CommandInteraction(self.dm_key, model_key=self.get_config_path(), probe=self.probe)
                            clii.add_basic_interaction('fmc_config/v1/domain/<domainUUID>/devices/devicerecords/<deviceUUID>/etherchannelinterfaces', self.vif, interface_executor, "", True)
                            clii.add_data_param({}, "ipv4")
                            clii.add_data_param({}, "securityZone")
                            clii.add_data_param('', "ifname")
                            clii.add_data_param('', "activeMACAddress")
                            #clii.add_data_param("NONE", "mode")
                            clii.add_data_param(True, "enabled")
                        else:
                            clii = CommandInteraction(self.dm_key, model_key=self.get_config_path(), probe=self.probe)
                            clii.add_basic_interaction("fmc_config/v1/domain/<domainUUID>/devices/devicerecords/<deviceUUID>/subinterfaces", self.vif, delete_interface_executor, "")
                            clii.name = self.vif+'.'+asciistr(self.vlan)
                            clii.params['GET'].param_value['vlanId'] = self.vlan
                        if need_to_delete_intf:
                            self.add_cli_to_array(clii, dm_cfg_list)

                # We need to delete all IPv4 static routes first before deleting interface due to dependency
                all_routes = self.probe.config_keeper.get('ipv4staticroutes')
                ifname_value = None
                if self.delta_ifc_cfg_value['value'].has_key((5, 'ifname', 'ifname')):
                    ifname_value = self.delta_ifc_cfg_value['value'][(5, 'ifname', 'ifname')]['value']
                elif self.delta_ifc_cfg_value['value'].has_key((13, 'ifname', 'ifname')):
                    ifname_value = self.delta_ifc_cfg_value['value'][(13, 'ifname', 'ifname')]['value']
                top_uuid = self.probe.config_keeper.get('top_uuid')
                if not ifname_value.endswith(top_uuid):
                    # create the full logical interface name by appending the '_Tenant_Device' suffix part
                    ifname_value = ifname_value + '_' + top_uuid
                for route in all_routes:
                    if ifname_value == route['interfaceName']:
                        route_id = route['id']
                        cld = self.get_child('StaticRoute').get_child('IPv4StaticRoute')
                        clii = CommandInteraction('IPv4StaticRoute', model_key=cld.get_config_path(), probe=self.probe)
                        clii.add_basic_interaction("fmc_config/v1/domain/<domainUUID>/devices/devicerecords/<deviceUUID>/routing/ipv4staticroutes", route_id, delete_device_object_executor, "", id=route_id)
                        clii.add_data_param(ifname_value, 'interfaceName')
                        clii.add_data_param(self.probe.config_keeper.get_network_value_from_ipv4_route(route), 'network')
                        clii.add_data_param(self.probe.config_keeper.get_host_value_from_ipv4_route(route), 'gateway')
                        metric_value = int(route['metricValue']) if route.has_key('metricValue') and route['metricValue'] is not None else 1
                        clii.add_data_param(metric_value, 'metric')
                        isTunneled_value = route['isTunneled'] if route.has_key('isTunneled') else False
                        clii.add_data_param(isTunneled_value, 'isTunneled')
                        clii.command = 'IPv4StaticRoute'
                        if need_to_delete_intf:
                            clii.state = State.DESTROY
                            self.add_cli_to_array(clii, dm_cfg_list)
                        else:
                            clii.state = State.NOCHANGE
                            # Note, we're not using self.add_cli_to_array function as above because it'll overwrite the clii state back to State.DESTROY which we don't want.
                            # Instead, we put the clii into the dm_cfg_list at the right place directly.
                            found_place_to_insert = False
                            for i in range(len(dm_cfg_list)):
                                if dm_cfg_list[i].state == State.DESTROY:
                                    dm_cfg_list.insert(i, clii)
                                    found_place_to_insert = True
                                    break
                            if not found_place_to_insert:
                                dm_cfg_list.append(clii)
        else: # self.get_state() == State.NOCHANGE or no state
            # when we do mini-audit for IPv4StaticRoute, we need to keep the config from APIC so it won't be deleted during audit
            super(InterfaceConfig, self).ifc2dm(no_dm_cfg_stack, local_dm_cfg_list)
            for item in local_dm_cfg_list:
                if item.toDict().has_key('command') and item.command == 'IPv4StaticRoute':
                    if not item in (dm_cfg_list):
                        item.state = State.CREATE
                        self.add_cli_to_array(item, dm_cfg_list)

class SecurityZoneInterfaceInline(CommandInteraction):
    """
    Interface Cli used to add the security zone into it. Used by the Inline Interface.
    """
    def __init__(self, interface_config, interface_clii, probe=None):
        super(SecurityZoneInterfaceInline, self).__init__("SecurityZoneInterfaceInline", interface_clii.model_key, probe=probe)
        self.state = State.MODIFY
        self.add_basic_interaction("fmc_config/v1/domain/<domainUUID>/devices/devicerecords/<deviceUUID>/etherchannelinterfaces", interface_clii.get_name(), interface_executor, NGIPSEtherChannelConfig.GENERAL_INTERFACE, True)
        self.probe = probe

        try:
            if interface_config.find('int_security_zone').get_state() != State.DESTROY:
                sz_id = interface_config.find("/SecurityZone/%s" % interface_clii.params['int_security_zone'].param_value['security_zone']).cli.idParam
                self.add_data_param({"id" : sz_id, "type" : "SecurityZone"}, "securityZone")
            else:
                self.add_data_param({}, "securityZone")
        except:
            pass


class NGIPSEtherChannelConfigs(InterfaceConfigs):
    def __init__(self, probe):
        super(NGIPSEtherChannelConfigs, self).__init__('InterfaceConfig', NGIPSEtherChannelConfig, probe=probe)
        self.probe = probe
